/* Copyright © 2024 Com2uS Platform Corp. All Rights Reserved. */
#include "Impl/HiveAnalyticsImpl.h"
#include "HiveUELogger.h"

hive::AnalyticsAdRevenue IHiveAnalyticsImpl::ConvertAnalyticsAdRevenue(const FHiveAnalyticsAdRevenue& AdRevenue)
{
    hive::AnalyticsAdRevenue analyticsAdRevenue;
    analyticsAdRevenue.revenue = AdRevenue.Revenue;
    analyticsAdRevenue.adPlatform = TCHAR_TO_UTF8(*AdRevenue.AdPlatform);
    analyticsAdRevenue.adUnitId = TCHAR_TO_UTF8(*AdRevenue.AdUnitId);
    analyticsAdRevenue.adType = TCHAR_TO_UTF8(*AdRevenue.AdType);
    analyticsAdRevenue.adPlacement = TCHAR_TO_UTF8(*AdRevenue.AdPlacement);
    analyticsAdRevenue.currency = TCHAR_TO_UTF8(*AdRevenue.Currency);
    return analyticsAdRevenue;
}

namespace
{
    hive::ConsentForm* CreateConsentFormFrom(const FHiveConsentForm& Form)
    {
        hive::ConsentForm* ConsentForm = new hive::ConsentForm();
        ConsentForm->id = TCHAR_TO_UTF8(*Form.Id);
        ConsentForm->title = TCHAR_TO_UTF8(*Form.Title);
        ConsentForm->content = TCHAR_TO_UTF8(*Form.Content);
        ConsentForm->retentionPeriod = Form.RetentionPeriod;

        if (Form.SubForms.Num() > 0) {
            for (const auto& SubForm : Form.SubForms) {
                ConsentForm->subForms.push_back(*(CreateConsentFormFrom(SubForm)));
            }
        }
        return ConsentForm;
    }
}

hive::ConsentMode* IHiveAnalyticsImpl::CreateConsentMode(const FHiveConsentMode& ConsentMode)
{
    hive::ConsentMode* consentMode = new hive::ConsentMode();
    consentMode->title = TCHAR_TO_UTF8(*ConsentMode.Title);
    consentMode->companyName = TCHAR_TO_UTF8(*ConsentMode.CompanyName);
    consentMode->privacyPolicy = TCHAR_TO_UTF8(*ConsentMode.PrivacyPolicy);

    if (ConsentMode.AdUserData.IsSet())
    {
        consentMode->adUserData = CreateConsentFormFrom(ConsentMode.AdUserData.GetValue());
    }

    if (ConsentMode.AdPersonalization.IsSet())
    {
        FHiveConsentForm _AdPersonalization = ConsentMode.AdPersonalization.GetValue();
        consentMode->adPersonalization = CreateConsentFormFrom(ConsentMode.AdPersonalization.GetValue());
    }

    if (ConsentMode.AnalyticsStorage.IsSet())
    {
        consentMode->analyticsStorage = CreateConsentFormFrom(ConsentMode.AnalyticsStorage.GetValue());
    }

    return consentMode;
}

void IHiveAnalyticsImpl::SendAnalyticsLog(const TSharedPtr<FJsonObject>& LogData)
{
    FString SerializedLogData;
    
    TSharedRef<TJsonWriter<>> JsonWriter = TJsonWriterFactory<>::Create(&SerializedLogData);
    if(FJsonSerializer::Serialize(LogData.ToSharedRef(), JsonWriter))
    {
        std::string strSerializedLogData = TCHAR_TO_UTF8(*SerializedLogData);
        picojson::value LogDataJsonValue;
        std::string err = picojson::parse(LogDataJsonValue, strSerializedLogData);

        if (!err.empty()) 
        {
            HIVE_LOG_WARNING(TEXT("picojson::parse error"));
            return;
        }

        if (!LogDataJsonValue.is<picojson::object>())
        {
            HIVE_LOG_WARNING(TEXT("picojson::parse error (not object)"));
            return;
        }

        picojson::object& LogDataJsonObject = LogDataJsonValue.get<picojson::object>();

        hive::Analytics::sendAnalyticsLog(LogDataJsonObject);
    }
    else
    {
        //TODO Warn Log
    }
}

void IHiveAnalyticsImpl::SetEnableTracker(const FString& Name, bool IsEnable)
{
    std::string StrName(TCHAR_TO_UTF8(*Name));
    hive::Analytics::setEnableTrackerWithName(StrName, IsEnable);
}

void IHiveAnalyticsImpl::SendEvent(const FString& EventName)
{
    std::string StrEventName(TCHAR_TO_UTF8(*EventName));
    hive::Analytics::sendEvent(StrEventName);
}

void IHiveAnalyticsImpl::SendEventWithAttributes(const FString& EventName, const FHiveAnalyticsAttributes& Attributes)
{
    hive::AnalyticsAttributes _attribute;

    FString SerializedCustomAttributes;

    TSharedRef<TJsonWriter<>> JsonWriter = TJsonWriterFactory<>::Create(&SerializedCustomAttributes);
    if (FJsonSerializer::Serialize(Attributes.CustomAttributes.ToSharedRef(), JsonWriter))
    {
        std::string strSerializedCustomAttributes = TCHAR_TO_UTF8(*SerializedCustomAttributes);
        picojson::value CustomAttributesJsonValue;
        std::string err = picojson::parse(CustomAttributesJsonValue, strSerializedCustomAttributes);

        if (!err.empty())
        {
            HIVE_LOG_WARNING(TEXT("picojson::parse error"));
            return;
        }

        if (!CustomAttributesJsonValue.is<picojson::object>())
        {
            HIVE_LOG_WARNING(TEXT("picojson::parse error (not object)"));
            return;
        }

        _attribute.customAttributes = CustomAttributesJsonValue.get<picojson::object>();
    }

    FString SerializedModuleSpecificAttributes;

    JsonWriter = TJsonWriterFactory<>::Create(&SerializedModuleSpecificAttributes);
    if (FJsonSerializer::Serialize(Attributes.ModuleSpecificAttributes.ToSharedRef(), JsonWriter))
    {
        std::string strSerializedModuleSpecificAttributes = TCHAR_TO_UTF8(*SerializedModuleSpecificAttributes);
        picojson::value ModuleSpecificAttributesJsonValue;
        std::string err = picojson::parse(ModuleSpecificAttributesJsonValue, strSerializedModuleSpecificAttributes);

        if (!err.empty())
        {
            HIVE_LOG_WARNING(TEXT("picojson::parse error"));
            return;
        }

        if (!ModuleSpecificAttributesJsonValue.is<picojson::object>())
        {
            HIVE_LOG_WARNING(TEXT("picojson::parse error (not object)"));
            return;
        }

        _attribute.moduleSpecificAttributes = ModuleSpecificAttributesJsonValue.get<picojson::object>();
    }

    std::string StrEventName(TCHAR_TO_UTF8(*EventName));
    hive::Analytics::sendEventWithAttributes(StrEventName, _attribute);
}

void IHiveAnalyticsImpl::SendAdRevenueEvent(const FHiveAnalyticsAdRevenue& AdRevenue)
{
    hive::Analytics::sendAdRevenueEvent(ConvertAnalyticsAdRevenue(AdRevenue));
}

void IHiveAnalyticsImpl::SendUserEntryFunnelsLogs(const FString& FunnelTrack, const FString& Tag)
{
    std::string StrFunnelTrack(TCHAR_TO_UTF8(*FunnelTrack));
    std::string StrTag(TCHAR_TO_UTF8(*Tag));
    hive::Analytics::sendUserEntryFunnelsLogs(StrFunnelTrack, StrTag);
}

void IHiveAnalyticsImpl::ShowConsentModeIfRequire(bool CheckCmp, const TOptional<FHiveConsentMode>& ConsentMode, const FHiveAnalyticsOnShowConsentModeDelegate& Delegate)
{
    hive::ConsentMode* consentMode = nullptr; // TODO(disker) convert shared_ptr
    if(ConsentMode.IsSet())
    {
        consentMode = CreateConsentMode(ConsentMode.GetValue());
    }

    hive::Analytics::showConsentModeIfRequire(CheckCmp, consentMode, [Delegate](hive::ResultAPI const & result, std::vector<hive::ConsentStatus> const & consentStatus) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);

            TArray<FHiveConsentStatus> ConsentStatusArray;

            if (consentStatus.size() > 0)
            {
                ConsentStatusArray.Reserve(consentStatus.size());
                
                for (const auto& element : consentStatus)
                {
                    FHiveConsentStatus ConsentStatus(element);
                    ConsentStatusArray.Emplace(ConsentStatus);
                }
            }

            Delegate.Execute(Result, ConsentStatusArray);            
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveAnalyticsImpl::ShowConsentMode(const TOptional<FHiveConsentMode>& ConsentMode, const FHiveAnalyticsOnShowConsentModeDelegate& Delegate)
{
    hive::ConsentMode* consentMode = nullptr;
    if(ConsentMode.IsSet())
    {
        consentMode = CreateConsentMode(ConsentMode.GetValue());
    }

    hive::Analytics::showConsentMode(consentMode, [Delegate](hive::ResultAPI const & result, std::vector<hive::ConsentStatus> const & consentStatus) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);

            TArray<FHiveConsentStatus> ConsentStatusArray;

            if (consentStatus.size() > 0)
            {
                ConsentStatusArray.Reserve(consentStatus.size());
                for (const auto& element : consentStatus)
                {
                    FHiveConsentStatus ConsentStatus(element);
                    ConsentStatusArray.Emplace(ConsentStatus);
                }
            }

            Delegate.Execute(Result, ConsentStatusArray);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}
