/* Copyright © 2024 Com2uS Platform Corp. All Rights Reserved. */
#include "Impl/HiveIAPV4Impl.h"
#include "HiveUELogger.h"

hive::IAPV4PurchaseParam IHiveIAPV4Impl::ConvertIAPV4PurchaseParam(const FHiveIAPV4PurchaseParam& Param)
{
    hive::IAPV4PurchaseParam result;
    result.marketPid = TCHAR_TO_UTF8(*Param.MarketPid);
    result.iapPayload = TCHAR_TO_UTF8(*Param.IapPayload);
    result.oldMarketPid = TCHAR_TO_UTF8(*Param.OldMarketPid);
    result.offerToken = TCHAR_TO_UTF8(*Param.OfferToken);
    return result;
}

void IHiveIAPV4Impl::MarketConnect(const FHiveIAPV4OnMarketConnectDelegate& Delegate)
{
    hive::IAPV4::marketConnect([Delegate](hive::ResultAPI const & result, std::vector<hive::IAPV4Type> const & marketIDs) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);

            TArray<EHiveIAPV4Type> IAPV4TypeArray;

            if (marketIDs.size() > 0)
            {
                IAPV4TypeArray.Reserve(marketIDs.size());
                
                for(const auto& element : marketIDs)
                {
                    IAPV4TypeArray.Add(static_cast<EHiveIAPV4Type>(element));
                }
            }
            
            Delegate.Execute(Result, IAPV4TypeArray);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveIAPV4Impl::GetMarketProductInfo(const TArray<FString>& MarketPidList, const FHiveIAPV4OnProductInfoDelegate& Delegate)
{
    std::vector<std::string> marketPidList;

    for(const auto& MarketPid : MarketPidList)
    {
        std::string StrMarketPid(TCHAR_TO_UTF8(*MarketPid));
        marketPidList.push_back(StrMarketPid);
    }

    hive::IAPV4::getMarketProductInfo(marketPidList, [Delegate](hive::ResultAPI const & result, std::vector<hive::IAPV4Product> const & iapV4ProductList, unsigned int balance) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);

            TArray<FHiveIAPV4Product> IAPV4ProductArray;

            if (iapV4ProductList.size() > 0)
            {
                IAPV4ProductArray.Reserve(iapV4ProductList.size());

                for(const auto& element : iapV4ProductList)
                {
                    FHiveIAPV4Product IAPV4Product(element);
                    IAPV4ProductArray.Emplace(IAPV4Product);
                }
            }
            
            Delegate.Execute(Result, IAPV4ProductArray, balance);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });

}

void IHiveIAPV4Impl::GetProductInfo(const FHiveIAPV4OnProductInfoDelegate& Delegate)
{
    hive::IAPV4::getProductInfo([Delegate](hive::ResultAPI const & result, std::vector<hive::IAPV4Product> const & iapV4ProductList, unsigned int balance) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);

            TArray<FHiveIAPV4Product> IAPV4ProductArray;

            if (iapV4ProductList.size() > 0)
            {
                IAPV4ProductArray.Reserve(iapV4ProductList.size());

                for(const auto& element : iapV4ProductList)
                {
                    FHiveIAPV4Product IAPV4Product(element);
                    IAPV4ProductArray.Emplace(IAPV4Product);
                }
            }
            
            Delegate.Execute(Result, IAPV4ProductArray, balance);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveIAPV4Impl::GetSubscriptionProductInfo(const FHiveIAPV4OnProductInfoDelegate& Delegate)
{
    hive::IAPV4::getSubscriptionProductInfo([Delegate](hive::ResultAPI const & result, std::vector<hive::IAPV4Product> const & iapV4ProductList, unsigned int balance) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);

            TArray<FHiveIAPV4Product> IAPV4ProductArray;

            if (iapV4ProductList.size() > 0)
            {
                IAPV4ProductArray.Reserve(iapV4ProductList.size());

                for(const auto& element : iapV4ProductList)
                {
                    FHiveIAPV4Product IAPV4Product(element);
                    IAPV4ProductArray.Emplace(IAPV4Product);
                }
            }
            
            Delegate.Execute(Result, IAPV4ProductArray, balance);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveIAPV4Impl::Purchase(const FString& MarketPid, const FString& IapPayload, const FHiveIAPV4OnPurchaseDelegate& Delegate)
{
    std::string StrMarketPid(TCHAR_TO_UTF8(*MarketPid));
    std::string StrIapPayload(TCHAR_TO_UTF8(*IapPayload));
    hive::IAPV4::purchase(StrMarketPid, StrIapPayload, [Delegate](hive::ResultAPI const & result, hive::IAPV4Receipt const & iapV4Receipt) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);

            FHiveIAPV4Receipt IAPV4Receipt(iapV4Receipt);
            
            Delegate.Execute(Result, IAPV4Receipt);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveIAPV4Impl::Purchase(const FString& MarketPid, const FString& IapPayload, int quantity, const FHiveIAPV4OnPurchaseDelegate& Delegate)
{
    std::string StrMarketPid(TCHAR_TO_UTF8(*MarketPid));
    std::string StrIapPayload(TCHAR_TO_UTF8(*IapPayload));
    hive::IAPV4::purchase(StrMarketPid, StrIapPayload, quantity, [Delegate](hive::ResultAPI const & result, hive::IAPV4Receipt const & iapV4Receipt) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);

            FHiveIAPV4Receipt IAPV4Receipt(iapV4Receipt);
            
            Delegate.Execute(Result, IAPV4Receipt);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveIAPV4Impl::PurchaseSubscription(const FHiveIAPV4PurchaseParam& Param, const FHiveIAPV4OnPurchaseDelegate& Delegate)
{
    hive::IAPV4::purchaseSubscription(ConvertIAPV4PurchaseParam(Param), [Delegate](hive::ResultAPI const & result, hive::IAPV4Receipt const & iapV4Receipt) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);

            FHiveIAPV4Receipt IAPV4Receipt(iapV4Receipt);
            
            Delegate.Execute(Result, IAPV4Receipt);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveIAPV4Impl::Restore(const FHiveIAPV4OnRestoreDelegate& Delegate)
{
    hive::IAPV4::restore([Delegate](hive::ResultAPI const & result, std::vector<std::reference_wrapper<hive::IAPV4Receipt>> const & receiptList){
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);

            TArray<FHiveIAPV4Receipt> IAPV4ReceiptList;

            if (receiptList.size() > 0)
            {
                IAPV4ReceiptList.Reserve(receiptList.size());

                for (const auto& item: receiptList)
                {
                    FHiveIAPV4Receipt Receipt(item);
                    IAPV4ReceiptList.Emplace(Receipt);
                }
            }
            
            Delegate.Execute(Result, IAPV4ReceiptList);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveIAPV4Impl::RestoreSubscription(const FHiveIAPV4OnRestoreDelegate& Delegate)
{
    hive::IAPV4::restoreSubscription([Delegate](hive::ResultAPI const & result, std::vector<std::reference_wrapper<hive::IAPV4Receipt>> const & receiptList){
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);

            TArray<FHiveIAPV4Receipt> IAPV4ReceiptList;

            if(receiptList.size() > 0)
            {
                IAPV4ReceiptList.Reserve(receiptList.size());

                for (const auto& item: receiptList)
                {
                    FHiveIAPV4Receipt Receipt(item);
                    IAPV4ReceiptList.Emplace(Receipt);
                }
            }
            
            Delegate.Execute(Result, IAPV4ReceiptList);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveIAPV4Impl::TransactionFinish(const FString& MarketPid, const FHiveIAPV4OnTransactionFinishDelegate& Delegate)
{
    std::string StrMarketPid(TCHAR_TO_UTF8(*MarketPid));
    hive::IAPV4::transactionFinish(StrMarketPid, [Delegate](hive::ResultAPI const & result, std::string marketPid) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);

            FString Pid = FString(UTF8_TO_TCHAR(marketPid.c_str()));
            
            Delegate.Execute(Result, Pid);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveIAPV4Impl::TransactionMultiFinish(const TArray<FString>& MarketPidList, const FHiveIAPV4OnTransactionMultiFinishDelegate& Delegate)
{
    std::vector<std::string> marketPidList;
    for (const auto& MarketPid: MarketPidList)
    {
        std::string StrMarketPid(TCHAR_TO_UTF8(*MarketPid));
        marketPidList.push_back(StrMarketPid);
    }

    hive::IAPV4::transactionMultiFinish(marketPidList, [Delegate](std::vector<hive::ResultAPI> const & resultList, std::vector<std::string> const & marketPidList) {
        if (Delegate.IsBound())
        {
            TArray<FHiveResultAPI> ResultList;
            if (resultList.size() > 0)
            {
                ResultList.Reserve(resultList.size());

                for (const auto& result: resultList)
                {
                    FHiveResultAPI Result(result);
                    ResultList.Emplace(Result);
                }
            }

            TArray<FString> MarketPidList;
            if (marketPidList.size() > 0)
            {
                MarketPidList.Reserve(marketPidList.size());

                for (const auto& marketPid: marketPidList)
                {
                    FString MarketPid = FString(UTF8_TO_TCHAR(marketPid.c_str()));
                    MarketPidList.Emplace(MarketPid);
                }
            }

            Delegate.Execute(ResultList, MarketPidList);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveIAPV4Impl::ShowMarketSelection(const FIAPV4OnMarketSelectionDelegate& Delegate)
{
    hive::IAPV4::showMarketSelection([Delegate](hive::ResultAPI const & result, std::vector<hive::IAPV4Type> const & marketIDs) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);

            TArray<EHiveIAPV4Type> IAPV4TypeArray;

            if (marketIDs.size() > 0)
            {
                IAPV4TypeArray.Reserve(marketIDs.size());

                for (const auto& element : marketIDs)
                {
                    IAPV4TypeArray.Add(static_cast<EHiveIAPV4Type>(element));
                }
            }
            
            Delegate.Execute(Result, IAPV4TypeArray);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveIAPV4Impl::GetBalanceInfo(const FHiveIAPV4OnBalanceDelegate& Delegate)
{
    hive::IAPV4::getBalanceInfo([Delegate](hive::ResultAPI const & result, int balance) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            Delegate.Execute(Result, balance);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveIAPV4Impl::ShowCharge(const FHiveIAPV4OnBalanceDelegate& Delegate)
{
    hive::IAPV4::showCharge([Delegate](hive::ResultAPI const & result, int balance) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            Delegate.Execute(Result, balance);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveIAPV4Impl::CheckPromotePurchase(const FHiveIAPV4OnCheckPromotePurchaseDelegate& Delegate)
{
    hive::IAPV4::checkPromotePurchase([Delegate](hive::ResultAPI const & result, std::string marketPid) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            FString MarketPid = FString(UTF8_TO_TCHAR(marketPid.c_str()));
            Delegate.Execute(Result, MarketPid);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

TOptional<EHiveIAPV4Type> IHiveIAPV4Impl::GetSelectedMarket()
{
    hive::IAPV4Type type = hive::IAPV4::getSelectedMarket();
    TOptional<EHiveIAPV4Type> Result = TOptional<EHiveIAPV4Type>(static_cast<EHiveIAPV4Type>(type));
    return Result;
}

FString IHiveIAPV4Impl::GetAccountUuid()
{
    std::string accountUuid = hive::IAPV4::getAccountUuid();
    FString AccountUuid = FString(UTF8_TO_TCHAR(accountUuid.c_str()));
    return AccountUuid;
}

void IHiveIAPV4Impl::ShowInAppMessages(int32 CategoryId, const FHiveIAPV4OnInAppMessagesDelegate& Delegate)
{
    hive::IAPV4::showInAppMessages(CategoryId, [Delegate](int responseCode) {
        if (Delegate.IsBound())
        {
            Delegate.Execute(responseCode);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveIAPV4Impl::GetConsumeInfoAgreement(const FHiveIAPV4OnGetConsumeInfoAgreementDelegate& Delegate)
{
    hive::IAPV4::getConsumeInfoAgreement([Delegate](hive::ResultAPI const& result, hive::IAPV4ConsumeInfoUserAgreement* const info) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            FHiveIAPV4ConsumeInfoUserAgreement Agreement = (info != nullptr) ? FHiveIAPV4ConsumeInfoUserAgreement(*info) : FHiveIAPV4ConsumeInfoUserAgreement();
            Delegate.Execute(Result, Agreement);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveIAPV4Impl::ShowConsumeInfoAgreement(const FHiveIAPV4OnShowConsumeInfoAgreementDelegate& Delegate)
{
    hive::IAPV4::showConsumeInfoAgreement([Delegate](hive::ResultAPI const& result, bool didAgree) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            Delegate.Execute(Result, didAgree);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}
