/* Copyright © 2024 Com2uS Platform Corp. All Rights Reserved. */
#include "Impl/HivePlatformHelperImpl.h"
#include "HiveUELogger.h"
#include "HIVESDK.h"

void IHivePlatformHelperImpl::RequestUserPermissions(const TArray<FString>& Requests, const FHivePlatformHelperOnUserPermissionsDelegate& Delegate)
{
    std::vector<std::string> requests;
    for (const auto& request : Requests)
    {
        std::string StrRequest(TCHAR_TO_UTF8(*request));
        requests.push_back(StrRequest);
    }

    hive::PlatformHelper::requestUserPermissions(requests,
        [Delegate](hive::ResultAPI const & result, std::vector<std::string> const & granted, std::vector<std::string> const & denied) {
            if (Delegate.IsBound())
            {
                FHiveResultAPI Result(result);

                TArray<FString> GrantedRequest;
                if (granted.size() > 0)
                {
                    GrantedRequest.Reserve(granted.size());

                    for (const auto& request : granted)
                    {
                        FString Request = FString(UTF8_TO_TCHAR(request.c_str()));
                        GrantedRequest.Emplace(Request);
                    }
                }

                TArray<FString> DeniedRequest;
                if (denied.size() > 0)
                {
                    DeniedRequest.Reserve(denied.size());
                    for (const auto& request : denied)
                    {
                        FString Request = FString(UTF8_TO_TCHAR(request.c_str()));
                        DeniedRequest.Emplace(Request);
                    }
                }

                Delegate.Execute(Result, GrantedRequest, DeniedRequest);
            }
            else
            {
                HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
            }
    });
}

void IHivePlatformHelperImpl::CompleteUpdate(int32 state)
{
    hive::PlatformHelper::completeUpdate(state);
}

void IHivePlatformHelperImpl::ShowInAppBrowser(const FHiveInAppBrowserParam& Param, const FHivePlatformHelperOnShowInAppBrowserDelegate& Delegate)
{
    std::string StrUrl(TCHAR_TO_UTF8(*Param.Url));
    
    hive::InAppBrowserParam param(StrUrl);

    if (!Param.ButtonColor.IsEmpty())
    {
        std::string StrButtonColor(TCHAR_TO_UTF8(*Param.ButtonColor));
        param.buttonColor = StrButtonColor;
    }

    if (!Param.NavigationColor.IsEmpty())
    {
        std::string StrNavigationColor(TCHAR_TO_UTF8(*Param.NavigationColor));
        param.navigationColor = StrNavigationColor;
    }

    param.urlBarHiding = Param.UrlBarHiding;
    param.autoRedirectToExternalBrowser = Param.AutoRedirectToExternalBrowser;

    hive::PlatformHelper::showInAppBrowser(param, [Delegate](hive::ResultAPI const & result) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            Delegate.Execute(Result);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });

}

void IHivePlatformHelperImpl::OpenBrowser(const FHiveOpenBrowserParam& Param)
{
    std::string StrUrl(TCHAR_TO_UTF8(*Param.Url));

    hive::OpenBrowserParam param(StrUrl);
    param.useIncognitoMode = Param.UseIncognitoMode;

    hive::PlatformHelper::openBrowser(std::move(param));
}

void IHivePlatformHelperImpl::ShowInAppWebView(const FHiveInAppWebViewParam& Param, const FHivePlatformHelperOnShowInAppWebViewDelegate& Delegate)
{
    std::string StrUrl(TCHAR_TO_UTF8(*Param.Url));
    hive::InAppWebViewParam param(StrUrl);
    param.postData = TCHAR_TO_UTF8(*Param.PostData);
    param.useUserSession = Param.UseUserSession;
    param.useGameWindow = Param.UseGameWindow;
    
    HIVESDKCEFImeEnable();
    hive::PlatformHelper::showInAppWebView(param, [Delegate](hive::ResultAPI const & result) {
        HIVESDKCEFImeDisable();
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            Delegate.Execute(Result);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHivePlatformHelperImpl::GetLaunchParameters(const FHivePlatformHelperOnGetLaunchParametersDelegate& Delegate)
{
    hive::PlatformHelper::getLaunchParameters([Delegate](hive::ResultAPI const& result, std::string parameters) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            FString Parameters(parameters.c_str());


            Delegate.Execute(Result, Parameters);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
        });
}
