/* Copyright © 2025 Com2uS Platform Corp. All Rights Reserved. */
#include "Types/HiveChatTypes.h"
#include "HiveChat.h"
#include "Impl/HiveChatImpl.h"

FString EHiveChannelTypeToString(EHiveChannelType type)
{
    switch (type)
    {
        case EHiveChannelType::PRIVATE:
            return TEXT("PRIVATE");
        case EHiveChannelType::PUBLIC:
            return TEXT("PUBLIC");
        case EHiveChannelType::GROUP:
            return TEXT("GROUP");
        case EHiveChannelType::UNKNOWN:
        default:
            return TEXT("UNKNOWN");
    }
}

EHiveChannelType FStringToChannelType(const FString& str)
{
    if (str == TEXT("PUBLIC"))
        return EHiveChannelType::PUBLIC;
    if (str == TEXT("PRIVATE"))
        return EHiveChannelType::PRIVATE;
    if (str == TEXT("GROUP"))
        return EHiveChannelType::GROUP;
    return EHiveChannelType::UNKNOWN;
}

FHiveChannel::FHiveChannel(const hive::Channel& Channel)
: channelId(UTF8_TO_TCHAR(Channel.channelId.c_str()))
, owner(UTF8_TO_TCHAR(Channel.owner.c_str()))
, channelName(UTF8_TO_TCHAR(Channel.channelName.c_str()))
, memberCount(Channel.memberCount)
, maxMemberCount(Channel.maxMemberCount)
, regTime(UTF8_TO_TCHAR(Channel.regTime.c_str()))
, regTimeMillis(Channel.regTimeMillis)
, chatHistoryAllowed(Channel.chatHistoryAllowed)
{
    switch (Channel.type) {
        case hive::ChannelType::UNKNOWN:
            type = EHiveChannelType::UNKNOWN;
            break;
        case hive::ChannelType::PRIVATE:
            type = EHiveChannelType::PRIVATE;
            break;
        case hive::ChannelType::PUBLIC:
            type = EHiveChannelType::PUBLIC;
            break;
        case hive::ChannelType::GROUP:
            type = EHiveChannelType::GROUP;
            break;
        default:
            type = EHiveChannelType::UNKNOWN;
            break;
    }
}

FString FHiveChannel::ToString() const
{
	return FString::Printf(TEXT("FHiveChannel { channelId: %s, type: %s, owner: %s, channelName: %s, memberCount: %d, maxMemberCount: %d, regTime: %s, regTimeMillis: %lld, chatHistoryAllowed: %s }"),
		*channelId,
		*(EHiveChannelTypeToString(type)),
		*owner,
		*channelName,
		memberCount,
		maxMemberCount,
		*regTime,
		regTimeMillis,
		chatHistoryAllowed ? TEXT("true") : TEXT("false")
	);
}

void FHiveChannel::Query(FHiveChannelMessageListQueryParams const & params, const FHiveChannelQueryDelegate& Delegate) const
{
    // CPP 타입으로 변환
    hive::ChannelMessageListQueryParams channelMessageListQueryParams = hive::ChannelMessageListQueryParams();
    channelMessageListQueryParams.prevSize = params.prevSize;
    channelMessageListQueryParams.nextSize = params.nextSize;
    FTCHARToUTF8 MessageIdConverter(*params.messageId);
    channelMessageListQueryParams.messageId = MessageIdConverter.Get();
    FTCHARToUTF8 OrderConverter(*params.order);
    channelMessageListQueryParams.order = OrderConverter.Get();
    
    // Channel 객체 생성 후 query 메서드 호출
    hive::Channel channel;
    FTCHARToUTF8 ChannelIdConverter(*channelId);
    channel.channelId = ChannelIdConverter.Get();
    
    channel.query(channelMessageListQueryParams, [Delegate](hive::ResultAPI const & result, hive::ChannelMessageListQueryResponse const & response) {
        FHiveResultAPI Result(result);
        FHiveChannelMessageListQueryResponse Response(response);
        if (Delegate)
        {
            Delegate(Result, Response);
        }
    });
}

void FHiveChannel::SetTranslationEnabled(bool enabled, const FHiveChannelTranslationSettingDelegate& Delegate) const
{
    // Channel 객체 생성 후 setTranslationEnabled 메서드 호출
    hive::Channel channel;
    FTCHARToUTF8 ChannelIdConverter(*channelId);
    channel.channelId = ChannelIdConverter.Get();
    
    channel.setTranslationEnabled(enabled, [Delegate](hive::ResultAPI const & result) {
        FHiveResultAPI Result(result);
        if (Delegate)
        {
            Delegate(Result);
        }
    });
}

void FHiveChannel::AddReaction(const FString& messageId, EHiveReactionType reactionType, const FHiveChannelReactionDelegate& Delegate) const
{
    // CPP 타입으로 변환
    hive::ReactionType _reactionType = hive::ReactionType::Unknown;
    switch (reactionType) {
        case EHiveReactionType::LIKE:
            _reactionType = hive::ReactionType::Like;
            break;
        case EHiveReactionType::UNKNOWN:
        default:
            _reactionType = hive::ReactionType::Unknown;
            break;
    }
    
    // Channel 객체 생성 후 addReaction 메서드 호출
    hive::Channel channel;
    FTCHARToUTF8 ChannelIdConverter(*channelId);
    channel.channelId = ChannelIdConverter.Get();
    FTCHARToUTF8 MessageIdConverter(*messageId);
    
    channel.addReaction(MessageIdConverter.Get(), _reactionType, [Delegate](hive::ResultAPI const & result, hive::ReactionType const & reactionType) {
        FHiveResultAPI Result(result);
        
        EHiveReactionType resultReactionType = EHiveReactionType::UNKNOWN;
        switch (reactionType) {
            case hive::ReactionType::Like:
                resultReactionType = EHiveReactionType::LIKE;
                break;
            case hive::ReactionType::Unknown:
            default:
                resultReactionType = EHiveReactionType::UNKNOWN;
                break;
        }
        
        if (Delegate)
        {
            Delegate(Result, resultReactionType);
        }
    });
}

void FHiveChannel::RemoveReaction(const FString& messageId, EHiveReactionType reactionType, const FHiveChannelReactionDelegate& Delegate) const
{
    // CPP 타입으로 변환
    hive::ReactionType _reactionType = hive::ReactionType::Unknown;
    switch (reactionType) {
        case EHiveReactionType::LIKE:
            _reactionType = hive::ReactionType::Like;
            break;
        case EHiveReactionType::UNKNOWN:
        default:
            _reactionType = hive::ReactionType::Unknown;
            break;
    }
    
    // Channel 객체 생성 후 removeReaction 메서드 호출
    hive::Channel channel;
    FTCHARToUTF8 ChannelIdConverter(*channelId);
    channel.channelId = ChannelIdConverter.Get();
    FTCHARToUTF8 MessageIdConverter(*messageId);
    
    channel.removeReaction(MessageIdConverter.Get(), _reactionType, [Delegate](hive::ResultAPI const & result, hive::ReactionType const & reactionType) {
        FHiveResultAPI Result(result);
        
        EHiveReactionType resultReactionType = EHiveReactionType::UNKNOWN;
        switch (reactionType) {
            case hive::ReactionType::Like:
                resultReactionType = EHiveReactionType::LIKE;
                break;
            case hive::ReactionType::Unknown:
            default:
                resultReactionType = EHiveReactionType::UNKNOWN;
                break;
        }
        
        if (Delegate)
        {
            Delegate(Result, resultReactionType);
        }
    });
}

FHiveChannelPage::FHiveChannelPage(const hive::ChannelPage& ChannelPage)
: size(ChannelPage.size)
, currentPage(ChannelPage.currentPage)
, totalElements(ChannelPage.totalElements)
, totalPages(ChannelPage.totalPages)
{ }

FString FHiveChannelPage::ToString() const
{
    return FString::Printf(TEXT("FHiveChannelPage { size: %d, currentPage: %d, totalElements: %d, totalPages: %d }"),
        size,
        currentPage,
        totalElements,
        totalPages
    );
}

FHiveMember::FHiveMember(const hive::Member& member)
: playerId(member.playerId)
, connectedTime(UTF8_TO_TCHAR(member.connectedTime.c_str()))
, connectedTimeMillis(member.connectedTimeMillis)
{ }

FString FHiveMember::ToString() const
{
    return FString::Printf(TEXT("FHiveMember { playerId: %lld, connectedTime: %s, connectedTimeMillis: %lld }"),
        playerId,
        *connectedTime,
        connectedTimeMillis
    );
}

FHiveBlockMember::FHiveBlockMember(const hive::BlockMember& BlockMember)
: playerId(BlockMember.playerId)
, blockedTime(UTF8_TO_TCHAR(BlockMember.blockedTime.c_str()))
, blockedTimeMillis(BlockMember.blockedTimeMillis)
{ }

FString FHiveBlockMember::ToString() const
{
    return FString::Printf(TEXT("FHiveBlockMember { playerId: %lld, blockedTime: %s, blockedTimeMillis: %lld }"),
        playerId,
        *blockedTime,
        blockedTimeMillis
    );
}

FHiveCreateChannelParams::FHiveCreateChannelParams(const hive::CreateChannelParams& CreateChannelParams)
: channelId(UTF8_TO_TCHAR(CreateChannelParams.channelId.c_str()))
, password(UTF8_TO_TCHAR(CreateChannelParams.password.c_str()))
, channelName(UTF8_TO_TCHAR(CreateChannelParams.channelName.c_str()))
, maxMemberCount(CreateChannelParams.maxMemberCount)
, chatHistoryAllowed(CreateChannelParams.chatHistoryAllowed)
{ 
    switch (CreateChannelParams.type) {
        case hive::ChannelType::UNKNOWN:
            type = EHiveChannelType::UNKNOWN;
            break;
        case hive::ChannelType::PRIVATE:
            type = EHiveChannelType::PRIVATE;
            break;
        case hive::ChannelType::PUBLIC:
            type = EHiveChannelType::PUBLIC;
            break;
        case hive::ChannelType::GROUP:
            type = EHiveChannelType::GROUP;
            break;
        default:
            type = EHiveChannelType::UNKNOWN;
            break;
    }
}

FHiveGetChannelsParams::FHiveGetChannelsParams(const hive::GetChannelsParams& GetChannelsParams)
: channelId(UTF8_TO_TCHAR(GetChannelsParams.channelId.c_str()))
, channelName(UTF8_TO_TCHAR(GetChannelsParams.channelName.c_str()))
, pageOrder(UTF8_TO_TCHAR(GetChannelsParams.pageOrder.c_str()))
, pageSize(GetChannelsParams.pageSize)
, pageNumber(GetChannelsParams.pageNumber)
{ 
    switch (GetChannelsParams.type) {
        case hive::ChannelType::UNKNOWN:
            type = EHiveChannelType::UNKNOWN;
            break;
        case hive::ChannelType::PRIVATE:
            type = EHiveChannelType::PRIVATE;
            break;
        case hive::ChannelType::PUBLIC:
            type = EHiveChannelType::PUBLIC;
            break;
        case hive::ChannelType::GROUP:
            type = EHiveChannelType::GROUP;
            break;
        default:
            type = EHiveChannelType::UNKNOWN;
            break;
    }
    
    switch (GetChannelsParams.sort) {
        case hive::SortType::ChannelId:
            sort = EHiveSortType::CHANNEL_ID;
            break;
        case hive::SortType::ChannelName:
            sort = EHiveSortType::CHANNEL_NAME;
            break;
        case hive::SortType::RegTime:
            sort = EHiveSortType::REG_TIME;
            break;
        case hive::SortType::Unknown:
        default:
            sort = EHiveSortType::UNKNOWN;
            break;
    }
}
FHiveEnterChannelParams::FHiveEnterChannelParams(const hive::EnterChannelParams& EnterChannelParams)
: channelId(UTF8_TO_TCHAR(EnterChannelParams.channelId.c_str()))
, password(UTF8_TO_TCHAR(EnterChannelParams.password.c_str()))
{ }

FHiveChannelSendMessageParams::FHiveChannelSendMessageParams(const hive::ChannelSendMessageParams& ChannelSendMessageParams)
: channelId(UTF8_TO_TCHAR(ChannelSendMessageParams.channelId.c_str()))
, message(UTF8_TO_TCHAR(ChannelSendMessageParams.message.c_str()))
, extraData(UTF8_TO_TCHAR(ChannelSendMessageParams.extraData.c_str()))
, replyMessageId(UTF8_TO_TCHAR(ChannelSendMessageParams.replyMessageId.c_str()))
{ 
    mentionedPlayerIds.Reserve(ChannelSendMessageParams.mentionedPlayerIds.size());
    for (const auto& playerId : ChannelSendMessageParams.mentionedPlayerIds)
    {
        mentionedPlayerIds.Add(playerId);
    }
}

FHiveDirectSendMessageParams::FHiveDirectSendMessageParams(const hive::DirectSendMessageParams& DirectSendMessageParams)
: toPlayerId(DirectSendMessageParams.toPlayerId)
, message(UTF8_TO_TCHAR(DirectSendMessageParams.message.c_str()))
, extraData(UTF8_TO_TCHAR(DirectSendMessageParams.extraData.c_str()))
{ }

FHiveEnteredMember::FHiveEnteredMember(const hive::EnteredMember& EnteredMember)
: channelId(UTF8_TO_TCHAR(EnteredMember.channelId.c_str()))
, playerId(EnteredMember.playerId)
, timestamp(UTF8_TO_TCHAR(EnteredMember.timestamp.c_str()))
, timestampMillis(EnteredMember.timestampMillis)
{ }

FString FHiveEnteredMember::ToString() const
{
    return FString::Printf(TEXT("FHiveEnteredMember { channelId: %s, playerId: %lld, timestamp: %s, timestampMillis: %lld }"),
        *channelId,
        playerId,
        *timestamp,
        timestampMillis
    );
}

FHiveExitedMember::FHiveExitedMember(const hive::ExitedMember& ExitedMember)
: channelId(UTF8_TO_TCHAR(ExitedMember.channelId.c_str()))
, playerId(ExitedMember.playerId)
, timestamp(UTF8_TO_TCHAR(ExitedMember.timestamp.c_str()))
, timestampMillis(ExitedMember.timestampMillis)
{ }

FString FHiveExitedMember::ToString() const
{
    return FString::Printf(TEXT("FHiveExitedMember { channelId: %s, playerId: %lld, timestamp: %s, timestampMillis: %lld }"),
        *channelId,
        playerId,
        *timestamp,
        timestampMillis
    );
}

FHiveDeletedChannel::FHiveDeletedChannel(const hive::DeletedChannel& DeletedChannel)
: channelId(UTF8_TO_TCHAR(DeletedChannel.channelId.c_str()))
, timestamp(UTF8_TO_TCHAR(DeletedChannel.timestamp.c_str()))
, timestampMillis(DeletedChannel.timestampMillis)
{ }

FString FHiveDeletedChannel::ToString() const
{
    return FString::Printf(TEXT("FHiveDeletedChannel { channelId: %s, timestamp: %s, timestampMillis: %lld }"),
        *channelId,
        *timestamp,
        timestampMillis
    );
}

FHiveChannelNoticeMessage::FHiveChannelNoticeMessage(const hive::ChannelNoticeMessage& ChannelNoticeMessage)
: messageId(UTF8_TO_TCHAR(ChannelNoticeMessage.messageId.c_str()))
, channelId(UTF8_TO_TCHAR(ChannelNoticeMessage.channelId.c_str()))
, from(UTF8_TO_TCHAR(ChannelNoticeMessage.from.c_str()))
, message(UTF8_TO_TCHAR(ChannelNoticeMessage.message.c_str()))
, timestamp(UTF8_TO_TCHAR(ChannelNoticeMessage.timestamp.c_str()))
, timestampMillis(ChannelNoticeMessage.timestampMillis)
{ }

FString FHiveChannelNoticeMessage::ToString() const
{
    return FString::Printf(TEXT("FHiveChannelNoticeMessage { channelId: %s, from: %s, message: %s, timestamp: %s, timestampMillis: %lld }"),
        *channelId,
        *from,
        *message,
        *timestamp,
        timestampMillis
    );
}

FHiveChannelMessage::FHiveChannelMessage(const hive::ChannelMessage& ChannelMessage)
: messageId(UTF8_TO_TCHAR(ChannelMessage.messageId.c_str()))
, from(ChannelMessage.from)
, extraData(UTF8_TO_TCHAR(ChannelMessage.extraData.c_str()))
, to(UTF8_TO_TCHAR(ChannelMessage.to.c_str()))
, message(UTF8_TO_TCHAR(ChannelMessage.message.c_str()))
, timestamp(UTF8_TO_TCHAR(ChannelMessage.timestamp.c_str()))
, timestampMillis(ChannelMessage.timestampMillis)
, translated(ChannelMessage.translated)
, translatedMessage(UTF8_TO_TCHAR(ChannelMessage.translatedMessage.c_str()))
, replyMessageId(UTF8_TO_TCHAR(ChannelMessage.replyMessageId.c_str()))
, replyMessage(UTF8_TO_TCHAR(ChannelMessage.replyMessage.c_str()))
, replyExtraData(UTF8_TO_TCHAR(ChannelMessage.replyExtraData.c_str()))
{ 
    // Convert reactions map
    for (const auto& reactionPair : ChannelMessage.reactions)
    {
        EHiveReactionType reactionType;
        switch (reactionPair.first) {
            case hive::ReactionType::Like:
                reactionType = EHiveReactionType::LIKE;
                break;
            case hive::ReactionType::Unknown:
            default:
                reactionType = EHiveReactionType::UNKNOWN;
                break;
        }
        TArray<int64> playerIds;
        for (const auto& playerId : reactionPair.second)
        {
            playerIds.Add(playerId);
        }
        reactions.Add(reactionType, playerIds);
    }
    
    // Convert mentionedPlayerIds array
    mentionedPlayerIds.Reserve(ChannelMessage.mentionedPlayerIds.size());
    for (const auto& playerId : ChannelMessage.mentionedPlayerIds)
    {
        mentionedPlayerIds.Add(playerId);
    }
}

FString FHiveChannelMessage::ToString() const
{
    FString reactionsStr = TEXT("{");
    for (const auto& reactionPair : reactions)
    {
        reactionsStr += FString::Printf(TEXT("%s: ["), *(EHiveReactionTypeToString(reactionPair.Key)));
        for (int32 i = 0; i < reactionPair.Value.Num(); ++i)
        {
            reactionsStr += FString::Printf(TEXT("%lld"), reactionPair.Value[i]);
            if (i < reactionPair.Value.Num() - 1) reactionsStr += TEXT(", ");
        }
        reactionsStr += TEXT("], ");
    }
    reactionsStr += TEXT("}");
    
    FString mentionedStr = TEXT("[");
    for (int32 i = 0; i < mentionedPlayerIds.Num(); ++i)
    {
        mentionedStr += FString::Printf(TEXT("%lld"), mentionedPlayerIds[i]);
        if (i < mentionedPlayerIds.Num() - 1) mentionedStr += TEXT(", ");
    }
    mentionedStr += TEXT("]");

    return FString::Printf(TEXT("FHiveChannelMessage { messageId: %s, from: %lld, extraData: %s, to: %s, message: %s, timestamp: %s, timestampMillis: %lld, translated: %s, translatedMessage: %s, reactions: %s, mentionedPlayerIds: %s, replyMessageId: %s, replyMessage: %s, replyExtraData: %s }"),
        *messageId,
        from,
        *extraData,
        *to,
        *message,
        *timestamp,
        timestampMillis,
        translated ? TEXT("true") : TEXT("false"),
        *translatedMessage,
        *reactionsStr,
        *mentionedStr,
        *replyMessageId,
        *replyMessage,
        *replyExtraData
    );
}

FHiveDirectMessage::FHiveDirectMessage(const hive::DirectMessage& DirectMessage)
: messageId(UTF8_TO_TCHAR(DirectMessage.messageId.c_str()))
, from(DirectMessage.from)
, extraData(UTF8_TO_TCHAR(DirectMessage.extraData.c_str()))
, to(DirectMessage.to)
, message(UTF8_TO_TCHAR(DirectMessage.message.c_str()))
, timestamp(UTF8_TO_TCHAR(DirectMessage.timestamp.c_str()))
, timestampMillis(DirectMessage.timestampMillis)
{ }

FString FHiveDirectMessage::ToString() const
{
    return FString::Printf(TEXT("FHiveDirectMessage { messageId: %s, from: %lld, extraData: %s, to: %lld, message: %s, timestamp: %s, timestampMillis: %lld }"),
        *messageId,
        from,
        *extraData,
        to,
        *message,
        *timestamp,
        timestampMillis
    );
}

FString EHiveReactionTypeToString(EHiveReactionType type)
{
    switch (type)
    {
        case EHiveReactionType::LIKE:
            return TEXT("LIKE");
        case EHiveReactionType::UNKNOWN:
        default:
            return TEXT("UNKNOWN");
    }
}

EHiveReactionType FStringToReactionType(const FString& str)
{
    if (str == TEXT("LIKE"))
        return EHiveReactionType::LIKE;
    return EHiveReactionType::UNKNOWN;
}

FHiveReaction::FHiveReaction(const hive::Reaction& Reaction)
: channelId(UTF8_TO_TCHAR(Reaction.channelId.c_str()))
, messageId(UTF8_TO_TCHAR(Reaction.messageId.c_str()))
, playerId(Reaction.playerId)
{
    switch (Reaction.type) {
        case hive::ReactionType::Like:
            type = EHiveReactionType::LIKE;
            break;
        case hive::ReactionType::Unknown:
        default:
            type = EHiveReactionType::UNKNOWN;
            break;
    }
}

FString FHiveReaction::ToString() const
{
    return FString::Printf(TEXT("FHiveReaction { channelId: %s, messageId: %s, playerId: %lld, type: %s }"),
        *channelId,
        *messageId,
        playerId,
        *(EHiveReactionTypeToString(type))
    );
}

FString EHiveSortTypeToString(EHiveSortType type)
{
    switch (type)
    {
        case EHiveSortType::CHANNEL_ID:
            return TEXT("CHANNEL_ID");
        case EHiveSortType::CHANNEL_NAME:
            return TEXT("CHANNEL_NAME");
        case EHiveSortType::REG_TIME:
            return TEXT("REG_TIME");
        case EHiveSortType::UNKNOWN:
        default:
            return TEXT("UNKNOWN");
    }
}

EHiveSortType FStringToSortType(const FString& str)
{
    if (str == TEXT("CHANNEL_ID") || str == TEXT("ChannelId"))
        return EHiveSortType::CHANNEL_ID;
    if (str == TEXT("CHANNEL_NAME") || str == TEXT("ChannelName"))
        return EHiveSortType::CHANNEL_NAME;
    if (str == TEXT("REG_TIME") || str == TEXT("RegTime"))
        return EHiveSortType::REG_TIME;
    return EHiveSortType::UNKNOWN;
}

FHiveTranslateParams::FHiveTranslateParams(const hive::TranslateParams& TranslateParams)
: message(UTF8_TO_TCHAR(TranslateParams.message.c_str()))
, sourceLanguage(UTF8_TO_TCHAR(TranslateParams.sourceLanguage.c_str()))
{
    targetLanguage.Reserve(TranslateParams.targetLanguage.size());
    for (const auto& lang : TranslateParams.targetLanguage)
    {
        targetLanguage.Add(UTF8_TO_TCHAR(lang.c_str()));
    }
}

FString FHiveTranslateParams::ToString() const
{
    FString targetLangs = TEXT("[");
    for (int32 i = 0; i < targetLanguage.Num(); ++i)
    {
        targetLangs += targetLanguage[i];
        if (i < targetLanguage.Num() - 1) targetLangs += TEXT(", ");
    }
    targetLangs += TEXT("]");
    
    return FString::Printf(TEXT("FHiveTranslateParams { message: %s, sourceLanguage: %s, targetLanguage: %s }"),
        *message,
        *sourceLanguage,
        *targetLangs
    );
}

FHiveTranslationData::FHiveTranslationData(const hive::TranslationData& TranslationData)
{
    for (const auto& translation : TranslationData.translations)
    {
        translations.Add(UTF8_TO_TCHAR(translation.first.c_str()), UTF8_TO_TCHAR(translation.second.c_str()));
    }
}

FString FHiveTranslationData::ToString() const
{
    FString translationsStr = TEXT("{");
    for (const auto& translation : translations)
    {
        translationsStr += FString::Printf(TEXT("%s: %s, "), *translation.Key, *translation.Value);
    }
    translationsStr += TEXT("}");
    
    return FString::Printf(TEXT("FHiveTranslationData { translations: %s }"), *translationsStr);
}

FHiveChannelMessageListQueryParams::FHiveChannelMessageListQueryParams(const hive::ChannelMessageListQueryParams& Params)
: prevSize(Params.prevSize)
, nextSize(Params.nextSize)
, messageId(UTF8_TO_TCHAR(Params.messageId.c_str()))
, order(UTF8_TO_TCHAR(Params.order.c_str()))
{ }

FString FHiveChannelMessageListQueryParams::ToString() const
{
    return FString::Printf(TEXT("FHiveChannelMessageListQueryParams { prevSize: %d, nextSize: %d, messageId: %s, order: %s }"),
        prevSize,
        nextSize,
        *messageId,
        *order
    );
}

FHiveChannelMessageListQueryResponse::FHiveChannelMessageListQueryResponse(const hive::ChannelMessageListQueryResponse& Response)
: hasNext(Response.hasNext)
, nextMessageId(UTF8_TO_TCHAR(Response.nextMessageId.c_str()))
, hasPrev(Response.hasPrev)
, prevMessageId(UTF8_TO_TCHAR(Response.prevMessageId.c_str()))
{
    content.Reserve(Response.content.size());
    for (const auto& message : Response.content)
    {
        content.Add(FHiveChannelMessage(message));
    }
}

FString FHiveChannelMessageListQueryResponse::ToString() const
{
    return FString::Printf(TEXT("FHiveChannelMessageListQueryResponse { hasNext: %s, nextMessageId: %s, hasPrev: %s, prevMessageId: %s, content.Num: %d }"),
        hasNext ? TEXT("true") : TEXT("false"),
        *nextMessageId,
        hasPrev ? TEXT("true") : TEXT("false"),
        *prevMessageId,
        content.Num()
    );
}

FHiveNoticeMessage::FHiveNoticeMessage(const hive::NoticeMessage& NoticeMessage)
: messageId(UTF8_TO_TCHAR(NoticeMessage.messageId.c_str()))
, from(UTF8_TO_TCHAR(NoticeMessage.from.c_str()))
, message(UTF8_TO_TCHAR(NoticeMessage.message.c_str()))
, timestamp(UTF8_TO_TCHAR(NoticeMessage.timestamp.c_str()))
, timestampMillis(NoticeMessage.timestampMillis)
{ }

FString FHiveNoticeMessage::ToString() const
{
    return FString::Printf(TEXT("FHiveNoticeMessage { messageId: %s, from: %s, message: %s, timestamp: %s, timestampMillis: %lld }"),
        *messageId,
        *from,
        *message,
        *timestamp,
        timestampMillis
    );
}
