/**
 * @file    HIVE_Analytics.h
 *
 * @date        2016-2019
 * @copyright  Copyright © GAMEVIL COM2US PLATFORM Inc. All Right Reserved.
 * @author     ryuvsken
 * @since      4.0.0 
 * @defgroup Analytics
 * @{
 * \~korean  앱과 사용자를 트래킹하고 분석하기 위한 기능 모음<br/>
 * (User Tracking Tool Wrapper & Callect Analytics Log API)<br/><br/>
 * \~english A collection of features for tracking and analyzing apps and users<br/>
 * (User Tracking Tool Wrapper & Callect Analytics Log API)<br/><br/>
 */

#ifndef __HIVE_ANALYTICS_H__
#define  __HIVE_ANALYTICS_H__


#include "HIVE_ResultAPI.h"
#include <map>

NS_HIVE_BEGIN

class AnalyticsAdRevenue;
class ConsentStatus;
class ConsentForm;
class ConsentMode;
class AnalyticsAttributes;
class IAnalyticsSpecific;
class AirbridgeSpecific;

#define HIVE_SDK_ANALYTICS_CONSENT_TYPE_AD_STORAGE "ad_storage"
#define HIVE_SDK_ANALYTICS_CONSENT_TYPE_AD_USER_DATA "ad_user_data"
#define HIVE_SDK_ANALYTICS_CONSENT_TYPE_AD_PERSONALIZATION "ad_personalization"
#define HIVE_SDK_ANALYTICS_CONSENT_TYPE_ANALYTICS_STORAGE "analytics_storage"
#define HIVE_SDK_ANALYTICS_CONSENT_GRANTED "granted"
#define HIVE_SDK_ANALYTICS_CONSENT_DENIED "denied"

/**
* \~korean 앱과 사용자를 트래킹하고 분석하기 위한 기능 모음<br/>
* (User Tracking Tool Wrapper & Callect Analytics Log API)<br/><br/>
*    
* \~english A collection of features for tracking and analyzing apps and users<br/>
* (User Tracking Tool Wrapper & Callect Analytics Log API)<br/><br/>
* \~
* @since       4.0.0 
* @author ryuvsken
* @ingroup Analytics
*/
class HIVESDK_DLLEXPORT Analytics
{
public:

	/**
 	* \~korean 사용자 분석을 위한 서드 파티 트래커 형태
	* 
	* \~english Tracker type for user analysis
	* \~
	* @ingroup Analytics
	* @author ryuvsken
	*/
	enum class TrackingType
	{
        ADJUST
		, SINGULAR
		, APPSFLYER
		, UNKNOWN
	};

    typedef std::function<void(ResultAPI const & result, std::vector<ConsentStatus> const & consentStatus)> onShowConsentMode;

	/**
	* \~korean 분석용 로그 전송.
	* 
	* @param logData 로그 데이터
    *
    * @return bool 로그 큐가 가득차 로그를 쌓지 못했을 경우 false를 반환한다.
	* 
	* \~english Send log for analysis.
	* 
	* @param logData Log data
    *
    * @return bool Returns false if the log queue is full and the log is not stacked.
	* \~
	* @ingroup Analytics
	*/
	static bool sendAnalyticsLog(picojson::object & logData);


	/**
	* \~korean 사용자 분석을 위한 사용자 정보 트래커 사용 유무 설정
	* 
	* @param trackingType 사용자 분석을 위한 사용자 정보 트래커 형태
	* @param isEnable 트래커 사용 유무
	* 
	* \~english Set whether to use User Information Tracker for user analysis
	* 
	* @param trackingType User information tracker type
	* @param isEnable whether to use User Information Tracker 
	* \~
	* @ingroup Analytics
	*/
	static void setEnableTracker(TrackingType trackingType, bool isEnable);


	/**
	* \~korean 사용자 분석을 위한 사용자 정보 트래커 사용 유무 설정
	* 
	* @param name 사용자 분석을 위한 사용자 정보 트래커 형태
	* @param isEnable 트래커 사용 유무
	* 
	* \~english Set whether to use User Information Tracker for user analysis
	* 
	* @param name User information tracker type
	* @param isEnable whether to use User Information Tracker 
	* \~
	* @ingroup Analytics
	*/
	static void setEnableTrackerWithName(std::string name, bool isEnable);
	


	/**
	* \~korean 사용자 분석을 위한 사용자 정보 이벤트 설정
	* 
	* @param eventName 사용자 정보 이벤트
	* 
	* \~english Send event for user analysis
	* 
	* @param eventName Event name
	* \~
	* @ingroup Analytics
	*/
	static void sendEvent(std::string eventName);
    
    /**
    * \~korean 사용자 분석을 위한 사용자 정보 이벤트 설정
    *
    * @param eventName 사용자 정보 이벤트
    * @param attributes 추가 데이터
    *
    * \~english Send event for user analysis
    *
    * @param eventName Event name
    * @param attributes Additional Data
    * \~
    * @ingroup Analytics
    */
    static void sendEventWithAttributes(std::string eventName, AnalyticsAttributes const& attributes);

    /**
    * \~korean 광고 수익 측정 이벤트 설정
    *
    * @param analyticsAdRevenue	광고 수익 측정 데이터
    *
    * \~english Send events for ad revenue measurement
    *
    * @param analyticsAdRevenue	Ad revenue data
    *
    * \~
    * @ingroup Analytics
    */
    static void sendAdRevenueEvent(AnalyticsAdRevenue analyticsAdRevenue);
    
    static unsigned int getRemainAnalyticsLogCount();
    
    
    /**
     * \~korean
     * @brief 사용자의 퍼널 분석을 위한 지표를 전송한다.
     *
     * 사전정의된 퍼널 목록
     * <table>
     *     <th>
     *         <td>구간명</td><td>퍼널값</td><td>HIVE SDK 자동전송 유무</td>
     *     </th>
     *     <tr>
     *         <td>권한고지</td><td>410</td><td>YES</td>
     *     </tr>
     *     <tr>
     *         <td>고지팝업</td><td>420</td><td>YES</td>
     *     </tr>
     *     <tr>
     *         <td>약관동의</td><td>430</td><td>YES</td>
     *     </tr>
     *     <tr>
     *         <td>게임서버선택</td><td>500</td><td>YES</td>
     *     </tr>
     *     <tr>
     *         <td>서버점검팝업</td><td>600</td><td>YES</td>
     *     </tr>
     *     <tr>
     *         <td>추가다운로드</td><td>700</td><td><b>NO</b></td>
     *     </tr>
     *     <tr>
     *         <td>추가다운로드 완료</td><td>800</td><td><b>NO</b></td>
     *     </tr>
     *     <tr>
     *         <td>로그인</td><td>900</td><td>YES</td>
     *     </tr>
     *     <tr>
     *         <td>전면배너</td><td>1000</td><td>YES</td>
     *     </tr>
     * </table>
     *
     * @param funnelTrack 사전정의된 퍼널의 값
     * @param optionTag 옵션으로 추가 전달할 값
     */
    static void sendUserEntryFunnelsLogs(std::string funnelTrack, std::string optionTag);
    
    /**
     * \~korean
     * 각 Provider 별로 서드 파티 동의 관리 플랫폼(CMP) 또는 맞춤 솔루션으로 구현한 최종 사용자 동의 상태를 설정한다.
     * Hive Analytics 에서 지원하는 동의 유형은 Analytics.CONSENT_TYPE_AD_STORAGE, Analytics.CONSENT_TYPE_AD_USER_DATA, Analytics.CONSENT_TYPE_AD_PERSONALIZATION, Analytics.CONSENT_TYPE_ANALYTICS_STORAGE 가 있다.
     * 각 유형별로 Analytics.CONSENT_GRANTED, Analytics.CONSENT_DENIED 설정을 할 수 있다.
     * 이 설정은 입력된 서드 파티 트래커에 적용된다.
     *
     * @param providerName: 지원하는 Provider의 이름
     * @param settings 지원되는 동의 유형 키는 CONSENT_TYPE_AD_STORAGE, CONSENT_TYPE_AD_USER_DATA, CONSENT_TYPE_AD_PERSONALIZATION, CONSENT_TYPE_ANALYTICS_STORAGE. 유효한 값은 CONSENT_GRANTED, CONSENT_DENIED.
     */
    static void setConsent(std::string providerName, std::map<std::string, std::string> settings);
    
    /**
     * \~korean
     * 각 provider별 setConsent 으로 설정한 동의 유형별 동의 상태를 반환한다.
     * 이 값은 setConsent 로 설정했던 최근값이다.
     *
     * @param providerName: 지원하는 Provider의 이름
     */
    static std::map<std::string, std::string> getConsent(std::string providerName);
    
    /**
     * \~korean
     * 개인 정보를 사용하는 데 최종 사용자 동의를 받기 위한 UI를 노출합니다.
     * 이미 동의를 한 경우 노출되지 않으며 기한이 만료되었거나 추가 동의가 필요한 경우 노출 됩니다.
     * [consentMode]를 null 로 입력할 경우 Hive SDK 의 Analytics Provider 추가 여부에 따른 동의 목록을 자동으로 생성합니다.
     * Hive SDK 에서 판단하는 [ConsentForm.id] 은"google", "adjust", "appsflyer", "singular", "airbridge" 입니다.
     *
     * @param checkCmp true 일 경우 서드 파티 동의 관리 플랫폼(CMP) 또는 맞춤 솔루션으로 TCF 규격에 맞게 저장된 파일이 있다면 해당 파일을 읽어 파일에 있는 업체는 동의 설정한 것으로 간주하게 됩니다.
     * @param consentMode UI에 띄울 내용과 Provider를 모두 커스텀 할 수 있게 됩니다.
     * @param listener [ConsentModeListener] 로 UI에서 동의한 결과를 [ConsentStatus] 배열로 응답합니다. 취소할 경우 [ResultAPI.CANCELED] [ResultAPI.Code.AnalyticsConsentModeCanceled] 로 응답합니다.
     *
     * @see [showConsentMode]
     */
    static void showConsentModeIfRequire(bool checkCmp, ConsentMode* consentMode, onShowConsentMode listener);
        
    /**
     * \~korean
     * 개인 정보를 사용하는 데 최종 사용자 동의를 받기 위한 UI를 노출합니다.
     * 기존에 동의한 내역에 따라 동의 상태값이 설정되어져 있습니다.
     * [consentMode]를 null 로 입력할 경우 Hive SDK 의 Analytics Provider 추가 여부에 따른 동의 목록을 자동으로 생성합니다.
     * Hive SDK 에서 판단하는 [ConsentForm.id] 은"google", "adjust", "appsflyer", "singular", "airbridge" 입니다.
     *
     * @param consentMode UI에 띄울 내용과 Provider를 모두 커스텀 할 수 있게 됩니다.
     * @param listener [ConsentModeListener] 로 UI에서 동의한 결과를 [ConsentStatus] 배열로 응답합니다. 취소할 경우 [ResultAPI.CANCELED] [ResultAPI.Code.AnalyticsConsentModeCanceled] 로 응답합니다.
     *
     * @see [showConsentModeIfRequire]
     */
    static void showConsentMode(ConsentMode* consentMode, onShowConsentMode listener);


	// Native 영역에서 호출된 요청을 처리하기 위한 플러그인 내부 코드
	static void executeEngine(picojson::value jsonParam);
};

class HIVESDK_DLLEXPORT AnalyticsAdRevenue
{
public:

	double revenue;				///< \~korean  광고 노출당 발생하는 광고 수익  \~english Ad impression revenue

	std::string adPlatform;		///< \~korean  광고 네트워크 플랫폼 \~english Ad network platform

	std::string adUnitId;		///< \~korean 광고 유닛 아이디  \~english Ad Unit Id

	std::string adType;    		///< \~korean 광고 노출 타입  \~english  The type of ads

	std::string adPlacement;    ///< \~korean 광고 노출 위치  \~english ad placement

	std::string currency;     	///< \~korean 통화 코드(ISO_4217 형식 문자열)  \~english ISO_4217 format string (ex. "USD")

	AnalyticsAdRevenue();

	AnalyticsAdRevenue(double revenue, std::string adPlatform, std::string adUnitId, std::string adType, std::string adPlacement, std::string currency);

	AnalyticsAdRevenue(picojson::value jsonParam);

	picojson::object toJson() const;

	std::string toString() const;
};

class HIVESDK_DLLEXPORT ConsentStatus {
public:
    std::string id;                   ///< \~korean  동의 대상에 해당하는 Provider 이름입니다. 따로 설정하지 않을 경우 "google", "adjust", "appsflyer", "singular" 중 하나가 설정됩니다.
    std::string granted;            ///< \~korean  동의 대상의 동의 여부 입니다. 최초로 아무 설정도 하지 않았을 경우 "not-set" 혹은 empty string 이 됩니다.
    long consentDate;                           ///< \~korean  동의를 결정한 시각입니다. 단위는 milli second 입니다.
    long expiredDate;                           ///< \~korean  동의가 만료되는 시각입니다. [ConsentForm.retentionPeriod] 설정에 따라 [consentDate] 에서 더해진 시각이 됩니다. 단위는 milli second 입니다.7
    
    ConsentStatus();
    ConsentStatus(picojson::value jsonParam);
    
    picojson::object toJson() const;
    std::string toString() const;
};

class HIVESDK_DLLEXPORT ConsentForm {
public:
    std::string id;                   ///< \~korean  동의 대상에 해당하는 Provider 이름입니다.
    int retentionPeriod;                        ///< \~korean  동의 후 만료까지의 시간을 설정합니다. 단위는 개월 수 입니다. [providerName] "main" 은 999 입니다.
    std::string title;                 ///< \~korean  타이틀 문구 입니다.
    std::string content;           ///< \~korean 세부 내용 입니다.
    std::vector<ConsentForm> subForms;           ///< \~korean 동의 대상의 하위 항목입니다.
    
    ConsentForm();
    ConsentForm(picojson::value jsonParam);
    
    picojson::object toJson() const;
    std::string toString() const;
};

class HIVESDK_DLLEXPORT ConsentMode {
public:
    std::string title;                          ///< \~korean  동의 배너의 상단에 표시되는 문구 입니다.
    std::string companyName;                    ///< \~korean  기본 정보로 노출시 회사이름이 필요한 [title] 에 설정됩니다.
    std::string privacyPolicy;                  ///< \~korean  동의 배너 하단의 PrivacyPolicy 버튼을 눌렀을때 동작하는 URL 입니다.
    ConsentForm* adUserData;                       ///< \~korean 사용자 정보 저장 동의 항목입니다.
    ConsentForm* adPersonalization;                       ///< \~korean  개인 정보를 공유할 업체별 동의 항목 입니다.
    ConsentForm* analyticsStorage;                       ///< \~korean  광고 정보 분석 동의 항목 입니다.
    
    ConsentMode();
    ~ConsentMode();
    ConsentMode(picojson::value jsonParam);
    
    picojson::object toJson() const;
    std::string toString() const;
};

/**
 * @brief Analytics 추가 데이터
 *
 * @see customAttribute: 커스텀 추가 데이터 / Custom attributes
*/
class HIVESDK_DLLEXPORT AnalyticsAttributes
{
public:
    picojson::object customAttributes;
    picojson::object moduleSpecificAttributes;
    
    AnalyticsAttributes();

    void addCustomAttribute(const std::string& key, const picojson::value& value);
    void addModuleSpecificAttribute(const IAnalyticsSpecific& specific);

    virtual picojson::object toJson() const;
    virtual std::string toString() const;
    
    virtual ~AnalyticsAttributes() = default; // 가상 소멸자
};

class HIVESDK_DLLEXPORT IAnalyticsSpecific
{
public:
    std::string providerName;
    
    IAnalyticsSpecific();
    
    virtual picojson::object toJson() const;
    virtual std::string toString() const;
    
    virtual ~IAnalyticsSpecific() = default; // 가상 소멸자
};

/**
 * @brief Analytics 추가 데이터 (Airbridge 추가 데이터 포함)
 *
 * @see semanticAttribute: Semantic 추가 데이터 / Semantic attributes
 * @see customAttribute: 커스텀 추가 데이터 / Custom attributes
*/
class HIVESDK_DLLEXPORT AirbridgeSpecific : public IAnalyticsSpecific
{
public:
    picojson::object semanticAttributes;

    AirbridgeSpecific();

    void setSemanticAction(const std::string& value);
    void setSemanticLabel(const std::string& value);
    void setSemanticValue(const double value);
    void addSemanticAttribute(const std::string& key, const picojson::value& value);
    picojson::object toJson() const override; // 부모 메서드 재정의
    std::string toString() const override;    // 부모 메서드 재정의
};

NS_HIVE_END        // namespace hive

#endif		// __HIVE_ANALYTICS_H__
/** @} */
