/**
 * @file    HIVE_Chat.h
 *
 * @date        2024
 * @copyright    Copyright © GAMEVIL COM2US PLATFORM Inc. All Right Reserved.
 * @author        herohjk
 * @since        4.25.0.0
 * @defgroup HiveChat
 * @{
 * \~korean
 * HIve Chat<br/><br/>
 * \~english
 * This class manage HIVE Chat Service<br/><br/>
 * \~
 */

#ifndef __HIVE_CHAT_H__
#define __HIVE_CHAT_H__

#include "HIVE_ResultAPI.h"
#include <cstdint>      // int64_t
#include <functional>   // std::function
#include <map>          // std::map
#include <string>       // std::string
#include <vector>       // std::vector
#include <mutex>


NS_HIVE_BEGIN

class ResultAPI;
enum class ChannelType;
enum class SortType;
enum class ReactionType;
class Channel;
class ChannelPage;
class Member;
class BlockMember;
class CreateChannelParams;
class GetChannelsParams;
class EnterChannelParams;
class ChannelSendMessageParams;
class DirectSendMessageParams;
class TranslateParams;
class ChannelListener;
class DirectMessageListener;
class UserListener;
class ConnectionListener;
class ChannelMessage;
class ChannelNoticeMessage;
class NoticeMssage;
class DirectMessage;
class EnteredMember;
class ExitedMember;
class DeletedChannel;
class ChannelMessageListQueryParams;
class ChannelMessageListQueryResponse;
class CustomDataListener;
class TranslationData;
class Reaction;

/**
 *  \~korean
 * HiveChat은 실시간, 그룹, 1:1 채팅, 유저 간 차단 기능 등 게임에서 원활한 채팅을 할 수 있도록 필요한 핵심 기능을 제공하기 때문에 유저 간 협력을 증가시켜 게임 플레이 만족도를 높일 수 있습니다.<br/>
 * 별도 복잡한 개발 없이도 Hive Chat에서 제공하는 AI 채팅 필터링 기능으로 금칙어 및 광고성 채팅을 감지하고 차단해 유저의 플레이 환경을 개선할 수 있습니다.<br/>
 *
 *  \~english
 * HiveChat provides essential features for smooth chatting in games, such as real-time, group, 1:1 chat, and user blocking functions, which can increase user collaboration and improve game play satisfaction.<br/>
 * Without additional complex development, you can use the AI chat filtering feature provided by Hive Chat to detect and block prohibited words and advertising chats, improving the user's play environment.<br/>
 *  \~
 *
 * @since   4.25.0.0
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT Chat {
public:
    // Callback types
    typedef std::function<void(ResultAPI const & result)> onResult;
    typedef std::function<void(ResultAPI const & result)> onCreateChannel;
    typedef std::function<void(ResultAPI const & result)> onDeleteChannel;
    typedef std::function<void(ResultAPI const & result)> onEnterChannel;
    typedef std::function<void(ResultAPI const & result)> onExitChannel;
    typedef std::function<void(ResultAPI const & result)> onBlockMember;
    typedef std::function<void(ResultAPI const & result)> onUnblockMember;
    typedef std::function<void(ResultAPI const & result)> onChannelTranslationSetting;
    typedef std::function<void(ResultAPI const & result)> onConnect;
    typedef std::function<void(ResultAPI const & result)> onDisconnect;
    typedef std::function<void(ResultAPI const & result, std::vector<std::string> const & channelIds, std::vector<std::string> const & failChannelIds)> onReconnect;
    typedef std::function<void(ResultAPI const & result, std::vector<Channel> const & channels, ChannelPage const & channelPage)> onGetChannels;
    typedef std::function<void(ResultAPI const & result, Channel const & channel, std::vector<Member> const & members)> onGetChannelInfo;
    typedef std::function<void(ResultAPI const & result, std::vector<Member> const & members)> onGetChannelMembers;
    typedef std::function<void(ResultAPI const & result, std::vector<Channel> const & channels)> onGetChannelsByUser;
    typedef std::function<void(ResultAPI const & result, std::vector<BlockMember> const & blockMembers)> onGetBlockMembers;
    typedef std::function<void(ResultAPI const & result, ChannelSendMessageParams const & params)> onChannelSendMessage;
    typedef std::function<void(ResultAPI const & result, DirectSendMessageParams const & params)> onDirectSendMessage;
    typedef std::function<void(ResultAPI const & result, ChannelMessageListQueryResponse const & response)> onGetChannelMessageHistory;
    typedef std::function<void(ResultAPI const & result, TranslationData const & translationData)> onTranslate;
    typedef std::function<void(ResultAPI const & result, ReactionType const & reactionType)> onAddReaction;
    typedef std::function<void(ResultAPI const & result, ReactionType const & reactionType)> onRemoveReaction;
    
    // MARK: Channel APIs
    /**
     * \~korean
     * 채널 생성<br/>
     *
     * 채널을 생성합니다.<br/>
     * 채널명, 최대 인원수, 타입을 지정할수 있으며, PUBLIC이 아니라면 비밀번호 또한 함께 지정해야합니다. <br/>
     * 채널 생성에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Create a channel<br/>
     * Channel Name, Maximum Number of People, and Type can be specified, and if it is not PUBLIC, a password must also be specified.<br/>
     * The result of creating a channel is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void createChannel(CreateChannelParams const & params, onCreateChannel listener);
    
    /**
     * \~korean
     * 채널 목록 조회<br/>
     * 접근 가능한 채널 목록을 조회합니다.<br/>
     * 채널 타입, 채널명, 페이지 순서, 페이지 사이즈, 페이지 번호를 지정할 수 있습니다.<br/>
     * 채널 목록 조회에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Get channel list<br/>
     * Retrieves a list of accessible channels.<br/>
     * You can specify the channel type, channel name, page order, page size, and page number.<br/>
     * The result of the channel list retrieval is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void getChannels(GetChannelsParams const * params, onGetChannels listener);
    
    /**
     * \~korean
     * 채널 정보 조회<br/>
     * 채널 ID를 입력받아, 해당 채널의 정보를 조회합니다.<br/>
     * 조회 결과로 해당 채널의 정보와, 채널에 참여중인 멤버들의 목록이 전달됩니다.<br/>
     *
     * \~english
     * Get channel information<br/>
     * Receives the channel ID and retrieves the information of the corresponding channel.<br/>
     * The result of the query includes information about the channel and a list of members participating in the channel.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void getChannelInfo(std::string const & channelId, onGetChannelInfo listener);
    
    /**
     * \~korean
     * 채널 멤버 조회<br/>
     * 채널 ID를 입력받아, 해당 채널의 멤버 목록을 조회합니다.<br/>
     * 조회 결과로 해당 채널의 멤버 목록이 전달됩니다.<br/>
     *
     * \~english
     * Get channel members<br/>
     * Receives the channel ID and retrieves the list of members in the corresponding channel.<br/>
     * The result of the query includes a list of members in the corresponding channel.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
    */
    static void getChannelMembers(std::string const & channelId, onGetChannelMembers listener);
    
    /**
     * \~korean
     * 채널 삭제<br/>
     * 채널 ID를 입력받아, 해당 채널을 삭제합니다.<br/>
     * 채널 삭제에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Delete channel<br/>
     * Receives the channel ID and deletes the corresponding channel.<br/>
     * The result of deleting the channel is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void deleteChannel(std::string const & channelId, onDeleteChannel listener);
    
    /**
     * \~korean
     * 채널 입장<br/>
     * 채널 ID를 입력받아, 채널에 접속합니다. <br/>
     * 공개(PUBLIC)채널이 아닌 경우, 비밀번호를 함께 입력해야합니다.<br/>
     * 채널 입장에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Enter channel<br/>
     * Receives the channel ID and connects to the channel.<br/>
     * If it is not a PUBLIC channel, you must enter the password.<br/>
     * The result of entering the channel is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void enterChannel(EnterChannelParams const & params, onEnterChannel listener);
    
    /**
     * \~korean
     * 채널 퇴장<br/>
     * 채널 ID를 입력받아, 해당 채널에서 퇴장합니다.<br/>
     * 채널 퇴장에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Exit channel<br/>
     * Receives the channel ID and exits the corresponding channel.<br/>
     * The result of exiting the channel is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void exitChannel(std::string const & channelId, onExitChannel listener);
    
    // MARK: User APIs
    /**
     * \~korean
     * 참여중인 채널 목록 조회<br/>
     * 참여중인 채널 목록을 조회합니다.<br/>
     * 채널 목록 조회에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Get a list of channels the user is participating in.<br/>
     * Retrieves a list of channels the user is participating in.<br/>
     * The result of the channel list retrieval is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void getChannelsByUser(onGetChannelsByUser listener);
    
    /**
     * \~korean
     * 차단된 사용자 목록 조회<br/>
     * 차단된 사용자 목록을 조회합니다.<br/>
     * 차단된 사용자 목록에는 차단된 사용자의 ID와 차단시간이 포함됩니다. <br/>
     * 차단된 사용자 목록 조회에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Get a list of blocked users.<br/>
     * Retrieves a list of blocked users.<br/>
     * The blocked user list includes the ID of the blocked user and the time of blocking.<br/>
     * The result of the blocked user list retrieval is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void getBlockMembers(onGetBlockMembers listener);
    
    /**
     * \~korean
     * 사용자 차단<br/>
     * 사용자 ID를 입력받아, 해당 사용자를 차단합니다.<br/>
     * 사용자 차단에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Block user<br/>
     * Receives the user ID and blocks the corresponding user.<br/>
     * The result of blocking the user is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void blockMember(int64_t blockPlayerId, onBlockMember listener);
    
    /**
     * \~korean
     * 사용자 차단 해제<br/>
     * 사용자 ID를 입력받아, 해당 사용자의 차단을 해제합니다.<br/>
     * 사용자 차단 해제에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Unblock user<br/>
     * Receives the user ID and unblocks the corresponding user.<br/>
     * The result of unblocking the user is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void unblockMember(int64_t blockPlayerId, onUnblockMember listener);
    
    /**
     * \~korean
     * 메시지 번역 요청<br/>
     * targetLanguage와 번역할 메시지를 입력받아, 해당 메시지를 번역합니다.<br/>
     * 하나의 메시지를 여러 언어로 번역할 수 있습니다.<br/>
     *
     * \~english
     * Message translation request.<br/>
     * Takes the target language and the message to be translated as input, and translates the message accordingly.<br/>
     * A single message can be translated into multiple languages.<br/>
     * \~
     *
     * @since 4.25.4.0
     * @ingroup Chat
     */
    static void translate(TranslateParams const & params, onTranslate listener);
    
    // MARK: Connection APIs
    /**
     * \~korean
     * 채팅서버에 연결 (더 이상 사용되지 않음)<br/>
     * 이 메서드는 더 이상 사용을 권장하지 않으며, 향후 버전에서 제거될 예정입니다.<br/>
     * 대신 connect(onResult)를 사용하세요<br/>
     *
     * \~english
     * Connects the Chat Server (Deprecated)<br/>
     * This method is no longer recommended and will be removed in future versions.<br/>
     * Use connect(onResult) instead.<br/>
     * \~
     *
     * @deprecated 4.25.2.0 버전부터 사용되지 않음.
     * @see connect(onResult)
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void connect(std::string const & extraData, onResult listener);
    
    /**
     * \~korean
     * 채팅서버에 연결<br/>
     * 채팅서버에  연결을 시도합니다.<br/>
     * 연결에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Connects the Chat Server<br/>
     * Attempts to connect to the Chat Server.<br/>
     * The result of the connection is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.2.0
     * @ingroup Chat
     */
    static void connect(onConnect listener);
    
    /**
     * \~korean
     * 채팅 서버 재 연결<br/>
     * Hive 채팅 서버와 재연결 합니다.<br/>
     * <br/>
     * 재연결은 기존의 접속이 끊긴지 10분이 지나지 않았다면, 기존에 접속되어있던 채널에 다시 접속합니다.<br/>
     * 마지막으로 연결이 끊긴지 10분이 지났거나, 이전 연결기록이 없다면 connect와 동일하게 동작합니다.
     * <br/>
     *
     * \~english
     * The reconnection will rejoin the previously connected channels if the disconnection occurred less than 10 minutes ago.<br/>
     * If the last disconnection was more than 10 minutes ago or there is no previous connection record, this will behave the same as connect.<br/>
     * <br/>
     * To detect connection status and receive status change events, you must register the Chat.addConnectionHandler event handler.<br>
     * \~
     *
     * @since 4.25.2.0
     * @ingroup Chat
     */
    static void reconnect(onReconnect listener);
    
    /**
     * \~korean
     * 채팅서버 연결 해제<br/>
     * 채팅서버와의 연결을 해제합니다.<br/>
     * 연결 해제에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Disconnects the Chat Server<br/>
     * Disconnects from the Chat Server.<br/>
     * The result of the disconnection is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void disconnect(onDisconnect listener);
    
    // MARK: Message APIs
    /**
     * \~korean
     * 채널 메시지 전송<br/>
     * 채널 ID와 메시지를 입력받아, 해당 채널에 메시지를 전송합니다.<br/>
     * 전달된 채널 메시지를 수신하기 위해 Chat.addChannelHandler 이벤트 핸들러를 등록해야 합니다.<br/>
     *
     * \~english
     * Send a channel message<br/>
     * Receives the channel ID and message and sends the message to the corresponding channel.<br/>
     * To receive the sent channel message, you need to register an event handler using Chat.addChannelHandler.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void sendMessageWithChannelSendMessageParams(ChannelSendMessageParams const & params);
    
    // MARK: Message APIs
    /**
     * \~korean
     * 채널 메시지 전송<br/>
     * 채널 ID와 메시지를 입력받아, 해당 채널에 메시지를 전송합니다.<br/>
     * 메시지 전송에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Send a channel message<br/>
     * Receives the channel ID and message and sends the message to the corresponding channel.<br/>
     * The result of sending the message is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void sendMessageWithChannelSendMessageParams(ChannelSendMessageParams const & params, onChannelSendMessage listener);
    
    /**
     * \~korean
     * 1:1 메시지 전송<br/>
     * 수신자 ID와 메시지를 입력받아, 해당 사용자에게 메시지를 전송합니다.<br/>
     * 전달된 1:1 메시지를 수신하기 위해 Chat.addDirectMessageHandler 이벤트 핸들러를 등록해야 합니다.<br/>
     *
     * \~english
     * Send a 1:1 message<br/>
     * Receives the recipient ID and message and sends the message to the corresponding user.<br/>
     * To receive the sent 1:1 message, you need to register an event handler using Chat.addDirectMessageHandler.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void sendMessageWithDirectSendMessageParams(DirectSendMessageParams const & params);
    
    /**
     * \~korean
     * 1:1 메시지 전송<br/>
     * 수신자 ID와 메시지를 입력받아, 해당 사용자에게 메시지를 전송합니다.<br/>
     * 메시지 전송에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Send a 1:1 message<br/>
     * Receives the recipient ID and message and sends the message to the corresponding user.<br/>
     * The result of sending the message is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void sendMessageWithDirectSendMessageParams(DirectSendMessageParams const & params, onDirectSendMessage listener);
    
    // MARK: Listener APIs
    /**
     * \~korean
     * 채널 리스너 등록<br/>
     * 채널 이벤트를 수신하기 위한 리스너를 등록합니다.<br/>
     * 등록된 리스너는 채널 이벤트(메시지, 공지메시지, 채널 접속 알림, 채널 퇴장 알림, 채널 삭제 알림)가 발생할 때마다 이벤트를 수신합니다.<br/>
     *
     * \~english
     * Register Channel Listener<br/>
     * Register a listener to receive channel events.<br/>
     * Registered listeners receive events (messages, notice messages, channel access notifications, channel exit notifications, channel deletion notifications) whenever they occur.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void addChannelListener(std::string const & uniqueKey, ChannelListener* listener);
    
    /**
     * \~korean
     * 1:1 메시지 리스너 등록<br/>
     * 1:1 메시지를 수신하기 위한 리스너를 등록합니다.<br/>
     * 등록된 리스너는 1:1 메시지가 발생할 때마다 이벤트를 수신합니다.<br/>
     *
     * \~english
     * Register Direct Message Listener<br/>
     * Register a listener to receive 1:1 messages.<br/>
     * Registered listeners receive events whenever a 1:1 message occurs.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void addDirectMessageListener(std::string const & uniqueKey, DirectMessageListener* listener);
    
    /**
     * \~korean
     * 유저 리스너 등록<br/>
     * 유저 이벤트를 수신하기 위한 리스너를 등록합니다.<br/>
     * 등록된 리스너는 1:1 메시지, 공지메시지 이벤트가 발생할 때마다 이벤트를 수신합니다.<br/>
     *
     * \~english
     * Register User Listener<br/>
     * Register a listener to receive user events.<br/>
     * Registered listeners receive events (1:1 messages, notice messages) whenever they occur.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void addUserListener(std::string const & uniqueKey, UserListener* listener);
    
    /**
     * \~korean
     * 연결 리스너 등록<br/>
     * 채팅서버 연결 상태를 수신하기 위한 리스너를 등록합니다.<br/>
     * 등록된 리스너는 채팅서버 연결 상태가 변경될 때마다 이벤트를 수신합니다.<br/>
     *
     * \~english
     * Register Connection Listener<br/>
     * Register a listener to receive the connection status of the Chat Server.<br/>
     * Registered listeners receive events whenever the connection status of the Chat Server changes.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void addConnectionListener(std::string const & uniqueKey, ConnectionListener* listener);
    
    /**
     * \~korean
     * 커스텀 데이터 리스너 등록<br/>
     *
     * \~english
     * Register Custom Data Listener<br/>
     * \~
     *
     * @since 4.25.4.0
     * @ingroup Chat
     */
    static void addCustomDataListener(std::string const & uniqueKey, CustomDataListener* listener);

    /**
     * \~korean
     * 채널 리스너 제거<br/>
     * 등록된 채널 리스너를 제거합니다.<br/>
     *
     * \~english
     * Remove Channel Listener<br/>
     * Remove the registered channel listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void removeChannelListener(std::string const & uniqueKey);
    
    /**
     * \~korean
     * 1:1 메시지 리스너 제거<br/>
     * 등록된 1:1 메시지 리스너를 제거합니다.<br/>
     *
     * \~english
     * Remove Direct Message Listener<br/>
     * Remove the registered 1:1 message listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void removeDirectMessageListener(std::string const & uniqueKey);
    
    /**
     * \~korean
     * 유저 리스너 제거<br/>
     * 등록된 유저 이벤트 리스너를 제거합니다.<br/>
     *
     * \~english
     * Remove User Listener<br/>
     * Remove the registered user event listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void removeUserListener(std::string const & uniqueKey);
    
    /**
     * \~korean
     * 연결 리스너 제거<br/>
     * 등록된 연결 리스너를 제거합니다.<br/>
     *
     * \~english
     * Remove Connection Listener<br/>
     * Remove the registered connection listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static void removeConnectionListener(std::string const & uniqueKey);
    
    /**
     * \~korean
     * 커스텀 데이터 리스너 제거<br/>
     *
     * \~english
     * Remove Custom Data Listener<br/>
     * \~
     *
     * @since 4.25.4.0
     * @ingroup Chat
     */
    static void removeCustomDataListener(std::string const & uniqueKey);

    // MARK: Socket Connected APIs
    /**
     * \~korean
     * 소켓 연결 상태 확인<br/>
     * 현재 소켓이 연결되어 있는지 확인합니다.<br/>
     *
     * \~english
     * Check the socket connection status<br/>
     * Check if the socket is currently connected.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup Chat
     */
    static bool isConnected();
    
    static void executeEngine(picojson::value jsonParam);
};

// MARK: Common
/**
 * @brief 채팅 채널 타입 정보<br/>
 *
 * @ingroup Chat
 */
enum class ChannelType {
    UNKNOWN,
    PRIVATE,
    PUBLIC,
    GROUP
};

/**
 * @brief 채널 정렬 타입 정보<br/>
 *
 * @ingroup Chat
 */
enum class SortType {
    Unknown,
    ChannelId,
    ChannelName,
    RegTime
};

enum class ReactionType {
    Unknown,
    Like
};

/**
 * @brief 채팅 채널 정보<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT Channel {
public:
    // 채널 ID / Channel ID
    std::string channelId;
    // 채널 타입 / Channel Type
    ChannelType type;
    // 채널 소유자 / Channel Owner
    std::string owner;
    // 채널명 / Channel Name
    std::string channelName;
    // 현재 채널 인원수 / Current Number of People in the Channel
    int memberCount;
    // 채널 최대 인원수 / Maximum Number of People in the Channel
    int maxMemberCount;
    // 채널 생성 시간 / Channel Creation Time
    std::string regTime;
    // 채널 생성 시간 UnixTime / Channel Creation UnixTime
    int64_t regTimeMillis;
    // 채널 메시지 조회 가능 여부 / Channel Message Query Availability
    bool chatHistoryAllowed; 
    
    Channel();
    bool parse(picojson::value const & json);
    picojson::object toJson() const;
    std::string toString() const;
    
    void query(ChannelMessageListQueryParams const & params, Chat::onGetChannelMessageHistory listener) const;
    void setTranslationEnabled(bool enabled, Chat::onChannelTranslationSetting listener) const;
    void addReaction(std::string const & messageId, ReactionType reactionType, Chat::onAddReaction listener) const;
    void removeReaction(std::string const & messageId, ReactionType reactionType, Chat::onRemoveReaction listener) const;
};

/**
 * @brief 채널 조회 페이지 정보<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT ChannelPage {
public:
    // 페이지 크기 / Page Size
    int size;
    // 현재 페이지 / Current Page
    int currentPage;
    // 전체 채널 수 / Total Number of Channels
    int totalElements;
    // 전체 페이지 수 / Total Number of Pages
    int totalPages;
    
    ChannelPage();
    bool parse(picojson::value const & json);
    picojson::object toJson() const;
};

/**
 * @brief 채널 멤버 정보<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT Member {
public:
    // 플레이어 ID / Player ID
    int64_t playerId;
    // 접속 시간 / Connection Time
    std::string connectedTime;
    // 접속 시간 UnixTime / Connection UnixTime
    int64_t connectedTimeMillis;
    
    Member();
    bool parse(picojson::value const & json);
    picojson::object toJson() const;
};

/**
 * @brief 차단된 사용자 정보<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT BlockMember {
public:
    // 차단된 사용자 ID / Blocked User ID
    int64_t playerId;
    // 차단 시간 / Blocked Time
    std::string blockedTime;
    // 차단 시간 UnixTime / Blocked UnixTime
    int64_t blockedTimeMillis;
    
    BlockMember();
    bool parse(picojson::value const & json);
    picojson::object toJson() const;
};

// MARK: Params
/**
 * @brief 채널 생성 파라미터<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT CreateChannelParams {
public:
    // 채널 ID / Channel ID
    std::string channelId;
    // 비밀번호 / Password
    std::string password;
    // 채널명 / Channel Name
    std::string channelName;
    // 최대 인원수 / Maximum Number of People
    int maxMemberCount;
    // 채널 타입 / Channel Type
    ChannelType type;
    // 채널 메시지 조회 가능 여부 / Channel Message Query Availability
    bool chatHistoryAllowed;
    
    CreateChannelParams();
    picojson::object toJson() const;
    std::string toString() const;
};

/**
 * @brief 채널 목록 조회 파라미터<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT GetChannelsParams {
public:
    // 채널 타입 / Channel Type
    ChannelType type;
    // 채널 ID / Channel ID
    std::string channelId;
    // 채널명 / Channel Name
    std::string channelName;
    // 정렬 타입 / Sort Type
    SortType sort;
    // 페이지 순서 (ASC, DESC) / Page Order (ASC, DESC)
    std::string pageOrder;
    // 페이지 크기 / Page Size
    int pageSize;
    // 페이지 번호 / Page Number
    int pageNumber;
    
    GetChannelsParams();
    picojson::object toJson() const;
    std::string toString() const;
};

/**
 * @brief 채널 입장 파라미터<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT EnterChannelParams {
public:
    // 채널 ID / Channel ID
    std::string channelId;
    // 비밀번호 / Password
    std::string password;
    
    EnterChannelParams();
    picojson::object toJson() const;
    std::string toString() const;
};

/**
 * @brief 채널 메시지 전송 파라미터<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT ChannelSendMessageParams {
public:
    // 채널 ID / Channel ID
    std::string channelId;
    // 메시지 / Message
    std::string message;
    // extraData
    std::string extraData;
    // 멘션 / Mentions
    std::vector<int64_t> mentionedPlayerIds;
    // 답글 Message ID / reply Message Id
    std::string replyMessageId;
    
    ChannelSendMessageParams();
    picojson::object toJson() const;
    std::string toString() const;
    bool parse(picojson::value const & json);
};

/**
 * @brief 1:1 메시지 전송 파라미터<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT DirectSendMessageParams {
public:
    // 수신자 플레이어 ID / Receiver Player ID
    int64_t toPlayerId;
    // 메시지 / Message
    std::string message;
    // extraData
    std::string extraData;
    
    DirectSendMessageParams();
    picojson::object toJson() const;
    std::string toString() const;
    bool parse(picojson::value const & json);
};

/**
 * @brief 채널 대화 이력 조회 파라미터<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT ChannelMessageListQueryParams {
public:
    // 입력받은 메시지 기준에서 이전 메시지 요청 개수 / Number of message requests before the received message
    int prevSize;
    // 입력받은 메시지 기준 이후 메시지 요청 개수 / Number of message requests after the received message
    int nextSize;
    // 메시지 index (없을 경우 가장 최신 메시지 조회) / Message Index (If not available, retrieve the most recent message)
    std::string messageId;
    // 메시지 순서 (ASC, DESC) / Message Sort (ASC, DESC)
    std::string order;

    ChannelMessageListQueryParams();
    picojson::object toJson() const;
    std::string toString() const;
};    

class HIVESDK_DLLEXPORT TranslateParams {
public:
    std::string message;
    std::string sourceLanguage;
    std::vector<std::string> targetLanguage;

    TranslateParams();
    picojson::object toJson() const;
    std::string toString() const;
};

// MARK: Listener Message Classes
/**
 * @brief 입장 멤버 정보<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT EnteredMember {
public:
    // 채널 ID / Channel ID
    std::string channelId;
    // 플레이어 ID / Player ID
    int64_t playerId;
    // 입장 시간 / Join Time
    std::string timestamp;
    // 입장 시간 UnixTime / Join UnixTime
    int64_t timestampMillis;
    
    EnteredMember();
    bool parse(picojson::value const & json);
};

/**
 * @brief 퇴장 멤버 정보<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT ExitedMember {
public:
    // 채널 ID / Channel ID
    std::string channelId;
    // 플레이어 ID / Player ID
    int64_t playerId;
    // 퇴장 시간 / Exit Time
    std::string timestamp;
    // 입장 시간 UnixTime / Join UnixTime
    int64_t timestampMillis;
    
    ExitedMember();
    bool parse(picojson::value const & json);
};

/**
 * @brief 삭제된 채널 정보<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT DeletedChannel {
public:
    // 채널 ID / Channel ID
    std::string channelId;
    // 삭제 시간 / Deletion Time
    std::string timestamp;
    // 입장 시간 UnixTime / Join UnixTime
    int64_t timestampMillis;
    
    DeletedChannel();
    bool parse(picojson::value const & json);
};


/**
 * @brief 채널 공지 메시지 정보<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT ChannelNoticeMessage {
public:
    // Message ID
    std::string messageId;
    // 채널 ID / Channel ID
    std::string channelId;
    // 발신자 / Sender
    std::string from;
    // 메시지 / Message
    std::string message;
    // 메시지 전송 시간 / Message Send Time
    std::string timestamp;
    // 입장 시간 UnixTime / Join UnixTime
    int64_t timestampMillis;
    
    ChannelNoticeMessage();
    bool parse(picojson::value const & json);
};

/**
 * @brief 채널 메시지 정보<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT ChannelMessage {
public:
    // Message ID
    std::string messageId;
    // 발신자 / Sender
    int64_t from;
    // 발신자 추가 데이터 (닉네임 등) / Sender Additional Data (Nickname, etc)
    std::string extraData;
    // 수신받은 채널 / Received Channel
    std::string to;
    // 메시지 / Message
    std::string message;
    // 메시지 전송 시간 / Message Send Time
    std::string timestamp;
    // 입장 시간 UnixTime / Join UnixTime
    int64_t timestampMillis;
    // 번역여부 / is Translated
    bool translated;
    // 번역 메시지 / Translated Message
    std::string translatedMessage;
    // 리액션 / reactions
    std::map<ReactionType, std::vector<int64_t>> reactions;
    // 멘션 / mentionedPlayerIds
    std::vector<int64_t> mentionedPlayerIds;
    // 답글 Meesage ID / Reply Message ID
    std::string replyMessageId;
    // 답글 메시지 / Reply Message
    std::string replyMessage;
    // 답글 Extra Data / Reply Extra Data
    std::string replyExtraData;
    
    ChannelMessage();
    bool parse(picojson::value const & json);
};

/**
 * @brief 1:1 메시지 정보<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT DirectMessage {
public:
    // Message ID
    std::string messageId;
    // 발신자 / Sender
    int64_t from;
    // 발신자 추가 데이터 (닉네임 등) / Sender Additional Data (Nickname, etc.)
    std::string extraData;
    // 수신자 / Receiver
    int64_t to;
    // 메시지
    std::string message;
    // 메시지 전송 시간 / Message Send Time
    std::string timestamp;
    // 입장 시간 UnixTime / Join UnixTime
    int64_t timestampMillis;
    
    DirectMessage();
    bool parse(picojson::value const & json);
};

/**
 * @brief 유저 대상 공지 메시지 정보<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT NoticeMessage {
public:
    // Message ID
    std::string messageId;
    // 발신자 / Sender
    std::string from;
    // 메시지
    std::string message;
    // 메시지 전송 시간 / Message Send Time
    std::string timestamp;
    // 메시지 전송 시간 / Message Send Time (unix time)
    int64_t timestampMillis;

    NoticeMessage();
    bool parse(picojson::value const & json);
};

/**
 * @brief 채널 대화 이력 정보<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT ChannelMessageListQueryResponse {
public:
    bool hasNext;
    std::string nextMessageId;
    bool hasPrev;
    std::string prevMessageId;
    std::vector<ChannelMessage> content;

    ChannelMessageListQueryResponse();
    bool parse(picojson::value const & json);
};

/**
 * @brief 메시지 번역 결과<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT TranslationData {
public:
    std::map<std::string, std::string> translations;

    TranslationData();
    bool parse(picojson::value const & json);
};

/**
 * @brief 리액션<br/>
 * 
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT Reaction {
public:
    std::string channelId;
    std::string messageId;
    int64_t playerId;
    ReactionType type;
    
    Reaction();
    bool parse(picojson::value const & json);
};
    
// MARK: Listener Classes
/**
 * @brief 채널 이벤트 리스너<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT ChannelListener {
public:
    // 채널 입장시 호출 / Called when a member enters the channel
    virtual void onEnteredMember(EnteredMember const & member) = 0;
    // 채널 퇴장시 호출 / Called when a member exits the channel
    virtual void onExitedMember(ExitedMember const & member) = 0;
    // 채널 삭제시 호출 / Called when a channel is deleted
    virtual void onDeletedChannel(DeletedChannel const & channel) = 0;
    // 채널 공지 메시지 수신시 호출 / Called when a channel notice message is received
    virtual void onNoticeMessage(ChannelNoticeMessage const & message) = 0;
    // 채널 메시지 수신시 호출 / Called when a channel message is received
    virtual void onChannelMessage(ChannelMessage const & message) = 0;
    // 리액션 추가 수신시 호출 / Called when a channel message reaction is received
    virtual void onAddReaction(Reaction const & reaction) = 0;
    // 리액션 제거 수신시 호출 / Called when a channel message reaction is deleted
    virtual void onRemoveReaction(Reaction const & reaction) = 0;
    
    virtual ~ChannelListener()  = default;
};

/**
 * @brief 1:1 메시지 리스너<br/>
 * @deprecated 4.25.4.0 버전부터 사용되지 않음.
 * 
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT DirectMessageListener {
public:
    // 1:1 메시지 수신시 호출 / Called when a 1:1 message is received
    virtual void onDirectMessage(DirectMessage const & message) = 0;
};

/**
 * @brief 유저 이벤트 리스너<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT UserListener {
public:
    // 1:1 메시지 수신시 호출 / Called when a 1:1 message is received
    virtual void onDirectMessage(DirectMessage const & message) = 0;
    // 개인 공지 메시지 수신 시 호출 / Called when a personal notice message is received
    virtual void onNoticeMessage(NoticeMessage const & message) = 0;

    virtual ~UserListener() = default;
};

/**
 * @brief 연결 리스너<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT ConnectionListener {
public:
    // 채팅서버 연결시 호출 / Called when connected to the Chat Server
    virtual void onConnected() = 0;
    // 채팅서버 재연결 시작 시 호출 / Called when reconnection to the Chat Server starts
    virtual void onReconnectStarted() = 0;
    // 채팅서버 재연결시 호출 / Called when reconnected to the Chat Server 
    virtual void onReconnected(ResultAPI const & result, std::vector<std::string> const & channelIds, std::vector<std::string> const & failChannelIds) = 0;
    // 채팅서버 연결 해제시 호출 / Called when disconnected from the Chat Server
    virtual void onDisconnected() = 0;
    
    virtual ~ConnectionListener() = default;
};

/**
 * @brief 커스텀 데이터 리스너<br/>
 *
 * @ingroup Chat
 */
class HIVESDK_DLLEXPORT CustomDataListener {
public:
    // 커스텀 데이터 수신시 호출 / Called when a custom data is received
    virtual void onCustomData(std::string const & data) = 0;

    virtual ~CustomDataListener() = default;
};

class HIVESDK_DLLEXPORT Unique64 {
public:

    static Unique64& getInstance();
    std::string generateKey();
private:
    std::mutex mutex_;
    uint64_t last_timestamp_ms_;
    uint16_t sequence_;
    
    static const uint16_t MAX_SEQUENCE = 0xFFFF;

    Unique64();
    uint64_t getCurrentTimestampMs();
    uint64_t waitUntilNextMillisecond(uint64_t current_ts_ms_ref);
};

template<typename Key, typename Value>
class ThreadSafeMap {
private:
    std::map<Key, Value> map_;
    mutable std::mutex mutex_;
    
public:
    ThreadSafeMap() = default;
    ThreadSafeMap(const ThreadSafeMap&) = delete;
    ThreadSafeMap& operator=(const ThreadSafeMap&) = delete;
    void insert(const Key& key, const Value& value);
    Value get(const Key& key) const;
    bool contains(const Key& key) const;
    size_t erase(const Key& key);
};

NS_HIVE_END

#endif // __HIVE_CHAT_H__
