/**
 * @file    HIVE_Configuration.h
 *
 * @date		2016-2019
 * @copyright	Copyright © GAMEVIL COM2US PLATFORM Inc. All Right Reserved.
 * @author		ryuvsken
 * @since		4.0.0
 * @defgroup Configuration
 * @{
 * \~korean
 * HIVE SDK 설정 관리<br/><br/>
 * \~english
 * This class manage HIVE SDK configuration<br/><br/>
 * \~
 */


#ifndef __HIVE_CONFIGURATION_H__
#define  __HIVE_CONFIGURATION_H__


#include "HIVE_ResultAPI.h"
#include "HIVE_Analytics.h"
#include "HIVE_PermissionView.h"



NS_HIVE_BEGIN


class ResultAPI;
enum class ZoneType;
enum class HIVELanguage;
enum class HiveThemeType;
enum class HiveConfigType;
class Tracker;


/**
 * \~korean
 * HIVE SDK 설정 관리<br/><br/>
 * \~english
 * This class manage HIVE SDK configuration<br/><br/>
 * \~
 * @since		4.0.0
 * @ingroup Configuration
 * @author ryuvsken
 */
class HIVESDK_DLLEXPORT Configuration {

public:

    /**
     *  \~korean
     * @brief MetaData 요청 결과 통지<br/>
     *
     * @param result        API 호출 결과
     * @param value         key에 매칭된 metadata 값
     *
     *  \~english
     * @brief MetaData request result callback<br/>
     *
     * @param result        Result of API call
     * @param value         The metadata value that matches the key
     *
     *  \~
     * @see #getMetaData(String, ConfigurationGetMetaDataListener)
     *
     * @ingroup Configuration
     */
    typedef std::function<void(ResultAPI const & result,std::string value)> onGetMetaData;
    typedef std::function<void(ResultAPI const& result, const char* value)> onGetMetaDataChar;

    static std::string getConfiguration();
    
    
	/**
	* \~korean 
	* @brief Hive SDK 버전 반환
	* 
	* @return Hive SDK 버전
	* \~english 
	* @brief Returns HIVE SDK Version
	*
	* @return HIVE SDK version
	* \~
	* @ingroup Configuration
	*/
	static std::string getHiveSDKVersion();
    
    
    /**
		* \~korean 
		* @brief HIVE SDK 가 참조하고 있는 SDK 의 버전 반환
		* 
		* @return HIVE SDK 가 참조하고 있는 SDK 의 버전
		* \~english 
		* @brief Return version of SDK referenced by HIVE SDK
		* 
		* @return Version of SDK referenced by HIVE SDK
		* \~
		* @ingroup Configuration
		*/
    static std::string getReferenceSDKVersion();


	/**
	* \~korean AppId 반환<br/>
	* (AppId 는 기본적으로 AndroidManifest.xml 파일의 package 값으로 설정하게 된다.<br/>
	* 그러나 테스트 설정등의 이유로 API 호출에 대한 변경을 지원한다.)
	* 
	* @return AppId
	* \~english Returns AppId
	* (By default, AppId will be set to the package name in the AndroidManifest.xml file. However, it supports changes to API calls for reasons such as test setup.)
	*
	* @return AppId
	* \~
	* @ingroup Configuration
	*/
	static std::string getAppId();


	/**
	* \~korean AppId 설정<br/>
	* (AppId 는 기본적으로 AndroidManifest.xml 파일의 package 값으로 설정하게 된다.<br/>
	* 그러나 테스트 설정등의 이유로 API 호출에 대한 변경을 지원한다.)
	* 
	* @param appId AppId
	* \~english Set AppId 
	* (By default, AppId will be set to the package name in the AndroidManifest.xml file. However, it supports changes to API calls for reasons such as test setup.)
	*
	* @param appId AppId
	* \~
	* @ingroup Configuration
	*/
	static void setAppId(std::string appId);


    static std::string getHiveCertificationKey();
    static void setHiveCertificationKey(std::string appKey);
    
    
	/**
	* \~korean Hive 플랫폼 서버 존 반환
	* 
	* @return Hive SDK 플랫폼 서버 존 (sandbox : 개발용, real : 실계용)
	* \~english Returns Hive platform server zone
	*
	* @return Hive platform server zone (sandbox : for development, real : for production)
	* \~
	* @ingroup Configuration
	*/
	static ZoneType getZone();


	/**
	* \~korean Hive 플랫폼 서버 존 설정
	* 
	* @param zone Hive SDK 플랫폼 서버 존 (sandbox : 개발용, real : 실계용)
	* \~english Set Hive Hive platform server zone
	*
	* @param zone Hive platform server zone (sandbox : for development, real : for production)
	* \~
	* @ingroup Configuration
	*/
	static void setZone(ZoneType zone);


	/**
	* \~korean 서버별 점검공지 팝업지원을 위한 serverId 반환<br/>
	* (백오피스 월드관리에 등록된 월드값을 서버에 따라 구분하여 입력이 되어야 한다.)
	* 
	* @return 서버별 점검공지 팝업지원을 위한 serverId
	* \~english Return serverId for server-specific maintenance popup support<br>
	* (The world value registered in the back office world management)
	*
	* @return serverId Server ID for server-specific maintenance popup support
	* \~
	* @ingroup Configuration
	*/
	static std::string getServerId();


	/**
	* \~korean 서버별 점검공지 팝업지원을 위한 serverId 설정<br/>
	* (백오피스 월드관리에 등록된 월드값을 서버에 따라 구분하여 입력이 되어야 한다.)
	* 
	* @param 서버별 점검공지 팝업지원을 위한 serverId
	* \~english Set serverId for server-specific maintenance popup support<br>
	* (The world value registered in the back office world management should be inputted according to the server.)
	*
	* @param serverId for server-specific maintenance popup support
	* \~
	* @ingroup Configuration
	*/
	static void setServerId(std::string serverId);

    /**
    * \~korean 서버별 점검공지 팝업지원을 위한 serverId 설정<br/>
    * (백오피스 월드관리에 등록된 월드값을 서버에 따라 구분하여 입력이 되어야 한다.)
    *
    * @param 서버별 점검공지 팝업지원을 위한 serverId
    * \~english Set serverId for server-specific maintenance popup support<br>
    * (The world value registered in the back office world management should be inputted according to the server.)
    *
    * @param serverId for server-specific maintenance popup support
    * \~
    * @ingroup Configuration
    */
    static void updateServerId(std::string serverId);
    
    /**
     * \~korean 현재 설정된 언어코드를 확인하기 위한 API<br/>
     * ISO 639-1 형식의 2자리 소문자를 반환한다.<br>
     * 기본값으로 기기언어가 설정되며 Hive SDK에서 지원하지 않는 언어일 경우 "en" 으로 설정 된다.
     *
     * \~english Gets the current language code setting.<br/>
     * Returns the language code as a 2-letter lowercase string (ISO 639-1 format).<br>
     * By default, the device language is set. If the language is not supported by the Hive SDK, it defaults to "en".
     * \~
     * @ingroup Configuration
     */
    static std::string getGameLanguage();

	/**
	* \~korean 게임에서 사용하는 언어코드를 모듈에도 반영하기 위한 API<br/>
	* 2자리 소문자로 ("en") 입력해주면 된다. ISO 639-1 형식.
	* <p>
	* 내부적으로 대문자도 소문자로 변환된다. 알파벳 대소문자 규칙은 US 규칙을 따른다.
	* 
	* @param language 게임에서 사용하는 언어코드
	* \~english API to reflect language code used in game in module<br>
	* Just type in two lowercase letters ("en"). ISO 639-1 format.<br>
	* <p>
	* Internally, uppercase characters are converted to lowercase characters. Alphabetic case rules follow US rules.
	* 
	* @param language Language code used in game. 
	* \~
	* @ingroup Configuration
	*/
	static void setGameLanguage(std::string language);

    /**
    * \~korean 게임에서 사용하는 언어코드를 모듈에도 반영하기 위한 API<br/>
    * 2자리 소문자로 ("en") 입력해주면 된다. ISO 639-1 형식.
    * <p>
    * 내부적으로 대문자도 소문자로 변환된다. 알파벳 대소문자 규칙은 US 규칙을 따른다.
    *
    * @param language 게임에서 사용하는 언어코드
    * \~english API to reflect language code used in game in module<br>
    * Just type in two lowercase letters ("en"). ISO 639-1 format.<br>
    * <p>
    * Internally, uppercase characters are converted to lowercase characters. Alphabetic case rules follow US rules.
    *
    * @param language Language code used in game.
    * \~
    * @ingroup Configuration
    */
    static void updateGameLanguage(std::string language);


	/**
	* \~korean Hive SDK 내부 로그 사용 여부 반환
	* 
	* @return Hive SDK 내부 로그 사용 여부
	* \~english Returns whether HIVE SDK internal log is used
	*
	* @return Whether HIVE SDK internal log is used
	* \~
	* @ingroup Configuration
	*/
	static bool getUseLog();


	/**
	* \~korean Hive SDK 내부 로그 사용 여부 설정
	* 
	* @param useLog Hive SDK 내부 로그 사용 여부
	* \~english Set whether HIVE SDK internal log is used
	*
	* @param useLog Whether HIVE SDK internal log is used
	* \~
	* @ingroup Configuration
	*/
	static void setUseLog(bool useLog);
    
    
    /**
		* \~korean 회사 반환
		*
		* @return 회사 (COM2US : 컴투스, GAMEVIL : 게임빌)
		* \~english Returns company.
		*
		* @return Company (COM2US : Com2us, GAMEVIL : Gamevil)
		*/
    static std::string getCompany();
    
    
    /**
		* \~korean 회사 설정
		*
		* @param 회사 (COM2US : 컴투스, GVI : 게임빌)
		* \~english Set company
		*
		* @param Company (COM2US : Com2us, GAMEVIL : Gamevil)
		* \~
		* @ingroup Configuration
		*/
    static void setCompany(std::string company);
    
    /**
     * \~korean 회사 인덱스 반환
     *
     * @return 회사 인덱스 (1 : 컴투스, 2 : 게임빌)
     * \~english Returns company index.
     *
     * @return CompanyIndex (1 : Com2us, 2 : Gamevil)
     */
    static std::string getCompanyIndex();
    
    
    /**
     * \~korean 회사 인덱스 설정
     *
     * @param 회사 인덱스 (1 : 컴투스, 2 : 게임빌)
     * \~english Set company
     *
     * @param CompanyIndex (1 : Com2us, 2 : Gamevil)
     * \~
     * @ingroup Configuration
     */
    static void setCompanyIndex(int companyIndex);
    
    
    /**
		* \~korean 채널 반환
		*
		* @return 채널 (HIVE : HIVE 플랫폼)
		* \~english Returns channel
		*
		* @return Channel (HIVE : HIVE Platform)
		* \~
		* @ingroup Configuration
		*/
    static std::string getChannel();
    
    
    /**
		* \~korean 채널 설정
		*
		* @param 채널 (HIVE : HIVE 플랫폼)
		* \~english Set channel
		*
		* @param Channel (HIVE : HIVE Platform)
		* \~
		* @ingroup Configuration
		*/
    static void setChannel(std::string channel);
    


	/**
	* \~korean HTTP Connect Timeout 의 기본 설정 값 반환 (초단위)
	* 
	* @return HTTP Timeout 의 기본 설정 값 (초단위)
	* \~english Returns the default value of HTTP Connect Timeout (in seconds)
	*
	* @return Default value of HTTP Connect Timeout (in seconds)
	* \~
	* @ingroup Configuration
	*/
	static int getHttpConnectTimeout();


	/**
	* \~korean HTTP Connect Timeout 의 기본 설정 값 설정 (초단위)
	* 
	* @param httpConnectTimeout HTTP Connect Timeout 의 기본 설정 값 (초단위)
	* \~english Set the value of HTTP Connect Timeout (in seconds)
	*
	* @param Value of HTTP Connect Timeout (in seconds)
	* \~
	* @ingroup Configuration
	*/
	static void setHttpConnectTimeout(int httpConnectTimeout);


	/**
	* \~korean HTTP Read Timeout 의 기본 설정 값 반환 (초단위)
	* 
	* @return HTTP Timeout 의 기본 설정 값 (초단위)
	* \~english Returns the default value of HTTP Read Timeout (in seconds)
	*
	* @return Default value of HTTP Read Timeout (in seconds)
	* \~
	* @ingroup Configuration
	*/
	static int getHttpReadTimeout();


	/**
	* \~korean HTTP Read Timeout 의 기본 설정 값 설정 (초단위)
	* 
	* @param httpReadTimeout HTTP Read Timeout 의 기본 설정 값 (초단위)
	* \~english Set the value of HTTP Read Timeout (in seconds)
	*
	* @param	Value of HTTP Read Timeout (in seconds)
	* \~
	* @ingroup Configuration
	*/
	static void setHttpReadTimeout(int httpReadTimeout);


	/**
	* \~korean 게임 로그 최대 저장 갯수 반환
	* 
	* @return 게임 로그 최대 저장 갯수
	* \~english Returns maximum number of game logs
	*
	* @return Maximum number of game logs
	* \~
	* @ingroup Configuration
	*/
	static int getMaxGameLogSize();


	/**
	* \~korean 게임 로그 최대 저장 갯수 설정<br/>
	* (특별한 경우가 아니면 변경 금지)
	* 
	* @param maxGameLogSize 게임 로그 최대 저장 갯수
	* \~english Set maximum number of game logs
	* (Note: No change unless special occasion)
	*
	* @param maxGameLogSize Maximum number of game logs
	* \~
	* @ingroup Configuration
	*/
	static void setMaxGameLogSize(int maxGameLogSize);


	/**
	* \~korean 결제 마켓 반환
	* 
	* @return 결제 마켓 (google, tstore, olleh, ozstore, googleplay_lebi)
	* \~english Return market
	* 
	* @return Market (google, tstore, olleh, ozstore, googleplay_lebi)
	* \~
	* @ingroup Configuration
	*/
	static std::string getMarket();


	/**
	* \~korean 결제 마켓 설정
	* 
	* @param market 결제 마켓 (google, tstore, olleh, ozstore, googleplay_lebi)
	* \~english Set market
	* 
	* @param market Market (google, tstore, olleh, ozstore, googleplay_lebi)
	* \~
	* @ingroup Configuration
	*/
	static void setMarket(std::string market);

    /**
		*  \~korean HIVE 서버에서 판단한 국가코드를 반환한다.
		*
		* @return ISO
		*  \~english HIVE 서버에서 판단한 국가코드를 반환한다.
		*  
		* @return ISO
		* \~
		* @ingroup Configuration
		*/
    static std::string getHiveCountry();
    
    /**
     *  \~korean HIVE 서버에서 판단한 TimeZone 정보를 반환한다.
     *
     * @return JSON String
     *  \~english HIVE 서버에서 판단한 TimeZone 정보를 반환한다.
     *
     * @return JSON String
     * \~
     * @ingroup Configuration
     */
    static std::string getHiveTimeZone();
    
    /**
     * 전송 주기마다 전송할 로그의 최대치.
     *
     * @return unsigned int 전송 주기마다 전송할 로그의 최대치.
     *
     * @ingroup Configuration
     */
    static unsigned int getAnalyticsSendLimit();
    
    /**
     * 전송 주기마다 전송할 로그의 최대치 설정.
     *
     * @param limit 전송주기마다 전송할 최대 로그의 양
     *
     * @ingroup Configuration
     */
    static void setAnalyticsSendLimit(unsigned int limit);
    
    /**
     * 최대로 쌓을수 있는 로그의 양.
     *
     * @return unsigned int 최대로 쌓을수 있는 로그의 양.
     *
     * @ingroup Configuration
     */
    static unsigned int getAnalyticsQueueLimit();
    
    /**
     * 최대로 쌓을 수 있는 로그의 수
     *
     * @param limit 최대 대기 가능한 로그의 수
     *
     * @ingroup Configuration
     */
    static void setAnalyticsQueueLimit(unsigned int limit);
    
    /**
     * 로그 전송 주기.
     *
     * @return float 전송주기
     *
     * @ingroup Configuration
     */
    static float getAnalyticsSendCycleSeconds();
    
    /**
     * 로그 전송 주기 설정.
     *
     * @param seconds 전송 주기 (초)
     *
     * @ingroup Configuration
     */
    static void setAnalyticsSendCycleSeconds(float seconds);

    /**
     * Hive SDK AgeGateU13 적용 여부 반환
     *
     * @return Hive SDK AgeGateU13 적용 여부
     */
    static bool getAgeGateU13();
    
    
    /**
     * Hive SDK AgeGateU13 적용 여부 설정
     *
     * @param ageGateU13 Hive SDK AgeGateU13 적용 여부 설정
     */
    static void setAgeGateU13(bool ageGateU13);

    /**
     * Hive SDK GDPR 국가에서 16세 미만 약관 동의 여부 반환
     *
     * @return Hive SDK GDPR 국가에서 16세 미만 약관 동의 여부
     */
    static bool getAgeGateU16Agree();
    
    /**
     * Hive SDK 법정대리인 동의 여부
     *
     * @return Hive SDK 법정대리인 동의 여부
     */
    static bool getLegalGuardianConsentAgree();

    /**
     * Hive SDK GDPR 국가에서 16세 미만 약관 동의 여부 설정 (not support)
     *
     * @param ageGateU16Agree Hive SDK GDPR 국가에서 16세 미만 약관 동의 여부 설정
     */
//    static void setAgeGateU16Agree(bool ageGateU16Agree);

    /**
     * Hive SDK 권한고지 팝업 노출 여부 설정
     *
     * @param isOn Hive SDK 권한고지 팝업 노출 여부 설정
     */
    static void setHivePermissionViewOn(bool isOn);

    /**
     * 커스텀 권한고지를 위한 데이터 설정
     *
     * @param language 타겟 언어
     *
     * @return PermissionViewData
     */
    static PermissionViewData getPermissionViewData(HIVELanguage language);

    
    /**
     * Hive 커스텀 테마 값 획득
     *
     * @param language 타겟 언어
     *
     * @return PermissionViewData
     */
    static HiveThemeType getHiveTheme();

    /**
     * Hive 커스텀 테마 설정
     *
     * @param HiveThemeType
     *
     */
    static void setHiveTheme(HiveThemeType theme);

	/**
	 * Hive Orientation 설정
	 *
	 * @param orientation 회전 방향
	 *
	 */
	static void setHiveOrientation(std::string orientation);


    static void setConfigurations(HiveConfigType configType,std::string value);

    /**
     * Game MetaData 요청
     *
     * @param key 요청 data 키
     * @param forceReload network 통신 여부
     * @param API 결과 통지
     *
     */
    static void getMetaData(std::string key, bool forceReload, onGetMetaData listener);

    /**
     * Hive CommunityUrl 설정
     *
     * @param url 커뮤니티 URL
     *
     */
    static void setHiveCommunityUrl(std::string url);

	/**
     * Hercules 사용 여부 설정
     *
     * @param enable Hercules 사용 여부 설정
     */
	static void setUseHercules(bool enable);
    
    /**
     * Hive 크로스프로모션 노출 여부 반환
     *
     * @return Hive 크로스프로모션 노출 여부
     */
    static bool getCrossPromotionBannerShowing();
    
    /**
     * Hive 크로스프로모션 노출 여부 설정
     *
     * @param bExpose  Hive 크로스프로모션 노출 여부 설정
     */
    static void setCrossPromotionBannerShowing(bool expose);
    
    /**
     * 시크릿 모드 설정
     *
     * @param use 시크릿 모드 사용 여부. true로 설정하면 시크릿 모드 사용, false로 설정하면 일반 모드로 작동
     *            별도 설정이 없으면 기본값으로 시크릿 모드로 작동
     *
     */
    static void setUsePrivateBrowsingForAuth(bool use);

    /**
     * 컨트롤러 사용 여부 반환
     *
     * @return enable 컨트롤러를 활성화 상태 true, 비활성화 상태 false 리턴
     */
    static bool getEnableGameController();

    /**
     * 컨트롤러 사용 여부 설정
     *
     * @param enable 컨트롤러를 활성화하려면 true, 비활성화하려면 false를 입력
     */
    static void setEnableGameController(bool enable);
    
    /**
     * HiveChat 타임아웃 설정 시간 반환
     *
     * @return HiveChat 타임아웃 설정 시간
     */
    static int getChatConnectionTimeout();
    
    /**
     * HiveChat 타임아웃 설정 시간 설정
     *
     * @param time HiveChat 타임아웃 설정 시간
     */
    static void setChatConnectionTimeout(int time);


	// Native 영역에서 호출된 요청을 처리하기 위한 플러그인 내부 코드
	static void executeEngine(picojson::value jsonParam);

private:
    friend class HiveTestUtils;
    static void setTestZone();
    static bool isTestZone();
};


/**
 * \~korean HIVE 플랫폼 서버 존 정의
 * \~english Hive Platform server zone type
 * @author seongjunkim
 * @ingroup Configuration
 */
enum class ZoneType {
    
    SANDBOX			///< Hive 플랫폼 서버 외부 개발
    , REAL			///< Hive 플랫폼 서버 실서비스
    
};

/**
 * \~korean HIVE 플랫폼 지원 언어
 * \~english HIVE Platform languages supported
 *
 * @author seokjinyong
 * @ingroup Configuration
 */
enum class HIVELanguage {
    HIVELanguageDE,
    HIVELanguageEN,
    HIVELanguageES,
    HIVELanguageFR,
    HIVELanguageID,
    HIVELanguageIT,
    HIVELanguageJA,
    HIVELanguageKO,
    HIVELanguagePT,
    HIVELanguageRU,
    HIVELanguageTH,
    HIVELanguageTR,
    HIVELanguageVI,
    HIVELanguageZHS,
    HIVELanguageZHT,
    HIVELanguageAR
};

/**
 * \~korean Hive 테마
 * \~english Hive theme
 *
 * @author daunjoung
 * @ingroup Configuration
 */
enum class HiveThemeType {
    hiveLight,
    hiveDark
};

enum class HiveConfigType {
    googleServerClientId,
	googlePlayGamesServerClientId,
    wechatSecret,
    wechatPaymentKey,
    adjustKey,
    singularKey,
    appsflyerKey,
    airbridgeKey,
    airbridgeAppName,
    airbridgeSecretId,
    airbridgeSecret

};
NS_HIVE_END		// namespace hive

#endif		// __HIVE_CONFIGURATION_H__


/** @} */



