/**
 * @file            IAPV4.h
 *
 * @date            2016-2023
 * @copyright       Copyright (C) 2023 Com2uS Platform Corporation. All Right Reserved.
 * @author          hife
 * @since           4.4.0
 * \brief           \copydoc     IAPV4
 */



/**
 * @defgroup        IAPV4
 * @{
 */


#ifndef __HIVE_IAPV4_H__
#define __HIVE_IAPV4_H__

#include "HIVE_ResultAPI.h"
#include <memory>

NS_HIVE_BEGIN


/**
 * @brief 결제 마켓 형태
 *
 * @ingroup IAPV4
 * @ingroup hive
 */
enum class IAPV4Type {
    NotSelected = 0
    ,AppleAppStore = 1
    ,GooglePlayStore = 2
    ,HIVELebi = 3
    ,OneStore = 4
    ,AmazonAppStore = 5
    ,SamsungGalaxyStore = 6
    ,HuaweiAppGallery = 7
    ,Funtap = 8
    ,OPPOAppMarket = 9
    ,VIVOAppStore = 10
    ,TencentMyapp = 11
    ,XiaomiAppStore = 12
    ,HuaweiAppGalleryChina = 13
    ,FacebookCloudGame = 14         // deprecated
    ,HiveStore = 15
    ,Steam = 16
    ,Nowgg = 17
};

class IAPV4Product;
class IAPV4Receipt;
class IAPV4PurchaseParam;
class IAPV4ConsumeInfoUserAgreement;


/**
 * \~korean
 * 현재 Hive IAP 가 제공하는 결제 마켓은 애플 앱스토어, 구글 플레이스토어, 원스토어, 삼성 갤럭시스토어, 아마존 앱스토어, 
 * 페이스북 클라우드 게임, 화웨이 앱커넥트, PG, Steam, Hive 러비(Lebi) 를 제공한다.<br/>
 * Hive 러비(Lebi) 상점은 중국 시장에서 상점을 구성하고 상품을 구매 할 수 있는 시스템으로서 Hive 에서 자체적으로 운영하고 있다.<br/><br/>
 * 
 * \~english
 * Currently, Hive IAP supports Apple AppStore, Google PlayStore, ONEStore, Samsung GalaxyStore,
 * Amazon AppStore, Facebook Cloud Games, Huawei AppConnect, PG, Steam and Hive Lebi.<br/>
 * Hive Lebi store is a system that can organize stores and purchase goods in Chinese market.
 * It is operated by Hive itself.<br/><br/>
 * \~
 * @since        4.4.0
 * @ingroup IAPV4
 * @author imsunghoon
 */
class HIVESDK_DLLEXPORT IAPV4 {
public:
    
    /**
     *  \~korean
     * @brief HIVE IAPV4 에서 마켓의 구성을 위한 정보 조회 결과 통지
     *
     * @param result        API 호출 결과
     * @param marketIDs    사용 가능한 마켓 정보의 목록<br/>이 마켓 정보의 목록은 게임 클라이언트에서 사용 가능한 마켓을 의미한다.<br/>마켓 목록이 2개 이상이라면 유저가 상점을 선택할 수 있도록 게임에서 유저에게 노출할 수 있는 화면을 제공해야 한다
     *
     *  \~english
     * @brief HIVE IAPV4 Market information request result callback
     *
     * @param result        Result of API call
     * @param marketIDs    사용 가능한 마켓 정보의 목록<br/>
     * This list of market information means the market available on the game client.<br/>
     * If there is more than one market list, you must provide a screen that display the available market list to the user so that the user can select a store. <br/>
     *
     *  \~
     * @see #marketConnect(IAPV4MarketInfoListener)
     *
     * @ingroup IAPV4
     */
    typedef std::function<void(ResultAPI const & result,std::vector<IAPV4Type> const & marketIDs)> onMarketConnect;

    /**
     *  \~korean
     * HIVE IAPV4 상품 정보 조회 결과 통지
     *
     * @param result            API 호출 결과
     * @param iapV4ProductList  상품 정보
     * @param balance           HIVE Lebi 마켓의 경우 잔액 정보
     *
     *  \~english
     * HIVE IAPV4 Product Information request callback
     *
     * @param result            Result of API call
     * @param iapV4ProductList  Product Information
     * @param balance           Balance information for HIVE Lebi Market If Lebi Market enabled.
     *
     *  \~
     * @see #getProductInfo(IAPV4ProductInfoListener)
     *
     * @ingroup IAPV4
     */
    typedef std::function<void(ResultAPI const & result,std::vector<IAPV4Product> const & iapV4ProductList,unsigned int balance)> onGetProductInfo;


    /**
     *  \~korean
     * @brief 상품 결제 시도 결과 통지<br/>
     * App Store, Google Store 등 에 등록된 마켓 상품(marketPid)의 구매를 요청 한다.<br/>
     * iapPayload 의 경우 구매 요청에 대한 보충 정보로 구매 후 영수증 검증에서 확인할 수 있다.<br/>
     * 프로모 코드 구매 처럼 앱 외부에서 구매 등의 경우에는 포함될 수 없게 된다.
     *
     * @param result            API 호출 결과
     * @param iapV4Receipt      구매 성공한 상품 정보 와 영수증
     *
     *  \~english
     * @brief Purchase request result callback<br/>
     * Request purchase of market product (marketPid) registered in App Store, Google Store, etc.<br/>
     * iapPayload is supplementary information for the purchase request, and it can be checked in Receipt Verification after purchase, <br/>
     * but, in case of purchasing outside of the app, such as a promo code purchase etc., it can not be included in verification result.
     *
     * @param result            Result of API call
     * @param iapV4Receipt      It includes the purchased product information and receipt.
     *
     *  \~
     * @see #purchase(String, String, IAPV4PurchaseListener)
     * @ingroup IAPV4
     */
    typedef std::function<void(ResultAPI const & result,IAPV4Receipt const & iapV4Receipt)> onPurchase;

    /**
     *  \~korean
     * @brief HIVE IAPV4 상품에 대한 미지급된 아이템 지급을 요청 결과 통지<br/>
     * 상품 구매 과정에서 여러가지 상황으로 인하여 아이템 지급을 실패하는 경우가 발생할 수 있다.<br/>
     * 이런 상황을 대비하기 위해서 미 지급된 아이템 지급을 요청 해야 한다.
     *
     * @param result            API 호출 결과
     * @param iapv4ReceiptList  구매 성공한 상품 정보 와 iapPayload 를 포함한 정보
     *
     *  \~english
     * @brief HIVE IAPV4 Restore request result callback<br/>
     * It may happen that the item delivery fails due to various situations in the product purchase process.<br/>
     * In order to prepare for this situation, you should request to deliver the item that has not been delivered.
     *
     * @param result            Result of API call
     * @param iapv4ReceiptList  It includes the purchased product information and iapPayload
     *
     *  \~
     * @see #restore(IAPV4RestoreListener)
     *
     * @ingroup IAPV4
     */
    typedef std::function<void(ResultAPI const & result,std::vector<std::reference_wrapper<IAPV4Receipt>> const & iapv4ReceiptList)> onRestore;

    /**
     *  \~korean
     * @brief HIVE IAP v4에서 상품에 대한 지급 완료 후 결제 트랜잭션을 종료하기 위한 요청 결과 통지<br/>
     *
     * @param result        API 호출 결과
     * @param marketPid    트랜잭션이 종료된 상품의 marketPid
     *
     *  \~english
     * @brief HIVE IAP v4 Transaction Finish request result callback<br/>
     *
     * @param result        Result of API call
     * @param marketPid     marketPid of the product whose transaction ended
     *
     *  \~
     * @see #transactionFinish(String, IAPV4TransactionFinishListener)
     *
     * @ingroup IAPV4
     */
    typedef std::function<void(ResultAPI const & result,std::string marketPid)> onTransactionFinish;
    
    typedef std::function<void(ResultAPI const& result, const char* marketPid)> onTransactionFinishWithChar;

    /**
     * @brief
     *
     * @param result        API 호출 결과
     * @param marketPid     Market PID
     * @ingroup IAPV4
     */
    typedef std::function<void(ResultAPI const & result,std::string marketPid)> onCheckPromotePurchase;


    /**
     *  \~korean
     * @brief HIVE IAP v4에서 상품에 대한 지급 완료 후 복수 개의 결제 트랜잭션을 종료하기 위한 요청 결과 통지<br/>
     *
     * @param resultList        트랜잭션의 종료 결과 목록
     * @param marketPidList     트랜잭션이 종료된 상품의 marketPid 목록
     *
     *  \~english
     * @brief HIVE IAP v4 Multiple Transaction Finish request result callback<br/>
     *
     * @param resultList        Result list of the transaction end
     * @param marketPidList     marketPid list of the transaction end
     *
     *  \~
     * @see #transactionMultiFinish(List, IAPV4TransactionMultiFinishListener)
     */
    typedef std::function<void(std::vector<ResultAPI> const & resultList,std::vector<std::string> const & marketPidList)> onTransactionMultiFinish;


    /**
     *  \~korean
     * @brief HIVE 러비 상점일 경우 잔액 확인 결과 통지
     *
     * @param result        API 호출 결과
     * @param balance       HIVE 러비 상점일 경우 잔액
     *
     *  \~english
     * @brief HIVE Lebi Balance request result callback, if it is a Rebi store.
     *
     * @param result        Result of API call
     * @param balance       HIVE Lebi balance
     *
     *  \~
     * @ingroup IAPV4
     */
    typedef std::function<void(ResultAPI const & result, int balance)> onBalance;

    /**
     * @brief Google In-App Offers
     * 
     * @param responseCode 2(NAVIGATE_TO_DEV_APP_STORE) 일 경우 유저가 제안(Offer)를 확인하고 사용하기를 원한 상황입니다.
     * 개발자는 유저가 구매할 인앱 항목을 선택할 수 있는 게임/앱 내 상점으로 (유저가 해당 화면에 없는 경우) 유저를 이동해야 합니다.
     * 0 은 아무런 액션을 하지 않아도 되고, 음수는 에러상황 입니다.
     */
    typedef std::function<void(int responseCode)> onInAppMessages;
    
    /**
     *  \~korean
     * @brief 소비정보 전송 약관에 대한 유저의 마지막 동의 여부 및 시기를 통지<br/>
     *
     * @param result        API 호출 결과
     * @param info       유저의 약관 동의 여부
     *
     *  \~english
     * @brief A callback to return the latest consent status and timing for the Consumption Information Transmission Terms<br/>
     *
     * @param result        Result of API call
     * @param info      User’s consent status and timing of agreement
     *
     *  \~
     * @ingroup IAPV4
     */
    typedef std::function<void(ResultAPI const & result, IAPV4ConsumeInfoUserAgreement* const info)> onGetConsumeInfoAgreement;
    
    /**
     *  \~korean
     * @brief 소비정보 전송 약관 팝업을 띄운 후 유저의 동의 여부를 통지<br/>
     *
     * @param result        API 호출 결과
     * @param didAgree       유저의 약관 동의 여부
     *
     *  \~english
     * @brief A callback to notify the user’s consent status as the outcome of the popup for Consumption Information transfer terms<br/>
     *
     * @param result        Result of API call
     * @param didAgree       User’s consent to terms and conditions
     *
     *  \~
     * @ingroup IAPV4
     */
    typedef std::function<void(ResultAPI const & result, bool didAgree)> onShowConsumeInfoAgreement;

    /**
     *  \~korean
     * @brief IAPV4 결재 API 초기화 요청<br/>
     * HIVE IAPV4 의 초기화를 수행하면 결과 통지시 사용 가능한 마켓 정보의 목록을 전달하게 된다.<br/>
     * 이 마켓 정보의 목록은 게임 클라이언트에서 사용 가능한 마켓을 의미한다.
     *
     * @param listener      API 결과 통지
     *
     *  \~english
     * @brief IAPV4 It initialize IAPV4 Purchase module.<br/>
     * When HIVE IAPV4 is initialized, it sends a list of available market information in the result callback.<br/>
     * This list of market information means the market available on the game client.
     *
     * @param listener      Result of API call
     *
     *  \~
     * @ingroup IAPV4
     */
    static void marketConnect(onMarketConnect listener);

    /**
     *  \~korean
     * @brief Apple Appstore 에서 상품의 구성을 위한 정보를 조회할 수 있다. <br/>
     *
     * @param listener          API 결과 통지
     *
     *  \~english
     * @brief Apple Appstore It returns information for composition of the product.<br/>
     *
     * @param listener          Result of API call
     *
     *  \~
     * @ingroup IAPV4
     */
    static void getMarketProductInfo(std::vector<std::string> const & marketPidList, onGetProductInfo listener);
    
    /**
     *  \~korean
     * @brief HIVE IAPV4 에서 상품의 구성을 위한 정보를 조회할 수 있다. <br/>
     *
     * @param listener          API 결과 통지
     *
     *  \~english
     * @brief HIVE IAPV4 It returns information for composition of the product.<br/>
     *
     * @param listener          Result of API call
     *
     *  \~
     * @ingroup IAPV4
     */
    static void getProductInfo(onGetProductInfo listener);
    
    /**
     *  \~korean
     * @brief HIVE IAPV4 에서 구독형 상품의 구성을 위한 정보를 조회할 수 있다. <br/>
     *
     * @param handler           API 결과 통지
     *
     *  \~english
     * @brief HIVE IAPV4 It returns information for composition of the product.<br/>
     *
     * @param handler           Result of API call
     *
     *  \~
     * @ingroup IAPV4
     * @since 4.11.6
     */
    static void getSubscriptionProductInfo(onGetProductInfo listener);
    
    /**
     *  \~korean
     * @brief App Store, Google Store 등 에 등록된 마켓 상품(marketPid)의 구매를 요청 한다.<br/>
     * iapPayload 의 경우 구매 요청에 대한 보충 정보로 구매 후 영수증 검증에서 확인할 수 있다.<br/>
     * 프로모 코드 구매 처럼 앱 외부에서 구매 등의 경우에는 포함될 수 없게 된다.
     *
     * @param marketPid         App Store, Google Store 등 에 등록된 하나의 상품을 지칭하는 고유 코드
     * @param iapPayload        구매 요청에 대한 보충 정보를 포함한 문자열로, 개발자가 지정합니다.
     * @param listener          API 결과 통지
     *
     *  \~english
     * @brief Request purchase of market product (marketPid) registered in App Store, Google Store, etc.<br/>
     * iapPayload is supplementary information for the purchase request, and it can be checked in Receipt Verification after purchase, <br/>
     * but, in case of purchasing outside of the app, such as a promo code purchase etc., it can not be included in verification result.
     *
     * @param marketPid         Unique code registered on the App Store, Google Store or etc.
     * @param iapPayload        A string containing supplementary information about the purchase request, specified by the developer.
     * @param listener          Result of API call
     *
     *  \~
     * @ingroup IAPV4
     */
    static void purchase(std::string marketPid,std::string iapPayload,onPurchase listener);
    
    /**
     *  \~korean
     * @brief App Store, Google Store 등 에 등록된 마켓 상품(marketPid)의 구매를 요청 한다.<br/>
     * iapPayload 의 경우 구매 요청에 대한 보충 정보로 구매 후 영수증 검증에서 확인할 수 있다.<br/>
     * 프로모 코드 구매 처럼 앱 외부에서 구매 등의 경우에는 포함될 수 없게 된다.
     *
     * @param marketPid         App Store, Google Store 등 에 등록된 하나의 상품을 지칭하는 고유 코드
     * @param iapPayload        구매 요청에 대한 보충 정보를 포함한 문자열로, 개발자가 지정합니다.
     * @param quantity          결제 수량
     * @param listener          API 결과 통지
     *
     *  \~english
     * @brief Request purchase of market product (marketPid) registered in App Store, Google Store, etc.<br/>
     * iapPayload is supplementary information for the purchase request, and it can be checked in Receipt Verification after purchase, <br/>
     * but, in case of purchasing outside of the app, such as a promo code purchase etc., it can not be included in verification result.
     *
     * @param marketPid         Unique code registered on the App Store, Google Store or etc.
     * @param iapPayload        A string containing supplementary information about the purchase request, specified by the developer.
     * @param quantity          payment quantity
     * @param listener          Result of API call
     *
     *  \~
     * @ingroup IAPV4
     */
    static void purchase(std::string marketPid,std::string iapPayload,int quantity,onPurchase listener);
    
    /**
     *  \~korean
     * @brief App Store, Google Store 등 에 등록된 마켓 상품(marketPid)의 구매를 요청 한다.<br/>
     * iapPayload 의 경우 구매 요청에 대한 보충 정보로 구매 후 영수증 검증에서 확인할 수 있다.<br/>
     * 프로모 코드 구매 처럼 앱 외부에서 구매 등의 경우에는 포함될 수 없게 된다.
     * 구독형 상품을 업그레이드하거나 다운그레이드를 수행하는 경우 기존에 구독 중인 상품의 고유코드 oldMarketPid를 필수로 넣어야 한다.
     *
     * @param marketPid         App Store, Google Store 에 등록된 하나의 상품을 지칭하는 고유 코드
     * @param oldMarketPid      기존에 구독 중인 상품의 고유코드
     * @param iapPayload        구매 요청에 대한 보충 정보를 포함한 문자열로, 개발자가 지정합니다.
     * @param listener          API 결과 통지
     *
     *  \~english
     * @brief Request purchase of market product (marketPid) registered in App Store, Google Store, etc.<br/>
     * iapPayload is supplementary information for the purchase request, and it can be checked in Receipt Verification after purchase, <br/>
     * but, in case of purchasing outside of the app, such as a promo code purchase etc., it can not be included in verification result.
     * When upgrading or downgrading a subscription-type product, the unique code oldMarketPid of the existing subscription product must be entered.
     *
     * @param marketPid         Unique code registered on the App Store or Google Store.
     * @param oldMarketPid      Unique code of existing subscription product
     * @param iapPayload        A string containing supplementary information about the purchase request, specified by the developer.
     * @param listener          Result of API call
     *
     *  \~
     * @ingroup IAPV4
     * @since 4.11.6
     */
    static void purchaseSubscriptionUpdate(std::string marketPid,std::string oldMarketPid,std::string iapPayload,onPurchase listener);
    
    /**
     *  \~korean
     * @brief App Store, Google Store 등 에 등록된 마켓 상품(marketPid)의 구매를 요청 한다.<br/>
     * iapPayload 의 경우 구매 요청에 대한 보충 정보로 구매 후 영수증 검증에서 확인할 수 있다.<br/>
     * 프로모 코드 구매 처럼 앱 외부에서 구매 등의 경우에는 포함될 수 없게 된다.
     * 구독형 상품을 업그레이드하거나 다운그레이드를 수행하는 경우 기존에 구독 중인 상품의 고유코드 oldMarketPid를 필수로 넣어야 한다.
     *
     * @param param             구매시 필요한 parameter 의 모음
     * @param listener          API 결과 통지
     *
     *  \~english
     * @brief Request purchase of market product (marketPid) registered in App Store, Google Store, etc.<br/>
     * iapPayload is supplementary information for the purchase request, and it can be checked in Receipt Verification after purchase, <br/>
     * but, in case of purchasing outside of the app, such as a promo code purchase etc., it can not be included in verification result.
     * When upgrading or downgrading a subscription-type product, the unique code oldMarketPid of the existing subscription product must be entered.
     *
     * @param param             A collection of parameters required for purchase.
     * @param listener          Result of API call
     *
     *  \~
     * @ingroup IAPV4
     * @since 4.16.4.0
     */
    static void purchaseSubscription(IAPV4PurchaseParam param, onPurchase listener);
    
    /**
     *  \~korean
     * @brief HIVE IAPV4 상품에 대한 미지급된 아이템 지급을 요청한다.<br/>
     * 상품 구매 과정에서 여러가지 상황으로 인하여 아이템 지급을 실패하는 경우가 발생할 수 있다.<br/>
     * 이런 상황을 대비하기 위해서 미 지급된 아이템 지급을 요청 해야 한다.<br/>
     *
     * @param listener            API 결과 통지
     *
     *  \~english
     * @brief HIVE IAPV4  It restore the item that has not been delivered.<br/>
     * It may happen that the item delivery fails due to various situations in the product purchase process.<br/>
     * In order to prepare for this situation, you should request to deliver the item that has not been delivered.
     *
     * @param listener            API 결과 통지
     *
     *  \~
     * @ingroup IAPV4
     */
    static void restore(onRestore listener);
    
    /**
     *  \~korean
     * @brief HIVE IAPV4 상품에 대한 미지급된 구독 상품 지급을 요청한다.<br/>
     * 상품 구매 과정에서 여러가지 상황으로 인하여 구독 상태를 갱신하지 못할 수 있다.<br/>
     * 구독 상태를 확인할 수 없는 경우, 구독 상태 복구를 요청 해야 한다.<br/>
     *
     * @param listener            API 결과 통지
     *
     *  \~english
     * @brief HIVE IAPV4  It restore the item that has not been delivered.<br/>
     *
     *
     *
     * @param listener            Result of API call
     *
     *  \~
     * @ingroup IAPV4
     * @since 4.11.6
     */
    static void restoreSubscription(onRestore listener);
    
    /**
     *  \~korean
     * @brief HIVE IAP v4에서 지급이 완료된 상품에 대한 구매 영수증에 대한 트랜잭션 완료 처리를 요청한다.<br/>
     * 구매나 복구 기능 사용 후 아이템 지급이 완료된 시점에 이 API를 호출하여 마켓에 구매 완료 처리를 해야한다.<br/>
     *
     * @warning 만약, 지급 완료 후에도 마켓으로의 구매 완료 처리를 하지 않으면, 미지급된 상품으로 남는다.<br/>
     * 미지급된 상품은 transactionFinish API 호출 전까지 계속 복구가 되므로, 영수증 중복 여부를 확인해야 한다.<br/>
     *
     * @param marketPid    지급이 완료되어 트랜잭션 완료 처리를 위한 상품의 고유 ID
     * @param listener    API 결과 통지
     *
     *  \~english
     * @brief HIVE IAP v4 It requests the transaction completion processing for the purchase receipt of the item that have been delivered.<br/>
     * At the completion of item delivery after purchasing or restoring, this API should be called to complete the purchase on the market.<br/>
     *
     * @warning If you do not notify the completion of the purchase to the market after payment has been made, it will be left as an unpaid item..<br/>
     * Since the unpaid items will continue to be recovered until the transactionFinish API call, you should check for duplicate receipts.<br/>
     *
     * @param marketPid    The unique ID of the item for which the transaction is to be completed.
     * @param listener    Result of API call
     *
     *  \~
     * @see IAPV4TransactionFinishListener#transactionFinish(String, IAPV4TransactionFinishListener)
     * @see IAPV4TransactionMultiFinishListener#transactionMultiFinish(List, IAPV4TransactionMultiFinishListener)
     *
     * @ingroup IAPV4
     */
    static void transactionFinish(std::string marketPid,onTransactionFinish listener);

    /**
     *  \~korean
     * @brief HIVE IAP v4에서 지급이 완료된 상품에 대한 구매 영수증에 대한 복수 개의 트랜잭션 완료 처리를 요청한다.<br/>
     * 구매나 복구 기능 사용 후 아이템 지급이 완료된 시점에 이 API를 호출하여 마켓에 구매 완료 처리를 해야한다.<br/>
     *
     * @warning 만약, 지급 완료 후에도 마켓으로의 구매 완료 처리를 하지 않으면, 미지급된 상품으로 남는다.<br/>
     * 미지급된 상품은 transactionFinish API 호출 전까지 계속 복구가 되므로, 영수증 중복 여부를 확인해야 한다.<br/>
     *
     * @param marketPidList    지급이 완료되어 트랜잭션 완료 처리를 위한 상품의 고유 ID 목록
     * @param listener            API 결과 통지
     *
     *  \~english
     * @brief HIVE IAP v4 It requests the multiple transaction completion processing for the purchase receipt of the item that have been delivered.<br/>
     * At the completion of item delivery after purchasing or restoring, this API should be called to complete the purchase on the market.<br/>
     *
     * @warning If you do not notify the completion of the purchase to the market after payment has been made, it will be left as an unpaid item..<br/>
     * Since the unpaid items will continue to be recovered until the transactionFinish API call, you should check for duplicate receipts.<br/>
     *
     * @param marketPidList    The list of unique ID of the item for which the transaction is to be completed.
     * @param listener    Result of API call
     *
     *  \~
     * @see IAPV4TransactionMultiFinishListener#transactionMultiFinish(List, IAPV4TransactionMultiFinishListener)
     * @see IAPV4TransactionFinishListener#transactionFinish(String, IAPV4TransactionFinishListener)
     *
     * @ingroup IAPV4
     */
    static void transactionMultiFinish(std::vector<std::string> const & marketPidList,onTransactionMultiFinish listener);

    /**
     *  \~korean
     * @brief 러비 상점이나 구글 상점을 선택하기 위한 창을 띄운다.<br/>
     * 만약 Google Store 와 HIVE 러비(Lebi) 상점을 동시에 사용 가능한 상황이라면 유저가 상점 선택을 할 수 있도록 상점 선택 화면을 노출해야 한다
     *
     * @param listener            API 결과 통지
     *
     *  \~english
     * @brief It shows a UI to select a Lebi store or a Google store..<br/>
     * If the Google Store and the HIVE Lebi store are available at the same time, you should expose the store selection screen to allow the user to select a store.
     *
     * @param listener            Result of API call
     *
     *  \~
     * @ingroup IAPV4
     */
    static void showMarketSelection(onMarketConnect listener);

    /**
     *  \~korean
     * @brief 러비 상점일 경우 잔액 정보 조회.
     *
     * @param listener            API 결과 통지
     *
     *  \~english
     * @brief It request Lebi balance information.
     *
     * @param listener            Result of API call
     *
     *  \~
     * @ingroup IAPV4
     */
    static void getBalanceInfo(onBalance listener);

    /**
     *  \~korean
     * @brief 러비 충전 페이지 노출
     *
     * @param listener            API 결과 통지
     *
     *  \~english
     * @brief It shows Lebi refill page.
     *
     * @param listener            Result of API call
     *
     *  \~
     * @ingroup IAPV4
     */
    static void showCharge(onBalance listener);
    
    /**
     @brief Apple AppStore Promote InApp Purchase 동작으로 얻어온 MarketPID를 요청한다.<br/>
     앱의 시작, 앱의 Background -> Foreground시에 이 API를 호출하여 MarketPID를 얻어오도록 한다.<br/>
     
     얻어온 MarketPID로 구매를 진행하도록 한다.<br/>
     
     @warning iOS 11 이상에서만 지원한다.<br/>
     
     @see #purchase:iapPayload:handler:
     @see HIVEIAPV4CheckPromotePurchaseHandler
     
     @param listener API 호출 결과 통지
     */
    static void checkPromotePurchase(onCheckPromotePurchase listener);

    /**
     * @brief 현재 선택된 마켓 타입을 반환한다. <br/>
     *
     * 얻어온 마켓 타입으로 구매를 진행하도록 한다.<br/>
     *
     * @see #marketConnect(IAPV4MarketInfoListener)
     *
     * @ingroup IAPV4
     */
    static IAPV4Type getSelectedMarket();

    /**
     * @brief 현재 계정의 UUIDv3 값을 반환한다. <br/>
     *
     * Returns the Hive account ID converted to a UUID version 3. return null, if not logged in.
     *
     * @ingroup IAPV4
     * @since 4.16.3.0
     *
     * @return accountUuid
     */
    static std::string getAccountUuid();

    /**
     * Google In-App Offers
     *
     * @param categoryId 3(PROMOTIONAL)
     *
     * @param listener Int(responseCode) 가 2(NAVIGATE_TO_DEV_APP_STORE) 일 경우 유저가 제안(Offer)를 확인하고 사용하기를 원한 상황입니다.
     * 개발자는 유저가 구매할 인앱 항목을 선택할 수 있는 게임/앱 내 상점으로 (유저가 해당 화면에 없는 경우) 유저를 이동해야 합니다.
     *
     * 0 은 아무런 액션을 하지 않아도 되고, 음수는 에러상황 입니다.
     * 
     * @since 4.21.0.0
     */
    static void showInAppMessages(int categoryId, onInAppMessages listener);
    
    /**
     * @brief 소비정보 전송 약관에 대한 유저의 마지막 동의 상태 및 시기를 반환한다. <br/>
     *
     * Returns the lastest consent status and timing for the Consumption Information Transmission Terms.
     *
     * @ingroup IAPV4
     * @since 4.25.0.0
     */
    static void getConsumeInfoAgreement(onGetConsumeInfoAgreement listener);
    
    /**
     * @brief 소비정보 전송 약관 팝업을 노출한다. <br/>
     *
     * Displays the Consumption Information Transmission Terms Popup.
     *
     * @ingroup IAPV4
     * @since 4.25.0.0
     */
    static void showConsumeInfoAgreement(onShowConsumeInfoAgreement listener);


    static void executeEngine(picojson::value jsonParam);
};

/**
 * @brief HIVE IAP 백오피스에서 생성한 상품 정보<br/>
 * 마켓에 등록된 상품(Market Pid) 의 정보를 포함한다.
 *
 * @ingroup IAPV4
 */
class HIVESDK_DLLEXPORT IAPV4Product{
public:
    IAPV4Product();
    IAPV4Product(picojson::value const& jsonParam);

    std::string productType;            ///< \~korean consumable, subscription     \~english     consumable, subscription
    std::string marketPid;              ///< \~korean 마켓에 등록된 고유 상품 ID     \~english     Unique product ID registered in Market
    std::string currency;               ///< \~korean 통화 코드 (ex, KRW / USD)     \~english     Currency code (ex, KRW / USD)
    double price;                       ///< \~korean 상품 가격         \~english     Product price
    std::string displayPrice;           ///< \~korean 상품 가격 문자열 (ex. $100.00)         \~english     Product price string (ex. $100.00)
    std::string title;                  ///< \~korean 상품 타이틀         \~english     Item Title
    std::string productDescription;     ///< \~korean 상품 설명         \~english     Product Description
    std::string originalMarketJson;     ///< \~korean 마켓에 등록된 원본 상품 정보         \~english     Original Product info in Market

    //Price before Promotion Sale
    std::string displayOriginalPrice;   /// 세일전 가격 문자열 Google, Apple Only
    double originalPrice;               /// 세일전 가격 Google, Apple Only
    /**
     * 32-bit PNG (with alpha)
     *
     * Dimensions: 512px by 512px
     *
     * Maximum file size: 1024KB
     */
    std::string iconURL;                /// 상품 아이콘 (512x512) Google Only
    int coinsReward;                    /// 구매시 추가 지급 코인량
};

/**
 * {@link #purchase(String, String, IAPV4PurchaseListener)} 나 {@link #restore(IAPV4RestoreListener)} API 에서 받을 수 있는 결과 정보.<br/>
 * {@link IAPV4Type} 에 따라 {@link IAPV4ReceiptGoogle} 등으로 형변환하여 사용할 수 있다.<br/>
 *
 * @see IAPV4ReceiptGoogle
 * @see IAPV4ReceiptLebi
 * @see IAPV4ReceiptOneStore
 *
 * @ingroup IAPV4
 */
class HIVESDK_DLLEXPORT IAPV4Receipt{
public:
    static std::shared_ptr<IAPV4Receipt> create(picojson::value const& jsonParam);
    IAPV4Receipt();
    virtual ~IAPV4Receipt();
    

    IAPV4Type type;                     ///< \~korean 영수증 마켓 정보    \~english  Receipt Market Information
    IAPV4Product product;               ///< \~korean 구매 성공한 상품 정보    \~english  Purchased product information.
    std::string iapPayload;             ///< \~korean 구매 요청시 추가로 입력받은 iapPayload. Restore 영수증은 서버 검증 응답으로 확인해야 한다.    
                                        ///  \~english iapPayload entered in purchase request. The restore receipt must be confirmed with the server validation response.
    std::string hiveiapReceipt;         ///< \~korean bypassInfo가 IAP서버 영수증 검증 요청시 보낼 데이터    \~english  Receipt Data to send when requesting validation to IAP server
    std::string bypassInfo;
    int quantity;                       ///< \~korean 상품 갯수 (Google Play 다중 구매 지원)    \~english Purchase quantity (Support Google Play multi-purchase)
    
    static IAPV4Type getIAPV4TypeFromJson(picojson::value const& jsonParam);
    
protected:
    IAPV4Receipt(picojson::value const& jsonParam);
};

class HIVESDK_DLLEXPORT IAPV4PurchaseParam{
public:
    std::string marketPid;
    std::string iapPayload;
    std::string oldMarketPid;
    std::string offerToken;
    
    IAPV4PurchaseParam();
    IAPV4PurchaseParam(picojson::value jsonParam);
    
    picojson::object toJson() const;
    std::string toString() const;};

class HIVESDK_DLLEXPORT IAPV4ConsumeInfoUserAgreement{
public:
    bool didAgree;
    int64_t firstAnsweredDate;
    int64_t lastAnsweredDate;
    
    IAPV4ConsumeInfoUserAgreement();
    IAPV4ConsumeInfoUserAgreement(picojson::value const jsonParam);
};
NS_HIVE_END

#endif /* __HIVE_IAPV4_H__ */
