//
//  HIVE_MatchMaking.hpp
//  HIVE_SDK_Plugin
//
//  Created by jschoi on 2024/09/26.
//


#ifndef __HIVE_MATCH_MAKING_H__
#define  __HIVE_MATCH_MAKING_H__

#include "HIVE_AuthV4.h"        //  hive::PlayerID
#include "HIVE_ResultAPI.h"
#include <map>

NS_HIVE_BEGIN

typedef long long PlayerID;

class ResultAPI;

class MatchMakingData;
class MatchMakingGroupData;
class MatchingMemberInfo;
class MatchingResultPlayerInfo;
class MatchingResultTeamInfo;

/**
 *  \~korean
 * Hive MatchMaking은 온라인 매치를 위해 플레이어를 연결하고 관련 데이터를 제공합니다. <br/>
 * Hive MatchMaking 기능을 통해 게임에서 구체적인 매치메이킹 로직을 구현하지 않아도 플레이어의 대전 상대를 간단하게 얻을 수 있습니다. <br/>
 * 플레이어가 비슷한 수준을 가진 상대와 매치되도록 하여 균형잡힌 게임 경험을 제공합니다. <br/>
 *
 *  \~english
 * Hive MatchMaking connects players for online matches and provides relevant data. <br/>
 * Hive MatchMaking feature makes it simple to get matches for your players without having to implement specific matchmaking logic in your game. <br/>
 * It provides a balanced gaming experience by matching players with opponents of similar skill level. <br/>
 *
 *  \~
 *
 * @since   4.24.5.0
 * @ingroup MatchMaking
 */
class HIVESDK_DLLEXPORT MatchMaking
{
public:

    typedef std::function<void(ResultAPI const & result)> onMatchMakingResult;

    typedef std::function<void(ResultAPI const & result, MatchMakingData const & matchMakingData)> onMatchMakingData;
    
    typedef std::function<void(ResultAPI const & result, MatchMakingGroupData const & matchMakingGroupData)> onMatchMakingGroupData;


    /**
     *  \~korean
     * 매칭 요청<br/>
     *
     * 매칭 요청 시 사용할 점수(point)를 입력할 수 있습니다.<br/>
     * 매칭에 사용할 부가 정보(닉네임, 레벨, 국가 등)(extraData)를 입력할 수 있습니다.<br/>
     * 부가 정보는 매칭 결과에 포함되어 전달됩니다.<br/>
     *
     *  \~english
     * Matching Request<br/>
     *
     * You can enter the score(point) to be used when requesting matching.<br/>
     * You can also enter additional information(nickname, level, country, etc.)(extraData) to be used for matching.<br/>
     * ExtraData is delivered as part of the matching results.<br/>
     *  \~
     *
     * @since   4.24.5.0
     * @ingroup MatchMaking
     */
    static void requestMatchMaking(int matchId, int point, const char * extraData, onMatchMakingData listener);

    /**
     *  \~korean
     * 매칭 상태 확인<br/>
     *
     * 요청한 매칭 상태를 확인할 수 있습니다. <br/>
     *
     *  \~english
     * Check Matching Status<br/>
     *
     * You can check the status of your requested matching.<br>
     *  \~
     *
     * @since   4.24.5.0
     * @ingroup MatchMaking
     */
    static void getRequestingStatus(int matchId, onMatchMakingData listener);

    /**
     *  \~korean
     * 매칭 요청 삭제<br/>
     *
     * 요청한 매칭을 삭제합니다.<br/>
     *
     *  \~english
     * Delete Matching Request<br/>
     *
     * Deletes the requested match.<br/>
     *  \~
     *
     * @since   4.24.5.0
     * @ingroup MatchMaking
     */
    static void deleteRequesting(int matchId, onMatchMakingResult listener);
    
    
    /**
     * \~korean
     * 매칭 그룹 생성<br/>
     *
     * 매칭 그룹을 생성합니다.<br/>
     *
     *  \~english
     *  Create a matching group<br/>
     *
     *  Create a matching group.<br/>
     *  \~
     *
     * @since 4.25.0.0
     * @ingroup MatchMaking
     */
    static void createGroup(int matchId, int point, const char * extraData, onMatchMakingGroupData listener);
    
    /**
     * \~korean
     * 매칭 그룹 참여<br/>
     *
     * 매칭 그룹에 참여합니다.<br/>
     *
     *  \~english
     * Join a matching group<br/>
     *
     * Join a matching group.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup MatchMaking
     */
    static void joinGroup(int matchId, const char * groupCode, int point, const char * extraData, onMatchMakingGroupData listener);
  
    /**
     * \~korean
     * 매칭 그룹 탈퇴<br/>
     *
     * 매칭 그룹에서 탈퇴합니다.<br/>
     *
     *  \~english
     * Leave a matching group<br/>
     *
     * Leave a matching group.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup MatchMaking
     */
    static void leaveGroup(int matchId, onMatchMakingResult listener);
    
    /**
     * \~korean
     * 그룹원 추방<br/>
     *
     * 매칭 그룹에서 그룹원을 추방시킵니다.<br/>
     *
     * \~english
     * Kick a group member<br/>
     *
     * Kick a group member<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup MatchMaking
     */
    static void kickGroupUser(int matchId, PlayerID targetPlayerId, onMatchMakingGroupData listener);
    
    /**
     * \~korean
     * 그룹 정보 조회 (그룹 내 유저 기준)<br/>
     *
     * 유저 정보를 기준으로 그룹 정보를 조회합니다.<br/>
     *
     * \~english
     * Get group information (based on group users)<br/>
     *
     * Get group information based on user information.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup MatchMaking
     */
    static void getGroupInfoByUser(int matchId, onMatchMakingGroupData listener);
    
    /**
     * \~korean
     * 그룹 정보 조회 (그룹 코드 기준)<br/>
     *
     * 그룹 코드를 기준으로 그룹 정보를 조회합니다.<br/>
     *
     * \~english
     * Get group information (based on group code)<br/>
     *
     * Get group information based on group code.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup MatchMaking
     */
    static void getGroupInfoByGroupCode(const char * groupCode, onMatchMakingGroupData listener);
    
    /**
     * \~korean
     * 멤버 정보 수정<br/>
     *
     *그룹에 속한 구성원이 자신의 정보를 변경시킵니다.<br/>
     *
     * \~english
     * Modify member information<br/>
     *
     * Group members change their information.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup MatchMaking
     */
    static void updateGroupUser(int matchId, bool ready, int point, const char * extraData, onMatchMakingGroupData listener);
    
    /**
     * \~korean
     * 그룹 매칭 요청<br/>
     *
     * 그룹 매칭을 요청합니다.<br/>
     *
     * \~english
     * Request group matching<br/>
     *
     * Request group matching.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup MatchMaking
     */
    static void requestGroupMatching(int matchId, onMatchMakingGroupData listener);
    
    /**
     * \~korean
     * 그룹 매칭 요청 취소<br/>
     *
     * 그룹 매칭 요청을 취소합니다.<br/>
     *
     * \~english
     * Cancel group matching request<br/>
     *
     * Cancel group matching request.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup MatchMaking
     */
    static void deleteGroupMatching(int matchId, onMatchMakingGroupData listener);


    // \~korean Native 영역에서 호출된 요청을 처리하기 위한 플러그인 내부 코드
    // \~english Plug-in internal code to handle requests invoked from the native code.
    static void executeEngine(picojson::value jsonParam);

};

/**
 * @brief 매치 메이킹 그룹 멤버 정보<br/>
 *
 * @ingroup MatchMaking
 */
class HIVESDK_DLLEXPORT MatchingMemberInfo {
public:
    // 플레이어 ID / Player ID
    hive::PlayerID playerId;
    // 준비 상태 (빈값인 경우 false) / Ready Status (false if empty)
    bool ready;
    // 매칭 포인트 / Matching Point
    int point;
    // 추가 데이터 (선택사항) / Extra Data (nickName, Level, Etc..)
    std::string extraData;

    MatchingMemberInfo();

    bool parse(picojson::value jsonParam);
    picojson::object toJson() const;
    std::string toString() const;
};

/**
 * @brief 매치 메이킹 그룹 데이터 정보<br/>
 *
 * @ingroup MatchMaking
 */
class HIVESDK_DLLEXPORT MatchMakingGroupData {
public:
    // 그룹 코드 / Group Code
    std::string groupCode;
    // 그룹 소유자 플레이어 ID / Group Owner Player ID
    hive::PlayerID ownerPlayerId;

    // 매칭 멤버 정보 리스트 / Matching Member Information List
    std::vector<MatchingMemberInfo> memberInfoList;
    
    // 게임인덱스 / Game Index
    int requestGameIndex;
    // 매치 ID / Match ID
    int requestMatchId;
    
    // 매칭 요청 상태 / Matching Request Status
    std::string requestStatus;
    // 요청 시각 (UTC) String / Request Time (UTC) String
    std::string requestTimeUtc;
    
    // 매칭 상태 / Matching Status
    std::string matchingStatus;
    // 매칭 ID / Matching ID
    std::string matchingId;
    // 매칭 타입 / Matching Type
    std::string matchingType;

    // 매칭 팀 정보 리스트 / Matching Team Information List
    std::vector<MatchingResultTeamInfo> matchingTeamInfoList;
    
    MatchMakingGroupData();

    bool parse(picojson::value jsonParam);
    picojson::object toJson() const;
    std::string toString() const;
};


/**
 * @brief Match Making 콜백 데이터 정보<br/>
 *
 * @ingroup MatchMaking
 */
class HIVESDK_DLLEXPORT MatchMakingData {
public:
    hive::PlayerID requestPlayerId;
    int requestGameIndex;
    int requestMatchId;
    
    std::string requestStatus;
    std::string requestTimeUtc;
    int requestPoint;
    std::string requestExtraData;
    
    std::string matchingStatus;
    std::string matchingId;
    std::string matchingType;
    
    std::vector<MatchingResultPlayerInfo> matchingPlayerInfoList;
    std::vector<MatchingResultTeamInfo> matchingTeamInfoList;

    MatchMakingData();

    bool parse(picojson::value jsonParam);
    picojson::object toJson() const;
    std::string toString() const;
};


class HIVESDK_DLLEXPORT MatchingResultPlayerInfo {
public:
    hive::PlayerID playerId;
    int point;
    std::string extraData;
    
    MatchingResultPlayerInfo();
    
    bool parse(picojson::value jsonParam);
    picojson::object toJson() const;
    std::string toString() const;
};

class HIVESDK_DLLEXPORT MatchingResultTeamInfo {
public:
    int teamIndex;
    std::vector<MatchingResultPlayerInfo> playerInfos;
    
    MatchingResultTeamInfo();
    
    bool parse(picojson::value jsonParam);
    picojson::object toJson() const;
    std::string toString() const;
};

// MARK: -
NS_HIVE_END        // namespace hive


#endif        // __HIVE_MATCH_MAKING_H__


/** @} */



