/**
 * @brief 플랫폼 사용 편의를 위한 헬퍼 기능들의 모음
 *
 * @author joosangkang
 *
 * @defgroup PlatformHelper
 * @{
 *
 */

#ifndef HIVE_PlatformHelper_hpp
#define HIVE_PlatformHelper_hpp

#include "HIVE_ResultAPI.h"

NS_HIVE_BEGIN

enum class PlatformShareType;
enum class PlatformShareDataType;
class PlatformShare;
class InAppBrowserParam;
class OpenBrowserParam;
class InAppWebViewParam;

/**
 * 플랫폼에서 지원하는 기능을 제공하는 클래스이다.
 * @ingroup PlatformHelper
 */
class HIVESDK_DLLEXPORT PlatformHelper
{
public:
    
    /**
     * @brief Share 관련 동작이 완료되었을 때 호출됨. (shareText, shareMedia)
     *
     * @param isSuccess Share 결과.
     *
     * @ingroup PlatformHelper
     */
    typedef std::function<void(bool isSuccess)> onHIVEPlatformHelperShareHandler;
    
    /**
     * @brief 하나 또는 복수 개의 미디어(이미지, 오디오, 비디오) 또는 텍스트를 공유한다.
     *
     * @param platformShare 공유할 미디어의 Uri(s)와 공유할 텍스트
     */
    static void share(PlatformShare platformShare, onHIVEPlatformHelperShareHandler handler);

    /**
     * @brief Android에서 재요청된 OS 권한동의에 대한 결과 값을 반환한다.
     *
     * @param granted 사용자가 수락한 권한 리스트, denied 사용자가 거부한 권한 리스트
     *
     * @ingroup PlatformHelper
     */
    typedef std::function<void(ResultAPI const & result, std::vector<std::string> const & granted, std::vector<std::string> const & denied)> onRequestUserPermissionsHandler;

    /**
     * @brief Android에서 사용자에게 OS 권한을 재요청.
     *
     * @param requests Android에서 요청할 권한들을 포함한 리스트.
     *
     * @ingroup PlatformHelper
     */
    static void requestUserPermissions(std::vector<std::string> requests, onRequestUserPermissionsHandler handler);

    /**
     * @brief 업데이트 팝업 설정으로 백그라운드에서 앱 다운로드가 완료되면 
     * 
     * UE에서 Promotion.EngagementEventType.APPUPDATE_DOWNLOADED 로 신호를 보낸다.
     * 
     * 이후 completeUpdate() 를 호출하면 completeState 값에 따라 새 버전으로 설치한다.
     * 
     * 호출 하지 않으면 기본 동작으로 재시작시 설치를 진행한다.
     * 
     * @param completeState 1 ImmediateUpdate, 2 RestartUpdate, 3 LaterUpdate. otherwise 1.
     */
    static void completeUpdate(int completeState);
    
    /**
     * @brief 인 앱 브라우저 콜백 리스너
     */
    typedef std::function<void(ResultAPI const & result)> onShowInAppBrowserHandler;
    
    /**
     * @brief 인 앱 브라우저를 오픈한다.
     *
     * @param param [InAppBrowserParam]
     * @param handler [onShowInAppBrowserHandler]
     *
     * @since 24.0.0
     */
    static void showInAppBrowser(InAppBrowserParam param, onShowInAppBrowserHandler handler);
    
    // Native 영역에서 호출된 요청을 처리하기 위한 플러그인 내부 코드
    static void executeEngine(picojson::value jsonParam);

    /**
     * \~korean
     * @brief 외부 브라우저를 오픈합니다.
     *
     * @param url 열고자 하는 웹사이트 주소
     * @param useIncognito 시크릿 모드 사용 여부
     * @return 없음
     *
     * \~english
     * @brief Opens an external browser.
     *
     * @param url URL of the website to open
     * @param useIncognito Whether to use incognito mode
     * @return None
     *
     * \~
     * @ingroup PlatformHelper
     */
    static void openBrowser(const OpenBrowserParam param);

    /**
     * @brief 인 앱 웹뷰 콜백 리스너
     */
    typedef std::function<void(ResultAPI const& result)> onShowInAppWebViewHandler;

    /**
     * @brief 인 앱 웹뷰를 오픈한다.
     *
     * @param param [InAppWebViewParam]
     * @param handler [onShowInAppWebViewHandler]
     *
     * @since 24.5.0
     */
    static void showInAppWebView(InAppWebViewParam param, onShowInAppWebViewHandler handler);

    /**
     * @brief 게임 실행 시 사용된 인수 정보를 반환하는 콜백 리스너
     *
     * @param result [ResultAPI] 처리 결과 정보
     * @param parameters [const char*] 게임 실행 시 사용된 인수 정보
     *
     */
    typedef std::function<void(ResultAPI const& result, const char* parameters)> onParameters;

    /**
     * @brief 게임 실행 시 사용된 인수 정보를 반환한다.
     *
     * 게임 실행 시 사용된 인수 정보를 리스너를 통해 전달한다.
     *
     * @param paramListener [onParameters] 게임 실행 시 사용된 인수 정보를 수신할 콜백 함수
     *
     * @since 25.3.0
     */
    static void getLaunchParameters(onParameters paramListener);
};

/**
 * \~korean  공유 타입
 *
 * \~english Types of Platform share
 * \~
 * @ingroup PlatformHelper
 */
enum class PlatformShareType
{
    TEXT     = 1       ///< \~korean 텍스트  \~english  Text
    , MEDIA  = 2      ///< \~korean 미디어       \~english Media
};

/**
 * \~korean  공유 데이터 타입
 *
 * \~english Types of Platform share data
 * \~
 * @ingroup PlatformHelper
 */
enum class PlatformShareDataType
{
    SUBJECT = 0,   ///< \~korean 주제    \~english  Subject
    TEXT,         ///< \~korean 텍스트       \~english Text
    MEDIA,         ///< \~korean 미디어       \~english Media
};

class HIVESDK_DLLEXPORT PlatformShare
{
public:
    PlatformShare();
    ~PlatformShare();
    
    PlatformShareType getShareType();
    int getShareTypeInt();
    void setShareType(PlatformShareType shareType);
    
    std::string getSubject();
    void setSubject(std::string subject);
    
    std::string getText();
    void setText(std::string text);
    
    std::vector<std::string> getMedia();
    void setMedia(std::vector<std::string> paths);
    
private:
    PlatformShareType shareType;
    
    std::string subject;
    std::string text;
    std::vector<std::string> paths;
};

/**
 * @brief 인앱 브라우저 설정을 위한 파라미터
 *
 * @see url: 연결 URL
 * @see buttonColor: 버튼 색상 (hex color)
 * @see navigationColor: 네비게이션 바 색상 (hex color)
 * @see urlBarHiding: 페이지 스크롤시 네비게이션 바 영역 최소화. 디폴트 값 true
 * @see autoRedirectToExternalBrowser: 외부 브라우저 자동 연결 기능 (Android only)
*/
class HIVESDK_DLLEXPORT InAppBrowserParam
{
public:
    std::string url;
    std::string buttonColor;
    std::string navigationColor;
    bool urlBarHiding;
    bool autoRedirectToExternalBrowser;
    
    InAppBrowserParam(std::string url);
    InAppBrowserParam(picojson::value jsonParam);
    
    picojson::object toJson() const;
    std::string toString() const;
};

/**
 * @brief 외부 브라우저를 열기 위한 클래스
 *
 * @see url: 연결할 URL을 나타냅니다.
 * @see useIncognito: 시크릿 모드 사용 여부를 나타냅니다. true일 경우 시크릿 모드로 브라우저를 엽니다.
 */
class HIVESDK_DLLEXPORT OpenBrowserParam {
public:
    std::string url;
    bool useIncognitoMode;

    OpenBrowserParam(const std::string& url);
};

/**
 * @brief 인앱 웹뷰 설정을 위한 파라미터
 *
 * @see url: 연결 URL
 * @see postData: 기본값은 empty. null or empty 일 경우 get 통신을 한다. 값이 있으면 post 통신을 한다.
 * @see useUserSession: 기본값은 false. true 시 playerId, playerToken 을 포함한 json 데이터로 post 통신한다.
 * @see useGameWindow: 안드로이드에서만 동작한다. true로 설정한 경우, 현재 window에 웹뷰를 노출한다. pause 상태가 되지 않으며 HiveTheme와 HiveOrientation이 무시된다. 기본값은 false 이다.
*/
class HIVESDK_DLLEXPORT InAppWebViewParam
{
public:
    std::string url;
    std::string postData;
    bool useUserSession;
    bool useGameWindow;
    
    InAppWebViewParam(std::string url);
    InAppWebViewParam(picojson::value jsonParam);
    
    picojson::object toJson() const;
    std::string toString() const;
};

NS_HIVE_END

#endif /* HIVE_PlatformHelper_hpp */


/** @} */
