/**
 * @file    HIVE_Promotion.h
 *
 * @author  ryuvsken
 * @date		2016-2019
 * @copyright	Copyright © GAMEVIL COM2US PLATFORM Inc. All Right Reserved.
 *  @since		4.0.0
 *  \brief		\copydoc 	Promotion
 */


/**
 * @defgroup Promotion
 * @{
 * \~korean  HIVE 프로모션은 게임을 이용하는 유저에게 게임의 새로운 소식이나 이벤트를 효과적으로 노출하는 기능을 제공한다<br/>
 * 프로모션 뷰 (공지, 이벤트), 보상 (쿠폰, 딥링크), 종료 팝업 (안드로이드), 무료 충전소, 리뷰 유도 팝업<br/><br/>
 * \~english HIVE Promotion gives users who use the game the ability to effectively expose new news or events in the game.<br/>
 * Promotion view (notice, event), reward (coupon, deep link), exit popup (Android), free charging station(Offerwall)<br/><br/>
 */

#ifndef __HIVE_PROMOTIONVIEW_H__
#define  __HIVE_PROMOTIONVIEW_H__


#include "HIVE_ResultAPI.h"
#include <vector>

NS_HIVE_BEGIN


class ResultAPI;

enum class PromotionType;
enum class PromotionCustomType;
enum class PromotionEventType;
enum class PromotionBadgeTarget;
enum class PromotionCampaignType;
enum class PromotionBannerType;
enum class OfferwallState;
enum class EngagementEventType;
enum class EngagementEventState;

class PromotionViewInfo;
class PromotionBadgeInfo;
class PromotionBannerInfo;
//class OfferwallReward;
class AppInvitationData;
class AppInvitationCampaign;
class AppInvitationCampaignStage;
class AppInvitationSenderInfo;


/**
 * \~korean  HIVE 프로모션은 게임을 이용하는 유저에게 게임의 새로운 소식이나 이벤트를 효과적으로 노출하는 기능을 제공한다<br/>
 * 프로모션 뷰 (공지, 이벤트), 보상 (쿠폰, 딥링크), 종료 팝업 (안드로이드), 무료 충전소, 리뷰 유도 팝업<br/><br/>
 *
 * \~english HIVE Promotion provides you with the ability to effectively expose new news or events to the game user.<br/>
 * Promotion view (notice, event), reward (coupon, deep link), exit popup (Android), free charging station(Offerwall)<br/><br/>
 * \~
 * @author ryuvsken
 * @since		4.0.0
 * @ingroup Promotion
 */

class HIVESDK_DLLEXPORT Promotion {

public:

    /**
     * \~korean  프로모션 뷰 API 결과 통지
     *
     * @param result				API 호출 결과
     * @param promotionEventType	프로모션 창 이벤트 형태
     *
     * \~english Promotion View API Result
     *
     * @param result				API call result
     * @param promotionEventType	Promotion event type
     * \~
     * @ingroup Promotion
     */
    typedef std::function<void(ResultAPI const & result, PromotionEventType promotionEventType)> onPromotionView;


    /**
     * \~korean  HIVE 프로모션 웹 뷰의 UI 를 게임 UI 의 컨셉에 맞추기 위해서 프로모션 웹 뷰를 게임에서 직접 구현하기 위한 데이터 반환
     *
     * @param result	API 호출 결과
     * @param promotionViewInfo	프로모션 웹 뷰 정보
     * \~english Returns HIVE Promotion Webview information so that your UI of webview is configured according to the concept of game UI.
     *
     * @param result	API call result
     * @param promotionViewInfo	HIVE Promotion Webview information
     * \~
     * @ingroup Promotion
     */
    typedef std::function<void(ResultAPI const & result, std::vector<PromotionViewInfo> const & promotionViewInfo)> onPromotionViewInfo;


    /**
     * \~korean  HIVE 프로모션 뱃지 정보 반환
     *
     * @param result	API 호출 결과
     * @param badgeInfoList		프로모션 뱃지 정보
     * \~english Returns HIVE Promotion badge information.
     *
     * @param result	API call result
     * @param badgeInfoList		HIVE Promotion badge information
     * \~
     * @ingroup Promotion
     */
    typedef std::function<void(ResultAPI const & result, std::vector<PromotionBadgeInfo> const & badgeInfoList)> onPromotionBadgeInfo;


    /**
     *  \~korean 프로모션 배너 정보 API 호출에 대한 결과 통지
     *
     * @param result    API 호출 결과
     * @param bannerInfoList   프로모션 배너 정보
     *  \~english HIVE Promotion banner information.
     *
     * @param result    API call result
     * @param bannerInfoList        HIVE Promotion banner information
     * \~
     * @ingroup Promotion
     */
    typedef std::function<void(ResultAPI const & result, std::vector<PromotionBannerInfo> const & bannerInfoList)> onPromotionBannerInfo;


    /**
     * \~korean  앱 초대 (UserAcquisition) 정보 요청에 대한 정보 반환
     *
     * @param result	API 호출 결과
     * @param appInvitationData		앱 초대 정보
     * \~english Return information about request for user invite (UserAcquisition)
     *
     * @param result	API call result
     * @param appInvitationData		User invite information.
     * \~
     * @ingroup Promotion
     */
    typedef std::function<void(ResultAPI const & result, AppInvitationData const & appInvitationData)> onAppInvitationData;

    /**
     * \~korean  SDK 가 특정한 조건에서 클라이언트에 개입 (Engagement) 하기 위한 이벤트 리스너.<br>
     * 여기서 특정한 조건은 모바일 메시지 (SMS), 푸시 알림 (Push Notification) 으로 전송된 메시지의 URL 클릭이나 프로모션 뷰에서 사용자 동작 등이 있다.
     *
     * @param result					API 호출 결과
     * @param engagementEventType		이벤트 타입
     * @param engagementEventState		이벤트 상태
     * @param param						이벤트 정보
     *
     * \~english  An event listener for the SDK to engage clients in certain conditions.<br>
     * The specific conditions are, for example, a mobile message (SMS), a URL click on a message in a push notification, or a user action in a promotional view.
     *
     * @param result					Result of API call
     * @param engagementEventType		Event type
     * @param engagementEventState		Event status
     * @param param						Event information
     * \~
     * @ingroup Promotion
     */
    typedef std::function<void(ResultAPI const & result, EngagementEventType engagementEventType, EngagementEventState engagementEventState, picojson::value const & param)> onEngagement;
    typedef std::function<void(ResultAPI const& result, EngagementEventType engagementEventType, EngagementEventState engagementEventState, std::string const& param)> onEngagementString;

    /**
     * \~korean  HIVE UA 공유 정보 반환
     *
     * @param result    API 호출 결과
     * \~english Returns HIVE UA Share information.
     *
     * @paraam result   API call result
     */
    typedef std::function<void(ResultAPI const & result)> onPromotionUAShare;

    /**
     *  \~korean UA를 통해 자신을 앱으로 최초로 초대 성공한 유저의 정보 반환 한다.
     *
     * @param result    API 호출 결과
     * @param senderInfo   자신을 앱으로 최초로 초대 성공한   유저의 정보
     *
     *  \~english  First sender's userInfo who sent UA share invitation
     *
     * @param result Result of API call
     * @param senderInfo First sender's userInfo
     * @ingroup Promotion
     */
    typedef std::function<void(ResultAPI const & result, AppInvitationSenderInfo const & senderInfo)> onAppInvitationSenderInfo;


    /**
     * \~korean  게임의 새로운 이벤트나 새로운 게임 소개등의 배너 화면을 노출
     *
     * @param promotionType		프로모션 뷰 창의 형태
     * @param isForced			type이 "notice" 와 "event" 일 때만 동작
     *
     * 							true일 경우 24시간 다시보지 않기를 무시한다.
     * 							(주의:24시간 다시보기가 적용되지 않으므로 버튼 액션으로 보는 등 특정 액션에서만 사용해야 함.)
     *
     * 							false이거나 생략할 경우 24시간 안보기가 정상동작함.
     * @param listener			API 결과 통지
     * \~english Shows banner such as new event of game or introduce new game
     *
     * @param promotionType		Promotion View type
     * @param isForced			If true, it only works when promotionType is "notice" and "event"
     *
     * 							and it ignore 'Not seeing it again for 24 hours'.
     * 							(Note: Since 'Not seeing it again for 24 hours' does not apply, you should only use certain actions, such as viewing as a button action.)
     *
     * 							If false or omitted, 'Not seeing it again for 24 hours' will operate normally.
     * @param listener			API call result listener
     * \~
     * @ingroup Promotion
     */
    static void showPromotion(PromotionType promotionType, bool isForced, onPromotionView listener);


    /**
     * \~korean 외부 컨텐츠를 사용하기 위해서 커스텀 웹 뷰 노출
     * @param promotionCustomType		PromotionCustomType Type
     * @param contentsKey				백오피스에 설정된 사용자 정의 페이지의 id
     * @param listener 					API 결과 통지 리스너
     *
     *  \~english Show custom Webview to use external content.
     * @param promotionCustomType		PromotionCustomType Type
     * @param contentsKey				ID of the custom web page set in the back office.
     * @param listener					API call result listener
     * \~
     * @ingroup Promotion
     */
    static void showCustomContents(PromotionCustomType customType, std::string contentsKey, onPromotionView listener);

    /**
     * \~korean 외부 컨텐츠를 사용하기 위해서 커스텀 웹 뷰 노출 (GameWindow)
     * @param promotionCustomType        PromotionCustomType Type
     * @param contentsKey                백오피스에 설정된 사용자 정의 페이지의 id
     * @param listener                     API 결과 통지 리스너
     *
     *  \~english Show custom Webview to use external content.
     * @param promotionCustomType        PromotionCustomType Type
     * @param contentsKey                ID of the custom web page set in the back office.
     * @param listener                    API call result listener
     * \~
     * @ingroup Promotion
     */

    static void showCustomContentsOnGameWindow(PromotionCustomType customType, std::string contentsKey, onPromotionView listener);

    /**
     * \~korean  게임내에서 무료 충전소를 노출하기 위한 버튼 UI 는 보여지거나 숨길 수 있도록 구성해야 한다.<br/>
     * 이 메서드는 게임내에서 오퍼월(무료 충전소) 을 호출 할 수 있는 버튼 노출 가능 여부를 반환한다.
     *
     * @return 오퍼월(무료 충전소) 을 호출 할 수 있는 버튼 노출 가능 여부
     * \~english Returns whether the button is available to invoke an offerwall (free recharging station) within the game.<br/>
     * The button UI for exposing free recharge stations within the game must be configured to be visible or hidden.
     *
     * @return whether the button is available.
     * \~
     * @ingroup Promotion
     */
    static OfferwallState getOfferwallState();


    /**
     * \~korean  무료 충전소 화면 노출 (Android only.)<br/>
     * 무료 충전소는 HIVE 게임 간의 광고 네트워크로 기존 HIVE 유저들이 새로운 HIVE 게임을 이용할 수 있도록 유도하는 기능이다.<br/>
     * 유저가 다운로드 받을 수 있는 게임을 목록으로 노출하고 게임을 다운로드 받아 실행하면 매체가 되는 게임에서 보상이 제공된다.<br/>
     *
     * @param listener	API 결과 통지
     * \~english Show Offerwall(Free recharging station) (Android only.)<br/>
     * The Offerwall (free recharge station) is an advertising network between HIVE games, which enables existing HIVE users to use the new HIVE game.<br/>
     * A list of the games that the user can download is exposed, and the game is downloaded and executed, and reward is provided in the game as the medium.<br/>
     *
     * @param listener	API call result listener
     * \~
     * @ingroup Promotion
     */
    static void showOfferwall(onPromotionView listener);

    /**
     * \~korean  게임의 새로운 이벤트나 새로운 게임 소개등의 새소식 화면을 노출
     * @param menu             최초 노출시 활성화할 메뉴 관리 명
     * @param handler          API 결과 통지
     * \~english Shows banner such as new event of game or introduce new game
     *
     * @param menu              Menu Management Name
     * @param handler          API call result handler
     * \~
     * @ingroup Promotion
     */
    static void showNews(std::string menu, std::vector<int> giftPidList, onPromotionView listener);

    /**
     * \~korean  리뷰 유도 팝업 노출<br/>
     * 게임 유저들의 긍정적인 평점 및 사용 후기는 다른 유저들이 게임을 이용하는데 영향을 미치게 된다.<br/>
     * 리뷰 유도 팝업을 제공하면 유저의 참여 건수가 5~10배 증가하는 것으로 알려져 있다.<br/>
     * iOS 10.3 이상의 기기에서는 앱 내부에서 평점 및 리뷰를 작성할 수 있는 팝업이 노출된다.
     *
     * \~english Show review popup <br/>
     * Positive ratings and reviews of game users will affect other users' use of the game.<br/>
     * It is known that the number of user participation increases by 5 ~ 10 times when the review<br/>
     * Review popup on the device with iOS 10.3 and later is available to rate and write reviews in apps. 
     *
     * \~
     * @since 4.10.0
     * @ingroup Promotion
     */
    static void showNativeReview(onPromotionView listener);

    /**
     * \~korean  리뷰 유도 팝업 노출<br/>
     * 게임 유저들의 긍정적인 평점 및 사용 후기는 다른 유저들이 게임을 이용하는데 영향을 미치게 된다.<br/>
     * 리뷰 유도 팝업을 제공하면 유저의 참여 건수가 5~10배 증가하는 것으로 알려져 있다.<br/>
     * Hive 콘솔 설정에 따라 Hive 일반 리뷰 팝업과 이미지 수정이 가능한 커스텀 리뷰 팝업이 노출된다.
     *
     * \~english Show review popup <br/>
     * Positive ratings and reviews of game users will affect other users' use of the game.<br/>
     * It is known that the number of user participation increases by 5 ~ 10 times when the review<br/>
     * Depending on the settings of Hive console, Hive normal review popup or customized image review popup will be appeared.
     *
     * \~
     * @ingroup Promotion
     */
    static void showReview(onPromotionView listener);


    /**
     * \~korean  종료 확인 팝업 노출 (Android only.)<br/>
     * 이용 중인 게임을 종료 하려는 유저에게 새로운 HIVE 게임의 다운로드를 유도하기 위해 '더 많은 게임'이란 버튼을 노출한다
     *
     * @param listener API 결과 통지
     * \~english Show exit popup (Android only)<br/>
     * Expose exit popup which include a button called "More games" to lead users to download a new HIVE game.
     *
     * @param listener API call result listener
     * \~
     * @ingroup Promotion
     */
    static void showExit(onPromotionView listener);


    /**
     * \~korean  HIVE 프로모션 웹 뷰의 UI 를 게임 UI 의 컨셉에 맞추기 위해서 프로모션 웹 뷰를 게임에서 직접 구현 할 수 있다<br/>
     * 이 메서드는 게임에서 HIVE 프로모션 웹 뷰를 커스터 마이징하기 위한 정보를 반환한다.
     *
     * @param listener API 결과 통지
     * \~english Request HIVE Promotion Webview information so that your UI of webview is configured according to the concept of game UI<br/>
     *
     * @param listener API call result listener
     * \~
     * @ingroup Promotion
     */
    static void getViewInfo(PromotionCustomType customType, std::string contentsKey, onPromotionViewInfo listener);


    /**
     * \~korean  뱃지 정보 조회<br/>
     * 프로모션 뱃지는 유저에게 새로운 프로모션 내용이 등록 되었음을 알려주기 위해서 게임의 버튼 UI 를 부각하는 정보이다
     *
     * @param listener API 결과 통지
     * \~english Request badge information<br/>
     * The promotional badge is information that highlights the button UI of the game to inform the user that a new promotion has been registered.
     *
     * @param listener API call result listener
     * \~
     * @ingroup Promotion
     */
    static void getBadgeInfo(onPromotionBadgeInfo listener);


    /**
     * \~korean  배너 정보 조회<br/>
     * 특정 프로모션에 대한 배너 정보를 요청한다. 게임 서버에서 배너 정보를 조회하기 위해 사용된다.
     *
     * @param campaignType "event", "notice", "all", "cross"
     * @param handler API 결과 통지
     *
     * \~english Request banner information<br/>
     * Request banner information for a specific promotion. Used to retrieve banner information from the game server.
     *
     * @param campaignType "event", "notice", "all", "cross"
     * @param handler API call result listener
     * \~
     * @ingroup Promotion
     */
    static void getBannerInfo(PromotionCampaignType campaignType, PromotionBannerType bannerType, onPromotionBannerInfo listener);


    /**
     * \~korean  배너 정보 조회<br/>
     * 특정 프로모션에 대한 배너 정보를 요청한다. 게임 서버에서 배너 정보를 조회하기 위해 사용된다.
     *
     * @param campaignType 커스텀 String campaign type
     * @param handler API 결과 통지
     *
     * \~english Request banner information<br/>
     * Request banner information for a specific promotion. Used to retrieve banner information from the game server.
     *
     * @param campaignType custom String campaign type
     * @param handler API call result listener
     * \~
     * @ingroup Promotion
     */
    static void getBannerInfoString(std::string campaignString, std::string bannerString, onPromotionBannerInfo listener);


    /**
     * \~korean  앱 초대를 위한 데이터 조회<br/>
     * 게임에서 더 많은 사용자를 유치하기 위해서 앱 설치 유도 기능을 제공한다.<br/>
     * 앱 설치 유도는 유저 애퀴지션 (User Acquisition) 이라고도 부른다.<br/>
     * 앱의 초대를 위한 데이터는 QR Code, 초대링크, 캠페인 등이 있다.<br/>
     * 캠페인은 초대에 대한 보상을 달상하기 위한 조건을 명시한 데이터이다.<br/>
     * 초대 URL 또는 QR코드를 통해 초대받은 유저가 게임을 설치하고, 특정 레벨 달성 등과 같은 조건(백오피스에서 조건 및 보상 설정 가능)을 달성했을 때 초대한 유저와 초대받은 유저 모두에게 보상을 제공한다.
     *
     * @param listener	API 호출 결과.
     * \~english Request user invite information<br/>
     * User acquisition information is provided to attract more users in the game.<br/>
     * Data for user acquisition include QR Code, invite links, and campaigns.<br/>
     * A campaign is data that specifies the conditions for rewarding an invite.<br/>
     * The invite URL or QR code provides rewards to both the invited and invited users when the invited user achieves the conditions such as installing the game and achieving a certain level (condition and reward can be set in the back office).
     *
     * @param listener	API call result.
     * \~
     * @ingroup Promotion
     */
    static void getAppInvitationData(onAppInvitationData listener);


    /**
     * \~korean  프로모션 뷰의 특정 이벤트 페이지를 노출하거나 프로모션 서버에 부가 정보를 전달하고 싶을 때 JSON 포맷의 문자열을 설정할 수 있다<br/>
     *
     * (필수)서버군이 따로 없는 단일 서버군이라도 아래 예제대로 넣어야 함<br/>
     * ex)"addtionalInfo":"{"server":"0","character":"0"}"
     *
     * @param additionalInfo	(필수) JSON 포맷의 부가 정보 데이터
     * \~english You can set a string in JSON format when you want to expose a specific event page in the Promotion View or want to pass additional information to the Promotion Server.<br/>
     *
     * (Required) Even a single server group without a server group should be properly put in the following example<br/>
     * ex)"addtionalInfo":"{"server":"0","character":"0"}"
     *
     * @param additionalInfo	(Required) Additional information data in JSON format
     * \~
     * @ingroup Promotion
     */
    static void setAdditionalInfo(std::string additionalInfo);


    /**
     * \~korean  Engagement 이벤트 처리가 가능한지(게임 서버에 DB가 생성된 경우 등) 여부를 설정한다.
     * true로 설정하려는 경우, <로그인 이후 & 리스너가 등록된 이후>의 조건을 만족한 상태여야 정상적으로 설정되며,
     * false로 설정하려는 경우는 항상 설정 가능하다.
     *
     * @param isReady Enganement 이벤트 처리 가능 여부.
     * @return API 호출 성공 여부.
     * \~english It sets whether Engagement event handling is enabled.(Such as when a DB is created in the game server)
     * If you want to set it to true, it must be in a state that satisfies the condition of <after login & after registering the listener>,
     * if you want to set it to false, you can always set it.
     *
     * @param isReady Whether Engagement events can be processed.
     * @return Whether the API call was successful.
     * \~
     * @ingroup Promotion
     */
    static ResultAPI setEngagementReady(bool isReady);


    /**
     * \~korean URI를 파싱하여 Event 타입으로 만든다.
     *
     * @param URI 파싱할 URI
     * @return 파싱에 문제가 없으면 true, 아니면 false.
     * \~english Parse the URI to make it an Event type.
     *
     * @param URI URI to parse
     * @return True if there is no problem parsing, false otherwise.
     * \~
     * @ingroup Promotion
     */
    static bool processURI(std::string URI);

    /**
     * \~korean  Engagement 리스너를 등록한다.
     * \~english  It register the Engagement listener.
     * \~
     * @ingroup Promotion
     * @param engagementHandler
     */
    static void setEngagementHandler(onEngagement engagementHandler);

    /**
     * \~korean 외부 SNS를 통해 UA url을 공유하기 위한 다이얼로그 노출
     * @param inviteMessage                UA 초대 메시지
     * @param inviteLink                UA 초대 링크
     * @param uaShareHandler                     API 결과 통지 리스너
     *
     *  \~english Show dialog to share UA url via SNS.
     * @param inviteMessage                UA invite message
     * @param inviteLink                UA invite link
     * @param uaShareHandler                    API call result listener
     * \~
     * @ingroup Promotion
     */
    static void showUAShare(std::string inviteMessage, std::string inviteLink, onPromotionUAShare uaShareHandler);
    
    /**
     * \~korean Promotion 정보 갱신 <br/>
     * Promotion 새소식 페이지 등의 노출에 필요한 정보를 갱신한다.
     *
     * \~english Update information of Promotion
     * Update promotion data to show Promotion UI.
     */
    static void  updatePromotionData();

    /**
     * \~korean UA를 통해 자신을 앱으로 최초 초대한 유저의 정보 반환 한다.
     * SDK 초기화, 로그인, setEngagementReady(true) 호출 이후 최초 초대자 정보 전달이 가능하다.
     * @param AppInvitationSenderInfoHandler API 결과 통지 리스터
     *
     * \~english return first sender's userInfo who sent UA share invitation
     * First sender's information can be returned after SDK initialize, login and setEngagementReady(true)
     * @param AppInvitationSenderInfoHandler API call result listener
     * \~
     * @ingroup Promotion
     */
    static void getAppInvitationSenderInfo(onAppInvitationSenderInfo appInvitationSenderInfoHandler);

    // Native 영역에서 호출된 요청을 처리하기 위한 플러그인 내부 코드
    static void executeEngine(picojson::value jsonParam);
    
};



/**
 * \~korean  프로모션 뷰 창의 형태
 *
 * \~english Types of Promotion view
 * \~
 * @ingroup Promotion
 * @author ryuvsken
 */
enum class PromotionType
{
    BANNERLegacy = -1   ///< \~korean 전면배너 (웹뷰만 사용하는 구 버전)  \~english  Full Banner
    , BANNER            ///< \~korean 전면배너	\~english Full banner
    , NEWS              ///< \~korean 새소식	\~english News
    , NOTICE            ///< \~korean 공지사항	\~english Notice

};


/**
 * \~korean  커스텀 컨텐츠의 형태
 *
 * \~english Types of Promotion CustomView
 * \~
 * @ingroup Promotion
 * @author kjkim
 */
enum class PromotionCustomType
{
    VIEW            ///< \~korean 커스텀 뷰	\~english Custom view
    , BOARD         ///< \~korean 커스텀 보드	\~english Custom board
    , SPOT          ///< \~korean 스팟 배너	\~english Spot banner
    , DIRECT        ///< \~korean 다이렉트 캠페인	\~english Direct campaign
    
};


/**
 * \~korean  프로모션 뷰 결과 통지시 이벤트 형태
 *
 * \~english Promotion view event types
 * \~
 * @ingroup Promotion
 * @author ryuvsken
 */
enum class PromotionEventType
{
    OPEN              ///< \~korean 프로모션 뷰가 열렸을 때	\~english When the Promotion View window opens.
    , CLOSE           ///< \~korean 프로모션 뷰가 닫혔을 때	\~english When the Promotion View window is closed.
    , START_PLAYBACK  ///< \~korean 영상 재생이 시작되었을 때	\~english When Playback starts.
    , FINISH_PLAYBACK ///< \~korean 영상 재생이 완료되었을 때	\~english When Playback is finished.
    , EXIT            ///< \~korean 종료(더 많은 게임 보기) 팝업에서 종료하기를 선택했을 때	\~english When you choose to quit from the Quit (see more games) popup.
    , GOBACK          ///
};


/**
 * \~korean  프로모션 뱃지 정보가 표시될 대상 정보<br/>
 * 만약 NOTICE 이면 공지사항 관련 뱃지를 표시해야한다
 *
 * \~english Information on which promotional badge is displayed<br/>
 * If NOTICE, it should indicate the badge related to the notice.
 * \~
 * @ingroup Promotion
 */
enum class PromotionBadgeTarget
{
    NEWS
    , NOTICE
    , CUSTOMVIEW
    , CUSTOMBOARD
    
};


/**
 * \~korean  게임내 오퍼월(무료 충전소) 을 호출 할 수 있는 버튼 노출 가능 상태 정보
 *
 * \~english Information of Offerwall(free charging station) button
 * \~
 * @ingroup Promotion
 */
enum class OfferwallState
{
    ENABLED          ///< \~korean 게임내 무료 충전 버튼 노출 가능	\~english  Offerwall Button can be exposed.

    , DISABLED       ///< \~korean 게임내 무료 충전 버튼 노출 불가능	\~english  Offerwall Button cannot be exposed.

    , UNKNOWN        ///< \~korean 버튼 노출 가능 상태를 알 수 없음	\~english  Unknown status whether available to expose buttons or not.
    
};

/**
 *  \~korean UserEngagement 기능 타입
 *
 *  \~english UserEngagement 기능 타입
 */
enum class EngagementEventType
{
    EVENT_TYPE              /// < kHIVEEngagementEventType Engagement의 전체 시작과 끝을 알리는 경우. (이름 변경 예정)
    
    , PROMOTION_VIEW        ///< kHIVEEngagementEventTypePromotionView Engagement에 의해 처리되는 PromotionView인 경우. EventState가 kHIVEEngagementStateEventEnd인 경우 PromotionView의 결과가 ResultAPI와 Param으로 전달된다.
    , OFFERWALL_VIEW
    , USER_ACQUISITION
    , COUPON                ///< kHIVEEngagementEventTypeCoupon Engagement에 의해 처리된 쿠폰 소모에 대한 결과. EventState가 kHIVEEngagementStateEventEnd인 경우 쿠폰사용의 결과가 ResultAPI와 Param으로 전달된다.
    
    , AUTH_LOGIN_VIEW       ///< kHIVEEngagementEventTypeAuthLogin 유저(클라이언트)에 의해 열리지 않은 로그인 프로세스에 의한 결과를 받는 콜백.

    , SOCIAL_INQUIRY_VIEW
    
    , EVENT                 ///< kHIVEEngagementEventTypeEvent Engagement에 의해 처리될 수 없는 이벤트(host가 game인 경우)를 전달해주는 콜백.
    
    , IAP_UPDATED           ///< Deprecated in HIVE SDK 4.5.0
    
    , IAP_PURCHASE          ///< kHIVEEngagementEventTypeIapPurchase 앱이 시작 혹은 실행 중일 때, 앱 외에서 상품을 구매시도시 불리게 되는 콜백.<br/> 해당 콜백이 불리면 구매 팝업이 노출되며 Engagement ready 상태가 false로 변경된다.
    , IAP_PROMOTE           ///< kHIVEEngagementEventTypeIapPromote 앱이 시작 혹은 실행 중일 때, 앱 외에서 상품을 구매시도시 불리게 되는 콜백.<br/> 해당 콜백이 불리면 게임에서는 아이템 구매가 가능한 적정한 시점에 checkPromotePurchase: 를 호출하여야 한다.
    , COMPANION             /// Promotion CPA Link received. SDK Will be send promotion companion.
    , SOCIAL_MYINQUIRY_VIEW ///< 내 문의 내역을 오픈
    , SOCIAL_PROFILE_VIEW   ///< 프로필 페이지 오픈
    , COMMUNITY_VIEW        ///< 커뮤니티 페이지를 오픈. Open the community page.
    , APPUPDATE_DOWNLOADED  ///< In-App Update Downloaded.

};

/**
 *  \~korean HIVE SDK 가 특정한 조건에서 클라이언트에 개입 (Engagement) 하는 상태
 *
 *  \~english HIVE SDK 가 특정한 조건에서 클라이언트에 개입 (Engagement) 하는 상태
 */

enum class EngagementEventState
{
    BEGIN           ///< Engagement에 의해 최초 기능이 수행되기 시작함을 알림.
    , FINISH           ///< Engagement에 의한 모든 기능 수행이 종료됨을 알림.
    , EVENT_START           ///< Engagement에 의해 특정 기능이 수행되기 시작함을 알림.
    , EVENT_END           ///< Engagement에 의한 특정 기능 수행이 종료됨을 알림.
};

/**
 * \~korean  롤링 배너 데이터 요청시 선택한 캠페인 타입
 *
 * \~english Campaign type selected before requesting rolling banner data.
 * \~
 * @author Seokjin Yong
 */
enum class PromotionCampaignType
{
    EVENT
    , NOTICE
    , ALL
    , CROSS
};

/**
 * \~korean  롤링 배너 데이터 요청시 선택한 배너 타입
 *
 * \~english Banner type selected before requesting rolling banner data.
 * \~
 * @author Seokjin Yong
 */
enum class PromotionBannerType
{
    GREAT
    , SMALL
    , ROLLING
};


/**
 * \~korean  Promotion 컨텐츠 화면 내용<br>
 * (HIVE 프로모션 웹 뷰의 UI 를 게임 UI 의 컨셉에 맞추기 위해서 프로모션 웹 뷰를 게임에서 직접 표시하기 위한 정보)
 *
 * \~english Information of Promotion webview<br>
 * (Information to customize your promotion Webview UI)
 * \~
 * @ingroup Promotion
 */
class HIVESDK_DLLEXPORT PromotionViewInfo
{
public:
    std::string postString;
    std::string url;
    
    PromotionViewInfo();
    
    bool parse(picojson::value jsonParam);
    
    picojson::object toJson() const;
    
    std::string toString() const;
};


/**
 * \~korean  유저에게 새로운 프로모션 내용이 등록 되었음을 알려주기 위한 정보
 *
 * \~english Badge information to inform users that new promotions are registered
 * \~
 * @ingroup Promotion
 * @author Joosang Kang
 */
class HIVESDK_DLLEXPORT PromotionBadgeInfo
{
public:
    
    /**
     * 뱃지를 표시해줘야하는 타겟<br/>
     * 예를 들어서 NOTICE 이면 공지사항 관련 뱃지를 표시해야한다<br/>
     * (NEWS, NOTICE, CUSTOMVIEW, CUSTOMBOARD 이 올 수 있다.)
     */
    PromotionBadgeTarget target;
    std::string contentsKey;    ///< target 이 커스텀 컨텐츠일 경우 백오피스에 등록된 프로모션 고유 KEY
    std::string badgeType;      ///< 뱃지 종류 ("new" or "none" 이 올 수 있다)
    
    PromotionBadgeInfo();
    
    bool parse(picojson::value jsonParam);
    
    picojson::object toJson() const;
    
    std::string toString() const;
};

/**
 * \~korean 특정 프로모션에 대한 배너 정보
 *
 * \~english Banner information for a specific promotion
 * \~
 * @ingroup Promotion
 */
class HIVESDK_DLLEXPORT PromotionBannerInfo
{
public:
    int pid;
    std::string imageUrl;
    std::string linkUrl;
    std::string displayStartDate;
    std::string displayEndDate;
    long utcStartDate;
    long utcEndDate;
    std::string typeLink;
    std::string typeBanner;
    std::string typeCampaign;

	std::string interworkData;
	
    PromotionBannerInfo();

    bool parse(picojson::value jsonParam);

    picojson::object toJson() const;

    std::string toString() const;
};


///**
// * @brief 보상 지급 요청에 대한 정보 정의<br/>
// * (showOfferwall() 호출 시, 보상 지급 요청을 보내고 응답을 받은 후, 화면에 wall을 표시해주는 구조임)
// *
// * @ingroup Promotion
// * @author ryuvsken
// */
//class HIVESDK_DLLEXPORT OfferwallReward
//{
//public:
//    int errorNo;
//    std::string errorMessage;
//    int result;
//    std::string eventID;
//    std::string assetCode;
//    int assetAmount;
//
//    OfferwallReward();
//
//    bool parse(picojson::value jsonParam);
//
//    picojson::object toJson() const;
//
//    std::string toString() const;
//};


/**
 * \~korean  앱 초대 (UserAcquisition)를 위한 정보.
 *
 * @see #getAppInvitationData(AppInvitationDataListener)
 *
 * \~english Invite information for UserAcquisition.
 *
 * @see #getAppInvitationData(AppInvitationDataListener)
 * \~
 * @ingroup Promotion
 */
class HIVESDK_DLLEXPORT AppInvitationData
{
public:
    std::vector<char> qrCode;                           ///< \~korean 앱의 초대 정보가 포함된 QR Code 이미지 데이터 \english QR Code image data with app invite information
    std::string inviteCommonLink;							///< \~korean 기본 초대 링크 \english Default invite link
    		std::string inviteHivemsgLink;						///< \~korean HIVE 인증 사용자용 초대 링크 \english Invite link for HIVE certified users
    		std::string inviteFacebookLink;						///< \~korean Facebook 인증 사용자용 초대 링크 \english Invite link for Facebook certified users
    std::vector<AppInvitationCampaign> eachCampaignList;		///< \~korean 캠페인 완료 발생시, 매번 보상을 지급하는 캠페인 목록 \english List of campaigns that will be rewarded every time when a campaign is completed.
    std::vector<AppInvitationCampaignStage> stageCampaignList;    ///< \~korean 캠페인 완료 목표 초대 수를 달성했을 때 보상을 지급하는 캠페인 목록 \english List of campaigns that will be rewarded when you reach the campaign completion goal of invites.
    std::string inviteMessage;
    
    AppInvitationData();
    
    bool parse(picojson::value jsonParam);
    
    picojson::object toJson() const;
    
    std::string toString() const;
};


/**
 * \~korean  앱 초대를 위한 캠패인 정보<br>
 * 캠페인은 초대에 대한 보상을 달상하기 위한 조건을 명시한 데이터.
 *
 * @see AppInvitationData
 *
 * \~english Campaign information for invite<br>
 * Campaigns are data that specifies the conditions for rewarding invite.
 *
 * @see AppInvitationData
 * \~
 * @ingroup Promotion
 */
class HIVESDK_DLLEXPORT AppInvitationCampaign
{
public:
    
    std::string title;              ///< \~korean 캠페인 타이틀 \~english Title
    std::string description;		///< \~korean 캠페인 설명 \~english Description
    std::string imageUrl;			///< \~korean 캠페인 이미지 URL \~english Image URL
    int order;                      ///< \~korean 캠페인 순서 \~english Order
    picojson::value item;           ///< \~korean 완료 보상 정보 ("key" 는 캠페인 설정에 따라 상이함) \~english Complete reward information ("key" depends on campaign settings)
    int count;                      ///< \~korean 초대 인원 중. 캠페인을 완료한 수 \~english Number of invitees who completed the campaign
    int limit;                      ///< \~korean 캠페인 최대 인원 수 \~english Maximum number of user of campaigns
    std::string campaignId;         ///< \~korean 캠페인 아이디 \~english Campaign Id
    
    AppInvitationCampaign();
    
    virtual ~AppInvitationCampaign(){}
    virtual bool parse(picojson::value jsonParam);
    
    picojson::object toJson() const;
    
    std::string toString() const;
};

class HIVESDK_DLLEXPORT AppInvitationCampaignStage: public AppInvitationCampaign
{
public:
	int goalCount;
	int goalTotal;
	
	AppInvitationCampaignStage();
	
	bool parse(picojson::value jsonParam) override;
};

/**
 * \~korean  UA를 통해 자신을 앱으로 최초 초대한 유저의 정보
 *
 * @see #getAppInvitationSenderInfo(AppInvitationSenderInfoHandler)
 *
 * \~english  First sender's userInfo who sent UA share invitation
 *
 * @see #getAppInvitationSenderInfo(AppInvitationSenderInfoHandler)
 * \~
 * @ingroup Promotion
 */
class HIVESDK_DLLEXPORT AppInvitationSenderInfo
{
public:
    std::string vid;

    AppInvitationSenderInfo();

    bool parse(picojson::value jsonParam);

    picojson::object toJson() const;

    std::string toString() const;
};

NS_HIVE_END		// namespace hive

#endif		// __HIVE_PROMOTIONVIEW_H__


/** @} */



