 /**
 * @file    HIVE_Push.h
 *
 * @author  ryuvsken
 * @date		2016-2019
 * @copyright	Copyright © GAMEVIL COM2US PLATFORM Inc. All Right Reserved.
 * @defgroup Push
  *  @since		4.0.0
  *  \brief		\copydoc 	Push
  */



/**
 * @defgroup Push
 * @{
 * \~korean  모바일 게임에서 푸시 통지 서비스는 게임 유저의 리텐션(잔존율)을 올리기 위한 중요 수단을 제공한다.<br/>
 * HIVE 푸시는 모바일 게임의 리텐션을 증대시키는 마케팅용 광고 푸시를 지원한다.<br/>
 * 또한 Google의 GCM, Apple의 APNS, Amazon의 ADM, 중국 지역의 Jpush를 지원하고, 언어별 발송 시간을 설정할 수 있어 글로벌 서비스 제공에도 대응하고 있다.<br/><br/>
 *
 * \~english In mobile games, push notification service provides an important method to increase the retention of game users.<br/>
 * HIVE Push supports push notification for marketing to increase the user retention of mobile games.<br/>
 * HIVE Push also responds to Google's GCM, Apple's APNS, Amazon's ADM, Jpush in China with language-specific time zones to support global services.<br/><br/>
 */

#ifndef __HIVE_PUSH_H__
#define  __HIVE_PUSH_H__


#include "HIVE_ResultAPI.h"


NS_HIVE_BEGIN


class ResultAPI;

class RemotePush;
class LocalPush;
class PushSetting;

/**
 * \~korean  모바일 게임에서 푸시 통지 서비스는 게임 유저의 리텐션(잔존율)을 올리기 위한 중요 수단을 제공한다.<br/>
 * HIVE 푸시는 모바일 게임의 리텐션을 증대시키는 마케팅용 광고 푸시를 지원한다.<br/>
 * 또한 Google의 GCM, Apple의 APNS, Amazon의 ADM, 중국 지역의 Jpush를 지원하고, 언어별 발송 시간을 설정할 수 있어 글로벌 서비스 제공에도 대응하고 있다.<br/><br/>
 *
 * \~english In mobile games, push notification service provides an important method to increase the retention of game users.<br/>
 * HIVE Push supports push notification for marketing to increase the user retention of mobile games.<br/>
 * HIVE Push also responds to Google's GCM, Apple's APNS, Amazon's ADM, Jpush in China with language-specific time zones to support global services.<br/><br/>
 * \~
 * @since		4.0.0
 * @ingroup Push
 * @author ryuvsken
 */
class HIVESDK_DLLEXPORT Push
{
public:

    /**
    		 * \~korean 유저의 푸시 수신 상태를 조회한 결과 통지
			 * 
		 	 * @param result		API 호출 결과
			 * @param remotePush		유저가 푸시를 수신하는 상태
		 	 *
		 	 * \~english Returns information of receiving the remote push
		 	 * 
		 	 * @param result		API call result
		 	 * @param remotePush		The status of receiving push notification
		 	 * \~
    		 * @ingroup Push
    		 */
    typedef std::function<void(ResultAPI const & result, RemotePush const & remotePush)> onRemotePush;
    
    
    /**
    		 * \~korean 로컬 푸시 등록 결과 통지
    		 *
    		 * @param result		API 호출 결과
    		 * @param localPush			로컬 푸시 등록 정보
    		 * \~english Returns information of receiving the local push
    		 *
    		 * @param result		API call result
    		 * @param localPush			Local push registration information.
    		 * \~
    		 * @ingroup Push
    		 */
	typedef std::function<void(ResultAPI const & result, LocalPush const & localPush)> onLocalPush;

    // MARK: #12076: Foreground Push Handler
    /**
     * \~korean 앱 활성화 시 푸시 설정 결과 통지
	 * 
	 * @param result		API 호출 결과
	 * @param pushSetting		푸시 설정 정보
	 *
	 * \~english Returns the result of push settings on the activated app.
	 * 
	 * @param result		API call result
	 * @param pushSetting		Push settings information
	 * \~
     * @ingroup Push
     */
    typedef std::function<void(ResultAPI const &result, PushSetting const &pushSetting)> onPushSetting;
    // MARK: -

    /**
    	 * \~korean 유저의 푸시 수신 상태 조회
		 * 
		 * @param listener	API 결과 통지
		 *
		 * \~english Request the status of receiving push notification.
		 * 
		 * @param listener	API call result
		 * \~
    	 * @ingroup Push
    	 */
    static void getRemotePush(onRemotePush listener);
    
    
    /**
    	 * \~korean 유저의 푸시 수신 상태 설정
 	     * 
		 * @param remotePush	리모트 푸시 정보
		 * @param listener  	API 결과 통지
		 *
		 * \~english Set the status of receiving push notification.
		 * 
		 * @param remotePush	Remote push information.
		 * @param listener  	API call result
    	 * \~
    	 * @ingroup Push
    	 */
    static void setRemotePush(RemotePush remotePush, onRemotePush listener);
    

    /**
    	 * \~korean 로컬 푸시 등록. iOS에서는 시스템 제한으로 최대 64개 까지 등록 가능
    	 *
    	 * @param localPush	로컬 푸시 등록 정보
    	 * @param listener	API 결과 통지
    	 * \~english Register Local push notification. In iOS, up to 64 can be registered due to the system limit.
    	 *
    	 * @param localPush	Local push registration information.
    	 * @param listener	API call result
    	 * \~
    	 * @ingroup Push
    	 */
	static void registerLocalPush(LocalPush localPush, onLocalPush listener);


    /**
    	 * \~korean 로컬 푸시 해제
    	 *
    	 * @param noticeID		로컬 푸시 등록 ID
    	 * \~english Unregister Local push notification.
    	 *
    	 * @param noticeID		Local push registration ID
    	 * \~
    	 * @ingroup Push
    	 */
	static void unregisterLocalPush(int noticeId);

    /**
         * \~korean 로컬 푸시 해제
         *
         * @param noticeIDs        로컬 푸시 등록 ID 배열
         * \~english Unregister Local push notification.
         *
         * @param noticeIDs        Array of Local push registration ID
         * \~
         * @ingroup Push
         */
    static void unregisterLocalPushes(std::vector<int> noticeIDs);

    /**
         * \~korean Hive 로컬푸시를 포함한 모든 로컬 푸시 해제
         *
         * \~english Unregister All local pushes including Hive local pushes.
         *
         * \~
         * @ingroup Push
         */
    static void unregisterAllLocalPushes();

    // MARK: #12076: Push Setting
    /**
     * \~korean 앱 활성화 시 푸시 수신 여부 설정 값 적용
     *
     * @param setting        유저가 앱 활성화 시 푸시를 수신하는 상태
     * @param listener       API 결과 통지
     * \~english Set the value whether to enable push notification or not on the activated app.
     *
     * @param setting        The status of receiving push notification on the activated app.
     * @param listener       API call result
     * \~
     * @ingroup Push
     */
    static void setForegroundPush(PushSetting setting, onPushSetting listener);
    
    /**
     * \~korean 앱 활성화 시 푸시 수신 여부 설정 값 조회
     *
     * @param listener       API 결과 통지
     *
     * \~english Get the value whether to enable push notification or not on the activated app.
     *
     * @param listener       API call result
     * \~
     * @ingroup Push
     */
    static void getForegroundPush(onPushSetting listener);
    
    /**
     * \~korean Push Token 명시적 권한 요청
     *
     * \~english Request Push Token
     *
     * \~
     * @ingroup Push
     */
    static void requestPushPermission();
    // MARK: -


	// \~korean Native 영역에서 호출된 요청을 처리하기 위한 플러그인 내부 코드
	// \~english Plug-in internal code to handle requests invoked from the native code.
	static void executeEngine(picojson::value jsonParam);

};




/**
	 * \~korean 유저가 푸시를 수신하는 상태 정보
	 *
	 * \~english Information about the status of receiving push notification.
	 * \~
	 * @ingroup Push
	 */
class HIVESDK_DLLEXPORT RemotePush {
    
public:
    
    bool isAgreeNotice;			///< \~korean 공지 푸시 수신 허용 여부 \~english Whether to receive announcement notification or not.
    
    bool isAgreeNight;			///< \~korean 야간 푸시 수신 허용 여부 \~english Whether to receive night-time notification or not. 
    
    
    RemotePush();
    
    RemotePush(bool isAgreeNotice, bool isAgreeNight);
    
    RemotePush(picojson::value jsonParam);
    
    picojson::object toJson() const;
    
    std::string toString() const;
};


/**
	 * \~korean 로컬 푸시 등록 정보
	 *
	 * \~english Local push registration information.
	 * \~
	 * @ingroup Push
	 */
class HIVESDK_DLLEXPORT LocalPush
{
public:
    
    int noticeId;			///< \~korean 로컬 푸시 통지 ID \~english Local push notification ID
    
    std::string title;		///< \~korean 로컬 푸시 타이틀 \~english Local push title
    
    std::string msg;		///< \~korean 로컬 푸시 메시지 \~english Local push message
    
    long after;				///< \~korean 알림 시점 \~english Notification timer

    std::string groupId;    ///< \~korean 알림 그룹 ID \~english Notification Group ID
    
    
    // \~korean 이하 Android에서 로컬 푸시를 커스터마이징하기 위한 필드 정의
    // \~english Followings are field definition for customizing local push on Android.
    
    std::string bigmsg;     ///< \~korean 큰 글씨 \~english Big-text
    
    std::string ticker;     ///< \~korean 메시지 티커 \~english Message Ticker
    
    std::string type;		///< \~korean 알림 형태 (bar, popup, toast 등) \~english Notification type (bar, popup, toast etc.)
    
    std::string icon;		///< \~korean 푸시 아이콘 \~english Push icon
    
    std::string sound;		///< \~korean 푸시 알림음 \~english Notification sound
    
    std::string active;		///< \~korean 수행할 동작 \~english Action to take
    
    std::string broadcastAction;
    int buckettype;
    int bucketsize;
    std::string bigpicture;
    std::string icon_color;
    
    
    LocalPush();
    
    LocalPush(int noticeID, std::string title, std::string msg, long after);
    
    LocalPush(picojson::value jsonParam);
    
    picojson::object toJson() const;
    
    std::string toString() const;
};

// MARK: #12076 Push Setting
/**
 * \~korean 푸시 설정 정보
 *
 * \~english Push setting information
 * \~
 * @ingroup Push
 */
class HIVESDK_DLLEXPORT PushSetting
{
public:
    
    bool useForegroundRemotePush;   ///< \~korean 앱 활성화 시 리모트 푸시 수신 여부 \~english Whether to enable remote notification or not on the activated app.
    bool useForegroundLocalPush;    ///< \~korean 앱 활성화 시 로컬 푸시 수신 여부 \~english Whether to enable local notification on the activated app.
    
    PushSetting();
    
    PushSetting(bool useForegroundRemotePush, bool useForegroundLocalPush);
    
    PushSetting(picojson::value jsonParam);
    
    picojson::object toJson() const;
    
    std::string toString() const;
};
// MARK: -


NS_HIVE_END		// namespace hive


#endif		// __HIVE_PUSH_H__


/** @} */



