/**
 * @file        HiveAuthV4.h
 *
 * @date        2025
 * @copyright  Copyright © 2025 Com2uS Platform Corp. All Rights Reserved.
 *
 *  @defgroup   FHiveAuthV4
 *  @{
 *  \~korean
 * HIVE SDK 초기화 및 HIVE 인증 기능을 제공한다 <br/>
 * HIVE SDK 는 xml 설정 파일을 기반으로 초기화를 수행한다.<br/><br/>
 * 이 클래스에서 제공하는 상세 기능은 다음과 같다. <br/><br/>
 *  - HIVE SDK 초기화<br/>
 *  - 고객 정보 수집 및 이용 약관 동의<br/>
 *  - 단말 고유 ID (DID : Device ID) 관리<br/>
 *  - User 다운로드 관리<br/>
 *  - 버전과 서버 점검 및 업데이트 관리<br/>
 *  - 사용자 제재<br/>
 *  - Guest / Provider 사인-인/아웃 수행<br/>
 *  - 프로필, 1:1문의 수행<br/>
 *  - 성인 인증 수행<br/>
 *
 *  \~english
 * @brief Provides HIVE SDK initialization and HIVE authentication functions<br/>
 * The HIVE SDK performs initialization based on the xml configuration file.<br/><br/>
 * Provides HIVE SDK initialization and HIVE authentication functions<br/>
 * The HIVE SDK performs initialization based on the xml configuration file.<br/><br/>
 * The detailed functions provided by this class are as follows. <br/><br/>
 * - HIVE SDK Initialization<br/>
 * - Collect user information and accept terms and conditions<br/>
 * - Device unique ID (DID : Device ID) management<br/>
 * - User download / Session management<br/>
 * - Version and server maintenance and update management<br/>
 * - User restriction<br/>
 * - Guest / IdP Log-in / Logout <br/>
 * - User Profile, 1:1 Inquiry <br/>
 * - Perform adult verification<br/><br/>
 *  \~
 *  \~
 */
#pragma once

#include "CoreMinimal.h"
#include "HiveInterface.h"
#include "Impl/HiveAuthV4Impl.h"
#include "Impl/HiveAuthV4HelperImpl.h"

/**
 *  \~korean
 * HIVE SDK 초기화 및 HIVE 인증 기능을 제공한다 <br/>
 * HIVE SDK 는 xml 설정 파일을 기반으로 초기화를 수행한다.<br/><br/>
 * 이 클래스에서 제공하는 상세 기능은 다음과 같다. <br/><br/>
 *  - HIVE SDK 초기화<br/>
 *  - 고객 정보 수집 및 이용 약관 동의<br/>
 *  - 단말 고유 ID (DID : Device ID) 관리<br/>
 *  - User 다운로드 관리<br/>
 *  - 버전과 서버 점검 및 업데이트 관리<br/>
 *  - 사용자 제재<br/>
 *  - Guest / Provider 사인-인/아웃 수행<br/>
 *  - 프로필, 1:1문의 수행<br/>
 *  - 성인 인증 수행<br/>
 *
 *  \~english
 * @brief Provides HIVE SDK initialization and HIVE authentication functions<br/>
 * The HIVE SDK performs initialization based on the xml configuration file.<br/><br/>
 * Provides HIVE SDK initialization and HIVE authentication functions<br/>
 * The HIVE SDK performs initialization based on the xml configuration file.<br/><br/>
 * The detailed functions provided by this class are as follows. <br/><br/>
 * - HIVE SDK Initialization<br/>
 * - Collect user information and accept terms and conditions<br/>
 * - Device unique ID (DID : Device ID) management<br/>
 * - User download / Session management<br/>
 * - Version and server maintenance and update management<br/>
 * - User restriction<br/>
 * - Guest / IdP Log-in / Logout <br/>
 * - User Profile, 1:1 Inquiry <br/>
 * - Perform adult verification<br/><br/>
 *  \~
 * @ingroup FHiveAuthV4
 */
class HIVESDK_API FHiveAuthV4 : public IHiveInterface<IHiveAuthV4Impl>
{
public:
    /**
    *  \~korean
    * @brief AuthV4 초기화를 수행한다.
    * Configuration 영역을 제외한 모든 API 중 가장 먼저 호출되어야 하며<br/>
    * 선 호출이 되지 않을시 일부 API 에서는 NEED_INITIALIZE 에러가 발생할 수 있다.<br/>
    * Android 의 경우 초기 퍼미션 요청 UI 동작이 추가되며<br/>
    * 이후 약관 동의, 다운로드 체크, DID 설정, config.xml 설정, ProviderList 설정 등이 진행 된다.<br/>
    * 최초 실행 시 DID 를 받아오지 못하거나, Provider List 를 받아오지 못하면 handler 결과로 실패가 내려간다.<br/>
    * handler 에 포함된 providerTypeList 를 통해 signIn 에 필요한 UI 를 구성해야 한다.<br/>
    * 직접 UI 를 구현하지 않을 시 showSignIn() 을 이용한다.<br/>
    * <br/>
    * @param Delegate FHiveAuthV4OnSetupDelegate AuthV4 초기화 결과 통지<br/>
    * 			               result : 최초 실행 시 DID 를 받아오지 못하거나, Provider List 를 받아오지 못하면 실패한다. 그 외 성공.<br/>
    * 			               isAutoSignIn : 로컬에 이전 세션이 남아있는지 여부. true 일 경우 SignIn(ProviderType.AUTO) 을 호출 한다.<br/>
    * 			               그 외의 경우 providerTypeList 중 하나로 SignIn 을 요청이 가능 하다.<br/>
    * 			               providerTypeList : 현재 단말에서 사인-인 가능한 Provider List 이다.<br/>
    * 			               단말의 현재 지역 (IP) 에 따라 다르게 보여질 수 있다.<br/>
    * 			               GUEST 를 포함하고 있으며 일부 지역 혹은 환경에서는 GUEST 도 불가능 할 수 있다. (분산서버)<br/>
    *
    *  \~english
    * @brief AuthV4 Perform initialization.
    * It should be called first among all the APIs except for the configuration area,<br/>
    * and if this is not called first, some API may cause a NEED_INITIALIZE error.<br/>
    * On Android, the permission request UI is exposed<br/>
    * Afterwards, consent to the Terms, download check, DID setting, config.xml setting, ProviderList setting and etc. are performed.<br/>
    * If the DID is not received on the first execution, or the provider list is not received, the result is sent to the handler as failure.<br/>
    * You need to configure the UI for signIn using the providerTypeList contained in handler.<br/>
    * If you do not implement the UI directly, use showSignIn().<br/>
    * <br/>
    * @param Delegate FHiveAuthV4OnSetupDelegate AuthV4 Setup request result callback<br/>
    *                 			result : If the DID is not received or the provider list is not received, it fails. Otherwise, success.<br/>
    *                 			isAutoSignIn : Whether an old session remains local storage. If true, call SignIn (ProviderType.AUTO).<br/>
    *                   		Otherwise, you can request SignIn as one of providerTypeList.<br/>
    *                 			providerTypeList : A provider list that can be signed in at the current device.<br/>
    *                   		And may be different depending on the current area (IP) of the device.<br/>
    *                   		GUEST is included, and in some areas or environments GUEST may not be possible.<br/>
    *
    *  \~
    * @ingroup FHiveAuthV4
    */
    static void Setup(const FHiveAuthV4OnSetupDelegate& Delegate);

    /**
    *  \~korean
    * @brief 주어진 providerType 에 따라 signIn 을 요청 한다.
    * 로컬에 저장된 세션이 있는데 AUTO 가 아니거나<br/>
    * 로컬에 저장된 세션이 없는데 AUTO 이면 INVALID_PARAM 이 발생한다.<br/>
    * 또한, isAutoSignIn() 호출로도 AUTO 인지 아닌지 체크할 수 있다.<br/>
    * 이미 sign 이 진행중이면 IN_PROGRESS 가 발생하며,<br/>
    * GUEST 가 아닌 다른 ProviderType 의 경우 외부 인증창이 한번 더 노출 될 수 있다.<br/>
    * <br/>
    * signIn 에 성공하게 되면 handler 에 포함되어있는 playerInfo 를 통해 유저 정보를 얻고<br/>
    * 연동된 provider 상태를 UI 에 표시해 주면 된다.<br/>
    * <br/>
    * signIn(AUTO) 의 경우 저장된 playerId 의 세션만으로 sign-in 되기 때문에<br/>
    * 묵시적 로그인을 수행하는 Provider가 실제 단말에도 로그인 되어있는지의 여부를 체크하고 계정을 동기화 하려면 checkProvider() 를 호출 하여야 한다.<br/>
    * 다를 경우 signOut() 후 signIn(Provider) 로 계정 전환을 할 수 있다.<br/>
    * <br/>
    * BLACKLIST 일 경우 SDK에서 제재 팝업을 띄우고 BLACKLIST 에러를 내려 준다.<br/>
    *
    * @param ProviderType signIn 요청할 ProviderType
    *
    * @param Delegate FHiveAuthV4OnSignInDelegate AuthV4 사인-인 결과 통지
    *
    *  \~english
    * @brief Requests signIn according to the given providerType.
    * If there is a locally stored session and it is not AUTO<br/>
    * or if there is no session stored locally and it is AUTO, it will cause INVALID_PARAM error.<br/>
    * You can also check whether it is AUTO or not with isAutoSignIn ().<br/>
    * IN_PROGRESS occurs when the sign-in is already in progress,<br/>
    * and the external authentication window can be exposed once again for ProviderType other than GUEST.<br/>
    * <br/>
    * If signIn succeeds, you can get the user information through the playerInfo contained in the handler<br/>
    * and display the status of the linked provider in the UI<br/>
    * <br/>
    * In the case of signIn (AUTO), since it is sign-in only with the session of the stored playerId,<br/>
    * you should call checkProvider () to check whether the provider performing the implicit login is also logged in the actual terminal and to synchronize the account. <br/>
    * If it is different, you can call signOut () and call signIn (Provider) to switch the account.<br/>
    * <br/>
    * In the case of BLACKLIST, SDK will pop up a restriction popup and issue a BLACKLIST error.<br/>
    *
    * @param ProviderType  ProviderType to request signIn
    *
    * @param Delegate FHiveAuthV4OnSignInDelegate AuthV4 signIn result callback
    *  \~
    * @ingroup FHiveAuthV4
    *
    */
    static void SignIn(const EHiveProviderType& ProviderType, const FHiveAuthV4OnSignInDelegate& Delegate);

    /**
    *  \~korean
    * @brief 자체 구현한 커스텀 로그인 후 획득한 authKey 값으로 signIn을 요청 한다.
    * <br/>
    * signIn 에 성공하게 되면 handler 에 포함되어있는 playerInfo 를 통해 유저 정보를 얻는다.<br/>
    *
    * @param Delegate FHiveAuthV4OnSignInDelegate AuthV4 사인-인 결과 통지
    *
    *  \~english
    * @brief Requests signIn with authKey gained from customized provider login implementation.
    * <br/>
    * If signIn succeeds, you can get the user information through the playerInfo contained in the handler<br/>
    *
    * @param Delegate FHiveAuthV4OnSignInDelegate AuthV4 signIn result callback
    *  \~
    * @ingroup FHiveAuthV4
    *
    */
    static void SignInWithAuthKey(const FString& AuthKey, const FHiveAuthV4OnSignInDelegate& Delegate);

    /**
    *  \~korean
    * @brief 로컬에 있는 세션을 제거 한다.
    * 로그인 된 모든 Provider 들을 로그아웃 시키며 결과와 상관 없이 로컬 세션을 제거하고 성공 콜백을 준다.<br/>
    * signIn/Out 이 진행중일 경우 IN_PROGRESS 가 발생할 수 있다.<br/>
    * <br/>
    * 게스트 상태인 player 의 경우 (연동된 Provider 가 하나도 없는 상태) 다시 사인-인 할 수 없게 되니 주의.<br/>
    * @param Delegate FHiveAuthV4OnSignOutDelegate 사인-아웃 결과 통지
    *
    *  \~english
    * @brief it remove the local session..
    * It logs out all the providers that are logged in, removes the local session and gives a success callback regardless of the result.<br/>
    * IN_PROGRESS may occur when signIn / Out is in progress<br/>
    * <br/>
    * Note that in the case of a player in the guest state (no connected Provider), the player will not be able to sign in again.<br/>
    * @param Delegate FHiveAuthV4OnSignOutDelegate Sign-out result callback
    *
    *  \~
    * @ingroup FHiveAuthV4
    *
    */
    static void SignOut(const FHiveAuthV4OnSignOutDelegate& Delegate);

    /**
     * \~korean
     * @brief 계정을 삭제하고 로컬에 있는 세션을 제거 한다.
     * 서버에 계정 삭제를 요청하고 연동된 모든 Provider 들을 해제 한다.
     * 성공 시 로그인 된 모든 Provider 들을 로그아웃 시키며 로컬 세션을 제거하고 성공 콜백을 준다.<br></br>
     * signIn/Out 이 진행중일 경우 IN_PROGRESS 가 발생할 수 있다.<br></br>
     * <br></br>
     * @param Delegate FHiveAuthV4OnSignOutDelegate 계정 삭제 결과 통지
     *
     * \~english
     * @brief Delete the account and remove the local session.
     * Request to the server to delete the account and disconnected all providers.
     * If success, all logged in providers are logged out, the local session is removed, and a success callback is given.
     * IN_PROGRESS may occur when signIn / Out is in progress<br></br>
     * <br></br>
     * @param Delegate FHiveAuthV4OnSignOutDelegate Delete account result callback
     *
     * \~
     *
     * @ingroup FHiveAuthV4
     */
    static void PlayerDelete(const FHiveAuthV4OnSignOutDelegate& Delegate);

    /**
    *  \~korean
    * @brief 현재 사인-인 된 유저에 요청한 provider 와 연동 한다.
    * 요청한 Provider 로 로그인을 시도 한 뒤, 로그인에 성공하면 인증서버에 연동 요청을 보낸다.<br/>
    * 해당 Provider 의 UserId 에 이미 매칭된 playerId 가 있을 경우 handler 의 result 에 CONFLICT_PLAYER 에러를 내려준다.<br/>
    * 이 경우 handler 에 포함되어 있는 conflictPlayer 객체의 충돌 유저 정보를 가지고 유저에게 계정 선택을 위한 UI 를 구성 해야한다.<br/>
    * 유저가 선택을 하면 선택된 유저 정보로 selectConflict() 를 호출해 주면 된다.<br/>
    * <br/>
    * 직접 UI 를 구성하지 않을 경우 showConflictSelection() 을 이용할 수 있다.<br/>
    * 이 경우 selectConflict() 까지 진행되며 사인-인 완료된 결과까지 받을 수 있다.<br/>
    * <br/>
    * 연동이 완료되면 해당 상태를 앱 내 UI 에 갱신해 준다.<br/>
    * <br/>
    * BLACKLIST 일 경우 SDK에서 제재 팝업을 띄우고 BLACKLIST 에러를 내려 준다.<br/>
    *
    * @param ProviderType connect 요청할 ProviderType.<br/>
    *                     AUTO 나 GUEST 는 INVALID_PARAM 에러를 발생시킨다.<br/>
    *
    * @param Delegate FHiveAuthV4OnConnectDelegate <br/>
    *                 <br/>
    *                 result : 지원하지 않는 ProviderType (INVALID_PARAM) 이거나 <br/>
    *                   Sign 이 진행중일 경우 (IN_PROGRESS) 실패가 될 수 있다.<br/>
    *                   SignIn 이 되지 않은 상태에서는 INVALID_SESSION 에러가 된다.<br/>
    *                   이미 연결되어있는 Provider 의 경우 INVALID_PARAM 에러가 되며<br/>
    *                   해당 Provider 에 대해 disconnect 를 먼저 호출 후 다시 connect 해야 한다.<br/>
    *                   연결을 시도한 Provider 가 이미 다른 playerId 에 연결되어 있을 경우 CONFLICT_PLAYER 가 되며<br/>
    *                   conflictPlayer 객체에 해당 충돌 유저 정보가 포함되어있다.<br/>
    *                 <br/>
    *                 conflictPlayer : result 가 CONFLICT_PLAYER 일 경우 충돌 유저에 대한 정보 이다.<br/>
    *                   playerId 와 충돌난 ProviderInfo 의 정보만 포함되어있다.<br/>
    *                   playerToken 은 비어 있음.<br/>
    *
    *  \~english
    * @brief It connects to the currently requested provider for the currently signed-in user.
    * After logging in with the requested provider, if it is successful, it sends an linking request to the authentication server.<br/>
    * If there is a playerId already linked to the UserId of the corresponding provider, a CONFLICT_PLAYER error is returned in the result of the handler.<br/>
    * In this case, you should configured an UI for account selection to the user with the conflict user information of the conflictPlayer object included in the handler.<br/>
    * When the user makes a selection, calls selectConflict () with the selected user information.<br/>
    * <br/>
    * If you do not customize the UI, you can use showConflictSelection().<br/>
    * In this case, when the user selects it, it can proceed to selectConflict() and receive the result of the sign-in completed.<br/>
    * <br/>
    * When the linking is completed, the corresponding state should be updated on the UI of the app.<br/>
    * <br/>
    * In the case of BLACKLIST, SDK will show a restriction popup and issue a BLACKLIST error.<br/>
    *
    * @param ProviderType ProviderType to request connect.<br/>
    *                     AUTO or GUEST causes an INVALID_PARAM error.<br/>
    *
    * @param Delegate FHiveAuthV4OnConnectDelegate <br/>
    *                 <br/>
    *                 result :If ProviderType is not supported (INVALID_PARAM) <br/>
    *                   or if the Sign-in is in progress (IN_PROGRESS), it may fail.<br/>
    *                   If sign-in fails, an INVALID_SESSION error is returned.<br/>
    *                   An INVALID_PARAM error will occur if the provider is already connected. <br/>
    *                   In this case, you must first call disconnect for the provider and then connect again.<br/>
    *                   CONFLICT_PLAYER is returned if the provider attempting to connect is already connected to another playerId<br/>
    *                   and the conflictPlayer object returned contains the corresponding conflict user information.<br/>
    *                 conflictPlayer : If result is CONFLICT_PLAYER, it contains information about the conflicting user.<br/>
    *                   Only the information of the ProviderInfo that conflicted with playerId is included.<br/>
    *                   playerToken is empty.<br/>
    *
    *  \~
    * @ingroup FHiveAuthV4
    *
    */
    static void Connect(const EHiveProviderType& ProviderType, const FHiveAuthV4OnConnectDelegate& Delegate);

    /**
    *  \~korean
    * @brief 현재 사인-인 된 유저에 요청한 provider 와 연동을 해제 한다.
    * 인증 서버에 disconnect 상황을 전달하고 이후 요청한 Provider 를 Logout 시킨다.<br/>
    * 인증 서버에 전달이 실패할 경우와 Sign 이 진행중이거나 setup 이 되지 않은 상황,<br/>
    * 또는 이미 disconnected 된 Provider 일 경우 실패가 될 수 있다.<br/>
    * <br/>
    * disconnected 로 인해 연동이 모두 해제될 경우 게스트 상태가 될 수 있으니 주의.<br/>
    *
    * @param ProviderType disconnect 요청할 ProviderType.<br/>
    *
    * @param Delegate FHiveAuthV4OnDisconnectDelegate
    *					                 	result : 인증 서버에 전달이 실패할 경우와 Sign 이 진행중이거나 setup 이 되지 않은 상황,<br/>
    *					                 	또는 이미 disconnected 된 Provider 일 경우 실패가 될 수 있다.<br/>
    *					                 	그 외 성공.<br/>
    *
    *
    *  \~english
    * @brief Releases the currently signed-in user from the requested provider.
    * It pass the disconnect status to the authentication server and then logout the requested provider.<br/>
    * If delivery fails to the authentication server, if the Sign or setup is in progress, <br/>
    * or if the provider is already disconnected, it may fail.<br/>
    * <br/>
    * Note that if all account link are disconnected due to 'disconnect', it may become a guest state.<br/>
    *
    * @param ProviderType disconnect 요청할 ProviderType.<br/>
    *
    * @param Delegate FHiveAuthV4OnDisconnectDelegate
    *                 					result : If delivery fails to the authentication server, if the Sign or setup is in progress, <br/>
    *                   				or if the provider is already disconnected, it may fail.<br/>
    *                   				Otherwise, success.<br/>
    *
    *  \~
    * @ingroup FHiveAuthV4
    */
    static void Disconnect(const EHiveProviderType& ProviderType, const FHiveAuthV4OnDisconnectDelegate& Delegate);

    /**
    *  \~korean
    * @brief 자체 구현한 커스텀 로그인 후 획득한 authKey 값으로 connect을 요청 한다.
    * 해당 Provider 의 UserId 에 이미 매칭된 playerId 가 있을 경우 handler 의 result 에 CONFLICT\_PLAYER 에러를 내려준다. <br/>
    * 이 경우 handler 에 포함되어 있는 conflictPlayer 객체의 충돌 유저 정보를 가지고 유저에게 계정 선택을 위한 UI 를 구성 해야한다. <br/>
    * <br/>
    * 유저가 선택을 하면 선택된 유저 정보로 selectConflict() 를 호출해 주면 된다.<br/>
    *
    * @param AuthKey connect 요청할 authKey.<br/>
    *
    * @param Delegate FHiveAuthV4OnConnectDelegate <br/>
    *                 <br/>
    *                 result : 지원하지 않는 ProviderType (INVALID_PARAM) 이거나 <br/>
    *                   Sign 이 진행중일 경우 (IN_PROGRESS) 실패가 될 수 있다.<br/>
    *                   SignIn 이 되지 않은 상태에서는 INVALID_SESSION 에러가 된다.<br/>
    *                   이미 연결되어있는 Provider 의 경우 INVALID_PARAM 에러가 되며<br/>
    *                   해당 Provider 에 대해 disconnect 를 먼저 호출 후 다시 connect 해야 한다.<br/>
    *                   연결을 시도한 Provider 가 이미 다른 playerId 에 연결되어 있을 경우 CONFLICT_PLAYER 가 되며<br/>
    *                   conflictPlayer 객체에 해당 충돌 유저 정보가 포함되어있다.<br/>
    *                 <br/>
    *                 conflictPlayer : result 가 CONFLICT_PLAYER 일 경우 충돌 유저에 대한 정보 이다.<br/>
    *                   playerId 와 충돌난 ProviderInfo 의 정보만 포함되어있다.<br/>
    *                   playerToken 은 비어 있음.<br/>
    *
    *  \~english
    * @brief Requests connect with authKey gained from customized provider login implementation.
    * If there is a playerId already linked to the UserId of the corresponding provider, a CONFLICT\_PLAYER error is returned in the result of the handler. <br/>
    * In this case, you should configured an UI for account selection to the user with the conflict user information of the conflictPlayer object included in the handler. <br/>
    * <br/>
    * When the user makes a selection, calls selectConflict () with the selected user information.<br/>
    *
    * @param AuthKey authKey to request connect <br/>
    *
    * @param Delegate FHiveAuthV4OnConnectDelegate <br/>
    *                 <br/>
    *                 result :If ProviderType is not supported (INVALID_PARAM) <br/>
    *                   or if the Sign-in is in progress (IN_PROGRESS), it may fail.<br/>
    *                   If sign-in fails, an INVALID_SESSION error is returned.<br/>
    *                   An INVALID_PARAM error will occur if the provider is already connected. <br/>
    *                   In this case, you must first call disconnect for the provider and then connect again.<br/>
    *                   CONFLICT_PLAYER is returned if the provider attempting to connect is already connected to another playerId<br/>
    *                   and the conflictPlayer object returned contains the corresponding conflict user information.<br/>
    *                 conflictPlayer : If result is CONFLICT_PLAYER, it contains information about the conflicting user.<br/>
    *                   Only the information of the ProviderInfo that conflicted with playerId is included.<br/>
    *                   playerToken is empty.<br/>
    *
    *  \~
    * @ingroup FHiveAuthV4
    */
    static void ConnectWithAuthKey(const FString& AuthKey, const FHiveAuthV4OnConnectDelegate& Delegate);

    /**
    *  \~korean
    * @brief 현재 사인-인 된 유저에 요청한 커스텀 provider 와 연동을 해제 한다.
    * 인증 서버에 disconnect 상황을 전달하고 이후 요청한 Provider 를 Logout 시킨다.<br/>
    * 인증 서버에 전달이 실패할 경우와 Sign 이 진행중이거나 setup 이 되지 않은 상황,<br/>
    * 또는 이미 disconnected 된 Provider 일 경우 실패가 될 수 있다.<br/>
    * <br/>
    * disconnected 로 인해 연동이 모두 해제될 경우 게스트 상태가 될 수 있으니 주의.<br/>
    *
    * @param ProviderName disconnect 요청할 ProviderName.<br/>
    *
    * @param Delegate FHiveAuthV4OnDisconnectDelegate
    *                                         result : 인증 서버에 전달이 실패할 경우와 Sign 이 진행중이거나 setup 이 되지 않은 상황,<br/>
    *                                         또는 이미 disconnected 된 Provider 일 경우 실패가 될 수 있다.<br/>
    *                                         그 외 성공.<br/>
    *
    *
    *  \~english
    * @brief Releases the currently signed-in user from the requested custom provider.
    * It pass the disconnect status to the authentication server and then logout the requested provider. <br/>
    * If delivery fails to the authentication server, if the Sign or setup is in progress, <br/>
    * or if the provider is already disconnected, it may fail.<br/>
    * <br/>
    * Note that if all account link are disconnected due to 'disconnect', it may become a guest state.<br/>
    *
    * @param ProviderName ProviderName to request disconnect.<br/>
    *
    * @param Delegate FHiveAuthV4OnDisconnectDelegate
    *                                     result : If delivery fails to the authentication server, if the Sign or setup is in progress, <br/>
    *                                   or if the provider is already disconnected, it may fail.<br/>
    *                                   Otherwise, success.<br/>
    *
    *  \~
    * @ingroup FHiveAuthV4
    */
    static void DisconnectWithName(const FString& ProviderName,  const FHiveAuthV4OnDisconnectDelegate& Delegate);

    /**
    *  \~korean
    * @brief connect 로 인한 계정 충돌 상황 (CONFLICT_PLAYER) 일 때 유저를 선택 한다.
    * connect 로 인해 계정이 충돌 된 상황 (CONFLICT_PLAYER) 에서 자체 UI 를 구성하였을 경우<br/>
    * 선택한 유저를 SDK에 통보해 주기 위해 호출 한다.<br/>
    * SDK 상태가 충돌 상태가 아니거나 signIn 이 진행중, 혹은 잘못된 playerId 일 경우 INVALID_PARAM 이 발생 할 수 있다.<br/>
    * 충돌 상태가 되었다고 해도 selectConflict() 가 호출 되어 signIn 되기 전까지는 기존 유저가 signIn 된 상태로 본다.<br/>
    * <br/>
    * showConflictSelection() 을 사용하였을 경우 호출할 필요가 없다.<br/>
    *
    * @param SelectedPlayerId 선택한 유저의 playerId
    *
    * @param Delegate FHiveAuthV4OnSignInDelegate<br/>
    *                 SUCCESS 가 되기 전까지는 기존 유저가 signIn 된 상태이다. 주의.<br/>
    *
    *  \~english
    * @brief When an account conflict (CONFLICT_PLAYER) occurs due to connect, it notify the selected user to the SDK.
    * If you use your own customized UI for an account conflict situation (CONFLICT_PLAYER), <br/>
    * you need to call it to notify the selected user when user select one.<br/>
    * INVALID_PARAM can occur if the SDK state is not in a conflict state, signIn is in progress, or the wrong playerId.<br/>
    * Even if a conflict occurs, the existing user is considered to be signIn until  selectConflict () is called then sign in.<br/>
    * <br/>
    * You do not need to call it if you use showConflictSelection().<br/>
    *
    * @param SelectedPlayerId PlayerId of the selected user
    *
    * @param Delegate FHiveAuthV4OnSignInDelegate<br/>
    *                 Note: The existing user is a signined user, until SUCCESS. <br/>
    *
    *  \~
    * @ingroup FHiveAuthV4
    */
    static void SelectConflict(int64 SelectedPlayerId, const FHiveAuthV4OnSignInDelegate& Delegate);

    /**
    *  \~korean
    * @brief 충돌상태에서 유저를 선택하는 UI 를 띄운다.
    * connect 시 CONFLICT_PLAYER 에러가 발생한 경우에 호출 하여 유저 선택이 가능한 UI 를 띄운다.<br/>
    * 충돌난 playerId 에 해당하는 유저가 인식할 수 있는 정보 (닉네임, 레벨 등) 들을 포함하여 호출 하여야 한다.<br/>
    * playerData 에는 "player_id" 를 키로 playerId 를 넣어주어야 한다.<br/>
    * 그 외의 정보는 "game_data" 를 키로 Map<String, Object> 형태 값을 넣어 주면 된다.<br/>
    * "player_id" 와 "game_data" 의 키는 필수이며 변경하면 안된다.<br/>
    * <br/>
    * ex) {"player_id":123, "game_data":{"Name":"CurrentPlayer", "Level":52}}<br/>
    * <br/>
    * 충돌난 유저로 선택 된 경우 다시 게임데이터를 불러와야 한다.<br/>
    *
    * @param Delegate FHiveAuthV4OnSignInDelegate
    *
    *  \~english
    * @brief It shows UI to select user in conflict state.
    * If a CONFLICT_PLAYER error occurs during connect() call, it shows the UI to display a user-selectable UI.<br/>
    * It should be called with information (nickname, level, etc.) that identifies the user corresponding to the conflicted playerId.<br/>
    * You should put the playerId in the "player_id" key in playerData.<br/>
    * For other information, add a value of type Map <String, Object> to the "game_data" key.<br/>
    * The keys name, "player_id" and "game_data" are required and should not be changed.<br/>
    * <br/>
    * ex) {"player_id":123, "game_data":{"Name":"CurrentPlayer", "Level":52}}<br/>
    * <br/>
    * If a conflicted user is selected, the game data corresponding to the user must be reloaded.<br/>
    *
    * @param Delegate FHiveAuthV4OnSignInDelegate
    *
    *  \~
    * @ingroup FHiveAuthV4
    *
    */
    static void ShowConflictSelection(const FHiveConflictViewData& ConflictViewData, const FHiveAuthV4OnSignInDelegate& Delegate);
    
    /**
    *  \~korean
    * @brief 요청한 Provider 의 상태를 체크 한다.
    * 요청한 Provider 로 Login 이 되어있지 않으면 Login 까지 시도한다.<br/>
    * Login 에 성공하면 providerUserId 까지 얻어 온다.<br/>
    * <br/>
    * 현재 playerId 와 connect 를 요청하지는 않는다.<br/>
    * <br/>
    * SUCCESS 는 조회에 성공했다는 뜻이며 providerInfo 에 providerUserId 여부로 로그인된 유저를 판단해야 한다.<br/>
    * 실제 provider 에 로그인 되어있다면 providerUserId 가 존재한다. 그 외 providerType 만 존재.<br/>
    * <br/>
    * Google Play Games, Apple Game Center 등 묵시적 사인-인 방식을 사용할 경우
    *
    * @param ProviderType 상태를 체크할 ProviderType
    * @param Delegate FHiveAuthV4OnCheckProviderDelegate
    *
    *  \~english
    * @brief It check the status of the requested Provider.
    * If the requested provider is not logged in, it tries to login.<br/>
    * If the login succeeds, it also brings providerUserId.<br/>
    * <br/>
    * It does not request connect with the current playerId.<br/>
    * <br/>
    * SUCCESS means that the query was successful and should determine whether or not the user is logged in with the presence of providerUserId in providerInfo.<br/>
    * If you are logged in to the actual provider, there is a providerUserId. Otherwise, only providerType exists.<br/>
    * <br/>
    * When using the implicit sign-in : Google Play Games, Apple Game Center etc.
    *
    * @param ProviderType ProviderType to check status
    * @param Delegate FHiveAuthV4OnCheckProviderDelegate
    *
    * @ingroup FHiveAuthV4
    *
    */
    static void CheckProvider(const EHiveProviderType& ProviderType, const FHiveAuthV4OnCheckProviderDelegate& Delegate);
    
    /**
    *  \~korean
    * @brief 점검 상태 표시 및 데이터 반환한다.
    * Configuration 에 설정한 ServerId 로 점검상태 여부를 체크한다.<br/>
    * <br/>
    * SDK 의 초기화가 완료되면 서버 점검 및 업데이트 상태를 확인해야 한다.<br/>
    * 서버 점검 및 업데이트는 게임 클라이언트의 업데이트 후에 하위 버전을 차단하거나, 게임 서버의 점검 시간 동안 게임 접속을 차단할 수 있다.<br/>
    * HIVE는 백오피스에 설정된 정보에 따라 서버 점검 혹은 공지 팝업을 노출하는 기능을 제공한다.<br/>
    * <br/>
    * 요청시 IsShow 가 true 였을 경우 점검 팝업 UI 가 노출된다.<br/>
    * <br/>
    * 결과가 SUCCESS 이지만 점검 공지 내용이 없다면 handler 의 maintenanceInfo 는 비어있게 된다.<br/>
    * 요청시 IsShow 를 false 로 주었다면 handler 의 maintenanceInfo 정보를 가지고 직접 UI 를 구성하여 노출하여야 한다.<br/>
    *
    * @param IsShow HIVE UI 를 사용할 지 여부
    * @param Delegate FHiveAuthV4OnMaintenanceInfoDelegate
    *
    *  \~english
    * @brief It display maintenance status and return its data.
    * It checks whether the server is in maintenance with the ServerId set in Configuration.<br/>
    * <br/>
    * Once the SDK is initialized, you should check the status of the server maintenance and update.<br/>
    * Server maintenance and Update can block lower version after update of game client, or block game connection during maintenance time of game server.<br/>
    * HIVE provides a function to expose server maintenance or update pop-up according to the information set in the back office.<br/>
    * <br/>
    * On request, a maintenance popup UI is exposed when IsShow is true.<br/>
    * <br/>
    * If the result is SUCCESS but there is no maintenance notification, the maintenanceInfo of the handler will be empty.<br/>
    * If IsShow is set to false on request, the UI should be configured and exposed by game developer with the maintenanceInfo information of the handler.<br/>
    *
    *  \~
    * @param IsShow Whether to use HIVE UI
    * @param Delegate FHiveAuthV4OnMaintenanceInfoDelegate
    *
    * @ingroup FHiveAuthV4
    */
    static void CheckMaintenance(bool IsShow, const FHiveAuthV4OnMaintenanceInfoDelegate& Delegate);

    /**
    *  \~korean
    * @brief 현재 유저의 제재 상태를 체크한다.
    * 기본적으로 signIn, connect 시에 자동으로 체크되며 제재 팝압을 띄우나 <br/>
    * 앱 내에서 필요한 시점에 실시간으로 체크가 필요할 때 사용할 수 있다.<br/>
    * <br/>
    * 요청시 isShow 가 true 였을 경우 팝업 UI 가 노출된다.<br/>
    * <br/>
    * 결과가 SUCCESS 이지만 제재 내용이 없다면 handler 의 maintenanceInfo 는 비어있게 된다.<br/>
    * 요청시 isShow 를 false 로 주었다면 handler 의 maintenanceInfo 정보를 가지고 직접 UI 를 구성하여 노출하여야 한다.<br/>
    *
    * @param IsShow HIVE UI 를 사용할 지 여부
    * @param Delegate FHiveAuthV4OnMaintenanceInfoDelegate
    *
    *  \~english
    * @brief it check the restriction status of the current user.
    * It is automatically checked when signIn, connect is called and shows a restriction popup,<br/>
    *  but you can use it when you need to check in realtime in the app.<br/>
    * <br/>
    * On request, the UI is exposed when isShow is true<br/>
    * <br/>
    * If the result is SUCCESS but there is no restriction notification, the maintenanceInfo of the handler will be empty.<br/>
    * If isShow is set to false on request, the UI should be configured and exposed by game developer with the maintenanceInfo information of the handler.<br/>
    *
    * @param IsShow Whether to use HIVE UI
    * @param Delegate FHiveAuthV4OnMaintenanceInfoDelegate
    *
    * @ingroup FHiveAuthV4
    */
    static void CheckBlacklist(bool IsShow, const FHiveAuthV4OnMaintenanceInfoDelegate& Delegate);

    /**
    *  \~korean
    * @brief 연동된 Provider 의 정보가 바뀌었는지 통보 받는다.
    * Google Play Games, Apple Game Center 의 경우 앱 외에서 계정 정보가 변경 될 수 있다.<br/>
    * 그렇기 때문에 앱이 resume (onStart) 될 때 유저 정보가 바뀌었는지 체크하게 되고,<br/>
    * 바뀌었다면 해당 콜백이 불리게 된다.<br/>
    * <br/>
    * 콜백은 checkProvider() 이후나 connect 등으로 직접 연동 시도 된 이후부터 동작하게 된다.<br/>
    * <br/>
    * 묵시적 사인-인 방식으로 구현할 경우 signIn() 완료 시점에서 checkProvider() 를 통해 직접 비교 체크 하여야 한다.<br/>
    *
    * @param Delegate FHiveAuthV4OnCheckProviderDelegate
    * 
    *  \~english
    * @brief It set a handler to be notified whether the information of the linked Provider has changed.
    * For Google Play Games and Apple Game Center, account information may be changed outside of the app.<br/>
    * That's why it checks to see if the user information has changed when the app resumes (onStart),<br/>
    * and if so, the callback is called.<br/>
    * <br/>
    * Callbacks will work after checkProvider() or attempting to connect directly with connect().<br/>
    * <br/>
    * If you implement implicit sign-in, you need to check directly with checkProvider () at the completion of signIn().<br/>
    *
    * @param Delegate FHiveAuthV4OnCheckProviderDelegate
    * 
    *  \~
    * @ingroup FHiveAuthV4
    *
    */
    static void SetProviderChangedListener(const FHiveAuthV4OnCheckProviderDelegate& Delegate);

    /**
    *  \~korean
    * @brief 사인-인 을 할 수 있는 Provider 목록이 있는 UI 를 띄운다.
    * setup 에서 받은 ProviderList 와 동일한 목록을 가진 사인-인 가능한 UI 를 띄운다.<br/>
    * 네트워크 지연 등으로 인해 GUEST 버튼이 노출되지 않을 수 있다.<br/>
    * AUTO 는 지원하지 않는다. isAutoSignIn() 이 true 라면 바로 signIn(AUTO) 를 호출 해 주면 된다.<br/>
    *
    * @param Delegate FHiveAuthV4OnSignInDelegate
    *
    *  \~english
    * @brief It shows the UI with a list of providers that can sign-in.
    * It will show a UI with the same list of providers as the ProviderList you received from setup() call.<br/>
    * A GUEST button may not be exposed due to network delays.<br/>
    * AUTO is not supported. If isAutoSignIn () is true, call signIn(AUTO) immediately.<br/>
    *
    * @param Delegate FHiveAuthV4OnSignInDelegate
    *
    *  \~
    * @ingroup FHiveAuthV4
    *
    */
    static void ShowSignIn(const FHiveAuthV4OnSignInDelegate& Delegate);

    /**
    *  \~korean
    * @brief 1:1 문의 UI 를 띄운다.
    * HIVE Membership 연동 여부와 상관없이 사용할 수 있으며<br/>
    * 연동되어 있다면 e-mail 부분이 자동으로 기입되어 있다.<br/>
    *
    * @param Delegate FHiveAuthV4OnShowInquiryDelegate
    *
    *  \~english
    * @brief It shows the 1: 1 query UI.
    * It can be used regardless of whether HIVE Membership is linked or not,<br/>
    * If it is linked, the e-mail part is automatically filled in.<br/>
    *
    * @param Delegate FHiveAuthV4OnShowInquiryDelegate
    *
    *  \~
    * @ingroup FHiveAuthV4
    *
    */
    static void ShowInquiry(const FHiveAuthV4OnShowInquiryDelegate& Delegate);

    /**
     * \~korean HIVE 챗봇 1:1 문의 화면 호출<br/>
     *
     * @param AdditionalInfo
     *            챗봇 페이지 바로가기 호출시 전달받기로한 약속된 JSON 형식의 String 데이터
     * @param Delegate
     *            API 호출 결과 통지
     * \~english Show HIVE Chatbot 1:1 inquiry <br/>
     *
     * @param AdditionalInfo
     *            Promised String data (JSON format) when you call chatbot page open API
     * @param Delegate
     *            API call result handler
     * \~
     * @ingroup FHiveAuthV4
     */
    static void ShowChatbotInquiry(const FString& AdditionalInfo, const FHiveAuthV4OnShowInquiryDelegate& Delegate);
    
    /**
     *  \~korean
     * @brief 1:1 내문의 UI 를 띄운다.
     * Hive 연동 되어 있는 경우에만 사용 할 수 있다.<br/>
     *
     * @param Delegate FHiveAuthV4OnShowInquiryDelegate
     *
     *  \~english
     * @brief It shows the 1: 1 query UI.
     * It can be used only when Hive is linked.<br/>
     *
     * @param Delegate FHiveAuthV4OnShowInquiryDelegate
     *
     *  \~
     * @ingroup FHiveAuthV4
     *
    */
    static void ShowMyInquiry(const FHiveAuthV4OnShowInquiryDelegate& Delegate);

    /**
    *  \~korean
    * @brief HIVE 약관 정보를 표시한다.
    * 약관 정보를 표시한다.<br/>
    * SDK 초기화 시 사용자에게 약관 동의 절차를 거치게 된다.<br/>
    * 이후 게임에서는 설정 창 등에서 개인 정보 처리 방침 및 이용 약관 정보를 확인할 수 있는 웹뷰를 노출하도록 구성해야 한다.<br/>
    *
    * @param Delegate 	FHiveAuthV4OnShowTermsDelegate<br/>
    *                 					API 호출 결과 통지
    *
    *  \~english
    * @brief  It shows HIVE Terms and Conditions.
    * It shows Terms and Conditions<br/>
    * At initialization of the SDK, the user will go through the agreement process.<br/>
    * After this, the game should be configured to expose the WebView os that user can check Privacy policy and Terms and Conditions in the game setting menu.<br/>
    *
    * @param Delegate 	FHiveAuthV4OnShowTermsDelegate<br/>
    *                 					API callback
    *
    *  \~
    * @ingroup FHiveAuthV4
    * 
    */
    static void ShowTerms(const FHiveAuthV4OnShowTermsDelegate& Delegate);

    /**
    *  \~korean
    * @brief 성인 인증을 요청한다.
    * 고스톱 / 포커류와 같이 성인 인증이 필요한 일부 게임의 경우 성인 인증 기능을 제공한다.
    *
    * @param Delegate FHiveAuthV4OnAdultConfirmDelegate<br/>
    *                 result : 사인-인 이 필요하기 때문에 사인-인 이 되어있지 않을 경우 INVALID_SESSION 이 올 수 있다.<br/>
    *                   성인인증에 정상적으로 진행되었을 경우 SUCCESS 가 된다.<br/>
    *
    *  \~english
    * @brief It request adult authentication.
    * For some games that require adult authentication, it provide adult authentication.
    *
    * @param Delegate FHiveAuthV4OnAdultConfirmDelegate<br/>
    *                 result : INVALID_SESSION can be given if it is not signed-in because it requires a sign-in.<br/>
    *                   SUCCESS will be made if the adult is successfully certified.<br/>
    *
    *  \~
    * @ingroup FHiveAuthV4
    *
    */
    static void ShowAdultConfirm(const FHiveAuthV4OnAdultConfirmDelegate& Delegate);

    /**
    *  \~korean
    * @brief 요청한 playerId 들의 프로필 정보를 반환한다.
    * playerId 중 사인-인 된 자신이 있을 경우 playerName 과 playerImageUrl 을 동기화 한다.<br/>
    *
    * @param PlayerIdArray ArrayList<Long>
    *
    * @param Delegate FHiveAuthV4OnGetProfileDelegate
    *
    *  \~english
    * @brief it returns the profile information of the requested playerIds.
    * It update playerName and playerImageUrl when the playerId list contains the signed-in oneself.<br/>
    *
    * @param PlayerIdArray ArrayList<Long>
    *
    * @param Delegate FHiveAuthV4OnGetProfileDelegate
    *
    *  \~
    * @ingroup FHiveAuthV4
    *
    */
    static void GetProfile(const TArray<int64> PlayerIdArray, const FHiveAuthV4OnGetProfileDelegate& Delegate);

    /**
    *  \~korean
    * @brief 요청한 playerId 의 프로필 UI 를 표시한다.
    * playerId 가 사인-인 된 자신이며 HIVE 와 연동된 유저일 경우 이 UI 에서 프로필 사진과 닉네임을 변경할 수 있다.<br/>
    * 이 경우 창이 닫히면 playerName 과 playerImageUrl 를 로컬 정보와 동기화 한다.<br/>
    *
    * @param PlayerId 프로필 UI 를 띄울 playerId
    *
    * @param Delegate FHiveAuthV4OnShowProfileDelegate
    *
    *  \~english
    * @brief It display the profile UI of the requested playerId.
    * If the playerId is a sign-in user's one and the user is a member of HIVE Membership, the user can change the profile picture and nickname in this UI.<br/>
    * In this case, when the window is closed, the playerName and playerImageUrl are synchronized with the local information.<br/>
    *
    * @param PlayerId PlayerId to show the profile UI
    *
    * @param Delegate FHiveAuthV4OnShowProfileDelegate
    *
    *  \~
    * @ingroup FHiveAuthV4
    *
    */
    static void ShowProfile(int64 PlayerId, const FHiveAuthV4OnShowProfileDelegate& Delegate);

    /**
    *  \~korean
    * @brief 약관동의 관련 데이터를 초기화 한다.
    *
    *  \~english
    * @brief Initialize data related to agreement to terms.
    *
    *  \~
    * @ingroup FHiveAuthV4
    *
    */
    static void ResetAgreement();

    /**
    *  \~korean
    * @brief signIn(AUTO) 가 가능한지 여부를 반환한다.
    * AuthV4.setup() 호출 시 알려주는 값과 동일 하며,<br/>
    * 로컬에 저장된 기존 세션만 체크 하기때문에 세션이 실제로 만료되지 않은 유효한 토큰인지는 알 수 없다.<br/>
    * true 일 경우 'TOUCH TO START' 등의 UI 를 통해 signIn(AUTO) 를 호출하고<br/>
    * 여기서 실패할 경우 다시 초기 사인-인 씬으로 돌아가야 한다.<br/>
    * <br/>
    * false 일 경우 AuthV4.setup() 호출 시 받은 providerList 를 통해 UI 를 구성하여 signIn 을 시도 하거나<br/>
    * showSignIn() 을 통해 HIVE 에서 제공하는 UI 를 사용하여도 된다.<br/>
    * <br/>
    * 앱 중간에 signOut 등 사인-인 씬으로 돌아왔을 경우 이 API 를 통해 UI 구성을 여부를 체크할 수 있다.<br/>
    *
    * @return boolean signIn(AUTO) 가 가능한지 여부
    *
    *  \~english
    * @brief It returns whether signIn (AUTO) is enabled.
    * It is the same value that you would get when you called setup(),<br/>
    * and it only checks for existing sessions that are stored locally, so you do not know if the session token is a valid token that the session has not been expired.<br/>
    * If true, then signIn (AUTO) should be called through UI such as 'TOUCH TO START', <br/>
    * and if signIn(AUTO) fails, it should return to the initial sign-in screen again.<br/>
    * <br/>
    * If it is false, you can customize the UI through the providerList provided at setup() call<br/>
    * or use the UI provided by HIVE via showSignIn()<br/>
    * <br/>
    * If you return to the sign-in scene such as signOut in the middle of your app, you can check whether you need to configure the UI through this API.<br/>
    *
    * @return boolean Whether signIn (AUTO) is enabled.
    *
    *  \~
    * @ingroup FHiveAuthV4
    *
    */
    static bool IsAutoSignIn();

    /**
    *  \~korean
    * @brief 현재 사인-인 된 유저의 정보를 반환 한다.
    * PlayerInfo 에는 playerToken 을 포함하고 있으며<br/>
    * 프로필 정보 (playerName, playerImageUrl) 까지 포함하고 있으나,<br/>
    * 요청시 실시간으로 서버에 요청하지 않고 로컬에 캐쉬되어 있는 프로필 정보를 반환 한다.<br/>
    * <br/>
    * getProfile 이나 showProfile 에서 다인-인 된 유저 자신일 경우 자동으로 캐쉬를 갱신한다..<br/>
    *
    * @return FHivePlayerInfo 현재 사인-인 된 유저의 정보. 사인-인 되어있지 않을 경우 null.
    *
    *  \~english
    * @brief It returns information about the currently signed-in user.
    * PlayerInfo includes playerToken and profile information (playerName, playerImageUrl),<br/>
    * but it returns the locally cached profile information instead of requesting it to the server in real time upon request.<br/>
    * <br/>
    * When getProfile or showProfile is called, it is automatically updated profile information if a signed-in user is it self.<br/>
    *
    * @return FHivePlayerInfo Information about the currently signed-in user. Null if not signed-in.
    *
    *  \~
    * @ingroup FHiveAuthV4
    *
    */
    static TOptional<FHivePlayerInfo> GetPlayerInfo();

    /**
    *  \~korean
    * @brief 입력받은 providerType 에 해당하는 연결된 사용자의 친구들의 playerId 목록 정보를 얻어온다.
    * 요청한 providerType의 providerId와 연결된 사용자의 playerId를 페어로 얻어온다.<br/>
    * kHIVEProviderTypeFACEBOOK 의 경우 최대 5000명.<br/>
    * <br/>
    * getProfile 을 사용하여 친구의 PlayerInfo 를 조회할 수 있다.<br/>
    *
    * @param ProviderType	친구목록을 조회할 연결된 ProviderType 값.
    * @param Delegate FHiveAuthV4OnGetProviderFriendsListDelegate
    *
    *  \~english
    * @brief It returns the playerId list information of the friends of the connected user corresponding to the input providerType..
    * It comes with a pair of information about the providerId of the requested providerType and the playerId of the associated user.<br/>
    * Up to 5000 people for kHIVEProviderTypeFACEBOOK.<br/>
    * <br/>
    * You can query your friend's PlayerInfo using getProfile.<br/>
    *  
    * @param ProviderType ProviderType to get friends list
    * @param Delegate FHiveAuthV4OnGetProviderFriendsListDelegate
    *
    *  \~
    * @ingroup FHiveAuthV4
    *
    */
    static void GetProviderFriendsList(const EHiveProviderType& ProviderType, const FHiveAuthV4OnGetProviderFriendsListDelegate& Delegate);
    
    /**
    *  \~korean
    * @brief connect 로 인해 계정이 충돌 된 상황 (CONFLICT_PLAYER) 을 해제 한다.
    * connect 로 인해 계정이 충돌 된 상황 (CONFLICT_PLAYER) 에서 자체 UI 를 구성하였을 경우에 사용<br/>
    * 충돌 된 상황 (CONFLICT_PLAYER) 을 해제 한다.<br/>
    * 호출시 해당하는 IDP (Provider) 를 로그아웃하게 된다.<br/>
    * 앱 내에서 로그아웃 할 수 없는 IDP (Apple GameCenter)의 경우 로그아웃은 불가능하고,<br/>
    * 내부 충돌 정보만 삭제 된다.
    *
    * @param Delegate FHiveAuthV4OnResolveConflictDelegate<br/>
    *                 실제 IDP 로그아웃에 실패했어도 내부 충돌 정보는 삭제 되다.
    *
    *  \~english
    * @brief It resolve the account conflicts(CONFLICT_PLAYER) caused by connect () call.
    * This is used when the customized UI is used in the account conflict state caused by connect() call.<br/>
    * It resolve the account conflicts(CONFLICT_PLAYER).<br/>
    * Upon calling, the corresponding IdP (Provider) is logged out.<br/>
    * For an Apple GameCenter (IDP) that can not log out of the app, you can not sign out,<br/>
    * and only delete internal conflict information.
    *
    * @param Delegate FHiveAuthV4OnResolveConflictDelegate<br/>
    *                 Even if the actual IdP logout fails, the internal conflict information is deleted.
    *
    *  \~
    *
    * @ingroup FHiveAuthV4
    *
    */
    static void ResolveConflict(const FHiveAuthV4OnResolveConflictDelegate& Delegate);

    // nothing
    static void ShowGameCenterLoginCancelDialog(const FHiveAuthV4OnAuthV4DialogDismissDelegate& Delegate);

    /**
     *  \~korean
     * @brief COPPA 13세 나이 제한 여부를 반환한다.<br/>
     * 기본값은 false이며, AuthV4.setup() 호출 시 COPPA 약관 동의에 따라 결정된다.<br/>
     * 정상적인 값을 받으려면 AuthV4.setup() 이후에 호출하여야 한다.<br/>
     *
     * @return boolean COPPA 13세 나이 제한 여부
     *
     *  \~english
     *  @brief Returns the value whether COPPA limits users under 13 or not.<br/>
     *  Default value is false, and it depends on the agreement on COPPA when calling AuthV4.setup() API.<br/>
     *  Normal process is calling this API after AuthV4.setup() API is completed.<br/>
     *
     * @return boolean the value whether COPPA limits users under 13 or not
     *  \~
     * @ingroup FHiveAuthV4
     *
     */
    static bool GetAgeGateU13();

    /**
     *  \~korean
     * @brief 기기 관리 서비스 UI 를 띄운다.
     * 미가입 계정이라면 기기 관리 서비스 가입 창을 띄운다.<br/>
     *
     * @param Delegate FHiveAuthV4OnShowDeviceManagementDelegate
     *
     *  \~english
     * @brief It shows the device management UI.
     * If unsigned account, the device management service subscription UI appears.<br/>
     *
     * @param Delegate FHiveAuthV4OnShowDeviceManagementDelegate
     *
     *  \~
     * @ingroup FHiveAuthV4
     *
     */
    static void ShowDeviceManagement(const FHiveAuthV4OnShowDeviceManagementDelegate& Delegate);

    /**
     *  \~korean
     * @brief HiveTalkPlus 로그인시 필요한 로그인 토큰을 얻어온다.
     *
     *  \~english
     * @brief Get the required login token when logging in to HiveTalkPlus.
     *
     *  \~
     * @ingroup FHiveAuthV4
     */
    static void GetHiveTalkPlusLoginToken(const FHiveAuthV4OnGetHiveTalkPlusLoginTokenDelegate& Delegate);

    /**
     *  \~korean
     * @brief Hive 로그인시 필요한 액세스 토큰을 얻어온다.
     *
     *  \~english
     * @brief Get the required access token when logging in to Hive.
     *
     *  \~
     * @ingroup AuthV4
     */
    static void RefreshAccessToken(const FHiveAuthV4OnRefreshAccessTokenDelegate& Delegate);

    /**
     *  \~korean
     * @brief PlayerID를 이용하여 로그인 요청을 수행한다.
     * 주어진 playerId를 사용하여 로그인 요청을 수행한다.
     *
     * @param playerId 로그인에 사용할 PlayerID
     * @param useAutoSignIn 자동 로그인 사용 여부
     * @param signInListener onSignIn AuthV4 사인-인 결과 통지
     *
     *  \~english
     * @brief Requests login using PlayerID.
     * Performs a login request using the given playerId.
     *
     * @param signInListener onSignIn AuthV4 signIn result callback
     * @param playerId PlayerID to use for login
     *  \~
     * @ingroup AuthV4
     */
    static void SignInWithStoredPlayerId(int64 PlayerId, bool useAutoSignIn, const FHiveAuthV4OnSignInDelegate& Delegate);

    /**
     *  \~korean
     * @brief 저장된 플레이어 ID 리스트를 반환한다.
     * signOut 시 저장된 플레이어 ID 리스트를 반환한다.
     *
     * @param playerIdListener onPlayerId 플레이어 ID 리스트 반환 결과 통지
     *
     *  \~english
     * @brief Returns the list of saved player IDs.
     * Returns the list of player IDs saved during signOut.
     *
     * @param playerIdListener onPlayerId player ID list result callback
     *  \~
     * @ingroup AuthV4
     */
    static void GetStoredPlayerIdList(const FHiveAuthV4OnGetStoredPlayerIdListDelegate& Delegate);

    /**
     *  \~korean
     * @brief 현재 로그인된 세션을 저장한다.
     * 현재 로그인된 세션 정보를 저장한다.
     *
     * @param addListener onAddSession 세션 저장 결과 통지
     *
     *  \~english
     * @brief Saves the current logged-in session.
     * Saves the current logged-in session information.
     *
     * @param addListener onAddSession session save result callback
     *  \~
     * @ingroup AuthV4
     */
    static void StoreCurrentPlayerId(const FHiveAuthV4OnStoreCurrentPlayerIdDelegate& Delegate);

    /**
     *  \~korean
     * @brief 저장된 세션 정보를 삭제한다.
     * signOut에서 저장된 세션 정보를 playerId를 이용하여 찾아서 삭제한다.
     *
     * @param removeListener removeSession 세션 정보 삭제 결과 통지
     * @param playerId 삭제할 세션 정보를 찾기 위한 PlayerID
     *
     *  \~english
     * @brief Deletes the saved session information.
     * Finds and deletes the session information saved during signOut using playerId.
     *
     * @param removeListener removeSession session information removal result callback
     * @param playerId PlayerID to find the session information to delete
     *  \~
     * @ingroup AuthV4
     */
    static void DeleteStoredPlayerId(int64 PlayerId, const FHiveAuthV4OnDeleteStoredPlayerIdDelegate& Delegate);

    /**
     *  \~korean
     * @brief 보호자 동의 정보를 반환한다.<br/>
     * 이용자의 생년월일과 보호자 이메일 주소가 포함된다.
     * 보호자 동의 이후에 호출하여야 한다.<br/>
     * 해당 정보는 보호자 동의 이후 앱 종료 이전까지만 유지된다.<br/>
     *
     * @return FHiveParentalConsentInfo 보호자 동의 정보
     *
     *  \~english
     *  @brief Returns the parent consent information.<br/>
     *  It includes the user's date of birth and the guardian's email address.<br/>
     *  It should be called after the parent consent is given.<br/>
     *  The information is retained only until the app is closed after the parent consent.<br/>
     *
     * @return FHiveParentalConsentInfo Parent Consent Information
     *
     *  \~
     * @ingroup FHiveAuthV4
     *
     */
    static TOptional<FHiveParentalConsentInfo> GetParentalConsentInfo();

    /**
     *  \~korean
     *  @brief 앱 스토어에서 확인한 사용자의 연령대 및 감독 상태를 반환한다.<br/>
     *  관할권 및 지역에 따라 제공 여부가 달라질 수 있으며,<br/>
     *  연령대 정보를 기반으로 앱 내 콘텐츠 제어가 가능하다.
     *
     *  @param FHiveAuthV4OnGetAgeRangeDelegate 연령대 정보를 수신하는 콜백
     *
     *  \~english
     *  @brief Returns the user's age range and supervision status verified by the App Store.<br/>
     *  Availability may vary depending on jurisdictions and regions.<br/>
     *  You can adjust in-app content based on the received age range information.
     *
     *  @param FHiveAuthV4OnGetAgeRangeDelegate Callback to receive age range information
     *
     *  \~
     *  @since 26.0.0
     */
    static void GetAgeRange(const FHiveAuthV4OnGetAgeRangeDelegate& Delegate);

    /**
     *  \~korean
     *  @brief 사용자의 연령대 정보를 기반으로, 미성년자인 경우 주요 앱 업데이트에 대한<br/>
     *  보호자 승인 요청 UI를 표시하고 그 결과를 콜백으로 반환한다.
     *
     *  @param Description 보호자 승인 요청 화면에 표시할 설명 문구
     *  @param FHiveAuthV4OnAgeRangeUpdatePermissionDelegate 승인 결과를 수신하는 콜백
     *
     *  \~english
     *  @brief Displays the guardian approval UI when the user is a minor based on the age range info.<br/>
     *  Shows the platform-provided consent prompt and returns the result (approval/denial) via callback.
     *
     *  @param Description Text description displayed on the guardian approval prompt
     *  @param FHiveAuthV4OnAgeRangeUpdatePermissionDelegate Callback to receive the approval result
     *
     *  \~
     *  @since 26.0.0
     */
    static void ShowAgeRangeUpdatePermission(const FString& Description, const FHiveAuthV4OnShowAgeRangeUpdatePermissionDelegate& Delegate);

    /**
     *  \~korean
     * @brief 현재 로그인된 PlayerId에 연결된 해시된 DI 값을 조회하여 반환한다.
     *
     * @param hashedDiListner onHashedDi 해시된 DI 값 반환 결과 통지
     *
     *  \~english
     * @brief Retrieves and returns the hashed DI value linked to the currently logged-in PlayerId.
     *
     * @param hashedDiListner onHashedDi hashed DI value result callback
     *  \~
     * @ingroup AuthV4
     */
    static void GetHashedDi(const FHiveAuthV4OnGetHashedDiDelegate& Delegate);

    /**
    *  \~korean
    * @brief 본인 확인을 요청한다.
    *
    * @param Delegate FHiveAuthV4OnIdentityVerificationDelegate<br/>
    *                 result : setup이 필요하기 때문에 되어있지 않을 경우 NEED_INITIALIZE 값이 올 수 있다.<br/>
    *                          본인 확인이 정상적으로 진행되었을 경우 SUCCESS가 된다.<br/>
    *
    *  \~english
    * @brief It request identity verification.
    *
    * @param Delegate FHiveAuthV4OnIdentityVerificationDelegate<br/>
    *                 result : NEED_INITIALIZE can come if you do not have a setup because you need a initialize.<br/>
    *                          SUCCESS, if you have successfully completed the identity verification.<br/>
    *
    *  \~
    * @ingroup FHiveAuthV4
    *
    */
    static void ShowIdentityVerification(const FHiveAuthV4OnIdentityVerificationDelegate& Delegate);

    /**
     *  \~korean
     * @brief 생일 확인 팝업을 호출한다.
     *
     * @param UseCloseButton 팝업에서 닫기 버튼을 노출할지 여부
     * @param CheckAgeGateListener 결과 콜백
     *
     *  \~english
     * @brief Shows the birthday verification popup.
     *
     * @param UseCloseButton Whether the popup shows the close button
     * @param CheckAgeGateListener Result callback
     *  \~
     * @ingroup AuthV4
     */
    static void CheckAgeGate(bool UseCloseButton, const FHiveAuthV4OnCheckAgeGateDelegate& Delegate);

    /**
     *  \~korean
     * @brief 자동로그인 여부를 설정한다.
     * 자동로그인 허용 여부를 지정하면, 이후 로그인 시 해당 설정에 따라 세션을 저장하여 다음 앱 실행 시 자동로그인 세션으로 사용할 수 있다.<br/>
     *
     * @param enabled 자동로그인 허용 여부 (true: 자동로그인 허용, false: 자동로그인 미허용)
     *
     *  \~english
     * @brief Sets whether auto-login is enabled.
     * When you set auto-login enabled, the session will be stored according to this setting and can be used for auto-login on the next app launch.<br/>
     *
     * @param enabled Whether to enable auto-login (true: enabled, false: disabled)
     *  \~
     * @ingroup AuthV4
     */
    static void SetAutoLoginEnabled(bool enabled, const FHiveAuthV4OnSetAutoLoginEnabledDelegate& Delegate);

    /**
     *  @file	HiveAuthV4.h
     *
     *  @copyright	Copyright © GAMEVIL COM2US PLATFORM Inc. All Right Reserved.
     *
     *  @defgroup FHiveAuthV4Helper
     *
     *  @{
     *  \~korean
     * @brief HIVE 인증의 고도화 된 기능을 제공한다 <br>
     * <br>
     * 이 클래스에서 제공하는 상세 기능은 다음과 같다. <br>
     *  - 사인-인/아웃<br>
     *  - Provider 연결/연결 해제<br>
     *  - 업적/리더보드 조회<br>
     *  - 계정 충돌 상태 조회/해결<br>
     *
     *  \~english
     * @brief Provides advanced features of HIVE Authentication.<br>
     * <br>
     * The detailed functions provided by this class are as follows. <br>
     *  - Sign-in/Sign-out<br>
     *  - Connect/Disconnect Providers<br>
     *  - Query Achievement/Leaderboard<br>
     *  - Query/Resolve account conflicts state<br>
     *  \~
     */
    class HIVESDK_API Helper : public IHiveInterface<IHiveAuthV4HelperImpl>
    {
    public:
        /**
         *  \~korean
         * @brief 제공된 API 외 경우에 계정의 충돌 여부를 확인할 수 있다.
         * <br>
         * ### 사용 조건
         *   -# HIVE SDK 초기화
         *   -# HIVE SignIn 완료
         * <br>
         * <br>
         * ### 주요 결과 코드
         <table>
         <tr>
         <td><strong>ErrorCode</strong></td>
         <td><strong>Code</strong></td>
         <td><strong>Description</strong></td>
         <td><strong>Solution</strong></td>
         </tr>
         <tr>
         <td>SUCCESS</td>
         <td>Success</td>
         <td>계정 상태 정상</td>
         <td> - </td>
         </tr>
         <tr>
         <td>NEED_INITIALIZE</td>
         <td>AuthV4NotInitialized</td>
         <td>HIVE SDK가 초기화되지 않은 경우</td>
         <td>AuthV4.setup() API로 HIVE SDK 초기화</td>
         </tr>
         <tr>
         <td>CONFLICT_PLAYER</td>
         <td>AuthV4ConflictPlayer</td>
         <td>로그인한 계정과 기기에 로그인된 PGS/GameCenter 계정이 다른 경우</td>
         <td>계정 충돌 상황 해결 방법 안내에 따라 해결</td>
         </tr>
         <tr>
         <td>NOT_SUPPORTED</td>
         <td>AuthV4NotSupportedProviderType</td>
         <td>PGS/GameCenter 외 Provider Type으로 계정 상태를 조회한 경우</td>
         <td>-</td>
         </tr>
         </table>
         *
         * @param providerType 동기화를 요청할 ProviderType
         *
         * @param listener onAuthV4Helper AuthV4Helper Sync Account 결과 통지
         *
         *  \~english
         * @brief Helps to inform whether accounts conflict or not when executing the provided as well as other APIs.
         * <br>
         * ### Condition of use
         *   -# Complete to initialize HIVE SDK
         *   -# Complete HIVE SignIn
         * <br>
         * <br>
         * ### Key result code
         <table>
         <tr>
         <td><strong>ErrorCode</strong></td>
         <td><strong>Code</strong></td>
         <td><strong>Description</strong></td>
         <td><strong>Solution</strong></td>
         </tr>
         <tr>
         <td>SUCCESS</td>
         <td>Success</td>
         <td>Account normal state</td>
         <td> - </td>
         </tr>
         <tr>
         <td>NEED_INITIALIZE</td>
         <td>AuthV4NotInitialized</td>
         <td>HIVE SDK not initialized</td>
         <td>Initialize HIVE SDK by implementing AuthV4.setup() API</td>
         </tr>
         <tr>
         <td>CONFLICT_PLAYER</td>
         <td>AuthV4ConflictPlayer</td>
         <td>Mismatched the signed-in account with PGS/GameCenter account on the device</td>
         <td>Follow the resolution in accordance with the type of account conflicts</td>
         </tr>
         <tr>
         <td>NOT_SUPPORTED</td>
         <td>AuthV4NotSupportedProviderType</td>
         <td>Queried account state with other provider types except PGS/GameCenter</td>
         <td>-</td>
         </tr>
         </table>
         *
         * @param providerType ProviderType to request syncronization
         *
         * @param listener onAuthV4Helper AuthV4Helper Sync Account result callback
         *  \~
         *
         * @ingroup FHiveAuthV4Helper
         *
         */
        static void SyncAccount(const EHiveProviderType& ProviderType, const FHiveAuthV4HelperDelegate& Delegate);

        /**
         *  \~korean
         * @brief 사용자 로그인
         * <br>
         * ### 사용 조건
         *   -# HIVE SDK 초기화
         * <br>
         * <br>
         * ### 주요 결과 코드
         <table>
         <tr>
         <td><strong>ErrorCode</strong></td>
         <td><strong>Code</strong></td>
         <td><strong>Description</strong></td>
         <td><strong>Solution</strong></td>
         </tr>
         <tr>
         <td>SUCCESS</td>
         <td>Success</td>
         <td>로그인 성공<br>
         <strong>playerInfo</strong>: 로그인 성공한 사용자의 정보</td>
         <td> - </td>
         </tr>
         <tr>
         <td>NEED_INITIALIZE</td>
         <td>AuthV4NotInitialized</td>
         <td>HIVE SDK가 초기화되지 않은 경우</td>
         <td>AuthV4.setup() API로 HIVE SDK 초기화</td>
         </tr>
         <tr>
         <td>IN_PROGRESS</td>
         <td>AuthV4InProgressSignIn<br>
         <td>Sign In 처리 중 재요청이 들어온 경우</td>
         <td>기존 SignIn 요청 결과 대기</td>
         </tr>
         <tr>
         <td>CONFLICT_PLAYER</td>
         <td>AuthV4ConflictPlayer</td>
         <td>Device에 로그인 된 계정과 현재 로그인 된 계정의 PGS/GameCenter 정보가 다르거나<br>
         Connect를 시도한 Provider의 Player ID가 이미 있는 경우<br>
         <strong>playerInfo</strong>: 충돌 계정 정보
         </td>
         <td>계정 충돌 상황 해결 방법 안내에 따라 해결</td>
         </tr>
         <tr>
         <td>INVALID_SESSION</td>
         <td>AuthV4HelperImplifiedLoginFail</td>
         <td>묵시적 로그인에 실패하여 명시적 로그인을 진행해야 하는 경우</td>
         <td>AuthV4.Helper.getIDPList()로 지원 Provider 목록 조회 후 명시적 로그인 UI를 구현하거나, <br>
         AuthV4.showSignIn() HIVE UI를 이용하여 명시적 로그인 동작 수행</td>
         </tr>
         </table>
         *
         * @param Delegate FHiveAuthV4HelperDelegate AuthV4Helper Sign In 결과 통지
         *
         *  \~english
         * @brief Player sign-in
         * <br>
         * ### Condition of use
         *   -# Complete to initialize HIVE SDK
         * <br>
         * <br>
         * ### Key result code
         <table>
         <tr>
         <td><strong>ErrorCode</strong></td>
         <td><strong>Code</strong></td>
         <td><strong>Description</strong></td>
         <td><strong>Solution</strong></td>
         </tr>
         <tr>
         <td>SUCCESS</td>
         <td>Success</td>
         <td>Succeeded to sign in<br>
         <strong>playerInfo</strong>: The information of signed-in player</td>
         <td> - </td>
         </tr>
         <tr>
         <td>NEED_INITIALIZE</td>
         <td>AuthV4NotInitialized</td>
         <td>HIVE SDK not initialized</td>
         <td>Initialize HIVE SDK by implementing AuthV4.setup() API</td>
         </tr>
         <tr>
         <td>IN_PROGRESS</td>
         <td>AuthV4InProgressSignIn<br>
         <td>Requested to process SignIn while sign-in is in progress</td>
         <td>Wait for the SignIn request in progress</td>
         </tr>
         <tr>
         <td>CONFLICT_PLAYER</td>
         <td>AuthV4ConflictPlayer</td>
         <td>The account signed in on user device is mismatched with the PGS/Game Center account on the game<br>
         or the player ID of the Provider to be connected already existed<br>
         <strong>playerInfo</strong>: The information of conflicted account
         </td>
         <td>Follow the resolution in accordance with the type of account conflicts</td>
         </tr>
         <tr>
         <td>INVALID_SESSION</td>
         <td>AuthV4HelperImplifiedLoginFail</td>
         <td>Required Explicit Login due to failure of Implicit Login</td>
         <td>Query the supported Provider lists by calling AuthV4.Helper.getIDPList() and implement Explicit Login UI, <br>
         or implement Explicit Login by calling AuthV4.showSignIn() of HIVE UI</td>
         </tr>
         </table>
         *
         * @param Delegate FHiveAuthV4HelperDelegate AuthV4Helper Sign In result callback
         *  \~
         *
         * @ingroup FHiveAuthV4Helper
         *
         */
        static void SignIn(const FHiveAuthV4HelperDelegate& Delegate);

        /**
         *  \~korean
         * @brief 사용자 로그아웃
         * <br>
         * ### 사용 조건
         *   -# HIVE SDK 초기화
         *   -# HIVE SignIn 완료
         * <br>
         * <br>
         * ### 주요 결과 코드
         <table>
         <tr>
         <td><strong>ErrorCode</strong></td>
         <td><strong>Code</strong></td>
         <td><strong>Description</strong></td>
         <td><strong>Solution</strong></td>
         </tr>
         <tr>
         <td>SUCCESS</td>
         <td>Success</td>
         <td>로그아웃 성공<br>
         <td> - </td>
         </tr>
         <tr>
         <td>NEED_INITIALIZE</td>
         <td>AuthV4NotInitialized</td>
         <td>HIVE SDK가 초기화되지 않은 경우</td>
         <td>AuthV4.setup() API로 HIVE SDK 초기화</td>
         </tr>
         </table>
         *
         * @param Delegate FHiveAuthV4HelperDelegate AuthV4Helper Sign Out 결과 통지
         *
         *  \~english
         * @brief Player sign-out
         * <br>
         * ### Condition of use
         *   -# Complete to initialize HIVE SDK
         *   -# Complete HIVE SignIn
         * <br>
         * <br>
         * ### Key result code
         <table>
         <tr>
         <td><strong>ErrorCode</strong></td>
         <td><strong>Code</strong></td>
         <td><strong>Description</strong></td>
         <td><strong>Solution</strong></td>
         </tr>
         <tr>
         <td>SUCCESS</td>
         <td>Success</td>
         <td>Succeeded to sign out<br>
         <td> - </td>
         </tr>
         <tr>
         <td>NEED_INITIALIZE</td>
         <td>AuthV4NotInitialized</td>
         <td>HIVE SDK not initialized</td>
         <td>Initialize HIVE SDK by implementing AuthV4.setup() API</td>
         </tr>
         </table>
         *
         * @param Delegate FHiveAuthV4HelperDelegate AuthV4Helper Sign Out result callback
         *  \~
         *
         * @ingroup FHiveAuthV4Helper
         *
         */
        static void SignOut(const FHiveAuthV4HelperDelegate& Delegate);

        /**
         *  \~korean
         * @brief 사용자 삭제
         * <br>
         * ### 사용 조건
         *   -# HIVE SDK 초기화
         *   -# HIVE SignIn 완료
         * <br>
         * <br>
         * ### 주요 결과 코드
        <table>
        <tr>
        <td><strong>ErrorCode</strong></td>
        <td><strong>Code</strong></td>
        <td><strong>Description</strong></td>
        <td><strong>Solution</strong></td>
        </tr>
        <tr>
        <td>SUCCESS</td>
        <td>Success</td>
        <td>로그아웃 성공<br>
        <td> - </td>
        </tr>
        <tr>
        <td>NEED_INITIALIZED</td>
        <td>AuthV4NotInitialized</td>
        <td>HIVE SDK가 초기화되지 않은 경우</td>
        <td>AuthV4.setup() API로 HIVE SDK 초기화</td>
        </tr>
        </table>
         *
         * @param Delegate FHiveAuthV4HelperDelegate AuthV4Helper PlayerDelete 결과 통지
         *
         *  \~english
         * @brief Player delete
         * <br>
         * ### Condition of use
         *   -# Complete to initialize HIVE SDK
         *   -# Complete HIVE SignIn
         * <br>
         * <br>
         * ### Key result code
        <table>
        <tr>
        <td><strong>ErrorCode</strong></td>
        <td><strong>Code</strong></td>
        <td><strong>Description</strong></td>
        <td><strong>Solution</strong></td>
        </tr>
        <tr>
        <td>SUCCESS</td>
        <td>Success</td>
        <td>Succeeded to sign out<br>
        <td> - </td>
        </tr>
        <tr>
        <td>NEED_INITIALIZED</td>
        <td>AuthV4NotInitialized</td>
        <td>HIVE SDK not initialized</td>
        <td>Initialize HIVE SDK by implementing AuthV4.setup() API</td>
        </tr>
        </table>
         *
         * @param Delegate FHiveAuthV4HelperDelegate AuthV4Helper PlayerDelete result callback
         *  \~
         *
         * @ingroup FHiveAuthV4Helper
         *
         */
        static void PlayerDelete(const FHiveAuthV4HelperDelegate& Delegate);

        /**
         *  \~korean
         * @brief Provider 연결
         * <br>
         * ### 사용 조건
         *   -# HIVE SDK 초기화
         *   -# HIVE SignIn 완료
         * <br>
         * <br>
         * ### 주요 결과 코드
         <table>
         <tr>
         <td><strong>ErrorCode</strong></td>
         <td><strong>Code</strong></td>
         <td><strong>Description</strong></td>
         <td><strong>Solution</strong></td>
         </tr>
         <tr>
         <td>SUCCESS</td>
         <td>Success</td>
         <td>Provider 연결 성공<br>
         <td> - </td>
         </tr>
         <tr>
         <td>NEED_INITIALIZE</td>
         <td>AuthV4NotInitialized</td>
         <td>HIVE SDK가 초기화되지 않은 경우</td>
         <td>AuthV4.setup() API로 HIVE SDK 초기화</td>
         </tr>
         <tr>
         <td>CANCELED</td>
         <td>AuthV4AppleLoginCancel</td>
         <td>Apple GameCenter 로그인창을 표시할 수 없거나 사용자가 GameCenter 로그인을 취소한 경우</td>
         <td>AuthV4.helper.showGameCenterLoginCancelDialog() API로 게임센터 로그인 안내 팝업을 표시하거나,<br>
         직접 구현하여 사용자에게 GameCenter 로그인 방법을 안내
         </td>
         </tr>
         <tr>
         <td>CONFLICT_PLAYER</td>
         <td>AuthV4ConflictPlayer</td>
         <td>Connect를 시도한 Provider의 Player ID가 이미 있는 경우<br>
         <strong>playerInfo</strong>: 충돌 계정 정보
         </td>
         <td>계정 충돌 상황 해결 방법 안내에 따라 해결</td>
         </tr>
         </table>
         *
         * @param Delegate FHiveAuthV4HelperDelegate AuthV4Helper Connect 결과 통지
         *
         * \~english
         * @brief Provider Connection
         * <br>
         * ### Condition of use
         *   -# Complete to initialize HIVE SDK
         *   -# Complete HIVE SignIn
         * <br>
         * <br>
         * ### Key result code
         <table>
         <tr>
         <td><strong>ErrorCode</strong></td>
         <td><strong>Code</strong></td>
         <td><strong>Description</strong></td>
         <td><strong>Solution</strong></td>
         </tr>
         <tr>
         <td>SUCCESS</td>
         <td>Success</td>
         <td>Succeeded to conenct with Provider<br>
         <td> - </td>
         </tr>
         <tr>
         <td>NEED_INITIALIZE</td>
         <td>AuthV4NotInitialized</td>
         <td>HIVE SDK not initialized</td>
         <td>Initialize HIVE SDK by implementing AuthV4.setup() API</td>
         </tr>
         <tr>
         <td>CANCELED</td>
         <td>AuthV4AppleLoginCancel</td>
         <td>If a sign-in window of Game Center is not displayed or the sign-in on Game Center is cancelled,</td>
         <td>use AuthV4.helper.showGameCenterLoginCancelDialog() API or implement directly to inform how to sign in the Game Center
         </td>
         </tr>
         <tr>
         <td>CONFLICT_PLAYER</td>
         <td>AuthV4ConflictPlayer</td>
         <td>the player ID of the Provider to be connected already existed<br>
         <strong>playerInfo</strong>: The information of conflicted account
         </td>
         <td>Follow the resolution in accordance with the type of account conflicts</td>
         </tr>
         </table>
         *
         * @param Delegate FHiveAuthV4HelperDelegate AuthV4Helper Connect result callback
         *  \~
         *
         * @ingroup FHiveAuthV4Helper
         *
         */
        static void Connect(const EHiveProviderType& ProviderType, const FHiveAuthV4HelperDelegate& Delegate);

        /**
         *  \~korean
         * @brief Provider 연결 해제
         * <br>
         * ### 사용 조건
         *   -# HIVE SDK 초기화
         *   -# HIVE SignIn 완료
         * <br>
         * <br>
         * ### 주요 결과 코드
         <table>
         <tr>
         <td><strong>ErrorCode</strong></td>
         <td><strong>Code</strong></td>
         <td><strong>Description</strong></td>
         <td><strong>Solution</strong></td>
         </tr>
         <tr>
         <td>SUCCESS</td>
         <td>Success</td>
         <td>Provider 연결 해제 성공<br>
         <td> - </td>
         </tr>
         <tr>
         <td>NEED_INITIALIZE</td>
         <td>AuthV4NotInitialized</td>
         <td>HIVE SDK가 초기화되지 않은 경우</td>
         <td>AuthV4.setup() API로 HIVE SDK 초기화</td>
         </tr>
         <tr>
         <td>INVALID_PARAM</td>
         <td>AuthV4ProviderAlreadtDisconnected</td>
         <td>해제 요청받은 Provider가 해당 계정에 연결되지 않은 경우</td>
         <td> - </td>
         </tr>
         </table>
         *
         * @param Delegate FHiveAuthV4HelperDelegate AuthV4Helper Disconnect 결과 통지
         *
         * \~english
         * @brief Provider disconnetion
         * <br>
         * ### Condition of use
         *   -# Complete to initialize HIVE SDK
         *   -# Complete HIVE SignIn
         * <br>
         * <br>
         * ### Key result code
         <table>
         <tr>
         <td><strong>ErrorCode</strong></td>
         <td><strong>Code</strong></td>
         <td><strong>Description</strong></td>
         <td><strong>Solution</strong></td>
         </tr>
         <tr>
         <td>SUCCESS</td>
         <td>Success</td>
         <td>Succeeded to discconect with Provider<br>
         <td> - </td>
         </tr>
         <tr>
         <td>NEED_INITIALIZE</td>
         <td>AuthV4NotInitialized</td>
         <td>HIVE SDK not initialized</td>
         <td>Initialize HIVE SDK by implementing AuthV4.setup() API</td>
         </tr>
         <tr>
         <td>INVALID_PARAM</td>
         <td>AuthV4ProviderAlreadtDisconnected</td>
         <td>The Provider requested to disconnect was not connected with the account</td>
         <td> - </td>
         </tr>
         </table>
         *
         * @param Delegate FHiveAuthV4HelperDelegate AuthV4Helper Disconnect result callback
         *  \~
         *
         * @ingroup FHiveAuthV4Helper
         */
        static void Disconnect(const EHiveProviderType& ProviderType, const FHiveAuthV4HelperDelegate& Delegate);

        /**
         *  \~korean
         * @brief 업적 조회
         * <br>
         * ### 사용 조건
         *   -# HIVE SDK 초기화
         *   -# HIVE SignIn 완료
         * <br>
         * <br>
         * ### 특이 사항
         *   -# 로그인 되어있는 계정이 PGS/GameCenter에 연결되어있지 않은 경우 자동으로 해당 서비스로 연결을 시도
         * <br>
         * <br>
         * ### 주요 결과 코드
         <table>
         <tr>
         <td><strong>ErrorCode</strong></td>
         <td><strong>Code</strong></td>
         <td><strong>Description</strong></td>
         <td><strong>Solution</strong></td>
         </tr>
         <tr>
         <td>SUCCESS</td>
         <td>Success</td>
         <td>업적 조회 요청 성공<br>
         <td> - </td>
         </tr>
         <tr>
         <td>NEED_INITIALIZE</td>
         <td>AuthV4NotInitialized</td>
         <td>HIVE SDK가 초기화되지 않은 경우</td>
         <td>AuthV4.setup() API로 HIVE SDK 초기화</td>
         </tr>
         <tr>
         <td>CANCELED</td>
         <td>AuthV4AppleLoginCancel</td>
         <td>Apple GameCenter 로그인창을 표시할 수 없거나 사용자가 GameCenter 로그인을 취소한 경우</td>
         <td>AuthV4.helper.showGameCenterLoginCancelDialog() API로 게임센터 로그인 안내 팝업을 표시하거나,<br>
         직접 구현하여 사용자에게 GameCenter 로그인 방법을 안내
         </td>
         </tr>
         <tr>
         <td>CONFLICT_PLAYER</td>
         <td>AuthV4ConflictPlayer</td>
         <td>Device에 로그인 된 계정과 현재 로그인 된 계정의 PGS/GameCenter 정보가 다르거나<br>
         Connect를 시도한 Provider의 Player ID가 이미 있는 경우
         </td>
         <td>계정 충돌 상황 해결 방법 안내에 따라 해결</td>
         </tr>
         </table>
         *
         * @param Delegate FHiveAuthV4HelperDelegate AuthV4Helper Show achievements 결과 통지
         *
         *  \~english
         * @brief Queries Achievements
         * <br>
         * ### Condition of use
         *   -# Complete to initialize HIVE SDK
         *   -# Complete HIVE SignIn
         * <br>
         * <br>
         * ### Reference
         *   -# It tries to execute the service automatically if the signed-in account is diconnected with PGS/GameCenter.
         * <br>
         * <br>
         * ### Key result code
         <table>
         <tr>
         <td><strong>ErrorCode</strong></td>
         <td><strong>Code</strong></td>
         <td><strong>Description</strong></td>
         <td><strong>Solution</strong></td>
         </tr>
         <tr>
         <td>SUCCESS</td>
         <td>Success</td>
         <td>Succeeded to request Acheivement<br>
         <td> - </td>
         </tr>
         <tr>
         <td>NEED_INITIALIZE</td>
         <td>AuthV4NotInitialized</td>
         <td>HIVE SDK not initialized</td>
         <td>Initialize HIVE SDK by implementing AuthV4.setup() API</td>
         </tr>
         <tr>
         <td>CANCELED</td>
         <td>AuthV4AppleLoginCancel</td>
         <td>If a sign-in window of Game Center is not displayed or the sign-in on Game Center is cancelled,</td>
         <td>use AuthV4.helper.showGameCenterLoginCancelDialog() API or implement directly to inform how to sign in the Game Center
         </td>
         </tr>
         <tr>
         <td>CONFLICT_PLAYER</td>
         <td>AuthV4ConflictPlayer</td>
         <td>The account signed in on user device is mismatched with the PGS/Game Center account on the game<br>
         or the player ID of the Provider to be connected already existed</td>
         </td>
         <td>Follow the resolution in accordance with the type of account conflicts</td>
         </tr>
         </table>
         *
         * @param Delegate FHiveAuthV4HelperDelegate AuthV4Helper Show achievements result callback
         *  \~
         *
         * @ingroup FHiveAuthV4Helper
         */
        static void ShowAchievements(const FHiveAuthV4HelperDelegate& Delegate);

        /**
         *  \~korean
         * @brief 리더보드 조회
         * <br>
         * ### 사용 조건
         *   -# HIVE SDK 초기화
         *   -# HIVE SignIn 완료
         * <br>
         * <br>
         * ### 특이 사항
         *   -# 로그인 되어있는 계정이 PGS/GameCenter에 연결되어있지 않은 경우 자동으로 해당 서비스로 연결을 시도
         * <br>
         * <br>
         * ### 주요 결과 코드
         <table>
         <tr>
         <td><strong>ErrorCode</strong></td>
         <td><strong>Code</strong></td>
         <td><strong>Description</strong></td>
         <td><strong>Solution</strong></td>
         </tr>
         <tr>
         <td>SUCCESS</td>
         <td>Success</td>
         <td>리더보드 조회 요청 성공<br>
         <td> - </td>
         </tr>
         <tr>
         <td>NEED_INITIALIZE</td>
         <td>AuthV4NotInitialized</td>
         <td>HIVE SDK가 초기화되지 않은 경우</td>
         <td>AuthV4.setup() API로 HIVE SDK 초기화</td>
         </tr>
         <tr>
         <td>CANCELED</td>
         <td>AuthV4AppleLoginCancel</td>
         <td>Apple GameCenter 로그인창을 표시할 수 없거나 사용자가 GameCenter 로그인을 취소한 경우</td>
         <td>AuthV4.helper.showGameCenterLoginCancelDialog() API로 게임센터 로그인 안내 팝업을 표시하거나,<br>
         직접 구현하여 사용자에게 GameCenter 로그인 방법을 안내
         </td>
         </tr>
         <tr>
         <td>CONFLICT_PLAYER</td>
         <td>AuthV4ConflictPlayer</td>
         <td>Device에 로그인 된 계정과 현재 로그인 된 계정의 PGS/GameCenter 정보가 다르거나<br>
         Connect를 시도한 Provider의 Player ID가 이미 있는 경우
         </td>
         <td>계정 충돌 상황 해결 방법 안내에 따라 해결</td>
         </tr>
         </table>
         *
         * @param Delegate FHiveAuthV4HelperDelegate AuthV4Helper Show leaderboard 결과 통지
         *
         * \~english
         * @brief Queries leaderboard
         * <br>
         * ### Condition of use
         *   -# Complete to initialize HIVE SDK
         *   -# Complete HIVE SignIn
         * <br>
         * <br>
         * ### Refernce
         *   -# It tries to execute the service automatically if the signed-in account is diconnected with PGS/GameCenter.
         * <br>
         * <br>
         * ### Key result code
         <table>
         <tr>
         <td><strong>ErrorCode</strong></td>
         <td><strong>Code</strong></td>
         <td><strong>Description</strong></td>
         <td><strong>Solution</strong></td>
         </tr>
         <tr>
         <td>SUCCESS</td>
         <td>Success</td>
         <td>Succeeded to request leaderboards<br>
         <td> - </td>
         </tr>
         <tr>
         <td>NEED_INITIALIZE</td>
         <td>AuthV4NotInitialized</td>
         <td>HIVE SDK not initialized</td>
         <td>Initialize HIVE SDK by implementing AuthV4.setup() API</td>
         </tr>
         <tr>
         <td>CANCELED</td>
         <td>AuthV4AppleLoginCancel</td>
         <td>If a sign-in window of Game Center is not displayed or the sign-in on Game Center is cancelled,</td>
         <td>use AuthV4.helper.showGameCenterLoginCancelDialog() API or implement directly to inform how to sign in the Game Center
         </td>
         </tr>
         <tr>
         <td>CONFLICT_PLAYER</td>
         <td>AuthV4ConflictPlayer</td>
         <td>The account signed in on user device is mismatched with the PGS/Game Center account on the game<br>
         or the player ID of the Provider to be connected already existed</td>
         <td>Follow the resolution in accordance with the type of account conflicts</td>
         </tr>
         </table>
         *
         * @param Delegate FHiveAuthV4HelperDelegate AuthV4Helper Show leaderboard result callback
         *  \~
         *
         * @ingroup FHiveAuthV4Helper
         */
        static void ShowLeaderboard(const FHiveAuthV4HelperDelegate& Delegate);

        /**
         *
         *  \~korean
         * @brief 숨겨진 업적 공개.
         * @details 업적이 0% 로 공개만 될 뿐 달성 되지는 않는다.
         * <br>
         * ### 사용 조건
         *   -# HIVE SDK 초기화
         *   -# HIVE SignIn 완료
         * <br>
         * <br>
         * ### 특이 사항
         *   -# 로그인 되어있는 계정이 PGS/GameCenter에 연결되어있지 않은 경우 자동으로 해당 서비스로 연결을 시도
         * <br>
         * <br>
         * ### 주요 결과 코드
        <table>
            <tr>
                <td><strong>ErrorCode</strong></td>
                <td><strong>Code</strong></td>
                <td><strong>Description</strong></td>
                <td><strong>Solution</strong></td>
            </tr>
            <tr>
                <td>SUCCESS</td>
                <td>Success</td>
                <td>숨겨진 업적 공개 요청 성공<br>
                <td> - </td>
            </tr>
            <tr>
                <td>RESPONSE_FAIL</td>
                <td>AuthV4GoogleResponseFailAchievementsReveal</td>
                <td>PGS API 함수 호출이 실패한 경우</td>
                <td> - </td>
            </tr>
        </table>
         *
         * @param achievementId 업적 키 값
         * @param Delegate FHiveAuthV4HelperDelegate achievementsReveal 결과 통지
         *
         *  \~english
         * @brief Reveal hidden achievement.
         * @details Achievements are only revealed at 0%, not achieved.
         * <br>
         * ### Condition of use
         *   -# Complete to initialize HIVE SDK
         *   -# Complete HIVE SignIn
         * <br>
         * <br>
         * ### Reference
         *   -# It tries to execute the service automatically if the signed-in account is diconnected with PGS/GameCenter.
         * <br>
         * <br>
         * ### Key result code
        <table>
            <tr>
                <td><strong>ErrorCode</strong></td>
                <td><strong>Code</strong></td>
                <td><strong>Description</strong></td>
                <td><strong>Solution</strong></td>
            </tr>
            <tr>
                <td>SUCCESS</td>
                <td>Success</td>
                <td>Succeeded to request achievementsReveal<br>
                <td> - </td>
            </tr>
            <tr>
                <td>RESPONSE_FAIL</td>
                <td>AuthV4GoogleResponseFailAchievementsReveal</td>
                <td>Failed to request achievementsReveal</td>
                <td> - </td>
            </tr>
        </table>
         *
         * @param achievementId achievements's key value
         * @param Delegate FHiveAuthV4HelperDelegate achievementsReveal result callback
         *
         * @ingroup FHiveAuthV4Helper
         */
        static void AchievementsReveal(const FString& achievementId, const FHiveAuthV4HelperDelegate& Delegate);

        /**
         *
         *  \~korean
         * @brief 업적 달성.
         * @details 숨겨져 있거나 공개된 여부와 상관없이 업적이 100% 로 달성 된다.
         * <br>
         * ### 사용 조건
         *   -# HIVE SDK 초기화
         *   -# HIVE SignIn 완료
         * <br>
         * <br>
         * ### 특이 사항
         *   -# 로그인 되어있는 계정이 PGS/GameCenter에 연결되어있지 않은 경우 자동으로 해당 서비스로 연결을 시도
         * <br>
         * <br>
         * ### 주요 결과 코드
        <table>
            <tr>
                <td><strong>ErrorCode</strong></td>
                <td><strong>Code</strong></td>
                <td><strong>Description</strong></td>
                <td><strong>Solution</strong></td>
            </tr>
            <tr>
                <td>SUCCESS</td>
                <td>Success</td>
                <td>업적 달성 요청 성공<br>
                <td> - </td>
            </tr>
            <tr>
                <td>RESPONSE_FAIL</td>
                <td>AuthV4GoogleResponseFailAchievementsUnlock</td>
                <td>PGS API 함수 호출이 실패한 경우</td>
                <td> - </td>
            </tr>
        </table>
         *
         * @param achievementId 업적 키 값
         * @param Delegate FHiveAuthV4HelperDelegate achievementsUnlock 결과 통지
         *
         *  \~english
         * @brief Unlock achievement.
         * @details Whether hidden or open, achievement is achieved at 100%.
         * <br>
         * ### Condition of use
         *   -# Complete to initialize HIVE SDK
         *   -# Complete HIVE SignIn
         * <br>
         * <br>
         * ### Reference
         *   -# It tries to execute the service automatically if the signed-in account is diconnected with PGS/GameCenter.
         * <br>
         * <br>
         * ### Key result code
        <table>
            <tr>
                <td><strong>ErrorCode</strong></td>
                <td><strong>Code</strong></td>
                <td><strong>Description</strong></td>
                <td><strong>Solution</strong></td>
            </tr>
            <tr>
                <td>SUCCESS</td>
                <td>Success</td>
                <td>Succeeded to request achievementsUnlock<br>
                <td> - </td>
            </tr>
            <tr>
                <td>RESPONSE_FAIL</td>
                <td>AuthV4GoogleResponseFailAchievementsUnlock</td>
                <td>Failed to request achievementsUnlock</td>
                <td> - </td>
            </tr>
        </table>
         *
         * @param achievementId achievements's key value
         * @param Delegate FHiveAuthV4HelperDelegate achievementsUnlock result callback
         *
         * @ingroup FHiveAuthV4Helper
         */
        static void AchievementsUnlock(const FString& achievementId, const FHiveAuthV4HelperDelegate& Delegate);

        /**
         *
         *  \~korean
         * @brief 업적 수치 증가.<br/>
         * @details 숨겨져 있거나 공개된 여부와 상관없이 업적이 100% 로 달성 된다.<br/>
         * 총 합산이 Max 가 될 경우 자동으로 업적이 달성되며, 계속 호출하여도 무방하다.<br/>
         * <br>
         * ### 사용 조건
         *   -# HIVE SDK 초기화
         *   -# HIVE SignIn 완료
         * <br>
         * <br>
         * ### 특이 사항
         *   -# 로그인 되어있는 계정이 PGS/GameCenter에 연결되어있지 않은 경우 자동으로 해당 서비스로 연결을 시도
         * <br>
         * <br>
         * ### 주요 결과 코드
        <table>
            <tr>
                <td><strong>ErrorCode</strong></td>
                <td><strong>Code</strong></td>
                <td><strong>Description</strong></td>
                <td><strong>Solution</strong></td>
            </tr>
            <tr>
                <td>SUCCESS</td>
                <td>Success</td>
                <td>업적 수치 증가 요청 성공<br>
                <td> - </td>
            </tr>
            <tr>
                <td>RESPONSE_FAIL</td>
                <td>AuthV4GoogleResponseFailAchievementsIncrement</td>
                <td>PGS API 함수 호출이 실패한 경우</td>
                <td> - </td>
            </tr>
        </table>
         *
         * @param AchievementId 업적 키 값
         * @param value 증가 값
         * @param Delegate FHiveAuthV4HelperDelegate achievementsIncrement 결과 통지
         *
         *  \~english
         * @brief Increases achievement figures.<br/>
         * @details Achievement figures is added as much as value set by the API call, not by setting.<br/>
         * If the total sum is Max, the achievement is automatically accomplished.<br/>
         * <br>
         * ### Condition of use
         *   -# Complete to initialize HIVE SDK
         *   -# Complete HIVE SignIn
         * <br>
         * <br>
         * ### Reference
         *   -# It tries to execute the service automatically if the signed-in account is diconnected with PGS/GameCenter.
         * <br>
         * <br>
         * ### Key result code
        <table>
            <tr>
                <td><strong>ErrorCode</strong></td>
                <td><strong>Code</strong></td>
                <td><strong>Description</strong></td>
                <td><strong>Solution</strong></td>
            </tr>
            <tr>
                <td>SUCCESS</td>
                <td>Success</td>
                <td>Succeeded to request achievementsIncrement<br>
                <td> - </td>
            </tr>
            <tr>
                <td>RESPONSE_FAIL</td>
                <td>AuthV4GoogleResponseFailAchievementsIncrement</td>
                <td>Failed to request achievementsIncrement</td>
                <td> - </td>
            </tr>
        </table>
         *
         * @param AchievementId achievements's key value
         * @param value value
         * @param Delegate FHiveAuthV4HelperDelegate achievementsIncrement result callback
         *
         * @ingroup FHiveAuthV4Helper
         */
        static void AchievementsIncrement(const FString& AchievementId, int64 value, const FHiveAuthV4HelperDelegate& Delegate);
        
        /**
         *
         *  \~korean
         * @brief 리더보드 점수를 갱신.<br/>
         * @details leaderboardId 에 해당하는 기록에 score 수치로 갱신된다.<br/>
         * <br>
         * ### 사용 조건
         *   -# HIVE SDK 초기화
         *   -# HIVE SignIn 완료
         * <br>
         * <br>
         * ### 특이 사항
         *   -# 로그인 되어있는 계정이 PGS/GameCenter에 연결되어있지 않은 경우 자동으로 해당 서비스로 연결을 시도
         * <br>
         * <br>
         * ### 주요 결과 코드
        <table>
            <tr>
                <td><strong>ErrorCode</strong></td>
                <td><strong>Code</strong></td>
                <td><strong>Description</strong></td>
                <td><strong>Solution</strong></td>
            </tr>
            <tr>
                <td>SUCCESS</td>
                <td>Success</td>
                <td>리더보드 점수 갱신 요청 성공<br>
                <td> - </td>
            </tr>
            <tr>
                <td>RESPONSE_FAIL</td>
                <td>AuthV4GoogleResponseFailLeaderboardsSubmitScore</td>
                <td>PGS API 함수 호출이 실패한 경우</td>
                <td> - </td>
            </tr>
        </table>
         *
         * @param LeaderboardId 리더보드 키 값
         * @param value 리더보드 점수
         * @param Delegate FHiveAuthV4HelperDelegate LeaderboardsSubmitScore 결과 통지
         *
         *  \~english
         * @brief It update the leaderboard score.<br/>
         * @details The score corresponding to the leaderboardId is updated with the score value.<br/>
         * <br>
         * ### Condition of use
         *   -# Complete to initialize HIVE SDK
         *   -# Complete HIVE SignIn
         * <br>
         * <br>
         * ### Reference
         *   -# It tries to execute the service automatically if the signed-in account is diconnected with PGS/GameCenter.
         * <br>
         * <br>
         * ### Key result code
        <table>
            <tr>
                <td><strong>ErrorCode</strong></td>
                <td><strong>Code</strong></td>
                <td><strong>Description</strong></td>
                <td><strong>Solution</strong></td>
            </tr>
            <tr>
                <td>SUCCESS</td>
                <td>Success</td>
                <td>Succeeded to request LeaderboardsSubmitScore<br>
                <td> - </td>
            </tr>
            <tr>
                <td>RESPONSE_FAIL</td>
                <td>AuthV4GoogleResponseFailLeaderboardsSubmitScore</td>
                <td>Failed to request LeaderboardsSubmitScore</td>
                <td> - </td>
            </tr>
        </table>
         *
         * @param LeaderboardId leaderboard's key value
         * @param value Leaderboard score
         * @param Delegate FHiveAuthV4HelperDelegate LeaderboardsSubmitScore result callback
         *
         * @ingroup FHiveAuthV4Helper
         */
        static void LeaderboardsSubmitScore(const FString& LeaderboardId, int64 value, const FHiveAuthV4HelperDelegate& Delegate);
        
        /**
         *  \~korean
         * @brief 충돌 상태를 알려주고 사용할 계정을 선택하는 HIVE UI 를 보여준다. <br>
         * ### 주요 결과 코드
         <table>
         <tr>
         <td><strong>ErrorCode</strong></td>
         <td><strong>Code</strong></td>
         <td><strong>Description</strong></td>
         </tr>
         <tr>
         <td>PLAYER_CHANGE</td>
         <td>AuthV4PlayerChange</td>
         <td>계정 충돌이 발생한 후 계정 전환에 성공한 경우<br>
         </tr>
         <tr>
         <td>CANCELED</td>
         <td>AuthV4PlayerResolved</td>
         <td>계정 충돌이 발생한 후 현재 로그인 된 계정을 유지하는 경우</td>
         </tr>
         <tr>
         <td>INVALID_PARAM</td>
         <td>AuthV4InvalidConflictInfo</td>
         <td>계정 충돌이 발생하지 않은 상태에서 계정 충돌 해결을 요청한 경우</td>
         </tr>
         </table>
         *
         * @param Delegate FHiveAuthV4HelperDelegate AuthV4Helper Show Conflict 결과 통지
         *
         *  \~english
         *  @brief Displays conflict status, and HIVE UI avilable to select an account.
         *  ### Key result code
         <table>
         <tr>
         <td><strong>ErrorCode</strong></td>
         <td><strong>Code</strong></td>
         <td><strong>Description</strong></td>
         </tr>
         <tr>
         <td>PLAYER_CHANGE</td>
         <td>AuthV4PlayerChange</td>
         <td>Succeeded to change the player after accounts conflict<br>
         </tr>
         <tr>
         <td>CANCELED</td>
         <td>AuthV4PlayerResolved</td>
         <td>Not changed the currently signed-in account after accounts conflict</td>
         </tr>
         <tr>
         <td>INVALID_PARAM</td>
         <td>AuthV4InvalidConflictInfo</td>
         <td>Requested to resolve account conflicts not happened</td>
         </tr>
         </table>
         *
         * @param Delegate FHiveAuthV4HelperDelegate AuthV4Helper Show Conflict result callback
         *  \~
         *
         * @ingroup FHiveAuthV4Helper
         */
        static void ShowConflict(const FHiveAuthV4HelperDelegate& Delegate);

        /**
         *  \~korean
         * @brief 충돌 상태를 알려주고 사용할 계정을 선택할 수 있고, 게임 정보를 같이 표현할 수 있는 HIVE UI 를 보여준다. <br>
         * ### 주요 결과 코드
         <table>
         <tr>
         <td><strong>ErrorCode</strong></td>
         <td><strong>Code</strong></td>
         <td><strong>Description</strong></td>
         </tr>
         <tr>
         <td>PLAYER_CHANGE</td>
         <td>AuthV4PlayerChange</td>
         <td>계정 충돌이 발생한 후 계정 전환에 성공한 경우<br>
         </tr>
         <tr>
         <td>CANCELED</td>
         <td>AuthV4PlayerResolved</td>
         <td>계정 충돌이 발생한 후 현재 로그인 된 계정을 유지하는 경우</td>
         </tr>
         <tr>
         <td>INVALID_PARAM</td>
         <td>AuthV4InvalidConflictInfo</td>
         <td>계정 충돌이 발생하지 않은 상태에서 계정 충돌 해결을 요청한 경우</td>
         </tr>
         </table>
         *
         * @param ViewData ConflictSingleViewInfo 충돌이 발생한 사용자의 게임 데이터
         * @param Delegate FHiveAuthV4HelperDelegate AuthV4Helper Show Conflict 결과 통지
         *
         *  \~english
         * @brief Displays conflict status, and HIVE UI which is available to select an account as well as shows game information. <br>
         * ### Key result code
         <table>
         <tr>
         <td><strong>ErrorCode</strong></td>
         <td><strong>Code</strong></td>
         <td><strong>Description</strong></td>
         </tr>
         <tr>
         <td>PLAYER_CHANGE</td>
         <td>AuthV4PlayerChange</td>
         <td>Succeeded to change the player after accounts conflict<br>
         </tr>
         <tr>
         <td>CANCELED</td>
         <td>AuthV4PlayerResolved</td>
         <td>Not changed the currently signed-in account after accounts conflict</td>
         </tr>
         <tr>
         <td>INVALID_PARAM</td>
         <td>AuthV4InvalidConflictInfo</td>
         <td>Requested to resolve account conflicts not happened</td>
         </tr>
         </table>
         *
         * @param ViewData ConflictSingleViewInfo Game information of the account-conflicted player
         * @param Delegate FHiveAuthV4HelperDelegate AuthV4Helper Show Conflict result callback
         * 
         *  \~
         * @ingroup FHiveAuthV4Helper
         * 
         */
        static void ShowConflict(const FHiveConflictSingleViewData& ViewData, const FHiveAuthV4HelperDelegate& Delegate);
        
        /**
         *  \~korean
         * @brief 계정 Conflict가 발생한 경우 현재 로그인 된 사용자를 로그아웃 시키고, <br>
         * 기기에 연결된 계정으로 로그인을 시도한다.<br>
         * ### 주요 결과 코드
         <table>
         <tr>
         <td><strong>ErrorCode</strong></td>
         <td><strong>Code</strong></td>
         <td><strong>Description</strong></td>
         </tr>
         <tr>
         <td>PLAYER_CHANGE</td>
         <td>AuthV4PlayerChange</td>
         <td>계정 충돌이 발생한 후 계정 전환에 성공한 경우<br>
         </tr>
         </table>
         *
         * @param Delegate FHiveAuthV4HelperDelegate AuthV4Helper Switch Account 결과 통지
         *
         *  \~english
         * @brief If accounts conflict, sign out the current player<br>
         * and try to sign in with the account signed in on the user device.<br>
         * ### Key result code
         <table>
         <tr>
         <td><strong>ErrorCode</strong></td>
         <td><strong>Code</strong></td>
         <td><strong>Description</strong></td>
         </tr>
         <tr>
         <td>PLAYER_CHANGE</td>
         <td>AuthV4PlayerChange</td>
         <td>Succeeded to change the player after accounts conflict<br>
         </tr>
         </table>
         *
         * @param Delegate FHiveAuthV4HelperDelegate AuthV4Helper Switch Account result callback
         *  \~
         *
         * @ingroup FHiveAuthV4Helper
         */
        static void SwitchAccount(const FHiveAuthV4HelperDelegate& Delegate);

        /**
         *  \~korean
         * @brief 계정 Conflict가 발생한 경우 기존 사용자를 유지하는 경우 사용한다. <br>
         * ### 주요 결과 코드
         <table>
         <tr>
         <td><strong>ErrorCode</strong></td>
         <td><strong>Code</strong></td>
         <td><strong>Description</strong></td>
         </tr>
         <tr>
         <td>CANCELED</td>
         <td>AuthV4PlayerResolved</td>
         <td>계정 충돌이 발생한 후 현재 로그인 된 계정을 유지하는 경우</td>
         </tr>
         </table>
         *
         *
         * @param Delegate FHiveAuthV4HelperDelegate AuthV4Helper Resolve Conflict결과 통지
         *
         *  \~english
         * @brief In case it is not changed the currently signed-in account after accounts conflict. <br>
         * ### Key result code
         <table>
         <tr>
         <td><strong>ErrorCode</strong></td>
         <td><strong>Code</strong></td>
         <td><strong>Description</strong></td>
         </tr>
         <tr>
         <td>CANCELED</td>
         <td>AuthV4PlayerResolved</td>
         <td>Not changed the currently signed-in account after accounts conflict</td>
         </tr>
         </table>
         *
         *
         * @param Delegate FHiveAuthV4HelperDelegate AuthV4Helper Resolve Conflict result callback
         *  \~
         *
         * @ingroup FHiveAuthV4Helper
         */
        static void ResolveConflict(const FHiveAuthV4HelperDelegate& Delegate);

        /**
         *  \~korean
         * @brief SDK에서 제공하는 Provider의 목록을 받는다. <br>
         * <br>
         *
         *  \~english
         * @brief Receives the Provider lists sent by SDK. <br>
         * <br>
         *  \~
         *
         * @ingroup FHiveAuthV4Helper
         */
        static TArray<EHiveProviderType> GetIDPList();

        /**
         *  \~korean
         * @brief Game Center 로그인창을 표시할 수 없는 경우, 해당 상태를 보여주고,<br>
         * Game Center 로그인 방법을 안내하는 UI를 띄운다.
         *
         * @param Delegate FHiveAuthV4HelperOnDialogDismissDelegate 계정 안내 팝업 Close 여부 통지
         *
         *  \~english
         * @brief If a sign-in window of Game Center is not displayed, show its status<br/>
         * and display a UI to inform how to sign in the Game Center.
         *
         * @param Delegate FHiveAuthV4HelperOnDialogDismissDelegate callback whether account notification popup is closed or not
         *
         *  \~
         * @ingroup FHiveAuthV4Helper
         * 
         */
        static void ShowGameCenterLoginCancelDialog(const FHiveAuthV4HelperOnDialogDismissDelegate& Delegate);
    };
};
