/**
 * @file        HiveConfiguration.h
 *
 * @date		2025
 * @copyright	Copyright © 2025 Com2uS Platform Corp. All Rights Reserved.
 * @defgroup FHiveConfiguration
 * @{
 * \~korean
 * HIVE SDK 설정 관리<br/><br/>
 * \~english
 * This class manage HIVE SDK configuration<br/><br/>
 * \~
 */
#pragma once

#include "CoreMinimal.h"
#include "HiveInterface.h"
#include "Impl/HiveConfigurationImpl.h"

/**
 * \~korean
 * HIVE SDK 설정 관리<br/><br/>
 * \~english
 * This class manage HIVE SDK configuration<br/><br/>
 * \~
 * @ingroup FHiveConfiguration
 */
class HIVESDK_API FHiveConfiguration : public IHiveInterface<IHiveConfigurationImpl>
{
public:
    static FString GetConfiguration();

    /**
    * \~korean
    * @brief Hive SDK 버전 반환
    *
    * @return Hive SDK 버전
    * \~english
    * @brief Returns Hive SDK Version
    *
    * @return Hive SDK version
    * \~
    * @ingroup FHiveConfiguration
    */
    static FString GetHiveSDKVersion();

    /**
    * \~korean
    * @brief Hive SDK 가 참조하고 있는 SDK 의 버전 반환
    *
    * @return Hive SDK 가 참조하고 있는 SDK 의 버전
    * \~english
    * @brief Return version of SDK referenced by Hive SDK
    *
    * @return Version of SDK referenced by Hive SDK
    * \~
    * @ingroup FHiveConfiguration
    */
    static FString GetReferenceSDKVersion();

    /**
    * \~korean AppId 반환<br/>
    * (AppId 는 기본적으로 AndroidManifest.xml 파일의 package 값으로 설정하게 된다.<br/>
    * 그러나 테스트 설정등의 이유로 API 호출에 대한 변경을 지원한다.)
    *
    * @return AppId
    * \~english Returns AppId
    * (By default, AppId will be set to the package name in the AndroidManifest.xml file. However, it supports changes to API calls for reasons such as test setup.)
    *
    * @return AppId
    * \~
    * @ingroup FHiveConfiguration
    */
    static FString GetAppId();

    /**
    * \~korean AppId 설정<br/>
    * (AppId 는 기본적으로 AndroidManifest.xml 파일의 package 값으로 설정하게 된다.<br/>
    * 그러나 테스트 설정등의 이유로 API 호출에 대한 변경을 지원한다.)
    *
    * @param AppId AppId
    * \~english Set AppId
    * (By default, AppId will be set to the package name in the AndroidManifest.xml file. However, it supports changes to API calls for reasons such as test setup.)
    *
    * @param AppId AppId
    * \~
    * @ingroup FHiveConfiguration
    */
    static void SetAppId(const FString& AppId);
    static FString GetHiveCertificationKey();
    static void SetHiveCertificationKey(const FString& Key);

    /**
    * \~korean Hive 플랫폼 서버 존 반환
    *
    * @return Hive SDK 플랫폼 서버 존 (sandbox : 개발용, real : 실계용)
    * \~english Returns Hive platform server zone
    *
    * @return Hive platform server zone (sandbox : for development, real : for production)
    * \~
    * @ingroup FHiveConfiguration
    */
    static EHiveZoneType GetZone();

    /**
    * \~korean Hive 플랫폼 서버 존 설정
    *
    * @param Type Hive SDK 플랫폼 서버 존 (sandbox : 개발용, real : 실계용)
    * \~english Set Hive Hive platform server zone
    *
    * @param Type Hive platform server zone (sandbox : for development, real : for production)
    * \~
    * @ingroup FHiveConfiguration
    */
    static void SetZone(const EHiveZoneType& Type);

    /**
    * \~korean 서버별 점검공지 팝업지원을 위한 serverId 반환<br/>
    * (백오피스 월드관리에 등록된 월드값을 서버에 따라 구분하여 입력이 되어야 한다.)
    *
    * @return 서버별 점검공지 팝업지원을 위한 serverId
    * \~english Return serverId for server-specific maintenance popup support<br>
    * (The world value registered in the back office world management)
    *
    * @return serverId Server ID for server-specific maintenance popup support
    * \~
    * @ingroup FHiveConfiguration
    */
    static FString GetServerId();

    /**
    * \~korean 서버별 점검공지 팝업지원을 위한 serverId 설정<br/>
    * (백오피스 월드관리에 등록된 월드값을 서버에 따라 구분하여 입력이 되어야 한다.)
    *
    * @param ServerId 서버별 점검공지 팝업지원을 위한 serverId
    * \~english Set serverId for server-specific maintenance popup support<br>
    * (The world value registered in the back office world management should be inputted according to the server.)
    *
    * @param ServerId for server-specific maintenance popup support
    * \~
    * @ingroup FHiveConfiguration
    */
    static void SetServerId(const FString& ServerId);

    /**
    * \~korean 서버별 점검공지 팝업지원을 위한 serverId 설정<br/>
    * (백오피스 월드관리에 등록된 월드값을 서버에 따라 구분하여 입력이 되어야 한다.)
    *
    * @param ServerId 서버별 점검공지 팝업지원을 위한 serverId
    * \~english Set serverId for server-specific maintenance popup support<br>
    * (The world value registered in the back office world management should be inputted according to the server.)
    *
    * @param ServerId for server-specific maintenance popup support
    * \~
    * @ingroup FHiveConfiguration
    */
    static void UpdateServerId(const FString& ServerId);

    /**
     * \~korean 현재 설정된 언어코드를 확인하기 위한 API<br/>
     * ISO 639-1 형식의 2자리 소문자를 반환한다.<br>
     * 기본값으로 기기언어가 설정되며 Hive SDK에서 지원하지 않는 언어일 경우 "en" 으로 설정 된다.
     *
     * \~english Gets the current language code setting.<br/>
     * Returns the language code as a 2-letter lowercase string (ISO 639-1 format).<br>
     * By default, the device language is set. If the language is not supported by the Hive SDK, it defaults to "en".
     * \~
     * @ingroup FHiveConfiguration
     */
    static FString GetGameLanguage();

    /**
    * \~korean 게임에서 사용하는 언어코드를 모듈에도 반영하기 위한 API<br/>
    * 2자리 소문자로 ("en") 입력해주면 된다. ISO 639-1 형식.
    * <p>
    * 내부적으로 대문자도 소문자로 변환된다. 알파벳 대소문자 규칙은 US 규칙을 따른다.
    *
    * @param Language 게임에서 사용하는 언어코드
    * \~english API to reflect language code used in game in module<br>
    * Just type in two lowercase letters ("en"). ISO 639-1 format.<br>
    * <p>
    * Internally, uppercase characters are converted to lowercase characters. Alphabetic case rules follow US rules.
    *
    * @param Language Language code used in game.
    * \~
    * @ingroup FHiveConfiguration
    */
    static void SetGameLanguage(const FString& Language);

    /**
    * \~korean 게임에서 사용하는 언어코드를 모듈에도 반영하기 위한 API<br/>
    * 2자리 소문자로 ("en") 입력해주면 된다. ISO 639-1 형식.
    * <p>
    * 내부적으로 대문자도 소문자로 변환된다. 알파벳 대소문자 규칙은 US 규칙을 따른다.
    *
    * @param Language 게임에서 사용하는 언어코드
    * \~english API to reflect language code used in game in module<br>
    * Just type in two lowercase letters ("en"). ISO 639-1 format.<br>
    * <p>
    * Internally, uppercase characters are converted to lowercase characters. Alphabetic case rules follow US rules.
    *
    * @param Language Language code used in game.
    * \~
    * @ingroup FHiveConfiguration
    */
    static void UpdateGameLanguage(const FString& Language);

    /**
    * \~korean Hive SDK 내부 로그 사용 여부 반환
    *
    * @return Hive SDK 내부 로그 사용 여부
    * \~english Returns whether HIVE SDK internal log is used
    *
    * @return Whether HIVE SDK internal log is used
    * \~
    * @ingroup FHiveConfiguration
    */
    static bool GetUseLog();

    /**
    * \~korean Hive SDK 내부 로그 사용 여부 설정
    *
    * @param bUseLog Hive SDK 내부 로그 사용 여부
    * \~english Set whether HIVE SDK internal log is used
    *
    * @param bUseLog Whether HIVE SDK internal log is used
    * \~
    * @ingroup FHiveConfiguration
    */
    static void SetUseLog(bool bUseLog);

    /**
    * \~korean 회사 반환
    *
    * @return 회사 (COM2US : 컴투스, GAMEVIL : 게임빌)
    * \~english Returns company.
    *
    * @return Company (COM2US : Com2us, GAMEVIL : Gamevil)
    * \~
    * @ingroup FHiveConfiguration
    */
    static FString GetCompany();

    /**
    * \~korean 회사 설정
    *
    * @param Company 회사 (COM2US : 컴투스, GVI : 게임빌)
    * \~english Set company
    *
    * @param Company Company(COM2US : Com2us, GAMEVIL : Gamevil)
    * \~
    * @ingroup FHiveConfiguration
    */
    static void SetCompany(const FString& Company);

    /**
    * \~korean 채널 반환
    *
    * @return Channel 채널 (HIVE : HIVE 플랫폼)
    * \~english Returns channel
    *
    * @return Channel Channel (HIVE : HIVE Platform)
    * \~
    * @ingroup FHiveConfiguration
    */
    static FString GetChannel();

    /**
    * \~korean 채널 설정
    *
    * @param Channel 채널 (HIVE : HIVE 플랫폼)
    * \~english Set channel
    *
    * @param Channel Channel (HIVE : HIVE Platform)
    * \~
    * @ingroup FHiveConfiguration
    */
    static void SetChannel(const FString& Channel);

    /**
    * \~korean HTTP Connect Timeout 의 기본 설정 값 반환 (초단위)
    *
    * @return HTTP Timeout 의 기본 설정 값 (초단위)
    * \~english Returns the default value of HTTP Connect Timeout (in seconds)
    *
    * @return Default value of HTTP Connect Timeout (in seconds)
    * \~
    * @ingroup FHiveConfiguration
    */
    static int32 GetHttpConnectTimeout();

    /**
    * \~korean HTTP Connect Timeout 의 기본 설정 값 설정 (초단위)
    *
    * @param second httpConnectTimeout HTTP Connect Timeout 의 기본 설정 값 (초단위)
    * \~english Set the value of HTTP Connect Timeout (in seconds)
    *
    * @param second Value of HTTP Connect Timeout (in seconds)
    * \~
    * @ingroup FHiveConfiguration
    */
    static void SetHttpConnectTimeout(int32 second);

    /**
    * \~korean HTTP Read Timeout 의 기본 설정 값 반환 (초단위)
    *
    * @return HTTP Timeout 의 기본 설정 값 (초단위)
    * \~english Returns the default value of HTTP Read Timeout (in seconds)
    *
    * @return Default value of HTTP Read Timeout (in seconds)
    * \~
    * @ingroup FHiveConfiguration
    */
    static int32 GetHttpReadTimeout();

    /**
    * \~korean HTTP Read Timeout 의 기본 설정 값 설정 (초단위)
    *
    * @param second httpReadTimeout HTTP Read Timeout 의 기본 설정 값 (초단위)
    * \~english Set the value of HTTP Read Timeout (in seconds)
    *
    * @param second Value of HTTP Read Timeout (in seconds)
    * \~
    * @ingroup FHiveConfiguration
    */
    static void SetHttpReadTimeout(int32 second);

    /**
    * \~korean 게임 로그 최대 저장 갯수 반환
    *
    * @return 게임 로그 최대 저장 갯수
    * \~english Returns maximum number of game logs
    *
    * @return Maximum number of game logs
    * \~
    * @ingroup FHiveConfiguration
    */
    static int32 GetMaxGameLogSize();

    /**
    * \~korean 게임 로그 최대 저장 갯수 설정<br/>
    * (특별한 경우가 아니면 변경 금지)
    *
    * @param size 게임 로그 최대 저장 갯수
    * \~english Set maximum number of game logs
    * (Note: No change unless special occasion)
    *
    * @param size Maximum number of game logs
    * \~
    * @ingroup FHiveConfiguration
    */
    static void SetMaxGameLogSize(int32 size);

    /**
    * \~korean 결제 마켓 반환
    *
    * @return 결제 마켓 (google, tstore, olleh, ozstore, googleplay_lebi)
    * \~english Return market
    *
    * @return Market (google, tstore, olleh, ozstore, googleplay_lebi)
    * \~
    * @ingroup FHiveConfiguration
    */
    static FString GetMarket();

    /**
    * \~korean 결제 마켓 설정
    *
    * @param Market 결제 마켓 (google, tstore, olleh, ozstore, googleplay_lebi)
    * \~english Set market
    *
    * @param Market Market (google, tstore, olleh, ozstore, googleplay_lebi)
    * \~
    * @ingroup FHiveConfiguration
    */
    static void SetMarket(const FString& Market);

    /**
    *  \~korean HIVE 서버에서 판단한 국가코드를 반환한다.
    *
    * @return ISO
    *  \~english HIVE 서버에서 판단한 국가코드를 반환한다.
    *
    * @return ISO
    * \~
    * @ingroup FHiveConfiguration
    */
    static FString GetHiveCountry();

    /**
     *  \~korean HIVE 서버에서 판단한 TimeZone 정보를 반환한다.
     *
     * @return JSON String
     *  \~english HIVE 서버에서 판단한 TimeZone 정보를 반환한다.
     *
     * @return JSON String
     * \~
     * @ingroup FHiveConfiguration
     */
    static FString GetHiveTimeZone();

    /**
     * 전송 주기마다 전송할 로그의 최대치.
     *
     * @return unsigned int 전송 주기마다 전송할 로그의 최대치.
     *
     * @ingroup FHiveConfiguration
     */
    static uint32 GetAnalyticsSendLimit();

    /**
     * 전송 주기마다 전송할 로그의 최대치 설정.
     *
     * @param Limit 전송주기마다 전송할 최대 로그의 양
     *
     * @ingroup FHiveConfiguration
     */
    static void SetAnalyticsSendLimit(uint32 Limit);

    /**
     * 최대로 쌓을수 있는 로그의 양.
     *
     * @return uint32 최대로 쌓을수 있는 로그의 양.
     *
     * @ingroup FHiveConfiguration
     */
    static uint32 GetAnalyticsQueueLimit();

    /**
     * 최대로 쌓을 수 있는 로그의 수
     *
     * @param Limit 최대 대기 가능한 로그의 수
     *
     * @ingroup FHiveConfiguration
     */
    static void SetAnalyticsQueueLimit(uint32 Limit);

    /**
     * 로그 전송 주기.
     *
     * @return float 전송주기
     *
     * @ingroup FHiveConfiguration
     */
    static float GetAnalyticsSendCycleSeconds();

    /**
     * 로그 전송 주기 설정.
     *
     * @param seconds 전송 주기 (초)
     *
     * @ingroup FHiveConfiguration
     */
    static void SetAnalyticsSendCycleSeconds(float seconds);

    /**
     * Hive SDK AgeGateU13 적용 여부 반환
     *
     * @return Hive SDK AgeGateU13 적용 여부
     * 
     * @ingroup FHiveConfiguration
     */
    static bool GetAgeGateU13();

    /**
     * Hive SDK AgeGateU13 적용 여부 설정
     *
     * @param bEnable Hive SDK AgeGateU13 적용 여부 설정
     * 
     * @ingroup FHiveConfiguration
     */
    static void SetAgeGateU13(bool bEnable);

    /**
     * Hive 커스텀 테마 값 획득
     *
     * @return Hive 커스텀 테마 값
     *
     * @ingroup FHiveConfiguration
     */
    static EHiveThemeType GetHiveTheme();

    /**
     * Hive 커스텀 테마 설정
     *
     * @param Type EHiveThemeType
     */
    static void SetHiveTheme(EHiveThemeType Type);

    /**
     * Hive Orientation 설정
     *
     * @param Orientation 회전 방향
     *
     * @ingroup FHiveConfiguration
     */
    static void SetHiveOrientation(const FString& Orientation);
    static void SetConfigurations(EHiveConfigType Type, const FString& Value);

    /**
     * Game MetaData 요청
     *
     * @param Key 요청 data 키
     * @param bForceReload network 통신 여부
     * @param Delegate API 결과 통지
     *
     * @ingroup FHiveConfiguration
     */
    static void GetMetaData(const FString& Key, bool bForceReload, const FHiveConfigurationOnMetaDataDelegate& Delegate);
    
    /**
     * Hercules 사용 여부 설정
     *
     * @param bEnable Hercules 사용 여부 설정
     */
    static void SetUseHercules(bool bEnable);

    /**
     * Hive 크로스프로모션 노출 여부 반환
     *
     * @return Hive 크로스프로모션 노출 여부
     */
    static bool GetCrossPromotionBannerShowing();

    /**
     * Hive 크로스프로모션 노출 여부 설정
     *
     * @param bExpose  Hive 크로스프로모션 노출 여부 설정
     */
    static void SetCrossPromotionBannerShowing(bool showing);

    /**
     * Hive SDK GDPR 국가에서 16세 미만 약관 동의 여부 반환
     *
     * @return Hive SDK GDPR 국가에서 16세 미만 약관 동의 여부
     */
    static bool GetAgeGateU16Agree();

    /**
     * Hive SDK 법정대리인 동의 여부
     *
     * @return Hive SDK 법정대리인 동의 여부
     */
    static bool GetLegalGuardianConsentMode();

    /**
     * 시크릿 모드 설정
     *
     * @param use 시크릿 모드 사용 여부. true로 설정하면 시크릿 모드 사용, false로 설정하면 일반 모드로 작동
     *            별도 설정이 없으면 기본값으로 시크릿 모드로 작동
     *
     */
    static void SetUsePrivateBrowsingForAuth(bool use);

    /**
     * 컨트롤러 사용 여부 반환
     *
     * @return enable 컨트롤러를 활성화 상태 true, 비활성화 상태 false 리턴
     */
    static bool GetEnableGameController();

    /**
     * 컨트롤러 사용 여부 설정
     *
     * @param enable 컨트롤러를 활성화하려면 true, 비활성화하려면 false를 입력
     */
    static void SetEnableGameController(bool enable);

    /**
     * HiveChat 타임아웃 설정 시간 설정
     *
     * @param time HiveChat 타임아웃 설정 시간
     */
    static void  SetChatConnectionTimeout(int32 Seconds);

    /**
     * HiveChat 타임아웃 설정 시간 반환
     *
     * @return HiveChat 타임아웃 설정 시간
     */
    static int32 GetChatConnectionTimeout();
};
