/**
 * @file        HivePromotion.h
 *
 * @date        2025
 * @copyright   Copyright © 2025 Com2uS Platform Corp. All Rights Reserved.
 * @defgroup FHivePromotion
 * @{
 * \~korean  HIVE 프로모션은 게임을 이용하는 유저에게 게임의 새로운 소식이나 이벤트를 효과적으로 노출하는 기능을 제공한다<br/>
 * 프로모션 뷰 (공지, 이벤트), 보상 (쿠폰, 딥링크), 종료 팝업 (안드로이드), 무료 충전소, 리뷰 유도 팝업<br/><br/>
 * 
 * \~english HIVE Promotion gives users who use the game the ability to effectively expose new news or events in the game.<br/>
 * Promotion view (notice, event), reward (coupon, deep link), exit popup (Android), free charging station(Offerwall)<br/><br/>
 */
#pragma once

#include "CoreMinimal.h"
#include "HiveInterface.h"
#include "Impl/HivePromotionImpl.h"

/**
 * \~korean  HIVE 프로모션은 게임을 이용하는 유저에게 게임의 새로운 소식이나 이벤트를 효과적으로 노출하는 기능을 제공한다<br/>
 * 프로모션 뷰 (공지, 이벤트), 보상 (쿠폰, 딥링크), 종료 팝업 (안드로이드), 무료 충전소, 리뷰 유도 팝업<br/><br/>
 *
 * \~english HIVE Promotion provides you with the ability to effectively expose new news or events to the game user.<br/>
 * Promotion view (notice, event), reward (coupon, deep link), exit popup (Android), free charging station(Offerwall)<br/><br/>
 *
 * \~
 * @ingroup FHivePromotion
 */
class HIVESDK_API FHivePromotion : public IHiveInterface<IHivePromotionImpl>
{
public:
    /**
     * \~korean  게임의 새로운 이벤트나 새로운 게임 소개등의 배너 화면을 노출
     *
     * @param PromotionType		프로모션 뷰 창의 형태
     * @param IsForced			type이 "notice" 와 "event" 일 때만 동작
     *
     * 							true일 경우 24시간 다시보지 않기를 무시한다.
     * 							(주의:24시간 다시보기가 적용되지 않으므로 버튼 액션으로 보는 등 특정 액션에서만 사용해야 함.)
     *
     * 							false이거나 생략할 경우 24시간 안보기가 정상동작함.
     * @param Delegate			API 결과 통지
     * \~english Shows banner such as new event of game or introduce new game
     *
     * @param PromotionType		Promotion View type
     * @param IsForced			If true, it only works when promotionType is "notice" and "event"
     *
     * 							and it ignore 'Not seeing it again for 24 hours'.
     * 							(Note: Since 'Not seeing it again for 24 hours' does not apply, you should only use certain actions, such as viewing as a button action.)
     *
     * 							If false or omitted, 'Not seeing it again for 24 hours' will operate normally.
     * @param Delegate			API call result listener
     * \~
     * @ingroup FHivePromotion
     */
    static void ShowPromotion(const EHivePromotionType& PromotionType, bool IsForced, const FHivePromotionViewDelegate& Delegate);
    
    /**
     * \~korean 외부 컨텐츠를 사용하기 위해서 커스텀 웹 뷰 노출
     * @param PromotionCustomType		PromotionCustomType Type
     * @param ContentsKey				백오피스에 설정된 사용자 정의 페이지의 id
     * @param Delegate 					API 결과 통지 리스너
     *
     *  \~english Show custom Webview to use external content.
     * @param PromotionCustomType		PromotionCustomType Type
     * @param ContentsKey				ID of the custom web page set in the back office.
     * @param Delegate					API call result listener
     *
     * \~
     * @ingroup FHivePromotion
     */
    static void ShowCustomContents(const EHivePromotionCustomType& PromotionCustomType, const FString& ContentsKey, const FHivePromotionViewDelegate& Delegate);
    
    /**
     * \~korean 외부 컨텐츠를 사용하기 위해서 커스텀 웹 뷰 노출 (GameWindow)
     * @param PromotionCustomType		PromotionCustomType Type
     * @param ContentsKey				백오피스에 설정된 사용자 정의 페이지의 id
     * @param Delegate 					API 결과 통지 리스너
     *
     *  \~english Show custom Webview to use external content. (GameWindow)
     * @param PromotionCustomType		PromotionCustomType Type
     * @param ContentsKey				ID of the custom web page set in the back office.
     * @param Delegate					API call result listener
     *
     * \~
     * @ingroup FHivePromotion
     */
    static void ShowCustomContentsOnGameWindow(const EHivePromotionCustomType& PromotionCustomType, const FString& ContentsKey, const FHivePromotionViewDelegate& Delegate);
    
    /**
     * \~korean  게임내에서 무료 충전소를 노출하기 위한 버튼 UI 는 보여지거나 숨길 수 있도록 구성해야 한다.<br/>
     * 이 메서드는 게임내에서 오퍼월(무료 충전소) 을 호출 할 수 있는 버튼 노출 가능 여부를 반환한다.
     *
     * @return 오퍼월(무료 충전소) 을 호출 할 수 있는 버튼 노출 가능 여부
     * \~english Returns whether the button is available to invoke an offerwall (free recharging station) within the game.<br/>
     * The button UI for exposing free recharge stations within the game must be configured to be visible or hidden.
     *
     * @return whether the button is available.
     * \~
     * @ingroup FHivePromotion
     */
    static EHiveOfferwallState GetOfferwallState();
    
    /**
     * \~korean  무료 충전소 화면 노출 (Android only.)<br/>
     * 무료 충전소는 HIVE 게임 간의 광고 네트워크로 기존 HIVE 유저들이 새로운 HIVE 게임을 이용할 수 있도록 유도하는 기능이다.<br/>
     * 유저가 다운로드 받을 수 있는 게임을 목록으로 노출하고 게임을 다운로드 받아 실행하면 매체가 되는 게임에서 보상이 제공된다.<br/>
     *
     * @param Delegate	API 결과 통지
     * \~english Show Offerwall(Free recharging station) (Android only.)<br/>
     * The Offerwall (free recharge station) is an advertising network between HIVE games, which enables existing HIVE users to use the new HIVE game.<br/>
     * A list of the games that the user can download is exposed, and the game is downloaded and executed, and reward is provided in the game as the medium.<br/>
     *
     * @param Delegate	API call result listener
     * \~
     * @ingroup FHivePromotion
     */
    static void ShowOfferwall(const FHivePromotionViewDelegate& Delegate);

    /**
     * \~korean  게임의 새로운 이벤트나 새로운 게임 소개등의 새소식 화면을 노출
     * @param Menu             최초 노출시 활성화할 메뉴 관리 명
     * @param Delegate          API 결과 통지
     * \~english Shows banner such as new event of game or introduce new game
     *
     * @param Menu              Menu Management Name
     * @param Delegate          API call result handler
     * \~
     * @ingroup FHivePromotion
     */
    static void ShowNews(const FString& Menu, const TArray<int32>& GiftPidList, const FHivePromotionViewDelegate& Delegate);
    
    /**
     * \~korean  리뷰 유도 팝업 노출<br/>
     * 게임 유저들의 긍정적인 평점 및 사용 후기는 다른 유저들이 게임을 이용하는데 영향을 미치게 된다.<br/>
     * 리뷰 유도 팝업을 제공하면 유저의 참여 건수가 5~10배 증가하는 것으로 알려져 있다.<br/>
     * iOS 10.3 이상의 기기에서는 앱 내부에서 평점 및 리뷰를 작성할 수 있는 팝업이 노출된다.
     *
     * \~english Show review popup <br/>
     * Positive ratings and reviews of game users will affect other users' use of the game.<br/>
     * It is known that the number of user participation increases by 5 ~ 10 times when the review<br/>
     * Review popup on the device with iOS 10.3 and later is available to rate and write reviews in apps.
     *
     * \~
     * @ingroup FHivePromotion
     */
    static void ShowNativeReview();

    /**
     * \~korean  리뷰 유도 팝업 노출<br/>
     * 게임 유저들의 긍정적인 평점 및 사용 후기는 다른 유저들이 게임을 이용하는데 영향을 미치게 된다.<br/>
     * 리뷰 유도 팝업을 제공하면 유저의 참여 건수가 5~10배 증가하는 것으로 알려져 있다.<br/>
     * Hive 콘솔 설정에 따라 Hive 일반 리뷰 팝업과 이미지 수정이 가능한 커스텀 리뷰 팝업이 노출된다.
     *
     * \~english Show review popup <br/>
     * Positive ratings and reviews of game users will affect other users' use of the game.<br/>
     * It is known that the number of user participation increases by 5 ~ 10 times when the review<br/>
     * Depending on the settings of Hive console, Hive normal review popup or customized image review popup will be appeared.
     *
     * \~
     * @ingroup FHivePromotion
     */
    static void ShowReview(const FHivePromotionViewDelegate& Delegate);

    /**
     * \~korean  종료 확인 팝업 노출 (Android only.)<br/>
     * 이용 중인 게임을 종료 하려는 유저에게 새로운 HIVE 게임의 다운로드를 유도하기 위해 '더 많은 게임'이란 버튼을 노출한다
     *
     * @param listener API 결과 통지
     * \~english Show exit popup (Android only)<br/>
     * Expose exit popup which include a button called "More games" to lead users to download a new HIVE game.
     *
     * @param listener API call result listener
     * \~
     * @ingroup FHivePromotion
     */
    static void ShowExit(const FHivePromotionViewDelegate& Delegate);

    /**
     * \~korean  HIVE 프로모션 웹 뷰의 UI 를 게임 UI 의 컨셉에 맞추기 위해서 프로모션 웹 뷰를 게임에서 직접 구현 할 수 있다<br/>
     * 이 메서드는 게임에서 HIVE 프로모션 웹 뷰를 커스터 마이징하기 위한 정보를 반환한다.
     *
     * @param listener API 결과 통지
     * \~english Request HIVE Promotion Webview information so that your UI of webview is configured according to the concept of game UI<br/>
     *
     * @param listener API call result listener
     * \~
     * @ingroup FHivePromotion
     */
    static void GetViewInfo(const EHivePromotionCustomType& PromotionCustomType, const FString& ContentsKey, const FHivePromotionOnViewInfoDelegate& Delegate);
    
    /**
     * \~korean  뱃지 정보 조회<br/>
     * 프로모션 뱃지는 유저에게 새로운 프로모션 내용이 등록 되었음을 알려주기 위해서 게임의 버튼 UI 를 부각하는 정보이다
     *
     * @param Delegate API 결과 통지
     * \~english Request badge information<br/>
     * The promotional badge is information that highlights the button UI of the game to inform the user that a new promotion has been registered.
     *
     * @param Delegate API call result listener
     * \~
     * @ingroup FHivePromotion
     */
    static void GetBadgeInfo(const FHivePromotionOnBadgeInfoDelegate& Delegate);

    /**
     * \~korean  배너 정보 조회<br/>
     * 특정 프로모션에 대한 배너 정보를 요청한다. 게임 서버에서 배너 정보를 조회하기 위해 사용된다.
     *
     * @param CampaignType "event", "notice", "all", "cross"
     * @param Delegate API 결과 통지
     *
     * \~english Request banner information<br/>
     * Request banner information for a specific promotion. Used to retrieve banner information from the game server.
     *
     * @param CampaignType "event", "notice", "all", "cross"
     * @param Delegate API call result listener
     * \~
     * @ingroup FHivePromotion
     */
    static void GetBannerInfo(const EHivePromotionCampaignType& CampaignType, const EHivePromotionBannerType& BannerType, const FHivePromotionOnBannerInfoDelegate& Delegate);
    
    /**
     * \~korean  배너 정보 조회<br/>
     * 특정 프로모션에 대한 배너 정보를 요청한다. 게임 서버에서 배너 정보를 조회하기 위해 사용된다.
     *
     * @param CampaignString 커스텀 String campaign type
     * @param Delegate API 결과 통지
     *
     * \~english Request banner information<br/>
     * Request banner information for a specific promotion. Used to retrieve banner information from the game server.
     *
     * @param CampaignString custom String campaign type
     * @param Delegate API call result listener
     * \~
     * @ingroup FHivePromotion
     */
    static void GetBannerInfoString(const FString& CampaignString, const FString& BannerString, const FHivePromotionOnBannerInfoDelegate& Delegate);
    
    /**
     * \~korean  앱 초대를 위한 데이터 조회<br/>
     * 게임에서 더 많은 사용자를 유치하기 위해서 앱 설치 유도 기능을 제공한다.<br/>
     * 앱 설치 유도는 유저 애퀴지션 (User Acquisition) 이라고도 부른다.<br/>
     * 앱의 초대를 위한 데이터는 QR Code, 초대링크, 캠페인 등이 있다.<br/>
     * 캠페인은 초대에 대한 보상을 달상하기 위한 조건을 명시한 데이터이다.<br/>
     * 초대 URL 또는 QR코드를 통해 초대받은 유저가 게임을 설치하고, 특정 레벨 달성 등과 같은 조건(백오피스에서 조건 및 보상 설정 가능)을 달성했을 때 초대한 유저와 초대받은 유저 모두에게 보상을 제공한다.
     *
     * @param Delegate	API 호출 결과.
     * \~english Request user invite information<br/>
     * User acquisition information is provided to attract more users in the game.<br/>
     * Data for user acquisition include QR Code, invite links, and campaigns.<br/>
     * A campaign is data that specifies the conditions for rewarding an invite.<br/>
     * The invite URL or QR code provides rewards to both the invited and invited users when the invited user achieves the conditions such as installing the game and achieving a certain level (condition and reward can be set in the back office).
     *
     * @param Delegate	API call result.
     * \~
     * @ingroup FHivePromotion
     */
    static void GetAppInvitationData(const FHivePromotionOnAppInvitationDataDelegate& Delegate);

    /**
     * \~korean  프로모션 뷰의 특정 이벤트 페이지를 노출하거나 프로모션 서버에 부가 정보를 전달하고 싶을 때 JSON 포맷의 문자열을 설정할 수 있다<br/>
     *
     * (필수)서버군이 따로 없는 단일 서버군이라도 아래 예제대로 넣어야 함<br/>
     * ex)"addtionalInfo":"{"server":"0","character":"0"}"
     *
     * @param AdditionalInfo	(필수) JSON 포맷의 부가 정보 데이터
     * \~english You can set a string in JSON format when you want to expose a specific event page in the Promotion View or want to pass additional information to the Promotion Server.<br/>
     *
     * (Required) Even a single server group without a server group should be properly put in the following example<br/>
     * ex)"addtionalInfo":"{"server":"0","character":"0"}"
     *
     * @param AdditionalInfo	(Required) Additional information data in JSON format
     * \~
     * @ingroup FHivePromotion
     */
    static void SetAdditionalInfo(const FString& AdditionalInfo);

    /**
     * \~korean  Engagement 이벤트 처리가 가능한지(게임 서버에 DB가 생성된 경우 등) 여부를 설정한다.
     * true로 설정하려는 경우, <로그인 이후 & 리스너가 등록된 이후>의 조건을 만족한 상태여야 정상적으로 설정되며,
     * false로 설정하려는 경우는 항상 설정 가능하다.
     *
     * @param IsReady Enganement 이벤트 처리 가능 여부.
     * @return API 호출 성공 여부.
     * \~english It sets whether Engagement event handling is enabled.(Such as when a DB is created in the game server)
     * If you want to set it to true, it must be in a state that satisfies the condition of <after login & after registering the listener>,
     * if you want to set it to false, you can always set it.
     *
     * @param IsReady Whether Engagement events can be processed.
     * @return Whether the API call was successful.
     * \~
     * @ingroup FHivePromotion
     */
    static TOptional<FHiveResultAPI> SetEngagementReady(bool IsReady);

    /**
     * \~korean URI를 파싱하여 Event 타입으로 만든다.
     *
     * @param URI 파싱할 URI
     * @return 파싱에 문제가 없으면 true, 아니면 false.
     * \~english Parse the URI to make it an Event type.
     *
     * @param URI URI to parse
     * @return True if there is no problem parsing, false otherwise.
     * \~
     * @ingroup FHivePromotion
     */
    static bool ProcessURI(const FString& URI);

    /**
     * \~korean  Engagement 리스너를 등록한다.
     * \~english  It register the Engagement listener.
     * \~
     * @ingroup FHivePromotion
     */
    static void SetEngagementDelegate(const FHivePromotionOnEngagementDelegate& Delegate);

    /**
     * \~korean 외부 SNS를 통해 UA url을 공유하기 위한 다이얼로그 노출
     * @param InviteMessage             UA 초대 메시지
     * @param InviteLink                UA 초대 링크
     * @param Delegate                  API 결과 통지 리스너
     *
     *  \~english Show dialog to share UA url via SNS.
     * @param InviteMessage             UA invite message
     * @param InviteLink                UA invite link
     * @param Delegate                  API call result listener
     * \~
     * @ingroup FHivePromotion
     */
    static void ShowUAShare(const FString& InviteMessage, const FString& InviteLink, const FHivePromotionOnUAShareDelegate& Delegate);
    
    /**
     * \~korean Promotion 정보 갱신 <br/>
     * Promotion 새소식 페이지 등의 노출에 필요한 정보를 갱신한다.
     *
     * \~english Update information of Promotion
     * Update promotion data to show Promotion UI.
     * 
     * \~
     * @ingroup FHivePromotion
     */
    static void UpdatePromotionData();

    /**
     * \~korean UA를 통해 자신을 앱으로 최초 초대한 유저의 정보 반환 한다.
     * SDK 초기화, 로그인, setEngagementReady(true) 호출 이후 최초 초대자 정보 전달이 가능하다.
     * @param Delegate API 결과 통지 리스터
     *
     * \~english return first sender's userInfo who sent UA share invitation
     * First sender's information can be returned after SDK initialize, login and setEngagementReady(true)
     * @param Delegate API call result listener
     * \~
     * @ingroup FHivePromotion
     */
    static void GetAppInvitationSenderInfo(const FHivePromotionOnAppInvitationSenderInfoDelegate& Delegate);
};
