/**
 * @file    HiveProviderGoogle.h
 *
*  @date        2025
*  @copyright   Copyright © 2025 Com2uS Platform Corp. All Rights Reserved.
*
*  @defgroup    FHiveProviderGoogle
*  @{
 *  \~korean
 * Google Play Games 의 기능 중 업적과 리더보드를 사용할 수 있다.<br/>
 * 연결되어있지 않을 경우 API 를 호출 할 때 자동으로 연결을 시도 한다.<br/>
 * AuthV4 의 Connect 와는 무관하게 독립적으로 사용할 수 있다.<br/>
 * 자동 연결이 되어도 현재 playerid 와 Connect 되는것은 아니다.
 *
 *  \~english
 * You can use achievements and leaderboards within Google Play Games.<br/>
 * If it is not connected to Google Play Games, it will try to connect automatically when calling API.<br/>
 * It can be used independently regardless of AuthV4 Connect.<br/>
 * Even if automatic connection is made, it is not connected with the current playerId.
*/
#pragma once

#include "CoreMinimal.h"
#include "HiveInterface.h"
#include "Impl/HiveProviderGoogleImpl.h"

/**
 *  \~korean
 * Google Play Games 의 기능 중 업적과 리더보드를 사용할 수 있다.<br/>
 * 연결되어있지 않을 경우 API 를 호출 할 때 자동으로 연결을 시도 한다.<br/>
 * AuthV4 의 Connect 와는 무관하게 독립적으로 사용할 수 있다.<br/>
 * 자동 연결이 되어도 현재 playerid 와 Connect 되는것은 아니다.
 *
 *  \~english
 * You can use achievements and leaderboards within Google Play Games.<br/>
 * If it is not connected to Google Play Games, it will try to connect automatically when calling API.<br/>
 * It can be used independently regardless of AuthV4 Connect.<br/>
 * Even if automatic connection is made, it is not connected with the current playerId.
 *
 *  \~
 * @ingroup FHiveProviderGoogle
 */
class HIVESDK_API FHiveProviderGoogle : public IHiveInterface<IHiveProviderGoogleImpl>
{
public:
/* Achievements */
    /**
     *  \~korean
     * 숨겨진 업적을 공개한다.<br/>
     * 업적이 0% 로 공개만 될 뿐 달성 되지는 않는다.<br/>
     *
     * @param AchievementId 공개할 achievementId
     * @param Delegate FHiveProviderGoogleOnAchievementsDelegate
     *
     *  \~english
     * It show hidden achievements.<br/>
     * Achievements are only revealed at 0%, not achieved.<br/>
     *
     * @param AchievementId achievementId to reveal
     * @param Delegate FHiveProviderGoogleOnAchievementsDelegate
     *
     *  \~
     * @ingroup FHiveProviderGoogle
     */
    static void AchievementsReveal(const FString& AchievementId, const FHiveProviderGoogleOnAchievementsDelegate& Delegate);
    
    /**
     *  \~korean
     * 업적을 달성한다.<br/>
     * 숨겨져 있거나 공개된 여부와 상관없이 업적이 100% 로 달성 된다.<br/>
     *
     * @param AchievementId 공개할 achievementId
     * @param Delegate FHiveProviderGoogleOnAchievementsDelegate
     *
     *  \~english
     * It achieve achievements.<br/>
     * Whether hidden or open, achievement is achieved at 100%.<br/>
     *
     * @param AchievementId achievementId to achieve
     * @param Delegate FHiveProviderGoogleOnAchievementsDelegate
     *
     *  \~
     * @ingroup FHiveProviderGoogle
     */
    static void AchievementsUnlock(const FString& AchievementId, const FHiveProviderGoogleOnAchievementsDelegate& Delegate);
    
    /**
     *  \~korean
     * 업적 수치를 증가 시킨다.<br/>
     * value 만큼 설정이 아닌 매 value 만큼 합산 된다.<br/>
     * 총 합산이 Max 가 될 경우 자동으로 업적이 달성되며, 계속 호출하여도 무방하다.<br/>
     *
     *  \~english
     * It increases achievement figures.<br/>
     * Achievement figures is added as much as value set by the API call, not by setting.<br/>
     * If the total sum is Max, the achievement is automatically accomplished.<br/>
     *
     *  \~
     * @param AchievementId incremental AchievementId
     * @param Value value
     * @param Delegate FHiveProviderGoogleOnAchievementsDelegate
     * @ingroup FHiveProviderGoogle
     */
    static void AchievementsIncrement(const FString& AchievementId, int32 Value, const FHiveProviderGoogleOnAchievementsDelegate& Delegate);
    
    /**
     *  \~korean
     * Google 업적 UI 를 띄운다.<br/>
     *
     *  \~english
     * It shows the Google achievement UI.<br/>
     *  \~
     * @param Delegate FHiveProviderGoogleOnAchievementsDelegate
     * @ingroup FHiveProviderGoogle
     */
    static void ShowAchievements(const FHiveProviderGoogleOnAchievementsDelegate& Delegate);


/* Leaderboards */
    /**
     *  \~korean
     * 리더보드 점수를 갱신한다.<br/>
     * leaderboardId 에 해당하는 기록에 score 수치로 갱신된다.<br/>
     *
     *  \~english
     * It update the leaderboard score.<br/>
     * The score corresponding to the leaderboardId is updated with the score value.<br/>
     *
     *  \~
     * @param LeaderboardId leaderboardId
     * @param Score score
     * @param Delegate FHiveProviderGoogleOnLeaderboardsDelegate
     * @ingroup FHiveProviderGoogle
     */
    static void LeaderboardsSubmitScore(const FString& LeaderboardId, int64 Score, const FHiveProviderGoogleOnLeaderboardsDelegate& Delegate);
    
    /**
     *  \~korean
     * Google 리더보드 UI 를 띄운다.<br/>
     *
     *  \~english
     * It shows Google Leaderboard UI.
     *  \~
     * @param Delegate FHiveProviderGoogleOnLeaderboardsDelegate
     * @ingroup FHiveProviderGoogle
     */
    static void ShowLeaderboard(const FHiveProviderGoogleOnLeaderboardsDelegate& Delegate);



    /**
     *  \~korean
     * Google PlayerId 및 authCode를 가져온다.<br/>
     *
     *  \~english
     * It get Google PlayerId and authCode
     *  \~
     * @param listener onGooglePlayerIdResult
     * @ingroup FHiveProviderGoogle
     */
    static void GetGooglePlayerId(const FHiveProviderGoogleOnGooglePlayerIdDelegate& Delegate);
};