/* Copyright © 2024 Com2uS Platform Corp. All Rights Reserved. */
#pragma once

/* Unreal Engine Header */
#include "CoreMinimal.h"
#include "Dom/JsonObject.h"
#include "Serialization/JsonReader.h"
#include "Serialization/JsonSerializer.h"

/* Hive SDK Header */
#include "CPP/HIVE_CPP.h"


class HIVESDK_API FHiveAnalyticsAdRevenue
{
public:
    double Revenue;             ///< \~korean  광고 노출당 발생하는 광고 수익  \~english Ad impression revenue
    FString AdPlatform;         ///< \~korean  광고 네트워크 플랫폼 \~english Ad network platform
    FString AdUnitId;           ///< \~korean 광고 유닛 아이디  \~english Ad Unit Id
    FString AdType;             ///< \~korean 광고 노출 타입  \~english  The type of ads
    FString AdPlacement;        ///< \~korean 광고 노출 위치  \~english ad placement
    FString Currency;           ///< \~korean 통화 코드(ISO_4217 형식 문자열)  \~english ISO_4217 format string (ex. "USD")

    FHiveAnalyticsAdRevenue() {};
    FHiveAnalyticsAdRevenue(const hive::AnalyticsAdRevenue& AdRevenue);
    virtual ~FHiveAnalyticsAdRevenue() = default;
};

class HIVESDK_API FHiveConsentStatus
{
public:
    FString Id;                         ///< \~korean  동의 항목의 고유 식별자
    FString Granted;                    ///< \~korean  동의 대상의 동의 여부 입니다. 최초로 아무 설정도 하지 않았을 경우 "not-set" 혹은 empty string 이 됩니다.
    int64 ConsentDate;                  ///< \~korean  동의를 결정한 시각입니다. 단위는 milli second 입니다.
    int64 ExpiredDate;                  ///< \~korean  동의가 만료되는 시각입니다. [ConsentForm.retentionPeriod] 설정에 따라 [consentDate] 에서 더해진 시각이 됩니다. 단위는 milli second 입니다.7

    FHiveConsentStatus() {};
    FHiveConsentStatus(const hive::ConsentStatus& ConsentStatus);
    virtual ~FHiveConsentStatus() = default;
};

class HIVESDK_API FHiveConsentForm
{
public:
    FString Id;                         ///< \~korean  동의 항목의 고유 식별자
    int32 RetentionPeriod;              ///< \~korean  동의 후 만료까지의 시간을 설정합니다. 단위는 개월 수 입니다. [providerName] "main" 은 999 입니다.
    FString Title;                      ///< \~korean  타이틀 문구 입니다.
    FString Content;                    ///< \~korean  세부 내용 입니다.
    TArray<FHiveConsentForm> SubForms;  ///< \~korean 동의 대상의 하위 항목입니다.

    FHiveConsentForm() {};
    FHiveConsentForm(const hive::ConsentForm& ConsentForm);
    virtual ~FHiveConsentForm() = default;
};

class HIVESDK_API FHiveConsentMode
{
public:
    FString Title;                          ///< \~korean  동의 배너의 상단에 표시되는 문구 입니다.
    FString CompanyName;                    ///< \~korean  기본 정보로 노출시 회사이름이 필요한 [title] 과 MainForm 의 [ConsentForm.dataUsageDescription] 에 설정됩니다.
    FString PrivacyPolicy;                  ///< \~korean  동의 배너 하단의 PrivacyPolicy 버튼을 눌렀을때 동작하는 URL 입니다.
    TOptional<FHiveConsentForm> AdUserData; ///< \~korean 사용자 정보 저장 동의 항목입니다.
    TOptional<FHiveConsentForm> AdPersonalization;  ///< \~korean  개인 정보를 공유할 업체별 동의 항목 입니다.
    TOptional<FHiveConsentForm> AnalyticsStorage;   ///< \~korean  광고 정보 분석 동의 항목 입니다.

    FHiveConsentMode() {};
    FHiveConsentMode(const hive::ConsentMode& ConsentMode);
    virtual ~FHiveConsentMode() = default;
};

class HIVESDK_API IHiveAnalyticsSpecific
{
public:
    virtual FString GetProviderName() const = 0;
    virtual TSharedPtr<FJsonObject> ToJson() const = 0;
    virtual FString ToString() const = 0;

    IHiveAnalyticsSpecific() {};
    virtual ~IHiveAnalyticsSpecific() = default;
};

class HIVESDK_API FHiveAirbridgeSpecific : public IHiveAnalyticsSpecific
{
public:
    TSharedPtr<FJsonObject> SemanticAttributes;

    void SetSemanticAction(const FString& Value);
    void SetSemanticLabel(const FString& Value);
    void SetSemanticValue(double Value);
    void AddSemanticAttribute(const FString& Key, TSharedPtr<FJsonValue> Value);

    virtual FString GetProviderName() const override;
    virtual TSharedPtr<FJsonObject> ToJson() const override;
    virtual FString ToString() const override;

    FHiveAirbridgeSpecific();
    ~FHiveAirbridgeSpecific() override = default;
};

class HIVESDK_API FHiveAnalyticsAttributes
{
public:
    TSharedPtr<FJsonObject> CustomAttributes;
    TSharedPtr<FJsonObject> ModuleSpecificAttributes;

    FHiveAnalyticsAttributes();
    virtual ~FHiveAnalyticsAttributes() = default;

    void AddCustomAttribute(const FString& Key, TSharedPtr<FJsonValue> Value);
    void AddModuleSpecificAttribute(const IHiveAnalyticsSpecific& Specific);

    virtual TSharedPtr<FJsonObject> ToJson() const;
    virtual FString ToString() const;
};