/* Copyright © 2025 Com2uS Platform Corp. All Rights Reserved. */
#pragma once

/* Unreal Engine Header */
#include "CoreMinimal.h"

/* Hive SDK Header */
#include "CPP/HIVE_CPP.h"
#include <HiveResultAPI.h>

// Forward declarations
class FHiveResultAPI;

enum class EHiveChannelType;
enum class EHiveSortType;
enum class EHiveReactionType;
class FHiveChannel;
class FHiveChannelPage;
class FHiveMember;
class FHiveBlockMember;
class FHiveCreateChannelParams;
class FHiveGetChannelsParams;
class FHiveEnterChannelParams;
class FHiveChannelSendMessageParams;
class FHiveDirectSendMessageParams;
class FHiveEnteredMember;
class FHiveExitedMember;
class FHiveDeletedChannel;
class FHiveChannelNoticeMessage;
class FHiveChannelMessage;
class FHiveDirectMessage;
class FHiveChannelListener;
class FHiveDirectMessageListener;
class FHiveConnectionListener;
class FHiveReaction;
class FHiveTranslateParams;
class FHiveTranslationData;
class FHiveChannelMessageListQueryParams;
class FHiveChannelMessageListQueryResponse;
class FHiveNoticeMessage;
class FHiveUserListener;
class FHiveCustomDataListener;

/**
 * @brief 채팅 채널 타입 정보<br/>
 *
 * @ingroup FHiveChat
 */
enum class EHiveChannelType
{
    UNKNOWN     = -1,
    PRIVATE     =  0,
    PUBLIC      =  1,
    GROUP       =  2
};

HIVESDK_API FString EHiveChannelTypeToString(EHiveChannelType type);
HIVESDK_API EHiveChannelType FStringToChannelType(const FString& str);

/**
 * @brief 리액션 타입 정보<br/>
 *
 * @ingroup FHiveChat
 */
enum class EHiveReactionType
{
    UNKNOWN = -1,
    LIKE    =  0
};

HIVESDK_API FString EHiveReactionTypeToString(EHiveReactionType type);
HIVESDK_API EHiveReactionType FStringToReactionType(const FString& str);

/**
 * @brief 채널 정렬 타입 정보<br/>
 *
 * @ingroup FHiveChat
 */
enum class EHiveSortType
{
    UNKNOWN       = -1,
    CHANNEL_ID    =  0,
    CHANNEL_NAME  =  1,
    REG_TIME      =  2
};

HIVESDK_API FString EHiveSortTypeToString(EHiveSortType type);
HIVESDK_API EHiveSortType FStringToSortType(const FString& str);


using FHiveChannelQueryDelegate = TFunction<void(const FHiveResultAPI&, const FHiveChannelMessageListQueryResponse&)>;
using FHiveChannelTranslationSettingDelegate = TFunction<void(const FHiveResultAPI&)>;
using FHiveChannelReactionDelegate = TFunction<void(const FHiveResultAPI&, EHiveReactionType)>;

/**
 * @brief 채팅 채널 정보<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveChannel 
{
public:
    FString channelId;
    EHiveChannelType type;
    FString owner;
    FString channelName;
    int memberCount;
    int maxMemberCount;
    FString regTime;
    int64 regTimeMillis;
    bool chatHistoryAllowed;

    FHiveChannel() {};
    FHiveChannel(const hive::Channel& Channel);
    virtual ~FHiveChannel() = default;

    FString ToString() const;
    
    void Query(FHiveChannelMessageListQueryParams const & params, const FHiveChannelQueryDelegate& Delegate) const;
    void SetTranslationEnabled(bool enabled, const FHiveChannelTranslationSettingDelegate& Delegate) const;
    void AddReaction(const FString& messageId, EHiveReactionType reactionType, const FHiveChannelReactionDelegate& Delegate) const;
    void RemoveReaction(const FString& messageId, EHiveReactionType reactionType, const FHiveChannelReactionDelegate& Delegate) const;
};

/**
 * @brief 채널 조회 페이지 정보<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveChannelPage 
{
public:
    int size;
    int currentPage;
    int totalElements;
    int totalPages;

    FHiveChannelPage() {};
    FHiveChannelPage(const hive::ChannelPage& ChannelPage);
    virtual ~FHiveChannelPage() = default;

    FString ToString() const;
};

/**
 * @brief 채널 멤버 정보<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveMember 
{
public:
    int64 playerId;
    FString connectedTime;
    int64 connectedTimeMillis;

    FHiveMember() {};
    FHiveMember(const hive::Member& Member);
    virtual ~FHiveMember() = default;

    FString ToString() const;
};

/**
 * @brief 차단된 사용자 정보<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveBlockMember
{
public:
    int64 playerId;
    FString blockedTime;
    int64 blockedTimeMillis;

    FHiveBlockMember() {};
    FHiveBlockMember(const hive::BlockMember& BlockMember);
    virtual ~FHiveBlockMember() = default;

    FString ToString() const;
};

/**
 * @brief 채널 생성 파라미터<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveCreateChannelParams
{
public:
    FString channelId;
    FString password;
    FString channelName;
    int maxMemberCount;
    EHiveChannelType type;
    bool chatHistoryAllowed;

    FHiveCreateChannelParams() {};
    FHiveCreateChannelParams(const hive::CreateChannelParams& CreateChannelParams);
    virtual ~FHiveCreateChannelParams() = default;
};

/**
 * @brief 채널 목록 조회 파라미터<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveGetChannelsParams
{
public:
    EHiveChannelType type;
    FString channelId;
    FString channelName;
    EHiveSortType sort;
    FString pageOrder;
    int pageSize;
    int pageNumber;

    FHiveGetChannelsParams() {};
    FHiveGetChannelsParams(const hive::GetChannelsParams& GetChannelsParams);
    virtual ~FHiveGetChannelsParams() = default;
};

/**
 * @brief 채널 입장 파라미터<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveEnterChannelParams
{
public:
    FString channelId;
    FString password;

    FHiveEnterChannelParams() {};
    FHiveEnterChannelParams(const hive::EnterChannelParams& EnterChannelParams);
    virtual ~FHiveEnterChannelParams() = default;
};

/**
 * @brief 채널 메시지 전송 파라미터<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveChannelSendMessageParams
{
public:
    FString channelId;
    FString message;
    FString extraData;
    TArray<int64> mentionedPlayerIds;
    FString replyMessageId;

    FHiveChannelSendMessageParams() {};
    FHiveChannelSendMessageParams(const hive::ChannelSendMessageParams& ChannelSendMessageParams);
    virtual ~FHiveChannelSendMessageParams() = default;
};

/**
 * @brief 1:1 메시지 전송 파라미터<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveDirectSendMessageParams
{
public:
    int64 toPlayerId;
    FString message;
    FString extraData;

    FHiveDirectSendMessageParams() {};
    FHiveDirectSendMessageParams(const hive::DirectSendMessageParams& DirectSendMessageParams);
    virtual ~FHiveDirectSendMessageParams() = default;
};

/**
 * @brief 입장 멤버 정보<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveEnteredMember
{
public:
    FString channelId;
    int64 playerId;
    FString timestamp;
    int64 timestampMillis;

    FHiveEnteredMember() {};
    FHiveEnteredMember(const hive::EnteredMember& EnteredMember);
    virtual ~FHiveEnteredMember() = default;

    FString ToString() const;
};

/**
 * @brief 퇴장 멤버 정보<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveExitedMember
{
public:
    FString channelId;
    int64 playerId;
    FString timestamp;
    int64 timestampMillis;

    FHiveExitedMember() {};
    FHiveExitedMember(const hive::ExitedMember& ExitedMember);
    virtual ~FHiveExitedMember() = default;

    FString ToString() const;
};

/**
 * @brief 삭제된 채널 정보<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveDeletedChannel
{
public:
    FString channelId;
    FString timestamp;
    int64 timestampMillis;

    FHiveDeletedChannel() {};
    FHiveDeletedChannel(const hive::DeletedChannel& DeletedChannel);
    virtual ~FHiveDeletedChannel() = default;

    FString ToString() const;
};

/**
 * @brief 채널 공지 메시지 정보<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveChannelNoticeMessage
{
public:
    FString messageId;
    FString channelId;
    FString from;
    FString message;
    FString timestamp;
    int64 timestampMillis;

    FHiveChannelNoticeMessage() {};
    FHiveChannelNoticeMessage(const hive::ChannelNoticeMessage& ChannelNoticeMessage);
    virtual ~FHiveChannelNoticeMessage() = default;

    FString ToString() const;
};

/**
 * @brief 채널 메시지 정보<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveChannelMessage
{
public:
    FString messageId;
    int64 from;
    FString extraData;
    FString to;
    FString message;
    FString timestamp;
    int64 timestampMillis;
    bool translated;
    FString translatedMessage;
    TMap<EHiveReactionType, TArray<int64>> reactions;
    TArray<int64> mentionedPlayerIds;
    FString replyMessageId;
    FString replyMessage;
    FString replyExtraData;

    FHiveChannelMessage() {};
    FHiveChannelMessage(const hive::ChannelMessage& ChannelMessage);
    virtual ~FHiveChannelMessage() = default;

    FString ToString() const;
};

/**
 * @brief 1:1 메시지 정보<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveDirectMessage
{
public:
    FString messageId;
    int64 from;
    FString extraData;
    int64 to;
    FString message;
    FString timestamp;
    int64 timestampMillis;

    FHiveDirectMessage() {};
    FHiveDirectMessage(const hive::DirectMessage& DirectMessage);
    virtual ~FHiveDirectMessage() = default;

    FString ToString() const;
};

/**
 * @brief 채널 이벤트 리스너<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveChannelListener
{
public:
    virtual void onEnteredMember(FHiveEnteredMember const & member) = 0;
    virtual void onExitedMember(FHiveExitedMember const & member) = 0;
    virtual void onDeletedChannel(FHiveDeletedChannel const & channel) = 0;
    virtual void onNoticeMessage(FHiveChannelNoticeMessage const & message) = 0;
    virtual void onChannelMessage(FHiveChannelMessage const & message) = 0;
    virtual void onAddReaction(FHiveReaction const & reaction) = 0;
    virtual void onRemoveReaction(FHiveReaction const & reaction) = 0;

    virtual ~FHiveChannelListener() = default;
};

/**
 * @brief 1:1 메시지 리스너<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveDirectMessageListener
{
public:
    virtual void onDirectMessage(FHiveDirectMessage const & message) = 0;

    virtual ~FHiveDirectMessageListener() = default;
};

/**
 * @brief 연결 리스너<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveConnectionListener
{
public:
    virtual void onConnected() = 0;
    virtual void onReconnectStarted() = 0;
    virtual void onReconnected(const FHiveResultAPI& result, const TArray<FString>& channelIds, const TArray<FString>& failChannelIds) = 0;
    virtual void onDisconnected() = 0;

    virtual ~FHiveConnectionListener() = default;
};

/**
 * @brief 리액션 정보<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveReaction
{
public:
    FString channelId;
    FString messageId;
    int64 playerId;
    EHiveReactionType type;

    FHiveReaction() = default;
    FHiveReaction(const hive::Reaction& Reaction);
    virtual ~FHiveReaction() = default;

    FString ToString() const;
};

/**
 * @brief 메시지 번역 파라미터<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveTranslateParams
{
public:
    FString message;
    FString sourceLanguage = TEXT("auto");
    TArray<FString> targetLanguage;

    FHiveTranslateParams() = default;
    FHiveTranslateParams(const hive::TranslateParams& TranslateParams);
    virtual ~FHiveTranslateParams() = default;

    FString ToString() const;
};

/**
 * @brief 메시지 번역 결과<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveTranslationData
{
public:
    TMap<FString, FString> translations;

    FHiveTranslationData() = default;
    FHiveTranslationData(const hive::TranslationData& TranslationData);
    virtual ~FHiveTranslationData() = default;

    FString ToString() const;
};

/**
 * @brief 채널 메시지 이력 조회 파라미터<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveChannelMessageListQueryParams
{
public:
    int prevSize = 0;
    int nextSize = 0;
    FString messageId;
    FString order;

    FHiveChannelMessageListQueryParams() = default;
    FHiveChannelMessageListQueryParams(const hive::ChannelMessageListQueryParams& Params);
    virtual ~FHiveChannelMessageListQueryParams() = default;

    FString ToString() const;
};

/**
 * @brief 채널 메시지 이력 조회 응답<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveChannelMessageListQueryResponse
{
public:
    bool hasNext;
    FString nextMessageId;
    bool hasPrev;
    FString prevMessageId;
    TArray<FHiveChannelMessage> content;

    FHiveChannelMessageListQueryResponse() = default;
    FHiveChannelMessageListQueryResponse(const hive::ChannelMessageListQueryResponse& Response);
    virtual ~FHiveChannelMessageListQueryResponse() = default;

    FString ToString() const;
};

/**
 * @brief 유저 대상 공지 메시지 정보<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveNoticeMessage
{
public:
    FString messageId;
    FString from;
    FString message;
    FString timestamp;
    int64 timestampMillis;

    FHiveNoticeMessage() = default;
    FHiveNoticeMessage(const hive::NoticeMessage& NoticeMessage);
    virtual ~FHiveNoticeMessage() = default;

    FString ToString() const;
};

/**
 * @brief 유저 이벤트 리스너<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveUserListener
{
public:
    virtual void onDirectMessage(FHiveDirectMessage const & message) = 0;
    virtual void onNoticeMessage(FHiveNoticeMessage const & message) = 0;

    virtual ~FHiveUserListener() = default;
};

/**
 * @brief 커스텀 데이터 리스너<br/>
 *
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveCustomDataListener
{
public:
    virtual void onCustomData(FString const & data) = 0;

    virtual ~FHiveCustomDataListener() = default;
};
