#include "HIVE_AuthV4.h"
#include "HIVE_CppPlugin.h"

#include <sstream>
#include <assert.h>
#if !WITH_EDITOR && PLATFORM_WINDOWS

NS_HIVE_BEGIN

static AuthV4::onSetup g_Authv4SetupListener = nullptr;
static AuthV4::onSignIn g_Authv4SignInListener = nullptr;
static AuthV4::onSignOut g_Authv4SignOutListener = nullptr;
static AuthV4::onSignOut g_Authv4PlayerDeleteListener = nullptr;
static AuthV4::onConnect g_Authv4ConnectListener = nullptr;
static AuthV4::onDisconnect g_Authv4DisconnectListener = nullptr;
static AuthV4::onSignIn g_Authv4SelectSignInListener = nullptr;
static AuthV4::onCheckProvider g_Authv4CheckProviderListener = nullptr;
static AuthV4::onMaintenance g_Authv4CheckMaintenanceListener = nullptr;
static AuthV4::onMaintenance g_Authv4CheckBlackListMaintenanceListener = nullptr;
static AuthV4::onCheckProvider g_Authv4SetProviderChangedHandler = nullptr;
static AuthV4::onSignIn g_Authv4ShowSignInListener = nullptr;
static AuthV4::onSignIn g_Authv4ConflictSelectionSignInListener = nullptr;
static AuthV4::onShowTerms g_Authv4ShowTermsListener = nullptr;
static AuthV4::onAdultConfirm g_Authv4ShowAdultConfirmListener = nullptr;
static AuthV4::onGetProfile g_Authv4GetProfileListener = nullptr;
static AuthV4::onShowProfile g_Authv4ShowProfileListener = nullptr;
static AuthV4::onShowInquiry g_AuthV4ShowInquiryListener = nullptr;
static AuthV4::onShowChatbotInquiry g_AuthV4ShowChatbotInquiryListener = nullptr;
static AuthV4::onShowMyInquiry g_AuthV4ShowMyInquiryListener = nullptr;
static AuthV4::onGetProviderFriendsList g_AuthV4GetProviderFriendsListListener = nullptr;
static AuthV4::onResolveConflict g_AuthV4ResolveConflict = nullptr;
static AuthV4::onAuthV4DialogDismiss g_AuthV4DialogDismiss = nullptr;
static AuthV4::onAuthV4RequestPermissionViewData g_onAuthV4RequestPermissionViewData = nullptr;
static AuthV4::onShowDeviceManagement g_AuthV4ShowDeviceManagementListener = nullptr;
static AuthV4::onGetHiveTalkPlusLoginToken g_AuthV4GetHiveTalkPlusLoginTokenListener = nullptr;
static AuthV4::onRefreshAccessToken g_AuthV4RefreshAccessTokenListener = nullptr;
static AuthV4::onSignIn g_AuthV4SignInWithStoredPlayerIdListener = nullptr;
static AuthV4::onPlayerIdList g_AuthV4GetStoredPlayerIdListListener = nullptr;
static AuthV4::onStorePlayerId g_AuthV4StoreCurrentPlayerIdListener = nullptr;
static AuthV4::onDeletePlayerId g_AuthV4DeleteStoredPlayerIdListener = nullptr;
static AuthV4::onHashedDi g_AuthV4GetHashedDiListener = nullptr;
static AuthV4::onIdentityVerification g_AuthV4ShowIdentityVerificationListener = nullptr;
static AuthV4::onGetAgeRange g_AuthV4GetAgeRangeListener = nullptr;
static AuthV4::onGetAgeRange g_AuthV4ShowAgeRangeUpdateListener = nullptr;
static AuthV4::onCheckAgeGate g_AuthV4OnCheckAgeGateListener = nullptr;
static AuthV4::onAutoLoginEnabled g_AuthV4SetAutoLoginEnabledListener = nullptr;

//AuthV4 Helper Callbacks
static AuthV4::Helper::onAuthV4Helper g_AuthV4HelperSyncAccount = nullptr;
static AuthV4::Helper::onAuthV4Helper g_AuthV4HelperSignIn = nullptr;
static AuthV4::Helper::onAuthV4Helper g_AuthV4HelperSignOut = nullptr;
static AuthV4::Helper::onAuthV4Helper g_AuthV4HelperPlayerDelete = nullptr;
static AuthV4::Helper::onAuthV4Helper g_AuthV4HelperConnect = nullptr;
static AuthV4::Helper::onAuthV4Helper g_AuthV4HelperDisconnect = nullptr;
static AuthV4::Helper::onAuthV4Helper g_AuthV4HelperShowAchievements = nullptr;
static AuthV4::Helper::onAuthV4Helper g_AuthV4HelperShowLeaderboard = nullptr;
static AuthV4::Helper::onAuthV4Helper g_AuthV4HelperAchievementsReveal = nullptr;
static AuthV4::Helper::onAuthV4Helper g_AuthV4HelperAchievementsUnlock = nullptr;
static AuthV4::Helper::onAuthV4Helper g_AuthV4HelperAchievementsIncrement = nullptr;
static AuthV4::Helper::onAuthV4Helper g_AuthV4HelperLeaderBoardsSubmitScore = nullptr;
static AuthV4::Helper::onAuthV4Helper g_AuthV4HelperShowConflict = nullptr;
static AuthV4::Helper::onAuthV4Helper g_AuthV4HelperSwitchAccount = nullptr;
static AuthV4::Helper::onAuthV4Helper g_AuthV4HelperResolveConflict = nullptr;
static AuthV4::onAuthV4DialogDismiss g_AuthV4HelperDialogDismiss = nullptr;
static std::map<std::string,AuthV4::Helper::onAuthV4Helper> g_AuthV4HelperCallbacks;

const char* AUTHV4_FUNC_NAME_SETUP = "setup";
const char* AUTHV4_FUNC_NAME_SIGNIN = "signIn";
const char* AUTHV4_FUNC_NAME_SIGNIN_WITH_AUTHKEY = "signInWithAuthKey";
const char* AUTHV4_FUNC_NAME_SIGNOUT = "signOut";
const char* AUTHV4_FUNC_NAME_PLAYERDELETE = "playerDelete";
const char* AUTHV4_FUNC_NAME_CONNECT = "connect";
const char* AUTHV4_FUNC_NAME_CONNECT_WITH_AUTHKEY = "connectWithAuthKey";
const char* AUTHV4_FUNC_NAME_DISCONNECT = "disconnect";
const char* AUTHV4_FUNC_NAME_DISCONNECT_WITH_NAME = "disconnectWithName";
const char* AUTHV4_FUNC_NAME_SELECTCONFLICT = "selectConflict";
const char* AUTHV4_FUNC_NAME_CHECKPROVIDER = "checkProvider";
const char* AUTHV4_FUNC_NAME_CHECKMAINTENANCE = "checkMaintenance";
const char* AUTHV4_FUNC_NAME_CHECKBLACKLIST = "checkBlacklist";
const char* AUTHV4_FUNC_NAME_SETPROVIDERCHANGEDLISTENER = "setProviderChangedListener";
const char* AUTHV4_FUNC_NAME_SHOWSIGNIN = "showSignIn";
const char* AUTHV4_FUNC_NAME_SHOWCONFLICTSELECTION = "showConflictSelection";
const char* AUTHV4_FUNC_NAME_SHOWINQUIRY = "showInquiry";
const char* AUTHV4_FUNC_NAME_SHOWCHATBOTINQUIRY = "showChatbotInquiry";
const char* AUTHV4_FUNC_NAME_SHOWMYINQUIRY = "showMyInquiry";
const char* AUTHV4_FUNC_NAME_SHOWTERMS = "showTerms";
const char* AUTHV4_FUNC_NAME_SHOWADULTCONFIRM = "showAdultConfirm";
const char* AUTHV4_FUNC_NAME_GETPROFILE = "getProfile";
const char* AUTHV4_FUNC_NAME_SHOWPROFILE = "showProfile";
const char* AUTHV4_FUNC_NAME_RESET_AGREEMENT = "resetAgreement";
const char* AUTHV4_FUNC_NAME_ISAUTOSIGNIN = "isAutoSignIn";
const char* AUTHV4_FUNC_NAME_GETPLAYERINFO = "getPlayerInfo";
const char* AUTHV4_FUNC_NAME_GETPROVIDERFRIENDSLIST = "getProviderFriendsList";
const char* AUTHV4_FUNC_NAME_RESOLVECONFLICT = "resolveConflict";
const char* AUTHV4_FUNC_NAME_SHOWGAMECENTERLOGINCANCELDIALOG = "showGameCenterLoginCancelDialog";
const char* AUTHV4_FUNC_NAME_GETAGEGATEU13 = "getAgeGateU13";
const char* AUTHV4_FUNC_NAME_REQUESTPERMISSIONVIEWDATA = "requestPermissionViewData";
const char* AUTHV4_FUNC_NAME_SHOWDEVICEMANAGEMENT = "showDeviceManagement";
const char* AUTHV4_FUNC_NAME_GET_HIVETALKPLUS_LOGIN_TOKEN = "getHiveTalkPlusLoginToken";
const char* AUTHV4_FUNC_NAME_REFRESH_ACCESSTOKEN = "refreshAccessToken";
const char* AUTHV4_FUNC_NAME_TERMINATEPROCESS = "terminateProcess";
const char* AUTHV4_FUNC_NAME_SIGNINWITHSTOREDPLAYERID = "signInWithStoredPlayerId";
const char* AUTHV4_FUNC_NAME_GETSTOREDPLAYERIDLIST = "getStoredPlayerIdList";
const char* AUTHV4_FUNC_NAME_STORECURRENTPLAYERID = "storeCurrentPlayerId";
const char* AUTHV4_FUNC_NAME_DELETESTOREDPLAYERID = "deleteStoredPlayerId";
const char* AUTHV4_FUNC_NAME_GETPARENTALCONSENTINFO = "getParentalConsentInfo";
const char* AUTHV4_FUNC_NAME_GETHASHEDDI = "getHashedDi";
const char* AUTHV4_FUNC_NAME_SHOWIDENTITYVERIFICATION = "showIdentityVerification";
const char* AUTHV4_FUNC_NAME_GETAGERANGE = "getAgeRange";
const char* AUTHV4_FUNC_NAME_SHOWAGERANGEUPDATEPERMISSION = "showAgeRangeUpdatePermission";
const char* AUTHV4_FUNC_NAME_CHECKAGEGATE = "checkAgeGate";
const char* AUTHV4_FUNC_NAME_SETAUTOLOGINENABLED = "setAutoLoginEnabled";

const char* AUTHV4HELPER_FUNC_NAME_SYNCACCOUNT = "syncAccount";
const char* AUTHV4HELPER_FUNC_NAME_SIGNIN = "signIn";
const char* AUTHV4HELPER_FUNC_NAME_SIGNOUT = "signOut";
const char* AUTHV4HELPER_FUNC_NAME_PLAYERDELETE = "playerDelete";
const char* AUTHV4HELPER_FUNC_NAME_CONNECT = "connect";
const char* AUTHV4HELPER_FUNC_NAME_DISCONNECT = "disconnect";
const char* AUTHV4HELPER_FUNC_NAME_SHOWACHIEVEMENTS = "showAchievements";
const char* AUTHV4HELPER_FUNC_NAME_SHOWLEADERBOARD = "showLeaderboard";
const char* AUTHV4HELPER_FUNC_NAME_ACHIEVEMENTSREVEAL = "achievementsReveal";
const char* AUTHV4HELPER_FUNC_NAME_ACHIEVEMENTSUNLOCK = "achievementsUnlock";
const char* AUTHV4HELPER_FUNC_NAME_ACHIEVEMENTSINCREMENT = "achievementsIncrement";
const char* AUTHV4HELPER_FUNC_NAME_LEADERBOARDS_SUBMITSCORE = "leaderboardsSubmitScore";
const char* AUTHV4HELPER_FUNC_NAME_SHOWCONFLICT = "showConflict";
const char* AUTHV4HELPER_FUNC_NAME_SWITCHACCOUNT = "switchAccount";
const char* AUTHV4HELPER_FUNC_NAME_RESOLVECONFLICT = "resolveConflict";
const char* AUTHV4HELPER_FUNC_NAME_SHOWGAMECENTERLOGINCANCELDIALOG = "showGameCenterLoginCancelDialog";
const char* AUTHV4HELPER_FUNC_NAME_GETIDPLIST = "getIDPList";


static picojson::value callNativeFunction(const char* className,const char* funcName,picojson::object params) {
    
    picojson::object jsonParam = HiveCppPlugin::createParam(className, funcName);
    
    if(params.size() > 0) {
        std::for_each(params.begin(), params.end(), [&jsonParam](picojson::object::value_type it){
            jsonParam[it.first] = it.second;
        });
    }
    return HiveCppPlugin::callNative(jsonParam);
}

static picojson::value callNativeFunction(const char* className,const char* funcName,std::string params) {
    
    picojson::object jsonParam = HiveCppPlugin::createParam(className, funcName);
    
    if(params.size()>0){
        std::string addingParam = "";
        if(jsonParam.size() > 0) {
            std::for_each(jsonParam.begin(), jsonParam.end(), [&addingParam](picojson::object::value_type it){
                addingParam += std::string(",")+"\""+it.first+"\":";
                if(it.second.is<std::string>()){
                    addingParam += "\""+it.second.get<std::string>()+"\"";
                }else if(it.second.is<bool>()){
                    addingParam += it.second.get<double>() ? "true":"false";
                }else if(it.second.is<double>()){
                    addingParam += to_string(it.second.get<double>());
                }
            });
        }
        
        size_t find = params.rfind("}");
        params.insert(find, addingParam);
        printf("****** conflict parms : %s",params.c_str());
        return HiveCppPlugin::callNative(params);
    }else {
        return HiveCppPlugin::callNative(jsonParam);
    }
}

static picojson::value callNativeAuthV4(const char* funcName,picojson::object params) {
    return callNativeFunction("AuthV4", funcName, params);
}
static picojson::value callNativeAuthV4(const char* funcName,std::string params) {
    return callNativeFunction("AuthV4", funcName, params);
}
static picojson::value callNativeAuthV4Helper(const char* funcName,picojson::object params) {
    return callNativeFunction("AuthV4Helper", funcName, params);
}
static picojson::value callNativeAuthV4Helper(const char* funcName,std::string params) {
    return callNativeFunction("AuthV4Helper", funcName, params);
}

static picojson::value callNativeAuthV4(const char* funcName) {
    return callNativeAuthV4(funcName, picojson::object());
}

static picojson::value callNativeAuthV4Helper(const char* funcName) {
    return callNativeAuthV4Helper(funcName, picojson::object());
}


void AuthV4::setup(onSetup setupListener){
    g_Authv4SetupListener = setupListener;
    callNativeAuthV4(AUTHV4_FUNC_NAME_SETUP);
}
void AuthV4::signIn(ProviderType providerType, onSignIn signInListener){
    g_Authv4SignInListener = signInListener;
    picojson::object jsonParam;
    jsonParam["providerType"] = picojson::value(ProviderInfo::stringProviderType(providerType));
    callNativeAuthV4(AUTHV4_FUNC_NAME_SIGNIN,jsonParam);
}
void AuthV4::signInWithAuthKey(std::string authKey, onSignIn signInListener) {
    g_Authv4SignInListener = signInListener;
    picojson::object jsonParam;
    jsonParam["authKey"] = picojson::value(authKey);
    callNativeAuthV4(AUTHV4_FUNC_NAME_SIGNIN_WITH_AUTHKEY, jsonParam);
}
void AuthV4::signOut(onSignOut signOutListener){
    g_Authv4SignOutListener = signOutListener;
    callNativeAuthV4(AUTHV4_FUNC_NAME_SIGNOUT);
}
void AuthV4::playerDelete(onSignOut playerDeleteListener){
    g_Authv4PlayerDeleteListener = playerDeleteListener;
    callNativeAuthV4(AUTHV4_FUNC_NAME_PLAYERDELETE);
}
void AuthV4::connect(ProviderType providerType, onConnect connectListener){
    g_Authv4ConnectListener = connectListener;
    picojson::object jsonParam;
    jsonParam["providerType"] = picojson::value(ProviderInfo::stringProviderType(providerType));
    callNativeAuthV4(AUTHV4_FUNC_NAME_CONNECT,jsonParam);
}
void AuthV4::connectWithAuthKey(std::string authKey, onConnect connectListener){
    g_Authv4ConnectListener = connectListener;
    picojson::object jsonParam;
    jsonParam["authKey"] = picojson::value(authKey);
    callNativeAuthV4(AUTHV4_FUNC_NAME_CONNECT_WITH_AUTHKEY,jsonParam);
}
void AuthV4::disconnect(ProviderType providerType, onDisconnect disconnectListener){
    g_Authv4DisconnectListener = disconnectListener;
    picojson::object jsonParam;
    jsonParam["providerType"] = picojson::value(ProviderInfo::stringProviderType(providerType));
    callNativeAuthV4(AUTHV4_FUNC_NAME_DISCONNECT,jsonParam);
}
void AuthV4::disconnectWithName(std::string providerName, onDisconnect disconnectListener){
    g_Authv4DisconnectListener = disconnectListener;
    picojson::object jsonParam;
    jsonParam["providerName"] = picojson::value(providerName);
    callNativeAuthV4(AUTHV4_FUNC_NAME_DISCONNECT_WITH_NAME,jsonParam);
}
void AuthV4::selectConflict(PlayerID selectedPlayerId, onSignIn signInListener){
    g_Authv4SelectSignInListener = signInListener;
    picojson::object jsonParam;
    //jsonParam["selectedPlayerId"] = picojson::value(to_string(selectedPlayerId));
    jsonParam["selectedPlayerId"] = picojson::value(static_cast<double>(selectedPlayerId));  //  Windows Plugin use long long type same as Unity.
    callNativeAuthV4(AUTHV4_FUNC_NAME_SELECTCONFLICT,jsonParam);
}
void AuthV4::checkProvider(ProviderType providerType, onCheckProvider checkProviderListener){
    g_Authv4CheckProviderListener = checkProviderListener;
    picojson::object jsonParam;
    jsonParam["providerType"] = picojson::value(ProviderInfo::stringProviderType(providerType));
    callNativeAuthV4(AUTHV4_FUNC_NAME_CHECKPROVIDER,jsonParam);
}
void AuthV4::checkMaintenance(bool isShow, onMaintenance maintenanceListener){
    g_Authv4CheckMaintenanceListener = maintenanceListener;
    picojson::object jsonParam;
    jsonParam["isShow"] = picojson::value(isShow);
    callNativeAuthV4(AUTHV4_FUNC_NAME_CHECKMAINTENANCE,jsonParam);
}
void AuthV4::checkBlacklist(bool isShow, onMaintenance maintenanceListener){
    g_Authv4CheckBlackListMaintenanceListener = maintenanceListener;
    picojson::object jsonParam;
    jsonParam["isShow"] = picojson::value(isShow);
    callNativeAuthV4(AUTHV4_FUNC_NAME_CHECKBLACKLIST,jsonParam);
}
void AuthV4::setProviderChangedListener(onCheckProvider checkProviderLisener){
    g_Authv4SetProviderChangedHandler = checkProviderLisener;
    callNativeAuthV4(AUTHV4_FUNC_NAME_SETPROVIDERCHANGEDLISTENER);
}
void AuthV4::showSignIn(onSignIn signInListener){
    g_Authv4ShowSignInListener = signInListener;
    callNativeAuthV4(AUTHV4_FUNC_NAME_SHOWSIGNIN);
}

void AuthV4::showConflictSelection(IConflictViewInfo const & viewInfo,onSignIn signInListener){
    g_Authv4ConflictSelectionSignInListener = signInListener;
    
    std::string json = "{";
    if(viewInfo.toJsonCurrent() != "")
        json += "\"currentPlayerData\":"+viewInfo.toJsonCurrent()+",";
    if(viewInfo.toJsonConflict() != "")
        json += "\"conflictPlayerData\":"+viewInfo.toJsonConflict();
    json += "}";
    callNativeAuthV4(AUTHV4_FUNC_NAME_SHOWCONFLICTSELECTION,json);
}

void AuthV4::showInquiry(onShowInquiry showInquiryListener){
    g_AuthV4ShowInquiryListener = showInquiryListener;
    callNativeAuthV4(AUTHV4_FUNC_NAME_SHOWINQUIRY);
}

void AuthV4::showChatbotInquiry(std::string additionalInfo, onShowChatbotInquiry showChatbotInquiryListener) {
    g_AuthV4ShowChatbotInquiryListener = showChatbotInquiryListener;
    picojson::object jsonParam;
    jsonParam["additionalInfo"] = picojson::value(additionalInfo);
    callNativeAuthV4(AUTHV4_FUNC_NAME_SHOWCHATBOTINQUIRY, jsonParam);
}

void AuthV4::showMyInquiry(onShowMyInquiry showInquiryListener) {
    g_AuthV4ShowMyInquiryListener = showInquiryListener;
    callNativeAuthV4(AUTHV4_FUNC_NAME_SHOWMYINQUIRY);
}
void AuthV4::showTerms(onShowTerms showTermsListener){
    g_Authv4ShowTermsListener = showTermsListener;
    callNativeAuthV4(AUTHV4_FUNC_NAME_SHOWTERMS);
}
void AuthV4::showAdultConfirm(onAdultConfirm adultConfirmListener){
    g_Authv4ShowAdultConfirmListener = adultConfirmListener;
    callNativeAuthV4(AUTHV4_FUNC_NAME_SHOWADULTCONFIRM);
}
void AuthV4::getProfile(std::vector<PlayerID> playerIdList, onGetProfile getProfileListener){
    g_Authv4GetProfileListener = getProfileListener;
    picojson::object jsonParam;
    picojson::array jsonArray;
    for (std::vector<PlayerID>::iterator it = playerIdList.begin(); it != playerIdList.end(); ++it) {
        jsonArray.push_back(picojson::value(*it));      //  unity use int64 instead of string.
    }
    jsonParam["playerIdList"] = picojson::value(jsonArray);
    
    callNativeAuthV4(AUTHV4_FUNC_NAME_GETPROFILE,jsonParam);
}
void AuthV4::showProfile(PlayerID playerId, onShowProfile showProfileListener){
    g_Authv4ShowProfileListener = showProfileListener;
    picojson::object jsonParam;
    //jsonParam["playerId"] = picojson::value(to_string(playerId));
    jsonParam["playerId"] = picojson::value(static_cast<double>(playerId));   //  Windows Plugin use long long type same as Unity.
    callNativeAuthV4(AUTHV4_FUNC_NAME_SHOWPROFILE,jsonParam);
}
void AuthV4::resetAgreement(){
    callNativeAuthV4(AUTHV4_FUNC_NAME_RESET_AGREEMENT);
}
bool AuthV4::isAutoSignIn(){
    picojson::value resJson = callNativeAuthV4(AUTHV4_FUNC_NAME_ISAUTOSIGNIN);
    bool isReadyResult = false;
    picojson::value jsonValue = resJson.get("isAutoSignIn");
    if(jsonValue.is<picojson::null>() == false)
        isReadyResult = jsonValue.evaluate_as_boolean();
    
    return isReadyResult;
}
PlayerInfo AuthV4::getPlayerInfo(){
    picojson::value resJson = callNativeAuthV4(AUTHV4_FUNC_NAME_GETPLAYERINFO);
    PlayerInfo playerInfo(resJson.get("getPlayerInfo"));
    return playerInfo;
}

ParentalConsentInfo AuthV4::getParentalConsentInfo(){
    picojson::value resJson = callNativeAuthV4(AUTHV4_FUNC_NAME_GETPARENTALCONSENTINFO);
    ParentalConsentInfo parentalConsentInfo(resJson.get(AUTHV4_FUNC_NAME_GETPARENTALCONSENTINFO));
    return parentalConsentInfo;
}

void AuthV4::getProviderFriendsList(ProviderType providerType, onGetProviderFriendsList getProviderFriendsListListener){
    g_AuthV4GetProviderFriendsListListener = getProviderFriendsListListener;
    picojson::object jsonParam;
    jsonParam["providerType"] = picojson::value(ProviderInfo::stringProviderType(providerType));
    callNativeAuthV4(AUTHV4_FUNC_NAME_GETPROVIDERFRIENDSLIST,jsonParam);
}

void AuthV4::resolveConflict(onResolveConflict resolveConflictListener) {
    g_AuthV4ResolveConflict = resolveConflictListener;
    callNativeAuthV4(AUTHV4_FUNC_NAME_RESOLVECONFLICT);
}

//4.7.0_ADDED
void AuthV4::showGameCenterLoginCancelDialog(onAuthV4DialogDismiss listener){
    g_AuthV4DialogDismiss = listener;
    callNativeAuthV4(AUTHV4_FUNC_NAME_SHOWGAMECENTERLOGINCANCELDIALOG);
}

bool AuthV4::getAgeGateU13() {
    picojson::value resJson = callNativeAuthV4(AUTHV4_FUNC_NAME_GETAGEGATEU13);
    
    picojson::value jsonValue = resJson.get("ageGateU13");
    if(jsonValue.is<picojson::null>() == false) {
        return jsonValue.evaluate_as_boolean();
    } else {
        return false;
    }
}

void AuthV4::requestPermissionViewData(onAuthV4RequestPermissionViewData listener) {
    g_onAuthV4RequestPermissionViewData = listener;
    callNativeAuthV4(AUTHV4_FUNC_NAME_REQUESTPERMISSIONVIEWDATA);
}

void AuthV4::showDeviceManagement(onShowDeviceManagement listener) {
    g_AuthV4ShowDeviceManagementListener = listener;
    callNativeAuthV4(AUTHV4_FUNC_NAME_SHOWDEVICEMANAGEMENT);
}

void AuthV4::getHiveTalkPlusLoginToken(onGetHiveTalkPlusLoginToken listener) {
    g_AuthV4GetHiveTalkPlusLoginTokenListener = listener;
    callNativeAuthV4(AUTHV4_FUNC_NAME_GET_HIVETALKPLUS_LOGIN_TOKEN);
}

void AuthV4::refreshAccessToken(onRefreshAccessToken listener) {
    g_AuthV4RefreshAccessTokenListener = listener;
    callNativeAuthV4(AUTHV4_FUNC_NAME_REFRESH_ACCESSTOKEN);
}

void AuthV4::terminateProcess() {
    // used by PC version.
    callNativeAuthV4(AUTHV4_FUNC_NAME_TERMINATEPROCESS);
}

void AuthV4::signInWithStoredPlayerId(PlayerID playerId, bool useAutoSignIn, onSignIn signInListener) {
    g_AuthV4SignInWithStoredPlayerIdListener = signInListener;
    picojson::object jsonParam;
    jsonParam["playerId"] = picojson::value(static_cast<double>(playerId));
    jsonParam["useAutoSignIn"] = picojson::value(useAutoSignIn);
    callNativeAuthV4(AUTHV4_FUNC_NAME_SIGNINWITHSTOREDPLAYERID, jsonParam);
}

void AuthV4::getStoredPlayerIdList(onPlayerIdList playerIdListListener) {
    g_AuthV4GetStoredPlayerIdListListener = playerIdListListener;
    callNativeAuthV4(AUTHV4_FUNC_NAME_GETSTOREDPLAYERIDLIST);
}

void AuthV4::storeCurrentPlayerId(onStorePlayerId storeListener) {
    g_AuthV4StoreCurrentPlayerIdListener = storeListener;
    callNativeAuthV4(AUTHV4_FUNC_NAME_STORECURRENTPLAYERID);
}

void AuthV4::deleteStoredPlayerId(PlayerID playerId, onDeletePlayerId deleteListener) {
    g_AuthV4DeleteStoredPlayerIdListener = deleteListener;
    picojson::object jsonParam;
    jsonParam["playerId"] = picojson::value(static_cast<double>(playerId));
    callNativeAuthV4(AUTHV4_FUNC_NAME_DELETESTOREDPLAYERID, jsonParam);
}

void AuthV4::getHashedDi(onHashedDi listener) {
    g_AuthV4GetHashedDiListener = listener;
    callNativeAuthV4(AUTHV4_FUNC_NAME_GETHASHEDDI);
}

void AuthV4::showIdentityVerification(onIdentityVerification listener) {
    g_AuthV4ShowIdentityVerificationListener = listener;
    callNativeAuthV4(AUTHV4_FUNC_NAME_SHOWIDENTITYVERIFICATION);
} 

void AuthV4::getAgeRange(onGetAgeRange listener) {
    g_AuthV4GetAgeRangeListener = listener;
    callNativeAuthV4(AUTHV4_FUNC_NAME_GETAGERANGE);
}

void AuthV4::showAgeRangeUpdatePermission(std::string description, onGetAgeRange listener) {
    g_AuthV4ShowAgeRangeUpdateListener = listener;
    picojson::object jsonParam;
    jsonParam["description"] = picojson::value(description);
    callNativeAuthV4(AUTHV4_FUNC_NAME_SHOWAGERANGEUPDATEPERMISSION, jsonParam);
}

void AuthV4::checkAgeGate(bool useCloseButton, onCheckAgeGate listener) {
    g_AuthV4OnCheckAgeGateListener = listener;
    picojson::object jsonParam;
    jsonParam["useCloseButton"] = picojson::value(useCloseButton);
    callNativeAuthV4(AUTHV4_FUNC_NAME_CHECKAGEGATE, jsonParam);
}

void AuthV4::setAutoLoginEnabled(bool enabled, onAutoLoginEnabled listener) {
    g_AuthV4SetAutoLoginEnabledListener = listener;
    picojson::object jsonParam;
    jsonParam["enabled"] = picojson::value(enabled);
    callNativeAuthV4(AUTHV4_FUNC_NAME_SETAUTOLOGINENABLED, jsonParam);
}

void AuthV4::executeEngine(picojson::value jsonParam) {
    ResultAPI result(jsonParam.get("resultAPI"));

    picojson::value jsonMethodValue;
    jsonMethodValue = jsonParam.get("method");
    if(jsonMethodValue.is<picojson::null>()) {
        return;
    }
    const bool isResultSuccess = result.isSuccess();
    
    std::string methodName = jsonMethodValue.to_str();
    
    if(methodName.compare(AUTHV4_FUNC_NAME_SETUP) == 0){
        if(g_Authv4SetupListener != nullptr) {
            
            bool isAuthSignIn = false;
            std::string did = "";
            std::vector<ProviderType> providerTypeList;
            
            if(isResultSuccess) {
                isAuthSignIn = jsonParam.get("isAutoSignIn").get<bool>();
                did = jsonParam.get("did").to_str();
				
				if(jsonParam.contains("providerTypeList") && !jsonParam.get("providerTypeList").is<picojson::null>()){
					picojson::array providerJsonArray = jsonParam.get("providerTypeList").get<picojson::array>();
					for(auto it = providerJsonArray.begin();it != providerJsonArray.end();++it) {
						auto provider = (*it).to_str();
						providerTypeList.push_back(ProviderInfo::providerTypeFromString(provider));
					}
				}
            }
            
            g_Authv4SetupListener(result,isAuthSignIn,did,providerTypeList);
        }
    }
    else if(methodName.compare(AUTHV4_FUNC_NAME_SIGNIN) == 0) {
        if(g_Authv4SignInListener != nullptr){
            PlayerInfo info;
            if(isResultSuccess) {
                info =PlayerInfo(jsonParam.get("playerInfo"));
            }
            g_Authv4SignInListener(result,info);
            
        }
    }
    else if(methodName.compare(AUTHV4_FUNC_NAME_SIGNIN_WITH_AUTHKEY) == 0) {
        if (g_Authv4SignInListener != nullptr) {
            PlayerInfo info;
            if (isResultSuccess) {
                info = PlayerInfo(jsonParam.get("playerInfo"));
            }
            g_Authv4SignInListener(result, info);
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_SIGNOUT) == 0) {
        if(g_Authv4SignOutListener != nullptr) {
            g_Authv4SignOutListener(result);
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_PLAYERDELETE) == 0) {
        if(g_Authv4PlayerDeleteListener != nullptr) {
            g_Authv4PlayerDeleteListener(result);
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_CONNECT) == 0) {
        if(g_Authv4ConnectListener != nullptr) {
            PlayerInfo info;
            bool bConflict = false;
            bConflict = jsonParam.get("conflictPlayer").is<picojson::null>() == false;
            if(bConflict)
                info = PlayerInfo(jsonParam.get("conflictPlayer"));
            g_Authv4ConnectListener(result,info);
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_DISCONNECT) == 0) {
        if(g_Authv4DisconnectListener != nullptr) {
            g_Authv4DisconnectListener(result);
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_CONNECT_WITH_AUTHKEY) == 0) {
        if(g_Authv4ConnectListener != nullptr) {
            PlayerInfo info;
            bool bConflict = false;
            bConflict = jsonParam.get("conflictPlayer").is<picojson::null>() == false;
            if(bConflict)
                info = PlayerInfo(jsonParam.get("conflictPlayer"));
            g_Authv4ConnectListener(result,info);
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_DISCONNECT_WITH_NAME) == 0) {
        if(g_Authv4DisconnectListener != nullptr) {
            g_Authv4DisconnectListener(result);
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_SELECTCONFLICT) == 0) {
        if(g_Authv4SelectSignInListener != nullptr) {
            PlayerInfo info;
            if(isResultSuccess)
                info = PlayerInfo(jsonParam.get("playerInfo"));
            g_Authv4SelectSignInListener(result, info);
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_CHECKPROVIDER) == 0) {
        if(g_Authv4CheckProviderListener != nullptr) {
            ProviderInfo info;
            if(isResultSuccess)
                info = ProviderInfo(jsonParam.get("providerInfo"));
            g_Authv4CheckProviderListener(result,info);
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_CHECKMAINTENANCE) == 0) {
        if(g_Authv4CheckMaintenanceListener != nullptr) {
            std::vector<AuthV4MaintenanceInfo> infos;
            if(isResultSuccess){
                picojson::array array = jsonParam.get("AuthV4MaintenanceInfoList").get<picojson::array>();
                for(auto it=array.begin();it!=array.end();++it) {
                    AuthV4MaintenanceInfo info((*it));
                    infos.push_back(info);
                }
            }
            
            g_Authv4CheckMaintenanceListener(result,infos);
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_CHECKBLACKLIST) == 0) {
        if(g_Authv4CheckBlackListMaintenanceListener != nullptr) {
            std::vector<AuthV4MaintenanceInfo> infos;
            if(isResultSuccess) {
                picojson::array array = jsonParam.get("AuthV4MaintenanceInfoList").get<picojson::array>();
                for(auto it=array.begin();it!=array.end();++it) {
                    AuthV4MaintenanceInfo info((*it));
                    infos.push_back(info);
                }
            }
            g_Authv4CheckBlackListMaintenanceListener(result,infos);
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_SETPROVIDERCHANGEDLISTENER) == 0) {
        if(g_Authv4SetProviderChangedHandler != nullptr) {
            ProviderInfo info;
            if(isResultSuccess)
                info = ProviderInfo(jsonParam.get("providerInfo"));
            g_Authv4SetProviderChangedHandler(result,info);
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_SHOWSIGNIN) == 0) {
        if(g_Authv4ShowSignInListener != nullptr) {
            PlayerInfo info;
            if(isResultSuccess)
                info = PlayerInfo(jsonParam.get("playerInfo"));
            g_Authv4ShowSignInListener(result,info);
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_SHOWCONFLICTSELECTION) == 0) {
        if(g_Authv4ConflictSelectionSignInListener != nullptr) {
            PlayerInfo info;
            if(isResultSuccess)
                info = PlayerInfo(jsonParam.get("playerInfo"));
            g_Authv4ConflictSelectionSignInListener(result,info);
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_SHOWINQUIRY) == 0) {
        if(g_AuthV4ShowInquiryListener != nullptr) {
            g_AuthV4ShowInquiryListener(result);
        }
    }
    else if(methodName.compare(AUTHV4_FUNC_NAME_SHOWCHATBOTINQUIRY) == 0) {
        if(g_AuthV4ShowChatbotInquiryListener != nullptr) {
            g_AuthV4ShowChatbotInquiryListener(result);
        }
        
    } else if(methodName.compare(AUTHV4_FUNC_NAME_SHOWMYINQUIRY) == 0) {
        if(g_AuthV4ShowMyInquiryListener != nullptr) {
            g_AuthV4ShowMyInquiryListener(result);
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_SHOWTERMS) == 0) {
        if(g_Authv4ShowTermsListener != nullptr) {
            g_Authv4ShowTermsListener(result);
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_SHOWADULTCONFIRM) == 0) {
        if(g_Authv4ShowAdultConfirmListener != nullptr) {
            g_Authv4ShowAdultConfirmListener(result);
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_GETPROFILE) == 0) {
        if(g_Authv4GetProfileListener != nullptr) {
            std::vector<ProfileInfo> infos;
            if(isResultSuccess){
                picojson::array array = jsonParam.get("profileInfoList").get<picojson::array>();
                for(auto it=array.begin();it!=array.end();++it) {
                    ProfileInfo info((*it));
                    infos.push_back(info);
                }
            }
            g_Authv4GetProfileListener(result,infos);
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_SHOWPROFILE) == 0) {
        if(g_Authv4ShowProfileListener != nullptr) {
            g_Authv4ShowProfileListener(result);
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_GETPROVIDERFRIENDSLIST) == 0) {
        if(g_AuthV4GetProviderFriendsListListener != nullptr) {
            std::map<std::string,PlayerID> providerUserIdList;
            ProviderType providerType = ProviderType::AUTO;
            
            if(!jsonParam.get("providerType").is<picojson::null>()){
                providerType = ProviderInfo::providerTypeFromString(jsonParam.get("providerType").get<std::string>());
            }
            
            if(!jsonParam.get("providerUserIdList").is<picojson::null>()){
                auto providerUserListJson = jsonParam.get("providerUserIdList").get<picojson::object>();
                for(auto it = providerUserListJson.begin();it != providerUserListJson.end();++it){
                    providerUserIdList.insert(std::make_pair(it->first, static_cast<PlayerID>(it->second.get<double>())));
                }
            }
            g_AuthV4GetProviderFriendsListListener(result,providerType,providerUserIdList);
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_RESOLVECONFLICT) == 0){
        if(g_AuthV4ResolveConflict != nullptr){
            g_AuthV4ResolveConflict(result);
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_SHOWGAMECENTERLOGINCANCELDIALOG) == 0){
        if(g_AuthV4DialogDismiss != nullptr){
            bool isDismiss = jsonParam.get("isDismiss").get<bool>();
            g_AuthV4DialogDismiss(isDismiss);
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_REQUESTPERMISSIONVIEWDATA) == 0){
        if(g_onAuthV4RequestPermissionViewData != NULL){
            g_onAuthV4RequestPermissionViewData(result,PermissionViewData(jsonParam.get("data")));
        }
    }else if(methodName.compare(AUTHV4_FUNC_NAME_SHOWDEVICEMANAGEMENT) == 0) {
        if(g_AuthV4ShowDeviceManagementListener != nullptr) {
            g_AuthV4ShowDeviceManagementListener(result);
        }
    }else if (methodName.compare(AUTHV4_FUNC_NAME_GET_HIVETALKPLUS_LOGIN_TOKEN) == 0) {
        if (g_AuthV4GetHiveTalkPlusLoginTokenListener != nullptr) {
            std::string loginToken = jsonParam.get("loginToken").get<string>();
            g_AuthV4GetHiveTalkPlusLoginTokenListener(result, loginToken);
        }
    }else if (methodName.compare(AUTHV4_FUNC_NAME_REFRESH_ACCESSTOKEN) == 0) {
        if (g_AuthV4RefreshAccessTokenListener != nullptr) {
            std::string accessToken = jsonParam.get("accessToken").get<string>();
            g_AuthV4RefreshAccessTokenListener(result, accessToken.c_str());
        }
    }else if (methodName.compare(AUTHV4_FUNC_NAME_SIGNINWITHSTOREDPLAYERID) == 0) {
        if (g_AuthV4SignInWithStoredPlayerIdListener != nullptr) {
            PlayerInfo info;
            if (isResultSuccess) {
                info = PlayerInfo(jsonParam.get("playerInfo"));
            }
            g_AuthV4SignInWithStoredPlayerIdListener(result, info);
        }
    }else if (methodName.compare(AUTHV4_FUNC_NAME_GETSTOREDPLAYERIDLIST) == 0) {
        if (g_AuthV4GetStoredPlayerIdListListener != nullptr) {
            std::vector<PlayerID> playerIdList;
            picojson::value playerIdListValue = jsonParam.get("playerIdList");
            if (!playerIdListValue.is<picojson::null>()) {
                picojson::array jsonArray = playerIdListValue.get<picojson::array>();
                for (auto& item : jsonArray) {
                    if (item.is<double>()) {
                        playerIdList.push_back(static_cast<PlayerID>(item.get<double>()));
                    }
                }
            }
            g_AuthV4GetStoredPlayerIdListListener(result, playerIdList);
        }
    }else if (methodName.compare(AUTHV4_FUNC_NAME_STORECURRENTPLAYERID) == 0) {
        if (g_AuthV4StoreCurrentPlayerIdListener != nullptr) {

            g_AuthV4StoreCurrentPlayerIdListener(result);
        }
    }else if (methodName.compare(AUTHV4_FUNC_NAME_DELETESTOREDPLAYERID) == 0) {
        if (g_AuthV4DeleteStoredPlayerIdListener != nullptr) {

            g_AuthV4DeleteStoredPlayerIdListener(result);
        }
    }
    else if (methodName.compare(AUTHV4_FUNC_NAME_GETHASHEDDI) == 0) {
        if (g_AuthV4GetHashedDiListener != nullptr) {
            std::string hashedDi = jsonParam.get("hashedDi").get<string>();
            g_AuthV4GetHashedDiListener(result, hashedDi.c_str());  
        }        
    }
    else if (methodName.compare(AUTHV4_FUNC_NAME_CHECKAGEGATE) == 0) {
        if (g_AuthV4OnCheckAgeGateListener != nullptr) {
            std::string dateOfBirth;
            picojson::value dateOfBirthValue = jsonParam.get("dateOfBirth");
            if (!dateOfBirthValue.is<picojson::null>()) {
                dateOfBirth = dateOfBirthValue.get<std::string>();
            }
            g_AuthV4OnCheckAgeGateListener(result, dateOfBirth);
        }
    } 
    else if (methodName.compare(AUTHV4_FUNC_NAME_SHOWIDENTITYVERIFICATION) == 0) {
        if (g_AuthV4ShowIdentityVerificationListener != nullptr) {
            Identity identity = Identity(jsonParam.get("identity"));
            g_AuthV4ShowIdentityVerificationListener(result, identity);
        }
    }
    else if (methodName.compare(AUTHV4_FUNC_NAME_GETAGERANGE) == 0) {
        if (g_AuthV4GetAgeRangeListener != nullptr) {
            AgeRange ageRange = AgeRange(jsonParam.get("ageRange"));
            g_AuthV4GetAgeRangeListener(result, ageRange);
        }
    }
    else if (methodName.compare(AUTHV4_FUNC_NAME_SHOWAGERANGEUPDATEPERMISSION) == 0) {
            if (g_AuthV4ShowAgeRangeUpdateListener != nullptr) {
                AgeRange ageRange = AgeRange(jsonParam.get("ageRange"));
                g_AuthV4ShowAgeRangeUpdateListener(result, ageRange);
            }
    }
    else if (methodName.compare(AUTHV4_FUNC_NAME_SETAUTOLOGINENABLED) == 0) {
        if (g_AuthV4SetAutoLoginEnabledListener != nullptr) {
            bool enabled = jsonParam.get("enabled").get<bool>();
            g_AuthV4SetAutoLoginEnabledListener(result, enabled);
        }
    }
}

ConflictSingleViewInfo::ConflictSingleViewInfo(PlayerID playerId)
{
    std::string strPlayerId = to_string(playerId);
    jsonData = "{\"player_id\":";
    jsonData.append(strPlayerId);
    jsonData.append(",\"game_data\":{ ");
}

ConflictSingleViewInfo::~ConflictSingleViewInfo(){
    
}

void ConflictSingleViewInfo::SetValue(const std::string key,const long long playerData) {
    jsonData += "\""+key+"\":"+to_string(playerData)+",";
}
void ConflictSingleViewInfo::SetValue(const std::string key,const double playerData){
    jsonData += "\""+key+"\":"+to_string(playerData)+",";
}
void ConflictSingleViewInfo::SetValue(const std::string key,const std::string playerData) {
    jsonData += "\""+key+"\":\""+playerData+"\""+",";
}
std::string ConflictSingleViewInfo::toJsonConflict() const {
    std::string tempString = jsonData;
    if( tempString[tempString.length() -1 ] == ',' ) //
    {
        tempString.erase( tempString.length()-1, 1 );
    }
    tempString.append("}}");
    return tempString;
}
std::string ConflictSingleViewInfo::toJsonCurrent() const {
    return "";
}

ConflictViewInfo::ConflictViewInfo(PlayerID currentPlayerId,PlayerID conflictPlayerId)
{
    std::string strCurrentID = to_string(currentPlayerId);      // prevent ios 8 Crash on Unreal Engine 4.15. memory allocate fail crash.
    jsonCurrent = "{\"player_id\":";
    jsonCurrent.append(strCurrentID);
    jsonCurrent.append(",\"game_data\":{ ");

    std::string strConflictID = to_string(conflictPlayerId);
    jsonConflict = "{\"player_id\":";
    jsonConflict.append(strConflictID);
    jsonConflict.append(",\"game_data\":{ ");
}
ConflictViewInfo::~ConflictViewInfo() {
    
}
void ConflictViewInfo::SetValue(const std::string key,const long long currentPlayerData,const long long conflictPlayerData) {
    jsonCurrent += "\""+key+"\":"+to_string(currentPlayerData)+",";
    jsonConflict += "\""+key+"\":"+to_string(conflictPlayerData)+",";
}
void ConflictViewInfo::SetValue(const std::string key,const double currentPlayerData,const double conflictPlayerData) {
    jsonCurrent += "\""+key+"\":"+to_string(currentPlayerData)+",";
    jsonConflict += "\""+key+"\":"+to_string(conflictPlayerData)+",";
}
void ConflictViewInfo::SetValue(const std::string key,const std::string currentPlayerData,const std::string conflictPlayerData) {
    jsonCurrent += "\""+key+"\":\""+currentPlayerData+"\""+",";
    jsonConflict += "\""+key+"\":\""+conflictPlayerData+"\""+",";
}

std::string ConflictViewInfo::toJsonConflict() const {
    std::string tempString = jsonConflict;
    if( tempString[tempString.length() -1 ] == ',' ) //
    {
        tempString.erase( tempString.length()-1, 1 );
    }
    tempString.append("}}");
    return tempString;
}
std::string ConflictViewInfo::toJsonCurrent() const {
    std::string tempString = jsonCurrent;
    if( tempString[tempString.length() -1 ] == ',' ) //
    {
        tempString.erase( tempString.length()-1, 1 );
    }
    tempString.append("}}");
    return tempString;
}

ProviderInfo::ProviderInfo() {
    
}
ProviderInfo::ProviderInfo(picojson::value object)
{
    if(object.is<picojson::null>())
        return;
    
    picojson::value jsonValue;
    
    jsonValue = object.get("providerType");
    if(jsonValue.is<picojson::null>() == false)
        this->providerType = static_cast<ProviderType>(jsonValue.get<double>());

    jsonValue = object.get("providerName");
    if(jsonValue.is<picojson::null>() == false)
        this->providerName = jsonValue.to_str();

    jsonValue = object.get("providerUserId");
    if(jsonValue.is<picojson::null>() == false)
        this->providerUserId = jsonValue.to_str();
    
    jsonValue = object.get("providerEmail");
    if(jsonValue.is<picojson::null>() == false)
        this->providerEmail = jsonValue.to_str();
}
std::string ProviderInfo::stringProviderType(ProviderType providerType) {
    switch(providerType){
        case ProviderType::GUEST:               return "GUEST";
        case ProviderType::HIVE:                return "HIVE";
        case ProviderType::FACEBOOK:    	    return "FACEBOOK";
        case ProviderType::GOOGLE:              return "GOOGLE";
        case ProviderType::QQ:                  return "QQ";
        case ProviderType::VK:                  return "VK";
        case ProviderType::WECHAT:      	    return "WECHAT";
        case ProviderType::APPLE:               return "APPLE";
        case ProviderType::SIGNIN_APPLE:        return "SIGNIN_APPLE";
        case ProviderType::LINE:                return "LINE";
        case ProviderType::WEVERSE:             return "WEVERSE";
        case ProviderType::GOOGLE_PLAY_GAMES:   return "GOOGLE_PLAY_GAMES";
        case ProviderType::HUAWEI:              return "HUAWEI";
        case ProviderType::STEAM:               return "STEAM";
        case ProviderType::X:                   return "X";
        case ProviderType::TELEGRAM:            return "TELEGRAM";
        case ProviderType::XIAOMI:              return "XIAOMI";
        case ProviderType::CUSTOM:              return "CUSTOM";
        case ProviderType::AUTO:                return "AUTO";
        default:
            return "";
    }
}

ProviderType ProviderInfo::providerTypeFromString(const std::string typeString) {
    static std::map<std::string,ProviderType> stot = {
        {"GUEST",               ProviderType::GUEST}
        ,{"HIVE",               ProviderType::HIVE}
        ,{"FACEBOOK",           ProviderType::FACEBOOK}
        ,{"GOOGLE",             ProviderType::GOOGLE}
        ,{"QQ",                 ProviderType::QQ}
        ,{"VK",                 ProviderType::VK}
        ,{"WECHAT",             ProviderType::WECHAT}
        ,{"APPLE",              ProviderType::APPLE}
        ,{"SIGNIN_APPLE",       ProviderType::SIGNIN_APPLE}
        ,{"LINE",               ProviderType::LINE}
        ,{"WEVERSE",            ProviderType::WEVERSE}
        ,{"GOOGLE_PLAY_GAMES",  ProviderType::GOOGLE_PLAY_GAMES}
        ,{"HUAWEI",             ProviderType::HUAWEI}
        ,{"STEAM",              ProviderType::STEAM}
        ,{"X",                  ProviderType::X}
        ,{"TELEGRAM",           ProviderType::TELEGRAM}
        ,{"XIAOMI",             ProviderType::XIAOMI}
        ,{"CUSTOM",             ProviderType::CUSTOM}
        ,{"AUTO",               ProviderType::AUTO}
    };
    return stot[typeString];
}
ProfileInfo::ProfileInfo() {
    
}

ProfileInfo::ProfileInfo(picojson::value object) {
    if(object.is<picojson::null>())
       return;
    
    picojson::value jsonValue;
    
    jsonValue = object.get("playerId");
    if(jsonValue.is<picojson::null>() == false)
        this->playerId = static_cast<PlayerID>(jsonValue.get<double>());
    
    jsonValue = object.get("playerName");
    if(jsonValue.is<picojson::null>() == false)
        this->playerName = jsonValue.to_str();
    
    jsonValue = object.get("playerImageUrl");
    if(jsonValue.is<picojson::null>() == false)
        this->playerImageUrl = jsonValue.to_str();
}

#if WITH_EDITOR
ProfileInfo::~ProfileInfo(){

}
#endif

PlayerInfo::PlayerInfo() {
    
}

PlayerInfo::PlayerInfo(picojson::value object) : ProfileInfo(object) {
    if(object.is<picojson::null>())
        return;
    
    picojson::value jsonValue;
    
    jsonValue = object.get("playerToken");
    if(jsonValue.is<picojson::null>() == false)
        this->playerToken = jsonValue.to_str();
    
    jsonValue = object.get("accessToken");
    if(jsonValue.is<picojson::null>() == false)
        this->accessToken = jsonValue.to_str();
    
    jsonValue = object.get("did");
    if(jsonValue.is<picojson::null>() == false)
        this->did = jsonValue.to_str();
    
    jsonValue = object.get("providerInfoData");
    if(jsonValue.is<picojson::null>() == false){
        auto map = jsonValue.get<picojson::object>();
        for(auto it = map.begin();it != map.end();++it){
            ProviderInfo data(it->second);
            this->providerInfoData.insert(std::make_pair(ProviderInfo::providerTypeFromString(it->first), data));
        }
    }

    jsonValue = object.get("customProviderInfoData");
    if (jsonValue.is<picojson::null>() == false) {
        auto map = jsonValue.get<picojson::object>();
        for (auto it = map.begin(); it != map.end(); ++it) {
            ProviderInfo data(it->second);
            this->customProviderInfoData.insert(std::make_pair(it->first, data));
        }
    }
}

#if WITH_EDITOR
PlayerInfo::~PlayerInfo() {

}
#endif

Identity::Identity(picojson::value object) {
    if(object.is<picojson::null>())
        return;
    
    picojson::value jsonValue;
    
    jsonValue = object.get("phoneNumber");
    if(jsonValue.is<picojson::null>() == false)
        this->phoneNumber = jsonValue.to_str();
    
    jsonValue = object.get("phoneCode");
    if(jsonValue.is<picojson::null>() == false)
        this->phoneCode = jsonValue.to_str();
    
    jsonValue = object.get("dateOfBirth");
    if(jsonValue.is<picojson::null>() == false)
        this->dateOfBirth = jsonValue.to_str();
    
    jsonValue = object.get("hashedDi");
    if(jsonValue.is<picojson::null>() == false){
        this->hashedDi = jsonValue.to_str();
    }
}

AuthV4MaintenanceExtraButton::AuthV4MaintenanceExtraButton(picojson::value object) {
    button = object.get("button").to_str();
    action = static_cast<hive::AuthV4MaintenanceActionType>(object.get("action").get<double>());
    url = object.get("url").to_str();
}

AuthV4MaintenanceInfo::AuthV4MaintenanceInfo(picojson::value object) {
    if(object.is<picojson::null>())
        return;
    
    picojson::value jsonValue;
    title = object.get("title").to_str();
    message = object.get("message").to_str();
    button = object.get("button").to_str();
    action = static_cast<AuthV4MaintenanceActionType>(object.get("action").get<double>());
    url = object.get("url").to_str();
    startDate = object.get("startDate").to_str();
    endDate = object.get("endDate").to_str();
    remainingTime = object.get("remainingTime").get<double>();
    customerButton = object.get("customerButton").to_str();
    customerLink = object.get("customerLink").to_str();
    
    picojson::value exButtonsValue = object.get("exButtons");

    // jsonExButtons가 null인지 확인
    if (!exButtonsValue.is<picojson::null>()) {
        picojson::array jsonExButtons = exButtonsValue.get<picojson::array>();
        
        // jsonExButtons가 비어있는지 확인
        if (!jsonExButtons.empty()) {
            for (const auto& btnValue : jsonExButtons) {
                AuthV4MaintenanceExtraButton btn(btnValue);
                btn.button = btnValue.get("button").to_str();
                btn.url = btnValue.get("url").to_str();
                btn.action = static_cast<AuthV4MaintenanceActionType>(btnValue.get("action").get<double>());
                exButtons.push_back(btn);
            }
        }
    }
}

AgeRange::AgeRange(picojson::value object) {
    if (object.is<picojson::null>())
        return;

    if (!object.is<picojson::object>() || object.get<picojson::object>().empty())
        return;

    picojson::value jsonValue;

    this->userStatus = static_cast<UserStatus>(object.get("userStatus").get<double>());
    this->ageLower = object.get("ageLower").get<double>();
    this->ageUpper = object.get("ageUpper").get<double>();

    jsonValue = object.get("mostRecentApprovalDate");
    if (jsonValue.is<picojson::null>() == false)
        this->mostRecentApprovalDate = jsonValue.to_str();

    jsonValue = object.get("ageRangeId");
    if (jsonValue.is<picojson::null>() == false)
        this->ageRangeId = jsonValue.to_str();
}

ParentalConsentInfo::ParentalConsentInfo(picojson::value object) {
    if(object.is<picojson::null>())
        return;
    
    picojson::value jsonValue;
    
    jsonValue = object.get("dateOfBirth");
    if(jsonValue.is<picojson::null>() == false)
        this->dateOfBirth = jsonValue.to_str();
    
    jsonValue = object.get("guardianEmail");
    if(jsonValue.is<picojson::null>() == false)
        this->guardianEmail = jsonValue.to_str();
}

void AuthV4::Helper::syncAccount(ProviderType providerType, onAuthV4Helper listener){
    g_AuthV4HelperCallbacks[AUTHV4HELPER_FUNC_NAME_SYNCACCOUNT] = listener;
    picojson::object jsonParam;
    jsonParam["providerType"] = picojson::value(ProviderInfo::stringProviderType(providerType));
    callNativeAuthV4Helper(AUTHV4HELPER_FUNC_NAME_SYNCACCOUNT,jsonParam);
}
void AuthV4::Helper::signIn(onAuthV4Helper listener){
    g_AuthV4HelperCallbacks[AUTHV4HELPER_FUNC_NAME_SIGNIN] = listener;
    callNativeAuthV4Helper(AUTHV4HELPER_FUNC_NAME_SIGNIN);
}
void AuthV4::Helper::signOut(onAuthV4Helper listener){
    g_AuthV4HelperCallbacks[AUTHV4HELPER_FUNC_NAME_SIGNOUT] = listener;
    callNativeAuthV4Helper(AUTHV4HELPER_FUNC_NAME_SIGNOUT);
}
void AuthV4::Helper::playerDelete(onAuthV4Helper listener){
    g_AuthV4HelperCallbacks[AUTHV4HELPER_FUNC_NAME_PLAYERDELETE] = listener;
    callNativeAuthV4Helper(AUTHV4HELPER_FUNC_NAME_PLAYERDELETE);
}
void AuthV4::Helper::connect(ProviderType providerType, onAuthV4Helper listener){
    g_AuthV4HelperCallbacks[AUTHV4HELPER_FUNC_NAME_CONNECT] = listener;
    picojson::object jsonParam;
    jsonParam["providerType"] = picojson::value(ProviderInfo::stringProviderType(providerType));
    callNativeAuthV4Helper(AUTHV4HELPER_FUNC_NAME_CONNECT,jsonParam);
}
void AuthV4::Helper::disconnect(ProviderType providerType, onAuthV4Helper listener){
    g_AuthV4HelperCallbacks[AUTHV4HELPER_FUNC_NAME_DISCONNECT] = listener;
    picojson::object jsonParam;
    jsonParam["providerType"] = picojson::value(ProviderInfo::stringProviderType(providerType));
    callNativeAuthV4Helper(AUTHV4HELPER_FUNC_NAME_DISCONNECT,jsonParam);
}
void AuthV4::Helper::showAchievements(onAuthV4Helper listener){
    g_AuthV4HelperCallbacks[AUTHV4HELPER_FUNC_NAME_SHOWACHIEVEMENTS] = listener;
    callNativeAuthV4Helper(AUTHV4HELPER_FUNC_NAME_SHOWACHIEVEMENTS);
}
void AuthV4::Helper::showLeaderboard(onAuthV4Helper listener){
    g_AuthV4HelperCallbacks[AUTHV4HELPER_FUNC_NAME_SHOWLEADERBOARD] = listener;
    callNativeAuthV4Helper(AUTHV4HELPER_FUNC_NAME_SHOWLEADERBOARD);
}

void AuthV4::Helper::achievementsReveal(const std::string achievementId, onAuthV4Helper listener){
    g_AuthV4HelperCallbacks[AUTHV4HELPER_FUNC_NAME_ACHIEVEMENTSREVEAL] = listener;
    picojson::object jsonParam;
    jsonParam["achievementId"] = picojson::value(achievementId);
    jsonParam["handler"] = picojson::value("true");
    callNativeAuthV4Helper(AUTHV4HELPER_FUNC_NAME_ACHIEVEMENTSREVEAL, jsonParam);
}

void AuthV4::Helper::achievementsUnlock(const std::string achievementId, onAuthV4Helper listener){
    g_AuthV4HelperCallbacks[AUTHV4HELPER_FUNC_NAME_ACHIEVEMENTSUNLOCK] = listener;
    picojson::object jsonParam;
    jsonParam["achievementId"] = picojson::value(achievementId);
    jsonParam["handler"] = picojson::value("true");
    callNativeAuthV4Helper(AUTHV4HELPER_FUNC_NAME_ACHIEVEMENTSUNLOCK, jsonParam);
}

void AuthV4::Helper::achievementsIncrement(const std::string achievementId, const int value, onAuthV4Helper listener){
    g_AuthV4HelperCallbacks[AUTHV4HELPER_FUNC_NAME_ACHIEVEMENTSINCREMENT] = listener;
    picojson::object jsonParam;
    jsonParam["incrementalAchievementId"] = picojson::value(achievementId);
    jsonParam["value"] = picojson::value(to_string(value));
    jsonParam["handler"] = picojson::value("true");
    callNativeAuthV4Helper(AUTHV4HELPER_FUNC_NAME_ACHIEVEMENTSINCREMENT, jsonParam);
}

void AuthV4::Helper::leaderboardsSubmitScore(const std::string leaderboardId, const long long score, onAuthV4Helper listener){
    g_AuthV4HelperCallbacks[AUTHV4HELPER_FUNC_NAME_LEADERBOARDS_SUBMITSCORE] = listener;
    picojson::object jsonParam;
    jsonParam["leaderboardId"] = picojson::value(leaderboardId);
    jsonParam["score"] = picojson::value(to_string(score));
    jsonParam["handler"] = picojson::value("true");
    callNativeAuthV4Helper(AUTHV4HELPER_FUNC_NAME_LEADERBOARDS_SUBMITSCORE, jsonParam);
}

void AuthV4::Helper::showConflict(onAuthV4Helper listener){
    g_AuthV4HelperCallbacks[AUTHV4HELPER_FUNC_NAME_SHOWCONFLICT] = listener;
    callNativeAuthV4Helper(AUTHV4HELPER_FUNC_NAME_SHOWCONFLICT);
}

void AuthV4::Helper::showConflict(ConflictSingleViewInfo const & viewInfo,onAuthV4Helper listener){
    
    g_AuthV4HelperCallbacks[AUTHV4HELPER_FUNC_NAME_SHOWCONFLICT] = listener;
    std::string json = "{";
    if(viewInfo.toJsonConflict() != "")
        json += "\"playerData\":"+viewInfo.toJsonConflict();
    json += "}";
    callNativeAuthV4Helper(AUTHV4HELPER_FUNC_NAME_SHOWCONFLICT,json);
}

void AuthV4::Helper::switchAccount(onAuthV4Helper listener){
    g_AuthV4HelperCallbacks[AUTHV4HELPER_FUNC_NAME_SWITCHACCOUNT] = listener;
    callNativeAuthV4Helper(AUTHV4HELPER_FUNC_NAME_SWITCHACCOUNT);
}
void AuthV4::Helper::resolveConflict(onAuthV4Helper listener){
    g_AuthV4HelperCallbacks[AUTHV4HELPER_FUNC_NAME_RESOLVECONFLICT] = listener;
    callNativeAuthV4Helper(AUTHV4HELPER_FUNC_NAME_RESOLVECONFLICT);
}
std::vector<ProviderType> AuthV4::Helper::getIDPList(){
    picojson::value resJson = callNativeAuthV4Helper(AUTHV4HELPER_FUNC_NAME_GETIDPLIST);
    
    std::vector<ProviderType> types;
    picojson::array array = resJson.get("idpList").get<picojson::array>();
    for(auto it=array.begin();it!=array.end();++it) {
        ProviderType type = ProviderInfo::providerTypeFromString((*it).to_str());
        types.push_back(type);
    }
    
    return types;
}
void AuthV4::Helper::showGameCenterLoginCancelDialog(onAuthV4DialogDismiss listener){
    g_AuthV4HelperDialogDismiss = listener;
    callNativeAuthV4Helper(AUTHV4HELPER_FUNC_NAME_SHOWGAMECENTERLOGINCANCELDIALOG);
}

void AuthV4::Helper::executeEngine(picojson::value jsonParam) {
    ResultAPI result(jsonParam.get("resultAPI"));
    
    picojson::value jsonMethodValue;
    jsonMethodValue = jsonParam.get("method");
    if(jsonMethodValue.is<picojson::null>()) {
        return;
    }
    
    std::string methodName = jsonMethodValue.to_str();
    
    if((methodName.compare(AUTHV4HELPER_FUNC_NAME_SYNCACCOUNT) == 0)
       || (methodName.compare(AUTHV4HELPER_FUNC_NAME_SIGNIN) == 0)
       || (methodName.compare(AUTHV4HELPER_FUNC_NAME_SIGNOUT) == 0)
       || (methodName.compare(AUTHV4HELPER_FUNC_NAME_PLAYERDELETE) == 0)
       || (methodName.compare(AUTHV4HELPER_FUNC_NAME_CONNECT) == 0)
       || (methodName.compare(AUTHV4HELPER_FUNC_NAME_DISCONNECT) == 0)
       || (methodName.compare(AUTHV4HELPER_FUNC_NAME_SHOWACHIEVEMENTS) == 0)
       || (methodName.compare(AUTHV4HELPER_FUNC_NAME_SHOWLEADERBOARD) == 0)
       || (methodName.compare(AUTHV4HELPER_FUNC_NAME_ACHIEVEMENTSREVEAL) == 0)
       || (methodName.compare(AUTHV4HELPER_FUNC_NAME_ACHIEVEMENTSUNLOCK) == 0)
       || (methodName.compare(AUTHV4HELPER_FUNC_NAME_ACHIEVEMENTSINCREMENT) == 0)
       || (methodName.compare(AUTHV4HELPER_FUNC_NAME_LEADERBOARDS_SUBMITSCORE) == 0)
       || (methodName.compare(AUTHV4HELPER_FUNC_NAME_SHOWCONFLICT) == 0)
       || (methodName.compare(AUTHV4HELPER_FUNC_NAME_SWITCHACCOUNT) == 0)
       || (methodName.compare(AUTHV4HELPER_FUNC_NAME_RESOLVECONFLICT) == 0)) {
        AuthV4::Helper::onAuthV4Helper callback = g_AuthV4HelperCallbacks[methodName];
        if(jsonParam.contains("playerInfo") && !jsonParam.get("playerInfo").is<picojson::null>()){
            std::shared_ptr<PlayerInfo> info = nullptr;
            if(jsonParam.get("playerInfo").is<picojson::object>()) {
                auto v = jsonParam.get("playerInfo");
                const picojson::object o = v.get<picojson::object>();
                if(!o.empty()) {
                    info = std::make_shared<PlayerInfo>(PlayerInfo(jsonParam.get("playerInfo")));
                }
            }
            if(callback){
                callback(result,info);
            }
        } else {
            std::shared_ptr<PlayerInfo> info = nullptr;
            if(callback) {
                callback(result,info);
            }
        }
    }
    else if(methodName.compare(AUTHV4HELPER_FUNC_NAME_SHOWGAMECENTERLOGINCANCELDIALOG) == 0){
        if(g_AuthV4HelperDialogDismiss != nullptr){
            bool isDismiss = jsonParam.get("isDismiss").get<bool>();
            g_AuthV4HelperDialogDismiss(isDismiss);
        }
    }

}
NS_HIVE_END

#endif