//
//  HIVE_CHAT.cpp
//  HIVE_SDK_Plugin
//
//  Created by herohjk on 12/18/24.
//
#include "HIVE_Chat.h"

#include "HIVE_CppPlugin.h"
#include <chrono>        // Unique64의 시간 관련 기능
#include <mutex>         // ThreadSafeMap과 Unique64
#include <thread>        // std::this_thread::sleep_for
#include <sstream>
#include <unordered_map> // enum 매핑
#if !WITH_EDITOR && PLATFORM_WINDOWS

NS_HIVE_BEGIN


namespace {
std::string channelTypeToString(ChannelType type);
ChannelType channelTypeFromString(std::string const & str);
std::string sortTypeToString(SortType type);
SortType sortTypeFromString(std::string const & str);
std::string reactionTypeToString(ReactionType type);
ReactionType reactionTypeFromString(std::string const & str);
}

static Chat::onResult g_onResult = NULL;
static Chat::onCreateChannel g_onCreateChannel = NULL;
static Chat::onDeleteChannel g_onDeleteChannel = NULL;
static Chat::onEnterChannel g_onEnterChannel = NULL;
static Chat::onExitChannel g_onExitChannel = NULL;
static Chat::onBlockMember g_onBlockMember = NULL;
static Chat::onUnblockMember g_onUnblockMember = NULL;
static Chat::onConnect g_onConnect = NULL;
static Chat::onDisconnect g_onDisconnect = NULL;
static Chat::onReconnect g_onReconnect = NULL;
static Chat::onGetChannels g_onGetChannels = NULL;
static Chat::onGetChannelInfo g_onGetChannelInfo = NULL;
static Chat::onGetChannelMembers g_onGetChannelMembers = NULL;
static Chat::onGetChannelsByUser g_onGetChannelsByUser = NULL;
static Chat::onGetBlockMembers g_onGetBlockMembers = NULL;
static Chat::onGetChannelMessageHistory g_onGetChannelMessageHistory = NULL;
static Chat::onChannelTranslationSetting g_onChannelTranslationSetting = NULL;
static Chat::onTranslate g_onTranslate = NULL;
static Chat::onAddReaction g_onChannelAddReaction = NULL;
static Chat::onRemoveReaction g_onChannelRemoveReaction = NULL;

static ThreadSafeMap<std::string, ChannelListener*> g_channelListeners;
static ThreadSafeMap<std::string, DirectMessageListener*> g_directMessageListeners;
static ThreadSafeMap<std::string, UserListener*> g_userListeners;
static ThreadSafeMap<std::string, ConnectionListener*> g_connectionListeners;
static ThreadSafeMap<std::string, CustomDataListener*> g_customDataListeners;

static ThreadSafeMap<std::string, Chat::onDirectSendMessage> g_directMessageTracker;
static ThreadSafeMap<std::string, Chat::onChannelSendMessage> g_channelMessageTracker;

// MARK: APIs
void Chat::createChannel(CreateChannelParams const & params, onCreateChannel listener) {
    g_onCreateChannel = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "createChannel");
    jsonParam["params"] = picojson::value(params.toString());
    
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::getChannels(GetChannelsParams const * params, onGetChannels listener) {
    g_onGetChannels = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "getChannels");
    if (params != nullptr) {
        jsonParam["params"] = picojson::value(params->toString());
    }
    
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::getChannelInfo(std::string const & channelId, onGetChannelInfo listener) {
    g_onGetChannelInfo = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "getChannelInfo");
    jsonParam["channelId"] = picojson::value(channelId);
    
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::getChannelMembers(std::string const & channelId, onGetChannelMembers listener) {
    g_onGetChannelMembers = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "getChannelMembers");
    jsonParam["channelId"] = picojson::value(channelId);
    
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::deleteChannel(std::string const & channelId, onDeleteChannel listener) {
    g_onDeleteChannel = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "deleteChannel");
    jsonParam["channelId"] = picojson::value(channelId);
    
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::enterChannel(EnterChannelParams const & params, onEnterChannel listener) {
    g_onEnterChannel = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "enterChannel");
    jsonParam["params"] = picojson::value(params.toString());
    
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::exitChannel(std::string const & channelId, onExitChannel listener) {
    g_onExitChannel = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "exitChannel");
    jsonParam["channelId"] = picojson::value(channelId);
    
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::getChannelsByUser(onGetChannelsByUser listener) {
    g_onGetChannelsByUser = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "getChannelsByUser");
    
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::getBlockMembers(onGetBlockMembers listener) {
    g_onGetBlockMembers = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "getBlockMembers");
    
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::blockMember(int64_t blockPlayerId, onBlockMember listener) {
    g_onBlockMember = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "blockMember");
    jsonParam["blockPlayerId"] = picojson::value((double)blockPlayerId);
    
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::unblockMember(int64_t blockPlayerId, onUnblockMember listener) {
    g_onUnblockMember = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "unblockMember");
    jsonParam["blockPlayerId"] = picojson::value((double)blockPlayerId);
    
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::translate(TranslateParams const & params, onTranslate listener) {
    g_onTranslate = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "translate");
    jsonParam["params"] = picojson::value(params.toString());
    
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::connect(std::string const & extraData, onResult listener) {
    connect(listener);
}

void Chat::connect(onConnect listener) {
    g_onConnect = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "connect");
    
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::reconnect(onReconnect listener) {
    g_onReconnect = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "reconnect");
    
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::disconnect(onDisconnect listener) {
    g_onDisconnect = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "disconnect");
    
    HiveCppPlugin::callNative(jsonParam);
}

bool Chat::isConnected() {
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "isConnected");
    picojson::value resJson = HiveCppPlugin::callNative(jsonParam);
    
    if (!resJson.is<picojson::null>()) {
        bool isConnected = resJson.get("isConnected").get<bool>();
        return isConnected;
    }
    
    return false;
}

// MARK: Params
ChannelSendMessageParams::ChannelSendMessageParams() {}

picojson::object ChannelSendMessageParams::toJson() const {
    picojson::object json;
    json["channelId"] = picojson::value(channelId);
    json["message"] = picojson::value(message);
    json["extraData"] = picojson::value(extraData);
    
    if (mentionedPlayerIds.size() > 0) {
        picojson::array mentionedPlayerIdsArray;
        for (int64_t mention : mentionedPlayerIds) {
            mentionedPlayerIdsArray.push_back(picojson::value((double)mention));
        }
        json["mentionedPlayerIds"] = picojson::value(mentionedPlayerIdsArray);
    }
    
    if (replyMessageId.length() > 0) {
        json["replyMessageId"] = picojson::value(replyMessageId);
        
    }
    
    return json;
}

std::string ChannelSendMessageParams::toString() const {
    picojson::object json = toJson();
    std::string jsonString = picojson::value(json).serialize();
    return jsonString;
}

bool ChannelSendMessageParams::parse(picojson::value const & json) {
    if (!json.is<picojson::object>()) return false;
    
    picojson::object obj = json.get<picojson::object>();
    
    if (obj.count("channelId") > 0 && obj.at("channelId").is<std::string>()) {
        channelId = obj["channelId"].get<std::string>();
    } else {
        channelId = "";
    }
    
    if (obj.count("message") > 0 && obj.at("message").is<std::string>()) {
        message = obj["message"].get<std::string>();
    } else {
        message = "";
    }
    
    if (obj.count("extraData") > 0 && obj.at("extraData").is<std::string>()) {
        extraData = obj["extraData"].get<std::string>();
    } else {
        extraData = "";
    }
    
    if (obj.count("mentionedPlayerIds") > 0 && obj.at("mentionedPlayerIds").is<picojson::array>()) {
        picojson::array mentionedPlayerIdsArray = obj["mentionedPlayerIds"].get<picojson::array>();
        for (picojson::array::iterator it = mentionedPlayerIdsArray.begin(); it != mentionedPlayerIdsArray.end(); ++it) {
            mentionedPlayerIds.push_back(it->get<double>());
        }
    }
    
    return true;
}

DirectSendMessageParams::DirectSendMessageParams() : toPlayerId(0) {}

picojson::object DirectSendMessageParams::toJson() const {
    picojson::object json;
    json["toPlayerId"] = picojson::value((double)toPlayerId);
    json["message"] = picojson::value(message);
    json["extraData"] = picojson::value(extraData);
    return json;
}

std::string DirectSendMessageParams::toString() const {
    picojson::object json = toJson();
    std::string jsonString = picojson::value(json).serialize();
    return jsonString;
}

bool DirectSendMessageParams::parse(picojson::value const & json) {
    if (!json.is<picojson::object>()) return false;
    
    picojson::object obj = json.get<picojson::object>();
    
    if (obj.count("toPlayerId") > 0 && obj.at("toPlayerId").is<double>()) {
        toPlayerId = obj["toPlayerId"].get<double>();
    } else {
        toPlayerId = 0;
    }
    
    if (obj.count("message") > 0 && obj.at("message").is<std::string>()) {
        message = obj["message"].get<std::string>();
    } else {
        message = "";
    }
    
    if (obj.count("extraData") > 0 && obj.at("extraData").is<std::string>()) {
        extraData = obj["extraData"].get<std::string>();
    } else {
        extraData = "";
    }
    
    return true;
}

ChannelMessageListQueryParams::ChannelMessageListQueryParams() : prevSize(0), nextSize(0), messageId(""), order("") {}

picojson::object ChannelMessageListQueryParams::toJson() const {
    picojson::object json;
    json["prevSize"] = picojson::value((double)prevSize);
    json["nextSize"] = picojson::value((double)nextSize);
    json["messageId"] = picojson::value(messageId);
    json["order"] = picojson::value(order);
    return json;
}

std::string ChannelMessageListQueryParams::toString() const {
    picojson::object json = toJson();
    std::string jsonString = picojson::value(json).serialize();
    return jsonString;
}

TranslateParams::TranslateParams() : sourceLanguage("auto") {}

picojson::object TranslateParams::toJson() const {
    picojson::object json;
    json["message"] = picojson::value(message);
    json["sourceLanguage"] = picojson::value(sourceLanguage);
    picojson::array targetLangs;
    for (const auto& target : targetLanguage) {
        targetLangs.emplace_back(target);
    }
    json["targetLanguage"] = picojson::value(targetLangs);
    return json;
}

std::string TranslateParams::toString() const {
    picojson::object json = toJson();
    std::string jsonString = picojson::value(json).serialize();
    return jsonString;
}

void Chat::sendMessageWithChannelSendMessageParams(ChannelSendMessageParams const & params) {
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "sendMessageWithChannelSendMessageParams");
    jsonParam["params"] = picojson::value(params.toString());
    
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::sendMessageWithChannelSendMessageParams(ChannelSendMessageParams const & params, onChannelSendMessage listener) {
    std::string uniqueKey = Unique64::getInstance().generateKey();
    g_channelMessageTracker.insert(uniqueKey, listener);
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "sendMessageWithChannelSendMessageParams");
    jsonParam["params"] = picojson::value(params.toString());
    jsonParam["uniqueKey"] = picojson::value(uniqueKey);
    
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::sendMessageWithDirectSendMessageParams(DirectSendMessageParams const & params) {
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "sendMessageWithDirectSendMessageParams");
    jsonParam["params"] = picojson::value(params.toString());
    
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::sendMessageWithDirectSendMessageParams(DirectSendMessageParams const & params, onDirectSendMessage listener) {
    std::string uniqueKey = Unique64::getInstance().generateKey();
    g_directMessageTracker.insert(uniqueKey, listener);
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "sendMessageWithDirectSendMessageParams");
    jsonParam["params"] = picojson::value(params.toString());
    jsonParam["uniqueKey"] = picojson::value(uniqueKey);
    
    HiveCppPlugin::callNative(jsonParam);
}

// MARK: Listener

void Chat::addChannelListener(std::string const & uniqueKey, ChannelListener* listener) {
    g_channelListeners.insert(uniqueKey, listener);
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "addChannelListener");
    jsonParam["uniqueKey"] = picojson::value(uniqueKey);
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::addDirectMessageListener(std::string const & uniqueKey, DirectMessageListener* listener) {
    g_directMessageListeners.insert(uniqueKey, listener);
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "addDirectMessageListener");
    jsonParam["uniqueKey"] = picojson::value(uniqueKey);
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::addUserListener(std::string const & uniqueKey, UserListener* listener) {
    g_userListeners.insert(uniqueKey, listener);
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "addUserListener");
    jsonParam["uniqueKey"] = picojson::value(uniqueKey);
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::addConnectionListener(std::string const & uniqueKey, ConnectionListener* listener) {
    g_connectionListeners.insert(uniqueKey, listener);
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "addConnectionListener");
    jsonParam["uniqueKey"] = picojson::value(uniqueKey);
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::addCustomDataListener(std::string const & uniqueKey, CustomDataListener* listener) {
    g_customDataListeners.insert(uniqueKey, listener);
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "addCustomDataListener");
    jsonParam["uniqueKey"] = picojson::value(uniqueKey);
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::removeChannelListener(std::string const & uniqueKey) {
    if (g_channelListeners.contains(uniqueKey)) {
        g_channelListeners.erase(uniqueKey);
    }
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "removeChannelListener");
    jsonParam["uniqueKey"] = picojson::value(uniqueKey);
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::removeDirectMessageListener(std::string const & uniqueKey) {
    if (g_directMessageListeners.contains(uniqueKey)) {
        g_directMessageListeners.erase(uniqueKey);
    }
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "removeDirectMessageListener");
    jsonParam["uniqueKey"] = picojson::value(uniqueKey);
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::removeUserListener(std::string const & uniqueKey) {
    if (g_userListeners.contains(uniqueKey)) {
        g_userListeners.erase(uniqueKey);
    }
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "removeUserListener");
    jsonParam["uniqueKey"] = picojson::value(uniqueKey);
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::removeConnectionListener(std::string const & uniqueKey) {
    if (g_connectionListeners.contains(uniqueKey)) {
        g_connectionListeners.erase(uniqueKey);
    }
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "removeConnectionListener");
    jsonParam["uniqueKey"] = picojson::value(uniqueKey);
    HiveCppPlugin::callNative(jsonParam);
}

void Chat::removeCustomDataListener(std::string const & uniqueKey) {
    if (g_customDataListeners.contains(uniqueKey)) {
        g_customDataListeners.erase(uniqueKey);
    }
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "removeCustomDataListener");
    jsonParam["uniqueKey"] = picojson::value(uniqueKey);
    HiveCppPlugin::callNative(jsonParam);
}


/// MARK: Channel
Channel::Channel() : type(ChannelType::PUBLIC), memberCount(0), maxMemberCount(0), chatHistoryAllowed(false) {}

bool Channel::parse(picojson::value const & json) {
    if (!json.is<picojson::object>()) return false;
    
    picojson::object obj = json.get<picojson::object>();
    
    if (obj.count("channelId") > 0 && obj.at("channelId").is<std::string>()) {
        channelId = obj["channelId"].get<std::string>();
    } else {
        channelId = "";
    }
    if (obj.count("type") > 0 && obj.at("type").is<std::string>()) {
        type = channelTypeFromString(obj["type"].get<std::string>());
    } else {
        type = ChannelType::UNKNOWN;
    }
    if (obj.count("owner") > 0 && obj.at("owner").is<std::string>()) {
        owner = obj["owner"].get<std::string>();
    } else {
        owner = "";
    }
    if (obj.count("channelName") > 0 && obj.at("channelName").is<std::string>()) {
        channelName = obj["channelName"].get<std::string>();
    } else {
        channelName = "";
    }
    if (obj.count("memberCount") > 0 && obj.at("memberCount").is<double>()) {
        memberCount = (int)obj["memberCount"].get<double>();
    } else {
        memberCount = 0;
    }
    if (obj.count("maxMemberCount") > 0 && obj.at("maxMemberCount").is<double>()) {
        maxMemberCount = (int)obj["maxMemberCount"].get<double>();
    } else {
        maxMemberCount = 0;
    }
    if (obj.count("regTime") > 0 && obj.at("regTime").is<std::string>()) {
        regTime = obj["regTime"].get<std::string>();
    } else {
        regTime = "";
    }
    if (obj.count("regTimeMillis") > 0 && obj.at("regTimeMillis").is<double>()) {
        regTimeMillis = (int64_t)obj["regTimeMillis"].get<double>();
    } else {
        regTimeMillis = 0;
    }
    if (obj.count("chatHistoryAllowed") > 0 && obj.at("chatHistoryAllowed").is<bool>()) {
        chatHistoryAllowed = obj["chatHistoryAllowed"].get<bool>();
    } else {
        chatHistoryAllowed = false;
    }
    
    return true;
}

picojson::object Channel::toJson() const {
    picojson::object json;
    json["channelId"] = picojson::value(channelId);
    json["type"] = picojson::value(channelTypeToString(type));
    json["owner"] = picojson::value(owner);
    json["channelName"] = picojson::value(channelName);
    json["memberCount"] = picojson::value((double)memberCount);
    json["maxMemberCount"] = picojson::value((double)maxMemberCount);
    json["regTime"] = picojson::value(regTime);
    json["regTimeMillis"] = picojson::value(regTimeMillis);
    json["chatHistoryAllowed"] = picojson::value(chatHistoryAllowed);
    return json;
}

std::string Channel::toString() const {
    picojson::object json = toJson();
    std::string jsonString = picojson::value(json).serialize();
    return jsonString;
}

void Channel::query(ChannelMessageListQueryParams const & params, Chat::onGetChannelMessageHistory listener) const {
    g_onGetChannelMessageHistory = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "ChannelMessageListQuery");
    jsonParam["channel"] = picojson::value(toString());
    jsonParam["params"] = picojson::value(params.toString());
    
    HiveCppPlugin::callNative(jsonParam);
}

void Channel::setTranslationEnabled(bool enabled, Chat::onChannelTranslationSetting listener) const {
    g_onChannelTranslationSetting = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "ChannelSetTranslationEnabled");
    jsonParam["channel"] = picojson::value(toString());
    jsonParam["isEnabled"] = picojson::value(enabled);
    
    HiveCppPlugin::callNative(jsonParam);
}

void Channel::addReaction(std::string const & messageId, ReactionType reactionType, Chat::onAddReaction listener) const {
    g_onChannelAddReaction = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "ChannelAddReaction");
    jsonParam["channel"] = picojson::value(toString());
    jsonParam["messageId"] = picojson::value(messageId);
    jsonParam["type"] = picojson::value(reactionTypeToString(reactionType));
    
    HiveCppPlugin::callNative(jsonParam);
}

void Channel::removeReaction(std::string const & messageId, ReactionType reactionType, Chat::onRemoveReaction listener) const {
    g_onChannelRemoveReaction = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam("Chat", "ChannelRemoveReaction");
    jsonParam["channel"] = picojson::value(toString());
    jsonParam["messageId"] = picojson::value(messageId);
    jsonParam["type"] = picojson::value(reactionTypeToString(reactionType));
    
    HiveCppPlugin::callNative(jsonParam);
}

/// MARK: ChannelPage
ChannelPage::ChannelPage() : size(0), currentPage(0), totalElements(0), totalPages(0) {}

bool ChannelPage::parse(picojson::value const & json) {
    if (!json.is<picojson::object>()) return false;
    
    picojson::object obj = json.get<picojson::object>();
    
    if (obj.count("size") > 0 && obj.at("size").is<double>()) {
        size = (int)obj["size"].get<double>();
    } else {
        size = 0;
    }
    if (obj.count("currentPage") > 0 && obj.at("currentPage").is<double>()) {
        currentPage = (int)obj["currentPage"].get<double>();
    } else {
        currentPage = 0;
    }
    if (obj.count("totalElements") > 0 && obj.at("totalElements").is<double>()) {
        totalElements = (int)obj["totalElements"].get<double>();
    } else {
        totalElements = 0;
    }
    if (obj.count("totalPages") > 0 && obj.at("totalPages").is<double>()) {
        totalPages = (int)obj["totalPages"].get<double>();
    } else {
        totalPages = 0;
    }
    
    return true;
}

picojson::object ChannelPage::toJson() const {
    picojson::object json;
    json["size"] = picojson::value((double)size);
    json["currentPage"] = picojson::value((double)currentPage);
    json["totalElements"] = picojson::value((double)totalElements);
    json["totalPages"] = picojson::value((double)totalPages);
    return json;
}

/// MARK: Member
Member::Member() : playerId(0) {}

bool Member::parse(picojson::value const & json) {
    if (!json.is<picojson::object>()) return false;
    
    picojson::object obj = json.get<picojson::object>();
    
    playerId = (int64_t)obj["playerId"].get<double>();
    connectedTime = obj["connectedTime"].get<std::string>();
    connectedTimeMillis = (int64_t)obj["connectedTimeMillis"].get<double>();
    
    return true;
}

picojson::object Member::toJson() const {
    picojson::object json;
    json["playerId"] = picojson::value((double)playerId);
    json["connectedTime"] = picojson::value(connectedTime);
    json["connectedTimeMillis"] = picojson::value((double)connectedTimeMillis);
    return json;
}

/// MARK: BlockMember
BlockMember::BlockMember() : playerId(0) {}

bool BlockMember::parse(picojson::value const & json) {
    if (!json.is<picojson::object>()) return false;
    
    picojson::object obj = json.get<picojson::object>();
    
    if (obj.count("playerId") > 0 && obj.at("playerId").is<double>()) {
        playerId = (int64_t)obj["playerId"].get<double>();
    } else {
        playerId = 0;
    }
    if (obj.count("blockedTime") > 0 && obj.at("blockedTime").is<std::string>()) {
        blockedTime = obj["blockedTime"].get<std::string>();
    } else {
        blockedTime = "";
    }
    if (obj.count("blockedTimeMillis") > 0 && obj.at("blockedTimeMillis").is<double>()) {
        blockedTimeMillis = (int64_t)obj["blockedTimeMillis"].get<double>();
    } else {
        blockedTimeMillis = 0;
    }
    
    return true;
}

picojson::object BlockMember::toJson() const {
    picojson::object json;
    json["playerId"] = picojson::value((double)playerId);
    json["blockedTime"] = picojson::value(blockedTime);
    json["blockedTimeMillis"] = picojson::value((double)blockedTimeMillis);
    return json;
}

/// MARK: CreateChannelParams
CreateChannelParams::CreateChannelParams() : maxMemberCount(100), type(ChannelType::PUBLIC), chatHistoryAllowed(false) {}

picojson::object CreateChannelParams::toJson() const {
    picojson::object json;
    json["channelId"] = picojson::value(channelId);
    json["password"] = picojson::value(password);
    json["channelName"] = picojson::value(channelName);
    json["maxMemberCount"] = picojson::value((double)maxMemberCount);
    json["type"] = picojson::value(channelTypeToString(type));
    json["chatHistoryAllowed"] = picojson::value(chatHistoryAllowed);
    return json;
}

std::string CreateChannelParams::toString() const {
    picojson::object json = toJson();
    std::string jsonString = picojson::value(json).serialize();
    return jsonString;
}

/// MARK: GetChannelsParams
GetChannelsParams::GetChannelsParams() : type(ChannelType::PUBLIC), pageSize(10), pageNumber(1) {}

picojson::object GetChannelsParams::toJson() const {
    picojson::object json;
    json["type"] = picojson::value(channelTypeToString(type));
    json["channelId"] = picojson::value(channelId);
    json["channelName"] = picojson::value(channelName);
    json["sort"] = picojson::value(sortTypeToString(sort));
    json["pageOrder"] = picojson::value(pageOrder);
    json["pageSize"] = picojson::value((double)pageSize);
    json["pageNumber"] = picojson::value((double)pageNumber);
    return json;
}

std::string GetChannelsParams::toString() const {
    picojson::object json = toJson();
    std::string jsonString = picojson::value(json).serialize();
    return jsonString;
}

/// MARK: EnterChannelParams
EnterChannelParams::EnterChannelParams() {}

picojson::object EnterChannelParams::toJson() const {
    picojson::object json;
    json["channelId"] = picojson::value(channelId);
    json["password"] = picojson::value(password);
    return json;
}

std::string EnterChannelParams::toString() const {
    picojson::object json = toJson();
    std::string jsonString = picojson::value(json).serialize();
    return jsonString;
}

EnteredMember::EnteredMember() : playerId(0) {}

bool EnteredMember::parse(picojson::value const & json) {
    if (!json.is<picojson::object>()) return false;
    
    picojson::object obj = json.get<picojson::object>();
    
    if (obj.count("channelId") > 0 && obj.at("channelId").is<std::string>()) {
        channelId = obj["channelId"].get<std::string>();
    } else {
        channelId = "";
    }
    if (obj.count("playerId") > 0 && obj.at("playerId").is<double>()) {
        playerId = (int64_t)obj["playerId"].get<double>();
    } else {
        playerId = 0;
    }
    if (obj.count("timestamp") > 0 && obj.at("timestamp").is<std::string>()) {
        timestamp = obj["timestamp"].get<std::string>();
    } else {
        timestamp = "";
    }
    if (obj.count("timestampMillis") > 0 && obj.at("timestampMillis").is<double>()) {
        timestampMillis = (int64_t)obj["timestampMillis"].get<double>();
    } else {
        timestampMillis = 0;
    }
    
    return true;
}

ExitedMember::ExitedMember() : playerId(0) {}

bool ExitedMember::parse(picojson::value const & json) {
    if (!json.is<picojson::object>()) return false;
    
    picojson::object obj = json.get<picojson::object>();
    
    if (obj.count("channelId") > 0 && obj.at("channelId").is<std::string>()) {
        channelId = obj["channelId"].get<std::string>();
    } else {
        channelId = "";
    }
    if (obj.count("playerId") > 0 && obj.at("playerId").is<double>()) {
        playerId = (int64_t)obj["playerId"].get<double>();
    } else {
        playerId = 0;
    }
    if (obj.count("timestamp") > 0 && obj.at("timestamp").is<std::string>()) {
        timestamp = obj["timestamp"].get<std::string>();
    } else {
        timestamp = "";
    }
    if (obj.count("timestampMillis") > 0 && obj.at("timestampMillis").is<double>()) {
        timestampMillis = (int64_t)obj["timestampMillis"].get<double>();
    } else {
        timestampMillis = 0;
    }
    
    return true;
}

DeletedChannel::DeletedChannel() {}

bool DeletedChannel::parse(picojson::value const & json) {
    if (!json.is<picojson::object>()) return false;
    
    picojson::object obj = json.get<picojson::object>();
    
    channelId = obj["channelId"].get<std::string>();
    timestamp = obj["timestamp"].get<std::string>();
    timestampMillis = (int64_t)obj["timestampMillis"].get<double>();
    
    return true;
}

ChannelNoticeMessage::ChannelNoticeMessage() {}

bool ChannelNoticeMessage::parse(picojson::value const & json) {
    if (!json.is<picojson::object>()) return false;
    
    picojson::object obj = json.get<picojson::object>();
    
    if (obj.count("messageId") > 0 && obj.at("messageId").is<std::string>()) {
        messageId = obj["messageId"].get<std::string>();
    }
    if (obj.count("channelId") > 0 && obj.at("channelId").is<std::string>()) {
        channelId = obj["channelId"].get<std::string>();
    } else {
        channelId = "";
    }
    if (obj.count("from") > 0 && obj.at("from").is<std::string>()) {
        from = obj["from"].get<std::string>();
    } else {
        from = "";
    }
    if (obj.count("message") > 0 && obj.at("message").is<std::string>()) {
        message = obj["message"].get<std::string>();
    } else {
        message = "";
    }
    if (obj.count("timestamp") > 0 && obj.at("timestamp").is<std::string>()) {
        timestamp = obj["timestamp"].get<std::string>();
    } else {
        timestamp = "";
    }
    if (obj.count("timestampMillis") > 0 && obj.at("timestampMillis").is<double>()) {
        timestampMillis = (int64_t)obj["timestampMillis"].get<double>();
    } else {
        timestampMillis = 0;
    }
    
    return true;
}

ChannelMessageListQueryResponse::ChannelMessageListQueryResponse() : hasNext(false), nextMessageId(""), hasPrev(false), prevMessageId("") {}

bool ChannelMessageListQueryResponse::parse(picojson::value const & json) {
    if (!json.is<picojson::object>()) return false;
    
    picojson::object obj = json.get<picojson::object>();
    
    hasNext = obj["hasNext"].get<bool>();
    nextMessageId = obj["nextMessageId"].get<std::string>();
    hasPrev = obj["hasPrev"].get<bool>();
    prevMessageId = obj["prevMessageId"].get<std::string>();
    auto msgs = obj["content"].get<picojson::array>();
    for (auto& msg : msgs) {
        ChannelMessage chmsg;
        chmsg.parse(msg);
        content.emplace_back(chmsg);
    }
    return true;
}

ChannelMessage::ChannelMessage() : from(0) {}

bool ChannelMessage::parse(picojson::value const & json) {
    if (!json.is<picojson::object>()) return false;
    
    picojson::object obj = json.get<picojson::object>();
    
    if (obj.count("messageId") > 0 && obj.at("messageId").is<std::string>()) {
        messageId = obj["messageId"].get<std::string>();
    }
    if (obj.count("from") > 0 && obj.at("from").is<double>()) {
        from = (int64_t)obj["from"].get<double>();
    } else {
        from = 0;
    }
    if (obj.count("extraData") > 0 && obj.at("extraData").is<std::string>()) {
        extraData = obj["extraData"].get<std::string>();
    } else {
        extraData = "";
    }
    if (obj.count("to") > 0 && obj.at("to").is<std::string>()) {
        to = obj["to"].get<std::string>();
    } else {
        to = "";
    }
    if (obj.count("message") > 0 && obj.at("message").is<std::string>()) {
        message = obj["message"].get<std::string>();
    } else {
        message = "";
    }
    if (obj.count("timestamp") > 0 && obj.at("timestamp").is<std::string>()) {
        timestamp = obj["timestamp"].get<std::string>();
    } else {
        timestamp = "";
    }
    if (obj.count("timestampMillis") > 0 && obj.at("timestampMillis").is<double>()) {
        timestampMillis = (int64_t)obj["timestampMillis"].get<double>();
    } else {
        timestampMillis = 0;
    }
    if (obj.count("translated") > 0 && obj.at("translated").is<bool>()) {
        translated = obj["translated"].get<bool>();
    } else {
        translated = false;
    }
    if (obj.count("translatedMessage") > 0 && obj.at("translatedMessage").is<std::string>()) {
        translatedMessage = obj["translatedMessage"].get<std::string>();
    } else {
        translatedMessage = "";
    }
    if (obj.count("reactions") > 0 && obj.at("reactions").is<picojson::object>()) {
        picojson::object reactionsObj = obj["reactions"].get<picojson::object>();
        reactions.clear();
        
        for (const auto& pair : reactionsObj) {
            ReactionType type = ReactionType::Unknown;
            std::string reactionStr = pair.first;
            if (reactionStr == "Like") {
                type = ReactionType::Like;
            }
            
            if (pair.second.is<picojson::array>()) {
                std::vector<int64_t> playerIds;
                picojson::array playerArray = pair.second.get<picojson::array>();
                
                for (const auto& playerId : playerArray) {
                    if (playerId.is<double>()) {
                        playerIds.push_back((int64_t)playerId.get<double>());
                    }
                }
                
                reactions[type] = playerIds;
            }
        }
    }
    if (obj.count("mentionedPlayerIds") > 0 && obj.at("mentionedPlayerIds").is<picojson::array>()) {
        picojson::array mentionedPlayerIdsArray = obj["mentionedPlayerIds"].get<picojson::array>();
        mentionedPlayerIds.clear();
        for (const auto& mention : mentionedPlayerIdsArray) {
            if (mention.is<int64_t>()) {
                mentionedPlayerIds.push_back(mention.get<double>());
            }
        }
    }
    if (obj.count("replyMessageId") > 0 && obj.at("replyMessageId").is<std::string>()) {
        replyMessageId = obj["replyMessageId"].get<std::string>();
        
        if (obj.count("replyMessage") > 0 && obj.at("replyMessage").is<std::string>()) {
            replyMessage = obj["replyMessage"].get<std::string>();
        }
        
        if (obj.count("replyExtraData") > 0 && obj.at("replyExtraData").is<std::string>()) {
            replyExtraData = obj["replyExtraData"].get<std::string>();
        }
    }
    
    return true;
}

DirectMessage::DirectMessage() : from(0), to(0) {}

bool DirectMessage::parse(picojson::value const & json) {
    if (!json.is<picojson::object>()) return false;
    
    picojson::object obj = json.get<picojson::object>();
    
    if (obj.count("messageId") > 0 && obj.at("messageId").is<std::string>()) {
        messageId = obj["messageId"].get<std::string>();
    }
    if (obj.count("from") > 0 && obj.at("from").is<double>()) {
        from = (int64_t)obj["from"].get<double>();
    } else {
        from = 0;
    }
    if (obj.count("extraData") > 0 && obj.at("extraData").is<std::string>()) {
        extraData = obj["extraData"].get<std::string>();
    } else {
        extraData = "";
    }
    if (obj.count("to") > 0 && obj.at("to").is<double>()) {
        to = (int64_t)obj["to"].get<double>();
    } else {
        to = 0;
    }
    if (obj.count("message") > 0 && obj.at("message").is<std::string>()) {
        message = obj["message"].get<std::string>();
    } else {
        message = "";
    }
    if (obj.count("timestamp") > 0 && obj.at("timestamp").is<std::string>()) {
        timestamp = obj["timestamp"].get<std::string>();
    } else {
        timestamp = "";
    }
    if (obj.count("timestampMillis") > 0 && obj.at("timestampMillis").is<double>()) {
        timestampMillis = (int64_t)obj["timestampMillis"].get<double>();
    } else {
        timestampMillis = 0;
    }
    
    return true;
}

NoticeMessage::NoticeMessage() {}

bool NoticeMessage::parse(picojson::value const & json) {
    if (!json.is<picojson::object>()) return false;
    
    picojson::object obj = json.get<picojson::object>();
    
    if (obj.count("messageId") > 0 && obj.at("messageId").is<std::string>()) {
        messageId = obj["messageId"].get<std::string>();
    }
    if (obj.count("from") > 0 && obj.at("from").is<std::string>()) {
        from = obj["from"].get<std::string>();
    } else {
        from = "";
    }
    if (obj.count("message") > 0 && obj.at("message").is<std::string>()) {
        message = obj["message"].get<std::string>();
    } else {
        message = "";
    }
    if (obj.count("timestamp") > 0 && obj.at("timestamp").is<std::string>()) {
        timestamp = obj["timestamp"].get<std::string>();
    } else {
        timestamp = "";
    }
    if (obj.count("timestampMillis") > 0 && obj.at("timestampMillis").is<double>()) {
        timestampMillis = (int64_t)obj["timestampMillis"].get<double>();
    } else {
        timestampMillis = 0;
    }
    
    return true;
}

TranslationData::TranslationData() : translations() {}

bool TranslationData::parse(picojson::value const & json) {
    if (!json.is<picojson::object>()) return false;
    
    const picojson::object& translation_details_obj = json.get<picojson::object>();
    
    this->translations.clear();
    
    for (const auto& pair : translation_details_obj) {
        const std::string & key = pair.first;
        const picojson::value & text = pair.second;
        
        if (text.is<std::string>()) {
            this->translations[key] = text.get<std::string>();
        }
    }
    
    return true;
}

Reaction::Reaction() : channelId(""), messageId(""), playerId(0), type(ReactionType::Like) {}

bool Reaction::parse(picojson::value const & json) {
    if (!json.is<picojson::object>()) return false;
    
    picojson::object obj = json.get<picojson::object>();
    
    if (obj.count("channelId") > 0 && obj.at("channelId").is<std::string>()) {
        channelId = obj["channelId"].get<std::string>();
    }
    if (obj.count("messageId") > 0 && obj.at("messageId").is<std::string>()) {
        messageId = obj["messageId"].get<std::string>();
    }
    if (obj.count("playerId") > 0 && obj.at("playerId").is<double>()) {
        playerId = (int64_t)obj["playerId"].get<double>();
    }
    if (obj.count("type") > 0 && obj.at("type").is<std::string>()) {
        type = reactionTypeFromString(obj["type"].get<std::string>());
    }
    
    return true;
}

enum class ChatEventType {
    Connected,
    OnReconnectStarted,
    OnReconnected,
    Disconnected,
    EnteredMember,
    ExitedMember,
    DeletedChannel,
    ChannelMessage,
    NoticeMessage,
    DirectMessage,
    CustomData,
    OnAddReaction,
    OnRemoveReaction
};

static const std::unordered_map<std::string, ChatEventType> eventTypes = {
    {"onConnected", ChatEventType::Connected},
    {"onReconnectStarted", ChatEventType::OnReconnectStarted},
    {"onReconnected", ChatEventType::OnReconnected},
    {"onDisconnected", ChatEventType::Disconnected},
    {"onEnteredMember", ChatEventType::EnteredMember},
    {"onExitedMember", ChatEventType::ExitedMember},
    {"onDeletedChannel", ChatEventType::DeletedChannel},
    {"onChannelMessage", ChatEventType::ChannelMessage},
    {"onNoticeMessage", ChatEventType::NoticeMessage},
    {"onDirectMessage", ChatEventType::DirectMessage},
    {"onCustomData", ChatEventType::CustomData},
    {"onAddReaction", ChatEventType::OnAddReaction},
    {"onRemoveReaction", ChatEventType::OnRemoveReaction}
};

enum class ChatMethodNames {
    AddConnectionListener,
    AddDirectMessageListener,
    AddChannelListener,
    AddUserListener,
    AddCustomDataListener,
    CreateChannel,
    GetChannels,
    GetChannelInfo,
    GetChannelMembers,
    DeleteChannel,
    EnterChannel,
    ExitChannel,
    GetChannelsByUser,
    GetBlockMembers,
    BlockMember,
    UnblockMember,
    Connect,
    Reconnect,
    Disconnect,
    SendMessageWithChannelSendMessageParams,
    SendMessageWithDirectSendMessageParams,
    ChannelMessageListQuery,
    ChannelSetTranslationEnabled,
    ChannelAddReaction,
    ChannelRemoveReaction,
    Translate,
};

static const std::unordered_map<std::string, ChatMethodNames> methodNames = {
    {"addConnectionListener", ChatMethodNames::AddConnectionListener},
    {"addDirectMessageListener", ChatMethodNames::AddDirectMessageListener},
    {"addChannelListener", ChatMethodNames::AddChannelListener},
    {"addUserListener", ChatMethodNames::AddUserListener},
    {"addCustomDataListener", ChatMethodNames::AddCustomDataListener},
    {"createChannel", ChatMethodNames::CreateChannel},
    {"getChannelInfo", ChatMethodNames::GetChannelInfo},
    {"getChannels", ChatMethodNames::GetChannels},
    {"getChannelMembers", ChatMethodNames::GetChannelMembers},
    {"deleteChannel", ChatMethodNames::DeleteChannel},
    {"enterChannel", ChatMethodNames::EnterChannel},
    {"exitChannel", ChatMethodNames::ExitChannel},
    {"getChannelsByUser", ChatMethodNames::GetChannelsByUser},
    {"getBlockMembers", ChatMethodNames::GetBlockMembers},
    {"blockMember", ChatMethodNames::BlockMember},
    {"unblockMember", ChatMethodNames::UnblockMember},
    {"connect", ChatMethodNames::Connect},
    {"reconnect", ChatMethodNames::Reconnect},
    {"disconnect", ChatMethodNames::Disconnect},    
    {"sendMessageWithChannelSendMessageParams", ChatMethodNames::SendMessageWithChannelSendMessageParams},
    {"sendMessageWithDirectSendMessageParams", ChatMethodNames::SendMessageWithDirectSendMessageParams},
    {"ChannelMessageListQuery", ChatMethodNames::ChannelMessageListQuery},
    {"ChannelSetTranslationEnabled", ChatMethodNames::ChannelSetTranslationEnabled},
    {"ChannelAddReaction", ChatMethodNames::ChannelAddReaction},
    {"ChannelRemoveReaction", ChatMethodNames::ChannelRemoveReaction},
    {"translate", ChatMethodNames::Translate},
};

void Chat::executeEngine(picojson::value jsonParam) {
    std::string methodName = jsonParam.get("method").to_str();
    std::string eventName = jsonParam.get("event").to_str();
    std::string uniqueKey = jsonParam.get("uniqueKey").to_str();
    
    auto method_iterator = methodNames.find(methodName);
    if (method_iterator == methodNames.end()) {
        return;
    }
    auto chatMethodName = method_iterator->second;
    
    auto event_iterator = eventTypes.find(eventName);
    
    switch (chatMethodName) {
        case ChatMethodNames::AddConnectionListener:{
            if (event_iterator != eventTypes.end()) {
                switch (event_iterator->second) {
                    case ChatEventType::Connected:
                        if (g_connectionListeners.contains(uniqueKey)) { g_connectionListeners.get(uniqueKey)->onConnected(); }
                        break;
                    case ChatEventType::OnReconnectStarted: {
                        if (g_connectionListeners.contains(uniqueKey)) { g_connectionListeners.get(uniqueKey)->onReconnectStarted(); }
                        break;
                    }
                    case ChatEventType::OnReconnected: {
                        std::vector<std::string> channelIds;
                        std::vector<std::string> failChannelIds;
                        ResultAPI result;
                        
                        if(!jsonParam.get("resultAPI").is<picojson::null>()) {
                            result = ResultAPI(jsonParam.get("resultAPI"));
                        }
                        
                        if(!jsonParam.get("channelIds").is<picojson::null>()) {
                            picojson::array arrayChannelIds = jsonParam.get("channelIds").get<picojson::array>();
                            for (auto iter = arrayChannelIds.begin(); iter != arrayChannelIds.end(); ++iter) {
                                channelIds.push_back(iter->to_str());
                            }
                        }
                        
                        if(!jsonParam.get("failChannelIds").is<picojson::null>()) {
                            picojson::array arrayFailChannelIds = jsonParam.get("failChannelIds").get<picojson::array>();
                            for (auto iter = arrayFailChannelIds.begin(); iter != arrayFailChannelIds.end(); ++iter) {
                                failChannelIds.push_back(iter->to_str());
                            }
                        }
                        if (g_connectionListeners.contains(uniqueKey)) { g_connectionListeners.get(uniqueKey)->onReconnected(result, channelIds, failChannelIds); }
                        break;
                    }
                    case ChatEventType::Disconnected:
                        if (g_connectionListeners.contains(uniqueKey)) { g_connectionListeners.get(uniqueKey)->onDisconnected(); }
                        break;
                    default: break;
                }
            }
        }
            break;
        case ChatMethodNames::AddChannelListener: {
            if (event_iterator != eventTypes.end()) {
                switch (event_iterator->second) {
                    case ChatEventType::EnteredMember: {
                        EnteredMember member;
                        member.parse(jsonParam.get("member"));
                        if (g_channelListeners.contains(uniqueKey)) { g_channelListeners.get(uniqueKey)->onEnteredMember(member); }
                        break;
                    }
                    case ChatEventType::ExitedMember: {
                        ExitedMember member;
                        member.parse(jsonParam.get("member"));
                        if (g_channelListeners.contains(uniqueKey)) { g_channelListeners.get(uniqueKey)->onExitedMember(member); }
                        break;
                    }
                    case ChatEventType::DeletedChannel: {
                        DeletedChannel channel;
                        channel.parse(jsonParam.get("channel"));
                        if (g_channelListeners.contains(uniqueKey)) { g_channelListeners.get(uniqueKey)->onDeletedChannel(channel); }
                        break;
                    }
                    case ChatEventType::ChannelMessage: {
                        ChannelMessage message;
                        message.parse(jsonParam.get("message"));
                        if (g_channelListeners.contains(uniqueKey)) { g_channelListeners.get(uniqueKey)->onChannelMessage(message); }
                        break;
                    }
                    case ChatEventType::NoticeMessage: {
                        ChannelNoticeMessage message;
                        message.parse(jsonParam.get("message"));
                        if (g_channelListeners.contains(uniqueKey)) { g_channelListeners.get(uniqueKey)->onNoticeMessage(message); }
                        break;
                    }
                    case ChatEventType::OnAddReaction: {
                        Reaction reaction;
                        reaction.parse(jsonParam.get("reaction"));
                        if (g_channelListeners.contains(uniqueKey)) { g_channelListeners.get(uniqueKey)->onAddReaction(reaction); }
                        break;
                    }
                    case ChatEventType::OnRemoveReaction: {
                        Reaction reaction;
                        reaction.parse(jsonParam.get("reaction"));
                        if (g_channelListeners.contains(uniqueKey)) { g_channelListeners.get(uniqueKey)->onRemoveReaction(reaction); }
                        break;
                    }
                    default: break;
                }
            }
        }
            break;
        case ChatMethodNames::AddDirectMessageListener: {
            if (event_iterator != eventTypes.end()) {
                switch (event_iterator->second) {
                    case ChatEventType::DirectMessage: {
                        DirectMessage message;
                        message.parse(jsonParam.get("message"));
                        if (g_directMessageListeners.contains(uniqueKey)) { g_directMessageListeners.get(uniqueKey)->onDirectMessage(message); }
                        break;
                    }
                    default: break;
                }
            }
        }
            break;
        case ChatMethodNames::AddUserListener: {
            if (event_iterator != eventTypes.end()) {
                switch (event_iterator->second) {
                    case ChatEventType::DirectMessage: {
                        DirectMessage message;
                        message.parse(jsonParam.get("message"));
                        if (g_userListeners.contains(uniqueKey)) { g_userListeners.get(uniqueKey)->onDirectMessage(message); }
                        break;
                    }
                    case ChatEventType::NoticeMessage: {
                        NoticeMessage message;
                        message.parse(jsonParam.get("message"));
                        if (g_userListeners.contains(uniqueKey)) { g_userListeners.get(uniqueKey)->onNoticeMessage(message); }
                        break;
                    }
                    default: break;
                }
            }
        }
            break;
        case ChatMethodNames::AddCustomDataListener: {
            if (event_iterator != eventTypes.end()) {
                switch (event_iterator->second) {
                    case ChatEventType::CustomData: {
                        auto data = jsonParam.get("data").to_str();
                        if (g_customDataListeners.contains(uniqueKey)) { g_customDataListeners.get(uniqueKey)->onCustomData(data); }
                        break;
                    }
                    default: break;
                }
            }
        }
            break;
        case ChatMethodNames::CreateChannel: {
            if (g_onCreateChannel != NULL) {
                ResultAPI result;
                if(!jsonParam.get("resultAPI").is<picojson::null>()) {
                    result = ResultAPI(jsonParam.get("resultAPI"));
                    onCreateChannel callback = g_onCreateChannel;
                    g_onCreateChannel = NULL;
                    callback(result);
                    return;
                }
            }
        }
            break;
        case ChatMethodNames::GetChannels:  {
            if (g_onGetChannels != NULL) {
                ResultAPI result;
                std::vector<Channel> channels;
                ChannelPage page;
                
                if(!jsonParam.get("resultAPI").is<picojson::null>()) {
                    result = ResultAPI(jsonParam.get("resultAPI"));
                }
                
                if(!jsonParam.get("channels").is<picojson::null>()) {
                    picojson::array arrayChannels = jsonParam.get("channels").get<picojson::array>();
                    for (auto iter = arrayChannels.begin(); iter != arrayChannels.end(); ++iter) {
                        Channel channel;
                        channel.parse(*iter);
                        channels.push_back(channel);
                    }
                }
                
                if(!jsonParam.get("channelPage").is<picojson::null>()) {
                    page.parse(jsonParam.get("channelPage"));
                }
                
                onGetChannels callback = g_onGetChannels;
                g_onGetChannels = NULL;
                callback(result, channels, page);
                return;
            }
        }
            break;
        case ChatMethodNames::GetChannelInfo:  {
            if (g_onGetChannelInfo != NULL) {
                ResultAPI result;
                Channel channel;
                std::vector<Member> members;
                
                if(!jsonParam.get("resultAPI").is<picojson::null>()) {
                    result = ResultAPI(jsonParam.get("resultAPI"));
                }
                
                if(!jsonParam.get("channel").is<picojson::null>()) {
                    channel.parse(jsonParam.get("channel"));
                }
                
                if(!jsonParam.get("members").is<picojson::null>()) {
                    picojson::array arrayMembers = jsonParam.get("members").get<picojson::array>();
                    for(auto iter = arrayMembers.begin(); iter != arrayMembers.end(); ++iter) {
                        Member member;
                        member.parse(*iter);
                        members.push_back(member);
                    }
                }
                
                onGetChannelInfo callback = g_onGetChannelInfo;
                g_onGetChannelInfo = NULL;
                callback(result, channel, members);
                return;
            }
        }
            break;
        case ChatMethodNames::GetChannelMembers: {
            if (g_onGetChannelMembers != NULL) {
                ResultAPI result;
                std::vector<Member> members;
                
                if(!jsonParam.get("resultAPI").is<picojson::null>()) {
                    result = ResultAPI(jsonParam.get("resultAPI"));
                }
                
                if(!jsonParam.get("members").is<picojson::null>()) {
                    picojson::array arrayMembers = jsonParam.get("members").get<picojson::array>();
                    for(auto iter = arrayMembers.begin(); iter != arrayMembers.end(); ++iter) {
                        Member member;
                        member.parse(*iter);
                        members.push_back(member);
                    }
                }
                
                onGetChannelMembers callback = g_onGetChannelMembers;
                g_onGetChannelMembers = NULL;
                callback(result, members);
                return;
            }
        }
            break;
        case ChatMethodNames::DeleteChannel:  {
            if (g_onDeleteChannel != NULL) {
                ResultAPI result;
                if(!jsonParam.get("resultAPI").is<picojson::null>()) {
                    result = ResultAPI(jsonParam.get("resultAPI"));
                    onDeleteChannel callback = g_onDeleteChannel;
                    g_onDeleteChannel = NULL;
                    callback(result);
                    return;
                }
            }
        }
            break;
        case ChatMethodNames::EnterChannel: {
            if (g_onEnterChannel != NULL) {
                ResultAPI result;
                if(!jsonParam.get("resultAPI").is<picojson::null>()) {
                    result = ResultAPI(jsonParam.get("resultAPI"));
                    onEnterChannel callback = g_onEnterChannel;
                    g_onEnterChannel = NULL;
                    callback(result);
                    return;
                }
            }
        }
            break;
        case ChatMethodNames::ExitChannel:  {
            if (g_onExitChannel != NULL) {
                ResultAPI result;
                if(!jsonParam.get("resultAPI").is<picojson::null>()) {
                    result = ResultAPI(jsonParam.get("resultAPI"));
                    onExitChannel callback = g_onExitChannel;
                    g_onExitChannel = NULL;
                    callback(result);
                    return;
                }
            }
        }
            break;
        case ChatMethodNames::GetChannelsByUser: {
            if (g_onGetChannelsByUser != NULL) {
                ResultAPI result;
                std::vector<Channel> channels;
                
                if(!jsonParam.get("resultAPI").is<picojson::null>()) {
                    result = ResultAPI(jsonParam.get("resultAPI"));
                }
                
                if(!jsonParam.get("channels").is<picojson::null>()) {
                    picojson::array arrayChannels = jsonParam.get("channels").get<picojson::array>();
                    for(auto iter = arrayChannels.begin(); iter != arrayChannels.end(); ++iter) {
                        Channel channel;
                        channel.parse(*iter);
                        channels.push_back(channel);
                    }
                }
                
                onGetChannelsByUser callback = g_onGetChannelsByUser;
                g_onGetChannelsByUser = NULL;
                callback(result, channels);
                return;
            }
        }
            break;
        case ChatMethodNames::GetBlockMembers: {
            if (g_onGetBlockMembers != NULL) {
                ResultAPI result;
                std::vector<BlockMember> blockMembers;
                
                if(!jsonParam.get("resultAPI").is<picojson::null>()) {
                    result = ResultAPI(jsonParam.get("resultAPI"));
                }
                
                if(!jsonParam.get("blockMembers").is<picojson::null>()) {
                    picojson::array arrayBlockMembers = jsonParam.get("blockMembers").get<picojson::array>();
                    for(auto iter = arrayBlockMembers.begin(); iter != arrayBlockMembers.end(); ++iter) {
                        BlockMember blockMember;
                        blockMember.parse(*iter);
                        blockMembers.push_back(blockMember);
                    }
                }
                
                onGetBlockMembers callback = g_onGetBlockMembers;
                g_onGetBlockMembers = NULL;
                callback(result, blockMembers);
                return;
            }
        }
            break;
        case ChatMethodNames::BlockMember: {
            if (g_onBlockMember != NULL) {
                ResultAPI result;
                if(!jsonParam.get("resultAPI").is<picojson::null>()) {
                    result = ResultAPI(jsonParam.get("resultAPI"));
                    onBlockMember callback = g_onBlockMember;
                    g_onBlockMember = NULL;
                    callback(result);
                    return;
                }
            }
        }
            break;
        case ChatMethodNames::UnblockMember: {
            if (g_onUnblockMember != NULL) {
                ResultAPI result;
                if(!jsonParam.get("resultAPI").is<picojson::null>()) {
                    result = ResultAPI(jsonParam.get("resultAPI"));
                    onUnblockMember callback = g_onUnblockMember;
                    g_onUnblockMember = NULL;
                    callback(result);
                    return;
                }
            }
        }
            break;
        case ChatMethodNames::ChannelMessageListQuery: {
            if (g_onGetChannelMessageHistory != NULL) {
                ResultAPI result;
                ChannelMessageListQueryResponse response;
                
                if(!jsonParam.get("resultAPI").is<picojson::null>()) {
                    result = ResultAPI(jsonParam.get("resultAPI"));
                }
                
                if(!jsonParam.get("response").is<picojson::null>()) {
                    response.parse(jsonParam.get("response"));
                }
                
                onGetChannelMessageHistory callback = g_onGetChannelMessageHistory;
                g_onGetChannelMessageHistory = NULL;
                callback(result, response);
                return;
            }
        }
            break;
        case ChatMethodNames::ChannelSetTranslationEnabled: {
            if (g_onChannelTranslationSetting != NULL) {
                ResultAPI result;
                
                if(!jsonParam.get("resultAPI").is<picojson::null>()) {
                    result = ResultAPI(jsonParam.get("resultAPI"));
                }
                
                onChannelTranslationSetting callback = g_onChannelTranslationSetting;
                g_onChannelTranslationSetting = NULL;
                callback(result);
                return;
            }
        }
        case ChatMethodNames::ChannelAddReaction: {
            if (g_onChannelAddReaction != NULL) {
                ResultAPI result;
                ReactionType type; 
                
                if(!jsonParam.get("resultAPI").is<picojson::null>()) {
                    result = ResultAPI(jsonParam.get("resultAPI"));
                }
                
                if(!jsonParam.get("type").is<picojson::null>()) {
                    type = reactionTypeFromString(jsonParam.get("type").to_str());
                }
                
                onAddReaction callback = g_onChannelAddReaction;
                g_onChannelAddReaction = NULL;
                callback(result, type);
                return;
            }
        }
            break;
        case ChatMethodNames::ChannelRemoveReaction: {
            if (g_onChannelRemoveReaction != NULL) {
                ResultAPI result;
                ReactionType type; 
                
                if(!jsonParam.get("resultAPI").is<picojson::null>()) {
                    result = ResultAPI(jsonParam.get("resultAPI"));
                }
                
                if(!jsonParam.get("type").is<picojson::null>()) {
                    type = reactionTypeFromString(jsonParam.get("type").to_str());
                }
                
                onRemoveReaction callback = g_onChannelRemoveReaction;
                g_onChannelRemoveReaction = NULL;
                callback(result, type);
                return;
            }
        }
            break;
            break;
        case ChatMethodNames::Translate: {
            if (g_onTranslate != NULL) {
                ResultAPI result;
                if(!jsonParam.get("resultAPI").is<picojson::null>()) {
                    result = ResultAPI(jsonParam.get("resultAPI"));
                }
                
                TranslationData resultData;
                resultData.parse(jsonParam.get("translations"));
                
                onTranslate callback = g_onTranslate;
                g_onTranslate = NULL;
                callback(result, resultData);
                return;
            }
        }
            break;
        case ChatMethodNames::Connect: {
            if (g_onConnect != NULL) {
                ResultAPI result;
                if(!jsonParam.get("resultAPI").is<picojson::null>()) {
                    result = ResultAPI(jsonParam.get("resultAPI"));
                    onConnect callback = g_onConnect;
                    g_onConnect = NULL;
                    callback(result);
                    return;
                }
            }
        }
            break;
        case ChatMethodNames::Reconnect: {
            if (g_onReconnect != NULL) {
                onReconnect callback = g_onReconnect;
                ResultAPI result;
                std::vector<std::string> channelIds;
                std::vector<std::string> failChannelIds;
                
                if(!jsonParam.get("resultAPI").is<picojson::null>()) {
                    result = ResultAPI(jsonParam.get("resultAPI"));
                }
                
                if(!jsonParam.get("channelIds").is<picojson::null>()) {
                    picojson::array arrayChannelIds = jsonParam.get("channelIds").get<picojson::array>();
                    for(auto iter = arrayChannelIds.begin(); iter != arrayChannelIds.end(); ++iter) {
                        std::string channelId = (*iter).to_str();
                        channelIds.push_back(channelId);
                    }
                }
                
                if(!jsonParam.get("failChannelIds").is<picojson::null>()) {
                    picojson::array arrayFailChannelIds = jsonParam.get("failChannelIds").get<picojson::array>();
                    for(auto iter = arrayFailChannelIds.begin(); iter != arrayFailChannelIds.end(); ++iter) {
                        std::string failChannelId = (*iter).to_str();
                        failChannelIds.push_back(failChannelId);
                    }
                }
                
                g_onReconnect = NULL;
                callback(result, channelIds, failChannelIds);
                return;
            }
        }
            break;
        case ChatMethodNames::Disconnect: {
            if (g_onDisconnect != NULL) {
                ResultAPI result;
                if(!jsonParam.get("resultAPI").is<picojson::null>()) {
                    result = ResultAPI(jsonParam.get("resultAPI"));
                    onDisconnect callback = g_onDisconnect;
                    g_onDisconnect = NULL;
                    callback(result);
                    return;
                }
            }
        }
            break;
        case ChatMethodNames::SendMessageWithChannelSendMessageParams: {
            std::string callbackKey;
            if (!jsonParam.get("uniqueKey").is<picojson::null>()) {
                callbackKey = jsonParam.get("uniqueKey").to_str();
            }
            
            ResultAPI result;
            if(!jsonParam.get("resultAPI").is<picojson::null>()) {
                result = ResultAPI(jsonParam.get("resultAPI"));
            }
            
            ChannelSendMessageParams params;
            if(!jsonParam.get("retryParams").is<picojson::null>()) {
                ChannelSendMessageParams channelSendMessageParams;
                channelSendMessageParams.parse(jsonParam.get("retryParams"));
                params = channelSendMessageParams;
            }
            
            if (!callbackKey.empty() && g_channelMessageTracker.contains(callbackKey)) {
                onChannelSendMessage callback = g_channelMessageTracker.get(callbackKey);
                callback(result, params);
                
                g_channelMessageTracker.erase(callbackKey);
            } 
            
            return;
        }
            break;
        case ChatMethodNames::SendMessageWithDirectSendMessageParams: {
            std::string callbackKey;
            if (!jsonParam.get("uniqueKey").is<picojson::null>()) {
                callbackKey = jsonParam.get("uniqueKey").to_str();
            }
            
            ResultAPI result;
            if(!jsonParam.get("resultAPI").is<picojson::null>()) {
                result = ResultAPI(jsonParam.get("resultAPI"));
            }
            
            DirectSendMessageParams params;
            if(!jsonParam.get("retryParams").is<picojson::null>()) {
                DirectSendMessageParams directSendMessageParams;
                directSendMessageParams.parse(jsonParam.get("retryParams"));
                params = directSendMessageParams;
            }
            
            if (!callbackKey.empty() && g_directMessageTracker.contains(callbackKey)) {
                onDirectSendMessage callback = g_directMessageTracker.get(callbackKey);
                callback(result, params);
                
                g_directMessageTracker.erase(callbackKey);
            }
            
            return;
        }
            break;
    }
}

// ChannelType <-> String 변환용 내부 함수
namespace {
std::string channelTypeToString(ChannelType type) {
    switch (type) {
        case ChannelType::PUBLIC:
            return "PUBLIC";
        case ChannelType::PRIVATE:
            return "PRIVATE";
        case ChannelType::GROUP:
            return "GROUP";
        default:
            return "UNKNOWN";
    }
}

ChannelType channelTypeFromString(std::string const & str) {
    if (str == "PUBLIC") return ChannelType::PUBLIC;
    if (str == "PRIVATE") return ChannelType::PRIVATE;
    if (str == "GROUP") return ChannelType::GROUP;
    return ChannelType::UNKNOWN;
}
}

// SrotType <-> String 변환용 내부 함수
namespace {
std::string sortTypeToString(SortType type) {
    switch(type) {
        case SortType::ChannelId:
            return "ChannelId";
        case SortType::ChannelName:
            return "ChannelName";
        case SortType::RegTime:
            return "RegTime";
        default:
            return "Unknown";
    }
}

SortType sortTypeFromString(std::string const & str) {
    if (str == "ChannelId") return SortType::ChannelId;
    if (str == "ChannelName") return SortType::ChannelName;
    if (str == "RegTime") return SortType::RegTime;
    return SortType::Unknown;
}

std::string reactionTypeToString(ReactionType type) {
    switch (type) {
        case ReactionType::Like:
            return "Like";
        default:
            return "Unknown";
    }
}

ReactionType reactionTypeFromString(std::string const & str) {
    if (str == "Like") return ReactionType::Like;
    return ReactionType::Unknown;
}
}

Unique64::Unique64() : last_timestamp_ms_(0), sequence_(0) { last_timestamp_ms_ = getCurrentTimestampMs(); }

Unique64& Unique64::getInstance() {
    static Unique64 instance;
    return instance;
}

std::string Unique64::generateKey() {
    std::lock_guard<std::mutex> guard(mutex_);
    uint64_t current_ts_ms = getCurrentTimestampMs();
    
    if (current_ts_ms < last_timestamp_ms_) { current_ts_ms = last_timestamp_ms_; }
    
    if (current_ts_ms == last_timestamp_ms_) {
        if (sequence_ == MAX_SEQUENCE) {
            current_ts_ms = waitUntilNextMillisecond(last_timestamp_ms_);
            sequence_ = 0;
        } else {
            sequence_++;
        }
    } else {
        sequence_ = 0;
    }
    
    last_timestamp_ms_ = current_ts_ms;
    
    const int SEQUENCE_BITS = 16;
    uint64_t id = (last_timestamp_ms_ << SEQUENCE_BITS) | sequence_;
    return std::to_string(id);
}

uint64_t Unique64::getCurrentTimestampMs() {
    using namespace std::chrono;
    return duration_cast<milliseconds>(system_clock::now().time_since_epoch()).count();
}

uint64_t Unique64::waitUntilNextMillisecond(uint64_t current_ts_ms_ref) {
    uint64_t next_ts_ms = getCurrentTimestampMs();
    while (next_ts_ms <= current_ts_ms_ref) {
        std::this_thread::sleep_for(std::chrono::microseconds(100)); // Sleep briefly
        next_ts_ms = getCurrentTimestampMs();
    }
    return next_ts_ms;    
}

template<typename Key, typename Value>
void ThreadSafeMap<Key, Value>::insert(const Key& key, const Value& value) {
    std::lock_guard<std::mutex> lock(mutex_);
    map_[key] = value;
}

template<typename Key, typename Value>
Value ThreadSafeMap<Key, Value>::get(const Key& key) const {
    std::lock_guard<std::mutex> lock(mutex_);
    auto it = map_.find(key);
    if (it != map_.end()) {
        return it->second;
    }
    return NULL;
}

template<typename Key, typename Value>
bool ThreadSafeMap<Key, Value>::contains(const Key& key) const {
    std::lock_guard<std::mutex> lock(mutex_);
    return map_.find(key) != map_.end();
}

template<typename Key, typename Value>
size_t ThreadSafeMap<Key, Value>::erase(const Key& key) {
    std::lock_guard<std::mutex> lock(mutex_);
    return map_.erase(key);
}


NS_HIVE_END
#endif