//
//  HIVE_MatchMaking.cpp
//  HIVE_SDK_Plugin
//
//  Created by jschoi on 2024/09/26.
//

#include "HIVE_MatchMaking.h"
#include "HIVE_CppPlugin.h"

#if !WITH_EDITOR && PLATFORM_WINDOWS

NS_HIVE_BEGIN

static MatchMaking::onMatchMakingData g_MatchMakingRequestMatchMaking = nullptr;
static MatchMaking::onMatchMakingData g_MatchMakingGetRequestingStatus = nullptr;
static MatchMaking::onMatchMakingResult g_MatchMakingDeleteRequesting = nullptr;
static MatchMaking::onMatchMakingGroupData g_MatchMakingCreateGroup = nullptr;
static MatchMaking::onMatchMakingGroupData g_MatchMakingJoinGroup = nullptr;
static MatchMaking::onMatchMakingResult g_MatchMakingLeaveGroup = nullptr;
static MatchMaking::onMatchMakingGroupData g_MatchMakingKickGroupUser = nullptr;
static MatchMaking::onMatchMakingGroupData g_MatchMakingGetGroupInfoByUser = nullptr;
static MatchMaking::onMatchMakingGroupData g_MatchMakingGetGroupInfoByGroupCode = nullptr;
static MatchMaking::onMatchMakingGroupData g_MatchMakingUpdateGroupUser = nullptr;
static MatchMaking::onMatchMakingGroupData g_MatchMakingRequestGroupMatching = nullptr;
static MatchMaking::onMatchMakingGroupData g_MatchMakingDeleteGroupMatching = nullptr;

const char* MATCHMAKING_CLASS_NAME = "MatchMaking";
const char* MATCHMAKING_FUNC_NAME_REQUEST_MATCH_MAKING = "requestMatchMaking";
const char* MATCHMAKING_FUNC_NAME_GET_REQUESTING_STATUS = "getRequestingStatus";
const char* MATCHMAKING_FUNC_NAME_DELETE_REQUESTING = "deleteRequesting";
const char* MATCHMAKING_FUNC_NAME_CREATE_GROUP = "createGroup";
const char* MATCHMAKING_FUNC_NAME_JOIN_GROUP = "joinGroup";
const char* MATCHMAKING_FUNC_NAME_LEAVE_GROUP = "leaveGroup";
const char* MATCHMAKING_FUNC_NAME_KICK_GROUP_USER = "kickGroupUser";
const char* MATCHMAKING_FUNC_NAME_GET_GROUP_INFO_BY_USER = "getGroupInfoByUser";
/**
 * \~korean
 *  그룹 코드로 그룹 정보 조회<br/>
 *
 *  그룹 코드를 기준으로 그룹 정보를 조회합니다.<br/>
 *
 * \~english
 * Get group information by group code<br/>
 *
 * Get group information based on group code.<br/>
 * \~
 *
 * @since 4.25.0.0
 * @ingroup MatchMaking
 */
const char* MATCHMAKING_FUNC_NAME_GET_GROUP_INFO_BY_GROUP_CODE = "getGroupInfoByGroupCode";
const char* MATCHMAKING_FUNC_NAME_UPDATE_GROUP_USER = "updateGroupUser";
const char* MATCHMAKING_FUNC_NAME_REQUEST_GROUP_MATCHING = "requestGroupMatching";
const char* MATCHMAKING_FUNC_NAME_DELETE_GROUP_MATCHING = "deleteGroupMatching";


void MatchMaking::requestMatchMaking(int matchId, int point, const char * extraData, onMatchMakingData listener) {
    g_MatchMakingRequestMatchMaking = listener;

    picojson::object jsonParam = HiveCppPlugin::createParam(MATCHMAKING_CLASS_NAME, MATCHMAKING_FUNC_NAME_REQUEST_MATCH_MAKING);
    
    jsonParam["matchId"] = picojson::value((double)matchId);
    jsonParam["point"] = picojson::value((double)point);
    jsonParam["extraData"] = picojson::value(extraData);

    HiveCppPlugin::callNative(jsonParam);
}

void MatchMaking::getRequestingStatus(int matchId, onMatchMakingData listener) {
    g_MatchMakingGetRequestingStatus = listener;

    picojson::object jsonParam = HiveCppPlugin::createParam(MATCHMAKING_CLASS_NAME, MATCHMAKING_FUNC_NAME_GET_REQUESTING_STATUS);
    jsonParam["matchId"] = picojson::value((double)matchId);

    HiveCppPlugin::callNative(jsonParam);
}

void MatchMaking::deleteRequesting(int matchId, onMatchMakingResult listener) {
    g_MatchMakingDeleteRequesting = listener;

    picojson::object jsonParam = HiveCppPlugin::createParam(MATCHMAKING_CLASS_NAME, MATCHMAKING_FUNC_NAME_DELETE_REQUESTING);
    jsonParam["matchId"] = picojson::value((double)matchId);

    HiveCppPlugin::callNative(jsonParam);
}

void MatchMaking::createGroup(int matchId, int point, const char * extraData, onMatchMakingGroupData listener) {
    g_MatchMakingCreateGroup = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam(MATCHMAKING_CLASS_NAME, MATCHMAKING_FUNC_NAME_CREATE_GROUP);
    jsonParam["matchId"] = picojson::value((double)matchId);
    jsonParam["point"] = picojson::value((double)point);
    jsonParam["extraData"] = picojson::value(extraData);
    
    HiveCppPlugin::callNative(jsonParam);
}

void MatchMaking::joinGroup(int matchId, const char * groupCode, int point, const char * extraData, onMatchMakingGroupData listener) {
    g_MatchMakingJoinGroup = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam(MATCHMAKING_CLASS_NAME, MATCHMAKING_FUNC_NAME_JOIN_GROUP);
    jsonParam["matchId"] = picojson::value((double)matchId);
    jsonParam["groupCode"] = picojson::value(groupCode);
    jsonParam["point"] = picojson::value((double)point);
    jsonParam["extraData"] = picojson::value(extraData);
    
    HiveCppPlugin::callNative(jsonParam);
}

void MatchMaking::leaveGroup(int matchId, onMatchMakingResult listener) {
    g_MatchMakingLeaveGroup = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam(MATCHMAKING_CLASS_NAME, MATCHMAKING_FUNC_NAME_LEAVE_GROUP);
    jsonParam["matchId"] = picojson::value((double)matchId);
    
    HiveCppPlugin::callNative(jsonParam);
}

void MatchMaking::kickGroupUser(int matchId, PlayerID targetPlayerId, onMatchMakingGroupData listener) {
    g_MatchMakingKickGroupUser = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam(MATCHMAKING_CLASS_NAME, MATCHMAKING_FUNC_NAME_KICK_GROUP_USER);
    jsonParam["matchId"] = picojson::value((double)matchId);
    jsonParam["targetPlayerId"] = picojson::value((double)targetPlayerId);
    
    HiveCppPlugin::callNative(jsonParam);
}


void MatchMaking::getGroupInfoByUser(int matchId, onMatchMakingGroupData listener) {
    g_MatchMakingGetGroupInfoByUser = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam(MATCHMAKING_CLASS_NAME, MATCHMAKING_FUNC_NAME_GET_GROUP_INFO_BY_USER);
    jsonParam["matchId"] = picojson::value((double)matchId);
    
    HiveCppPlugin::callNative(jsonParam);
}

void MatchMaking::getGroupInfoByGroupCode(const char * groupCode, onMatchMakingGroupData listener) {
    g_MatchMakingGetGroupInfoByGroupCode = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam(MATCHMAKING_CLASS_NAME, MATCHMAKING_FUNC_NAME_GET_GROUP_INFO_BY_GROUP_CODE);
    jsonParam["groupCode"] = picojson::value(groupCode);
    
    HiveCppPlugin::callNative(jsonParam);
}

void MatchMaking::updateGroupUser(int matchId, bool ready, int point, const char * extraData, onMatchMakingGroupData listener) {
    g_MatchMakingUpdateGroupUser = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam(MATCHMAKING_CLASS_NAME, MATCHMAKING_FUNC_NAME_UPDATE_GROUP_USER);
    jsonParam["matchId"] = picojson::value((double)matchId);
    jsonParam["ready"] = picojson::value(ready);
    jsonParam["point"] = picojson::value((double)point);
    jsonParam["extraData"] = picojson::value(extraData);
    
    HiveCppPlugin::callNative(jsonParam);
}

void MatchMaking::requestGroupMatching(int matchId, onMatchMakingGroupData listener) {
    g_MatchMakingRequestGroupMatching = listener;

    picojson::object jsonParam = HiveCppPlugin::createParam(MATCHMAKING_CLASS_NAME, MATCHMAKING_FUNC_NAME_REQUEST_GROUP_MATCHING);
    jsonParam["matchId"] = picojson::value((double)matchId);

    HiveCppPlugin::callNative(jsonParam);
}


void MatchMaking::deleteGroupMatching(int matchId, onMatchMakingGroupData listener) {
    g_MatchMakingDeleteGroupMatching = listener;
    
    picojson::object jsonParam = HiveCppPlugin::createParam(MATCHMAKING_CLASS_NAME, MATCHMAKING_FUNC_NAME_DELETE_GROUP_MATCHING);
    jsonParam["matchId"] = picojson::value((double)matchId);
    
    HiveCppPlugin::callNative(jsonParam);
}
    

void MatchMaking::executeEngine(picojson::value jsonParam) {
    ResultAPI result(jsonParam.get("resultAPI"));

    std::string methodName = jsonParam.get("method").to_str();
    if (methodName.compare(MATCHMAKING_FUNC_NAME_REQUEST_MATCH_MAKING) == 0) {
        if (g_MatchMakingRequestMatchMaking != nullptr) {
            MatchMakingData matchMakingData;
            matchMakingData.parse(jsonParam.get("matchMakingData"));

            g_MatchMakingRequestMatchMaking(result, matchMakingData);
            return;
        }
    }
    else if (methodName.compare(MATCHMAKING_FUNC_NAME_GET_REQUESTING_STATUS) == 0) {
        if (g_MatchMakingGetRequestingStatus != nullptr) {
            MatchMakingData matchMakingData;
            matchMakingData.parse(jsonParam.get("matchMakingData"));

            g_MatchMakingGetRequestingStatus(result, matchMakingData);
            return;
        }
    }
    else if (methodName.compare(MATCHMAKING_FUNC_NAME_DELETE_REQUESTING) == 0) {
        if (g_MatchMakingDeleteRequesting != nullptr) {
            g_MatchMakingDeleteRequesting(result);
            return;
        }
    }
    else if (methodName.compare(MATCHMAKING_FUNC_NAME_CREATE_GROUP) == 0) {
        if (g_MatchMakingCreateGroup != nullptr) {
            MatchMakingGroupData matchMakingGroupData;
            matchMakingGroupData.parse(jsonParam.get("matchMakingGroupData"));
            
            g_MatchMakingCreateGroup(result, matchMakingGroupData);
            return;
        }
    }
    else if (methodName.compare(MATCHMAKING_FUNC_NAME_JOIN_GROUP) == 0) {
        if (g_MatchMakingJoinGroup != nullptr) {
            MatchMakingGroupData matchMakingGroupData;
            matchMakingGroupData.parse(jsonParam.get("matchMakingGroupData"));
            
            g_MatchMakingJoinGroup(result, matchMakingGroupData);
            return;
        }
    }
    else if (methodName.compare(MATCHMAKING_FUNC_NAME_LEAVE_GROUP) == 0) {
        if (g_MatchMakingLeaveGroup != nullptr) {
            g_MatchMakingLeaveGroup(result);
            return;
        }
    }
    else if (methodName.compare(MATCHMAKING_FUNC_NAME_KICK_GROUP_USER) == 0) {
        if (g_MatchMakingKickGroupUser != nullptr) {
            MatchMakingGroupData matchMakingGroupData;
            matchMakingGroupData.parse(jsonParam.get("matchMakingGroupData"));
            
            g_MatchMakingKickGroupUser(result, matchMakingGroupData);
            return;
        }
    }
    else if (methodName.compare(MATCHMAKING_FUNC_NAME_GET_GROUP_INFO_BY_USER) == 0) {
        if (g_MatchMakingGetGroupInfoByUser != nullptr) {
            MatchMakingGroupData matchMakingGroupData;
            matchMakingGroupData.parse(jsonParam.get("matchMakingGroupData"));
            
            g_MatchMakingGetGroupInfoByUser(result, matchMakingGroupData);
            return;
        }
    }
    else if (methodName.compare(MATCHMAKING_FUNC_NAME_GET_GROUP_INFO_BY_GROUP_CODE) == 0) {
        if (g_MatchMakingGetGroupInfoByGroupCode != nullptr) {
            MatchMakingGroupData matchMakingGroupData;
            matchMakingGroupData.parse(jsonParam.get("matchMakingGroupData"));
            
            g_MatchMakingGetGroupInfoByGroupCode(result, matchMakingGroupData);
            return;
        }
    }
    else if (methodName.compare(MATCHMAKING_FUNC_NAME_UPDATE_GROUP_USER) == 0) {
        if (g_MatchMakingUpdateGroupUser != nullptr) {
            MatchMakingGroupData matchMakingGroupData;
            matchMakingGroupData.parse(jsonParam.get("matchMakingGroupData"));
            
            g_MatchMakingUpdateGroupUser(result, matchMakingGroupData);
            return;
        }
    }
    else if (methodName.compare(MATCHMAKING_FUNC_NAME_REQUEST_GROUP_MATCHING) == 0) {
        if (g_MatchMakingRequestGroupMatching != nullptr) {
            MatchMakingGroupData matchMakingGroupData;
            matchMakingGroupData.parse(jsonParam.get("matchMakingGroupData"));
            
            g_MatchMakingRequestGroupMatching(result, matchMakingGroupData);
            return;
        }
    }
    else if (methodName.compare(MATCHMAKING_FUNC_NAME_DELETE_GROUP_MATCHING) == 0) {
        if (g_MatchMakingDeleteGroupMatching != nullptr) {
            MatchMakingGroupData matchMakingGroupData;
            matchMakingGroupData.parse(jsonParam.get("matchMakingGroupData"));
            
            g_MatchMakingDeleteGroupMatching(result, matchMakingGroupData);
            return;
        }
    }

}

MatchingMemberInfo::MatchingMemberInfo() {
    this->playerId = 0;
    this->ready = false;
    this->point = 0;
    this->extraData = "";
}

bool MatchingMemberInfo::parse(picojson::value jsonParam) {
    if(jsonParam.is<picojson::null>())
        return false;

    // jsonParam이 string인 경우 먼저 파싱
    std::string jsonString;
    if (jsonParam.is<std::string>()) {
        jsonString = jsonParam.get<std::string>();
        picojson::value parseValue;
        std::string err = picojson::parse(parseValue, jsonString);
        if (!err.empty()) {
            return false;
        }
        jsonParam = parseValue;
    }

    // object가 아닌 경우 실패
    if (!jsonParam.is<picojson::object>()) {
        return false;
    }

    picojson::object jsonObject = jsonParam.get<picojson::object>();
    
    // playerId 파싱
    auto playerIdIt = jsonObject.find("playerId");
    if (playerIdIt != jsonObject.end() && !playerIdIt->second.is<picojson::null>()) {
        this->playerId = static_cast<PlayerID>(playerIdIt->second.get<double>());
    }

    // ready 파싱 
    auto readyIt = jsonObject.find("ready");
    if (readyIt != jsonObject.end() && !readyIt->second.is<picojson::null>()) {
        if (readyIt->second.is<bool>()) {
            this->ready = readyIt->second.get<bool>();
        } else if (readyIt->second.is<std::string>()) {
            // "true" 또는 "false" 문자열 처리
            std::string readyStr = readyIt->second.get<std::string>();
            this->ready = (readyStr == "true");
        }
    }
    
    // point 파싱
    auto pointIt = jsonObject.find("point"); 
    if (pointIt != jsonObject.end() && !pointIt->second.is<picojson::null>()) {
        this->point = static_cast<int>(pointIt->second.get<double>());
    }
    
    // extraData 파싱
    auto extraDataIt = jsonObject.find("extraData");
    if (extraDataIt != jsonObject.end() && !extraDataIt->second.is<picojson::null>()) {
        this->extraData = extraDataIt->second.get<std::string>();
    }

    return true;
}

picojson::object MatchingMemberInfo::toJson() const {
    picojson::object resJson;
    resJson["playerId"] = picojson::value((double)this->playerId);
    resJson["ready"] = picojson::value(this->ready);
    resJson["point"] = picojson::value((double)this->point);
    resJson["extraData"] = picojson::value(this->extraData);
    return resJson;
}

std::string MatchingMemberInfo::toString() const {
    picojson::object resJson = this->toJson();
    return picojson::value(resJson).serialize();
}

MatchingResultPlayerInfo::MatchingResultPlayerInfo() {
    this->playerId = 0;
    this->point = 0;
    this->extraData = "";
}

bool MatchingResultPlayerInfo::parse(picojson::value jsonParam) {
    if(jsonParam.is<picojson::null>())
        return false;

    // jsonParam이 string인 경우 먼저 파싱
    std::string jsonString;
    if (jsonParam.is<std::string>()) {
        jsonString = jsonParam.get<std::string>();
        picojson::value parseValue;
        std::string err = picojson::parse(parseValue, jsonString);
        if (!err.empty()) {
            return false;
        }
        jsonParam = parseValue;
    }

    // object가 아닌 경우 실패
    if (!jsonParam.is<picojson::object>()) {
        return false;
    }

    picojson::object jsonObject = jsonParam.get<picojson::object>();

    // playerId 파싱
    auto playerIdIt = jsonObject.find("playerId");
    if (playerIdIt != jsonObject.end() && !playerIdIt->second.is<picojson::null>()) {
        this->playerId = static_cast<PlayerID>(playerIdIt->second.get<double>());
    }

    // point 파싱
    auto pointIt = jsonObject.find("point"); 
    if (pointIt != jsonObject.end() && !pointIt->second.is<picojson::null>()) {
        this->point = static_cast<int>(pointIt->second.get<double>());
    }

    // extraData 파싱
    auto extraDataIt = jsonObject.find("extraData");
    if (extraDataIt != jsonObject.end() && !extraDataIt->second.is<picojson::null>()) {
        this->extraData = extraDataIt->second.get<std::string>();
    }

    return true;
}

picojson::object MatchingResultPlayerInfo::toJson() const {
    picojson::object resJson;
    resJson["playerId"] = picojson::value((double)this->playerId);
    resJson["point"] = picojson::value((double)this->point);
    resJson["extraData"] = picojson::value(this->extraData);
    return resJson;
}

std::string MatchingResultPlayerInfo::toString() const {
    picojson::object resJson = this->toJson();
    return picojson::value(resJson).serialize();
}

MatchingResultTeamInfo::MatchingResultTeamInfo() {
    this->teamIndex = 0;
}

bool MatchingResultTeamInfo::parse(picojson::value jsonParam) {
    if(jsonParam.is<picojson::null>())
        return false;

    // jsonParam이 string인 경우 먼저 파싱
    std::string jsonString;
    if (jsonParam.is<std::string>()) {
        jsonString = jsonParam.get<std::string>();
        picojson::value parseValue;
        std::string err = picojson::parse(parseValue, jsonString);
        if (!err.empty()) {
            return false;
        }
        jsonParam = parseValue;
    }

    // object가 아닌 경우 실패
    if (!jsonParam.is<picojson::object>()) {
        return false;
    }

    picojson::object jsonObject = jsonParam.get<picojson::object>();

    // teamIndex 파싱
    auto teamIndexIt = jsonObject.find("teamIndex");
    if (teamIndexIt != jsonObject.end() && !teamIndexIt->second.is<picojson::null>()) {
        this->teamIndex = static_cast<int>(teamIndexIt->second.get<double>());
    }

    // playerInfos 파싱
    auto playerInfosIt = jsonObject.find("playerInfos");
    if (playerInfosIt != jsonObject.end() && playerInfosIt->second.is<picojson::array>()) {
        picojson::array playerInfoArray = playerInfosIt->second.get<picojson::array>();
        this->playerInfos.clear();
        for(const auto& value : playerInfoArray) {
            MatchingResultPlayerInfo playerInfo;
            if(playerInfo.parse(value)) {
                this->playerInfos.push_back(playerInfo);
            }
        }
    }

    return true;
}

picojson::object MatchingResultTeamInfo::toJson() const {
    picojson::object resJson;
    resJson["teamIndex"] = picojson::value((double)this->teamIndex);
    
    picojson::array playerInfoArray;
    for(const auto& playerInfo : this->playerInfos) {
        playerInfoArray.push_back(picojson::value(playerInfo.toJson()));
    }
    resJson["playerInfos"] = picojson::value(playerInfoArray);
    
    return resJson;
}

std::string MatchingResultTeamInfo::toString() const {
    picojson::object resJson = this->toJson();
    return picojson::value(resJson).serialize();
}

MatchMakingGroupData::MatchMakingGroupData() {
    this->groupCode = "";
    this->ownerPlayerId = 0;
    this->requestGameIndex = 0;
    this->requestMatchId = 0;
    this->requestStatus = "";
    this->requestTimeUtc = "";
    this->matchingStatus = "";
    this->matchingId = "";
    this->matchingType = "";
    this->memberInfoList.clear();
    this->matchingTeamInfoList.clear();
}

bool MatchMakingGroupData::parse(picojson::value jsonParam) {
    if(jsonParam.is<picojson::null>())
        return false;

    // jsonParam이 string인 경우 먼저 파싱
    std::string jsonString;
    if (jsonParam.is<std::string>()) {
        jsonString = jsonParam.get<std::string>();
        picojson::value parseValue;
        std::string err = picojson::parse(parseValue, jsonString);
        if (!err.empty()) {
            return false;
        }
        jsonParam = parseValue;
    }

    // object가 아닌 경우 실패
    if (!jsonParam.is<picojson::object>()) {
        return false;
    }

    picojson::object jsonObject = jsonParam.get<picojson::object>();

    // 각 필드 파싱
    auto groupCodeIt = jsonObject.find("groupCode");
    if (groupCodeIt != jsonObject.end() && !groupCodeIt->second.is<picojson::null>()) {
        this->groupCode = groupCodeIt->second.get<std::string>();
    }

    auto ownerPlayerIdIt = jsonObject.find("ownerPlayerId");
    if (ownerPlayerIdIt != jsonObject.end() && !ownerPlayerIdIt->second.is<picojson::null>()) {
        this->ownerPlayerId = static_cast<PlayerID>(ownerPlayerIdIt->second.get<double>());
    }

    auto requestGameIndexIt = jsonObject.find("requestGameIndex");
    if (requestGameIndexIt != jsonObject.end() && !requestGameIndexIt->second.is<picojson::null>()) {
        this->requestGameIndex = static_cast<int>(requestGameIndexIt->second.get<double>());
    }

    auto requestMatchIdIt = jsonObject.find("requestMatchId");
    if (requestMatchIdIt != jsonObject.end() && !requestMatchIdIt->second.is<picojson::null>()) {
        this->requestMatchId = static_cast<int>(requestMatchIdIt->second.get<double>());
    }

    auto requestStatusIt = jsonObject.find("requestStatus");
    if (requestStatusIt != jsonObject.end() && !requestStatusIt->second.is<picojson::null>()) {
        this->requestStatus = requestStatusIt->second.get<std::string>();
    }

    auto requestTimeUtcIt = jsonObject.find("requestTimeUtc");
    if (requestTimeUtcIt != jsonObject.end() && !requestTimeUtcIt->second.is<picojson::null>()) {
        this->requestTimeUtc = requestTimeUtcIt->second.get<std::string>();
    }

    auto matchingStatusIt = jsonObject.find("matchingStatus");
    if (matchingStatusIt != jsonObject.end() && !matchingStatusIt->second.is<picojson::null>()) {
        this->matchingStatus = matchingStatusIt->second.get<std::string>();
    }

    auto matchingIdIt = jsonObject.find("matchingId");
    if (matchingIdIt != jsonObject.end() && !matchingIdIt->second.is<picojson::null>()) {
        this->matchingId = matchingIdIt->second.get<std::string>();
    }

    auto matchingTypeIt = jsonObject.find("matchingType");
    if (matchingTypeIt != jsonObject.end() && !matchingTypeIt->second.is<picojson::null>()) {
        this->matchingType = matchingTypeIt->second.get<std::string>();
    }

    // memberInfoList 파싱
    auto memberInfoListIt = jsonObject.find("memberInfoList");
    if (memberInfoListIt != jsonObject.end() && memberInfoListIt->second.is<picojson::array>()) {
        picojson::array memberArray = memberInfoListIt->second.get<picojson::array>();
        this->memberInfoList.clear();
        for(const auto& value : memberArray) {
            MatchingMemberInfo memberInfo;
            if(memberInfo.parse(value)) {
                this->memberInfoList.push_back(memberInfo);
            }
        }
    }

    // MatchMakingGroupData 파싱
    auto matchingTeamInfoListIt = jsonObject.find("matchingTeamInfoList");
    if (matchingTeamInfoListIt != jsonObject.end() && matchingTeamInfoListIt->second.is<picojson::array>()) {
        picojson::array teamInfoArray = matchingTeamInfoListIt->second.get<picojson::array>();
        this->matchingTeamInfoList.clear();
        for(const auto& value : teamInfoArray) {
            MatchingResultTeamInfo teamInfo;
            if(teamInfo.parse(value)) {
                this->matchingTeamInfoList.push_back(teamInfo);
            }
        }
    }

    return true;
}

picojson::object MatchMakingGroupData::toJson() const {
    picojson::object resJson;

    resJson["groupCode"] = picojson::value(this->groupCode);
    resJson["ownerPlayerId"] = picojson::value((double)this->ownerPlayerId);
    resJson["requestGameIndex"] = picojson::value((double)this->requestGameIndex);
    resJson["requestMatchId"] = picojson::value((double)this->requestMatchId);

    resJson["requestStatus"] = picojson::value(this->requestStatus);
    resJson["requestTimeUtc"] = picojson::value(this->requestTimeUtc);
    resJson["matchingStatus"] = picojson::value(this->matchingStatus);
    resJson["matchingId"] = picojson::value(this->matchingId);
    resJson["matchingType"] = picojson::value(this->matchingType);

    picojson::array memberInfoArray;
    for(const auto& memberInfo : this->memberInfoList) {
        memberInfoArray.push_back(picojson::value(memberInfo.toJson()));
    }
    resJson["memberInfoList"] = picojson::value(memberInfoArray);

    picojson::array teamInfoArray;
    for(const auto& teamInfo : this->matchingTeamInfoList) {
        teamInfoArray.push_back(picojson::value(teamInfo.toJson()));
    }
    resJson["matchingTeamInfoList"] = picojson::value(teamInfoArray);

    return resJson;
}

std::string MatchMakingGroupData::toString() const {
    picojson::object resJson = this->toJson();
    std::string resJsonString = picojson::value(resJson).serialize();
    return resJsonString;
}

MatchMakingData::MatchMakingData() {
    this->requestPlayerId = 0;
    this->requestGameIndex = 0;
    this->requestMatchId = 0;
    this->requestStatus = "";
    this->requestTimeUtc = "";
    this->requestPoint = 0;
    this->requestExtraData = "";
    this->matchingStatus = "";
    this->matchingId = "";
    this->matchingType = "";
    this->matchingPlayerInfoList.clear();
    this->matchingTeamInfoList.clear();
}

bool MatchMakingData::parse(picojson::value jsonParam) {
    if(jsonParam.is<picojson::null>())
        return false;

    // jsonParam이 string인 경우 먼저 파싱
    std::string jsonString;
    if (jsonParam.is<std::string>()) {
        jsonString = jsonParam.get<std::string>();
        picojson::value parseValue;
        std::string err = picojson::parse(parseValue, jsonString);
        if (!err.empty()) {
            return false;
        }
        jsonParam = parseValue;
    }

    // object가 아닌 경우 실패
    if (!jsonParam.is<picojson::object>()) {
        return false;
    }

    picojson::object jsonObject = jsonParam.get<picojson::object>();

    // 각 필드 파싱
    auto requestPlayerIdIt = jsonObject.find("requestPlayerId");
    if (requestPlayerIdIt != jsonObject.end() && !requestPlayerIdIt->second.is<picojson::null>()) {
        this->requestPlayerId = static_cast<PlayerID>(requestPlayerIdIt->second.get<double>());
    }

    auto requestGameIndexIt = jsonObject.find("requestGameIndex");
    if (requestGameIndexIt != jsonObject.end() && !requestGameIndexIt->second.is<picojson::null>()) {
        this->requestGameIndex = static_cast<int>(requestGameIndexIt->second.get<double>());
    }

    auto requestMatchIdIt = jsonObject.find("requestMatchId");
    if (requestMatchIdIt != jsonObject.end() && !requestMatchIdIt->second.is<picojson::null>()) {
        this->requestMatchId = static_cast<int>(requestMatchIdIt->second.get<double>());
    }

    auto requestStatusIt = jsonObject.find("requestStatus");
    if (requestStatusIt != jsonObject.end() && !requestStatusIt->second.is<picojson::null>()) {
        this->requestStatus = requestStatusIt->second.get<std::string>();
    }

    auto requestTimeUtcIt = jsonObject.find("requestTimeUtc");
    if (requestTimeUtcIt != jsonObject.end() && !requestTimeUtcIt->second.is<picojson::null>()) {
        this->requestTimeUtc = requestTimeUtcIt->second.get<std::string>();
    }

    auto requestPointIt = jsonObject.find("requestPoint");
    if (requestPointIt != jsonObject.end() && !requestPointIt->second.is<picojson::null>()) {
        this->requestPoint = static_cast<int>(requestPointIt->second.get<double>());
    }

    auto requestExtraDataIt = jsonObject.find("requestExtraData");
    if (requestExtraDataIt != jsonObject.end() && !requestExtraDataIt->second.is<picojson::null>()) {
        this->requestExtraData = requestExtraDataIt->second.get<std::string>();
    }

    auto matchingStatusIt = jsonObject.find("matchingStatus");
    if (matchingStatusIt != jsonObject.end() && !matchingStatusIt->second.is<picojson::null>()) {
        this->matchingStatus = matchingStatusIt->second.get<std::string>();
    }

    auto matchingIdIt = jsonObject.find("matchingId");
    if (matchingIdIt != jsonObject.end() && !matchingIdIt->second.is<picojson::null>()) {
        this->matchingId = matchingIdIt->second.get<std::string>();
    }

    auto matchingTypeIt = jsonObject.find("matchingType");
    if (matchingTypeIt != jsonObject.end() && !matchingTypeIt->second.is<picojson::null>()) {
        this->matchingType = matchingTypeIt->second.get<std::string>();
    }

    // matchingPlayerInfoList 파싱
    auto playerInfoListIt = jsonObject.find("matchingPlayerInfoList");
    if (playerInfoListIt != jsonObject.end() && playerInfoListIt->second.is<picojson::array>()) {
        picojson::array playerArray = playerInfoListIt->second.get<picojson::array>();
        this->matchingPlayerInfoList.clear();
        for(const auto& value : playerArray) {
            MatchingResultPlayerInfo playerInfo;
            if(playerInfo.parse(value)) {
                this->matchingPlayerInfoList.push_back(playerInfo);
            }
        }
    }

    // matchingTeamInfoList 파싱
    auto teamInfoListIt = jsonObject.find("matchingTeamInfoList");
    if (teamInfoListIt != jsonObject.end() && teamInfoListIt->second.is<picojson::array>()) {
        picojson::array teamArray = teamInfoListIt->second.get<picojson::array>();
        this->matchingTeamInfoList.clear();
        for(const auto& value : teamArray) {
            MatchingResultTeamInfo teamInfo;
            if(teamInfo.parse(value)) {
                this->matchingTeamInfoList.push_back(teamInfo);
            }
        }
    }

    return true;
}

picojson::object MatchMakingData::toJson() const {
    picojson::object resJson;

    resJson["requestPlayerId"] = picojson::value((double)this->requestPlayerId);
    resJson["requestGameIndex"] = picojson::value((double)this->requestGameIndex);
    resJson["requestMatchId"] = picojson::value((double)this->requestMatchId);

    resJson["requestStatus"] = picojson::value(this->requestStatus);
    resJson["requestTimeUtc"] = picojson::value(this->requestTimeUtc);
    resJson["requestPoint"] = picojson::value((double)this->requestPoint);
    resJson["requestExtraData"] = picojson::value(this->requestExtraData);

    resJson["matchingStatus"] = picojson::value(this->matchingStatus);
    resJson["matchingId"] = picojson::value(this->matchingId);
    resJson["matchingType"] = picojson::value(this->matchingType);

    picojson::array playerInfoArray;
    for(const auto& playerInfo : this->matchingPlayerInfoList) {
        playerInfoArray.push_back(picojson::value(playerInfo.toJson()));
    }
    resJson["matchingPlayerInfoList"] = picojson::value(playerInfoArray);

    picojson::array teamInfoArray;
    for(const auto& teamInfo : this->matchingTeamInfoList) {
        teamInfoArray.push_back(picojson::value(teamInfo.toJson()));
    }
    resJson["matchingTeamInfoList"] = picojson::value(teamInfoArray);

    return resJson;
}

std::string MatchMakingData::toString() const {
    picojson::object resJson = this->toJson();
    std::string resJsonString = picojson::value(resJson).serialize();
    return resJsonString;
}

NS_HIVE_END

#endif