/* Copyright © 2024 Com2uS Platform Corp. All Rights Reserved. */
#include "Impl/HiveAuthV4Impl.h"
#include "HiveUELogger.h"

void IHiveAuthV4Impl::Setup(const FHiveAuthV4OnSetupDelegate& Delegate)
{
    hive::AuthV4::setup([Delegate](hive::ResultAPI const & result, bool isAutoSignIn, std::string did, std::vector<hive::ProviderType> const & providerTypeList)
    {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);

            FString Did(did.c_str());

            TArray<EHiveProviderType> ProviderTypeArray;

            if (providerTypeList.size() > 0)
            {
                ProviderTypeArray.Reserve(providerTypeList.size());

                for(const auto& element : providerTypeList)
                {
                    ProviderTypeArray.Add(static_cast<EHiveProviderType>(element));
                }
            }
            
            Delegate.Execute(Result, isAutoSignIn, Did, ProviderTypeArray);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }

    });    
}


void IHiveAuthV4Impl::SignIn(const EHiveProviderType& ProviderType, const FHiveAuthV4OnSignInDelegate& Delegate)
{
    hive::AuthV4::signIn(static_cast<hive::ProviderType>(ProviderType), [Delegate](hive::ResultAPI const & result, hive::PlayerInfo const & playerInfo)
    {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            FHivePlayerInfo PlayerInfo(playerInfo);

            Delegate.Execute(Result, PlayerInfo);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::SignInWithAuthKey(const FString& AuthKey, const FHiveAuthV4OnSignInDelegate& Delegate)
{
    std::string StrAuthKey(TCHAR_TO_UTF8(*AuthKey));

    hive::AuthV4::signInWithAuthKey(StrAuthKey, [Delegate](hive::ResultAPI const & result, hive::PlayerInfo const & playerInfo)
    {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            FHivePlayerInfo PlayerInfo(playerInfo);

            Delegate.Execute(Result, PlayerInfo);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    }); 
}


void IHiveAuthV4Impl::SignOut(const FHiveAuthV4OnSignOutDelegate& Delegate)
{
    hive::AuthV4::signOut([Delegate](hive::ResultAPI const & result)
    {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            Delegate.Execute(Result);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::PlayerDelete(const FHiveAuthV4OnSignOutDelegate& Delegate)
{
    hive::AuthV4::playerDelete([Delegate](hive::ResultAPI const & result)
    {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            Delegate.Execute(Result);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::Connect(const EHiveProviderType& ProviderType, const FHiveAuthV4OnConnectDelegate& Delegate)
{
    hive::AuthV4::connect(static_cast<hive::ProviderType>(ProviderType), [Delegate](hive::ResultAPI const & result, hive::PlayerInfo const & conflictPlayer)
    {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            FHivePlayerInfo ConflictPlayerInfo(conflictPlayer);
            Delegate.Execute(Result, ConflictPlayerInfo);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::Disconnect(const EHiveProviderType& ProviderType, const FHiveAuthV4OnDisconnectDelegate& Delegate)
{
    hive::AuthV4::disconnect(static_cast<hive::ProviderType>(ProviderType), [Delegate](hive::ResultAPI const & result) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            Delegate.Execute(Result);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::ConnectWithAuthKey(const FString& AuthKey, const FHiveAuthV4OnConnectDelegate& Delegate)
{
    std::string StrAuthKey(TCHAR_TO_UTF8(*AuthKey));
    hive::AuthV4::connectWithAuthKey(StrAuthKey, [Delegate](hive::ResultAPI const& result, hive::PlayerInfo const& conflictPlayer) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            FHivePlayerInfo ConflictPlayerInfo(conflictPlayer);
            Delegate.Execute(Result, ConflictPlayerInfo);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::DisconnectWithName(const FString& ProviderName,  const FHiveAuthV4OnDisconnectDelegate& Delegate)
{
    std::string StrProviderName(TCHAR_TO_UTF8(*ProviderName));
    hive::AuthV4::disconnectWithName(StrProviderName, [Delegate](hive::ResultAPI const& result) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            Delegate.Execute(Result);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::SelectConflict(int64 SelectedPlayerId, const FHiveAuthV4OnSignInDelegate& Delegate)
{
    hive::AuthV4::selectConflict(SelectedPlayerId, [Delegate](hive::ResultAPI const & result, hive::PlayerInfo const & playerInfo) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            FHivePlayerInfo PlayerInfo(playerInfo);
            Delegate.Execute(Result, PlayerInfo);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::ShowConflictSelection(const FHiveConflictViewData& ConflictViewData, const FHiveAuthV4OnSignInDelegate& Delegate)
{
    const FString* CurrentPlayerId = ConflictViewData.CurrentPlayerData.Find("player_id");
    const FString* ConflictPlayerId = ConflictViewData.ConflictPlayerData.Find("player_id");
    
    if (CurrentPlayerId == nullptr || ConflictPlayerId == nullptr)
    {
        HIVE_LOG_WARNING(TEXT("PlayerId is nullptr."));
        return;
    }
    
    int64 iCurrentPlayerId = FCString::Atoi64(**CurrentPlayerId);
    int64 iConflictPlayerId = FCString::Atoi64(**ConflictPlayerId);
    
    hive::ConflictViewInfo ConflictViewInfo(iCurrentPlayerId, iConflictPlayerId);
    
    for (const auto& Pair : ConflictViewData.CurrentPlayerData)
    {
        if (Pair.Key == TEXT("player_id"))
        {
            continue;
        }
        
        const FString* CurrentPlayerValue = ConflictViewData.CurrentPlayerData.Find(Pair.Key);
        const FString* ConflictPlayerValue = ConflictViewData.ConflictPlayerData.Find(Pair.Key);
        
        if (CurrentPlayerValue == nullptr || ConflictPlayerValue == nullptr)
        {
            HIVE_LOG_WARNING(TEXT("Value is nullptr."));
            continue;
        }
        
        std::string strKey = TCHAR_TO_UTF8(*(Pair.Key));
        std::string strCurrentPlayerValue = TCHAR_TO_UTF8(**CurrentPlayerValue);
        std::string strConflictPlayerValue = TCHAR_TO_UTF8(**ConflictPlayerValue);
        
        ConflictViewInfo.SetValue(strKey, strCurrentPlayerValue, strConflictPlayerValue);
    }
    
    hive::AuthV4::showConflictSelection(ConflictViewInfo, [Delegate](hive::ResultAPI const & result, hive::PlayerInfo const & playerInfo) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            FHivePlayerInfo PlayerInfo(playerInfo);
            Delegate.Execute(Result, PlayerInfo);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::CheckProvider(const EHiveProviderType& ProviderType, const FHiveAuthV4OnCheckProviderDelegate& Delegate)
{
    hive::AuthV4::checkProvider(static_cast<hive::ProviderType>(ProviderType), [Delegate](hive::ResultAPI const & result, hive::ProviderInfo const & providerInfo){
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            FHiveProviderInfo ProviderInfo(providerInfo);
            Delegate.Execute(Result, ProviderInfo);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::CheckMaintenance(bool IsShow, const FHiveAuthV4OnMaintenanceInfoDelegate& Delegate)
{
    hive::AuthV4::checkMaintenance(IsShow, [Delegate](hive::ResultAPI const & result, std::vector<hive::AuthV4MaintenanceInfo> const & maintenanceInfolist) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);

            TArray<FHiveAuthV4MaintenanceInfo> AuthV4MaintenanceInfoArray;

            if (maintenanceInfolist.size() > 0)
            {
                AuthV4MaintenanceInfoArray.Reserve(maintenanceInfolist.size());

                for (const auto& element : maintenanceInfolist)
                {
                    FHiveAuthV4MaintenanceInfo Info(element);
                    AuthV4MaintenanceInfoArray.Emplace(Info);
                }
            }
            
            Delegate.Execute(Result, AuthV4MaintenanceInfoArray);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::CheckBlacklist(bool IsShow, const FHiveAuthV4OnMaintenanceInfoDelegate& Delegate)
{
    hive::AuthV4::checkBlacklist(IsShow, [Delegate](hive::ResultAPI const & result, std::vector<hive::AuthV4MaintenanceInfo> const & maintenanceInfolist) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);

            TArray<FHiveAuthV4MaintenanceInfo> AuthV4MaintenanceInfoArray;

            if (maintenanceInfolist.size() > 0)
            {
                AuthV4MaintenanceInfoArray.Reserve(maintenanceInfolist.size());
                
                for (const auto& element : maintenanceInfolist)
                {
                    FHiveAuthV4MaintenanceInfo Info(element);
                    AuthV4MaintenanceInfoArray.Emplace(Info);
                }
            }
            
            Delegate.Execute(Result, AuthV4MaintenanceInfoArray);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::SetProviderChangedListener(const FHiveAuthV4OnCheckProviderDelegate& Delegate)
{
    hive::AuthV4::setProviderChangedListener([Delegate](hive::ResultAPI const & result, hive::ProviderInfo const & providerInfo) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            FHiveProviderInfo ProviderInfo(providerInfo);
            Delegate.Execute(Result, ProviderInfo);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::ShowSignIn(const FHiveAuthV4OnSignInDelegate& Delegate)
{
    HIVESDKCEFImeEnable();
    
    hive::AuthV4::showSignIn([Delegate](hive::ResultAPI const & result, hive::PlayerInfo const & playerInfo) {
        
        HIVESDKCEFImeDisable();
        
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            FHivePlayerInfo PlayerInfo(playerInfo);

            Delegate.Execute(Result, PlayerInfo);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::ShowInquiry(const FHiveAuthV4OnShowInquiryDelegate& Delegate)
{
    HIVESDKCEFImeEnable();
    
    hive::AuthV4::showInquiry([Delegate](hive::ResultAPI const & result) {
        
        HIVESDKCEFImeDisable();
        
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            Delegate.Execute(Result);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::ShowChatbotInquiry(const FString& AdditionalInfo, const FHiveAuthV4OnShowInquiryDelegate& Delegate)
{
    HIVESDKCEFImeEnable();
    
    std::string StrAdditionalInfo(TCHAR_TO_UTF8(*AdditionalInfo));
    hive::AuthV4::showChatbotInquiry(StrAdditionalInfo, [Delegate](hive::ResultAPI const & result) {
        
        HIVESDKCEFImeDisable();
        
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            Delegate.Execute(Result);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::ShowMyInquiry(const FHiveAuthV4OnShowInquiryDelegate& Delegate)
{
    HIVESDKCEFImeEnable();
    
    hive::AuthV4::showMyInquiry([Delegate](hive::ResultAPI const & result) {
        
        HIVESDKCEFImeDisable();
        
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            Delegate.Execute(Result);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::ShowTerms(const FHiveAuthV4OnShowTermsDelegate& Delegate)
{
    hive::AuthV4::showTerms([Delegate](hive::ResultAPI const & result) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            Delegate.Execute(Result);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::ShowAdultConfirm(const FHiveAuthV4OnAdultConfirmDelegate& Delegate)
{
    hive::AuthV4::showAdultConfirm([Delegate](hive::ResultAPI const & result) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            Delegate.Execute(Result);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::GetProfile(const TArray<int64> PlayerIdArray, const FHiveAuthV4OnGetProfileDelegate& Delegate)
{
    std::vector<hive::PlayerID> playerIdList;

    for(const auto& PlayerId : PlayerIdArray)
    {
        playerIdList.push_back(PlayerId);
    }

    hive::AuthV4::getProfile(playerIdList, [Delegate](hive::ResultAPI const & result, std::vector<hive::ProfileInfo> const & profileInfoList) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            TArray<FHiveProfileInfo> FHiveProfileInfoArray;

            if (profileInfoList.size() > 0)
            {
                FHiveProfileInfoArray.Reserve(profileInfoList.size());

                for (const auto& data : profileInfoList)
                {
                    FHiveProfileInfo ProfileInfo(data);
                    FHiveProfileInfoArray.Emplace(ProfileInfo);
                }
            }

            Delegate.Execute(Result, FHiveProfileInfoArray);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::ShowProfile(int64 PlayerId, const FHiveAuthV4OnShowProfileDelegate& Delegate)
{
    HIVESDKCEFImeEnable();
    
    hive::AuthV4::showProfile(PlayerId, [Delegate](hive::ResultAPI const & result) {
        
        HIVESDKCEFImeDisable();
        
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            Delegate.Execute(Result);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::ResetAgreement()
{
    hive::AuthV4::resetAgreement();
}


bool IHiveAuthV4Impl::IsAutoSignIn()
{
    return hive::AuthV4::isAutoSignIn();
}


TOptional<FHivePlayerInfo> IHiveAuthV4Impl::GetPlayerInfo()
{
    hive::PlayerInfo playerInfo = hive::AuthV4::getPlayerInfo();
    TOptional<FHivePlayerInfo> PlayerInfo = TOptional<FHivePlayerInfo>(FHivePlayerInfo(playerInfo));
    return PlayerInfo;
}


TOptional<FHiveParentalConsentInfo> IHiveAuthV4Impl::GetParentalConsentInfo()
{
    HIVE_LOG_WARNING(TEXT("This API is only supported on the Windows platform"));
    return TOptional<FHiveParentalConsentInfo>(); // only support Windows platform
}


void IHiveAuthV4Impl::GetProviderFriendsList(const EHiveProviderType& ProviderType, const FHiveAuthV4OnGetProviderFriendsListDelegate& Delegate)
{
    hive::AuthV4::getProviderFriendsList(static_cast<hive::ProviderType>(ProviderType), [Delegate](hive::ResultAPI const & result, hive::ProviderType providerType, std::map<std::string, hive::PlayerID> providerUserIdList) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            EHiveProviderType _ProviderType = static_cast<EHiveProviderType>(providerType);
            ProviderFriendsMap ProviderUserIdList;

            if (!providerUserIdList.empty())
            {
                for (const auto& Pair : providerUserIdList)
                {
                    FString Key = FString(UTF8_TO_TCHAR(Pair.first.c_str()));
                    int64 Value = Pair.second;
                    ProviderUserIdList.Emplace(Key, Value);
                }
            }

            Delegate.Execute(Result, _ProviderType, ProviderUserIdList);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::ResolveConflict(const FHiveAuthV4OnResolveConflictDelegate& Delegate)
{
    hive::AuthV4::resolveConflict([Delegate](hive::ResultAPI const & result) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            Delegate.Execute(Result);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::ShowGameCenterLoginCancelDialog(const FHiveAuthV4OnAuthV4DialogDismissDelegate& Delegate)
{
    hive::AuthV4::showGameCenterLoginCancelDialog([Delegate](bool isDismiss) {
        if (Delegate.IsBound())
        {
            Delegate.Execute(isDismiss);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


bool IHiveAuthV4Impl::GetAgeGateU13()
{
    return hive::AuthV4::getAgeGateU13();
}


void IHiveAuthV4Impl::ShowDeviceManagement(const FHiveAuthV4OnShowDeviceManagementDelegate& Delegate)
{
    HIVESDKCEFImeEnable();
    
    hive::AuthV4::showDeviceManagement([Delegate](hive::ResultAPI const & result) {
        
        HIVESDKCEFImeDisable();
        
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            Delegate.Execute(Result);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::GetHiveTalkPlusLoginToken(const FHiveAuthV4OnGetHiveTalkPlusLoginTokenDelegate& Delegate)
{
    hive::AuthV4::getHiveTalkPlusLoginToken([Delegate](hive::ResultAPI const & result, std::string loginToken) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            FString LoginToken = FString(UTF8_TO_TCHAR(loginToken.c_str()));
            Delegate.Execute(Result, LoginToken);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::RefreshAccessToken(const FHiveAuthV4OnRefreshAccessTokenDelegate& Delegate)
{
    hive::AuthV4::refreshAccessToken([Delegate](hive::ResultAPI const& result, const char* accessToken) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            FString AccessToken = FString(UTF8_TO_TCHAR(accessToken));
            Delegate.Execute(Result, AccessToken);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveAuthV4Impl::SignInWithStoredPlayerId(int64 PlayerId, bool useAutoSignIn, const FHiveAuthV4OnSignInDelegate& Delegate)
{
    hive::AuthV4::signInWithStoredPlayerId(PlayerId, useAutoSignIn, [Delegate](hive::ResultAPI const& result, hive::PlayerInfo const& playerInfo) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            FHivePlayerInfo PlayerInfo(playerInfo);

            Delegate.Execute(Result, PlayerInfo);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
        });
}

void IHiveAuthV4Impl::GetStoredPlayerIdList(const FHiveAuthV4OnGetStoredPlayerIdListDelegate& Delegate)
{
    hive::AuthV4::getStoredPlayerIdList([Delegate](hive::ResultAPI const& result, std::vector<int64> const& playerIdList) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            TArray<int64> PlayerIdListArray;
            if (playerIdList.size() > 0) {
                PlayerIdListArray.Reserve(playerIdList.size());
                for (const auto& id : playerIdList)
                {
                    PlayerIdListArray.Emplace(id);
                }
            }

            Delegate.Execute(Result, PlayerIdListArray);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
        });
}

void IHiveAuthV4Impl::StoreCurrentPlayerId(const FHiveAuthV4OnStoreCurrentPlayerIdDelegate& Delegate)
{
    hive::AuthV4::storeCurrentPlayerId([Delegate](hive::ResultAPI const& result) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            Delegate.Execute(Result);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveAuthV4Impl::DeleteStoredPlayerId(int64 PlayerId, const FHiveAuthV4OnDeleteStoredPlayerIdDelegate& Delegate)
{
    hive::AuthV4::deleteStoredPlayerId(PlayerId, [Delegate](hive::ResultAPI const& result) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            Delegate.Execute(Result);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveAuthV4Impl::GetHashedDi(const FHiveAuthV4OnGetHashedDiDelegate& Delegate)
{
    hive::AuthV4::getHashedDi([Delegate](hive::ResultAPI const& result, std::string hashedDi) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            FString HashedDi = FString(UTF8_TO_TCHAR(hashedDi.c_str()));
            Delegate.Execute(Result, HashedDi);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveAuthV4Impl::ShowIdentityVerification(const FHiveAuthV4OnIdentityVerificationDelegate& Delegate)
{
    hive::AuthV4::showIdentityVerification([Delegate](hive::ResultAPI const& result, hive::Identity const& identity) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            FHiveIdentity Identity(identity);
            Delegate.Execute(Result, identity);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveAuthV4Impl::GetAgeRange(const FHiveAuthV4OnGetAgeRangeDelegate& Delegate)
{
    hive::AuthV4::getAgeRange([Delegate](hive::ResultAPI const& result, hive::AgeRange const& ageRange) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            FHiveAuthV4AgeRange AgeRange(ageRange);
            Delegate.Execute(Result, AgeRange);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveAuthV4Impl::ShowAgeRangeUpdatePermission(const FString& Description, const FHiveAuthV4OnShowAgeRangeUpdatePermissionDelegate& Delegate)
{
    std::string StrDescription(TCHAR_TO_UTF8(*Description));
    hive::AuthV4::showAgeRangeUpdatePermission(StrDescription, [Delegate](hive::ResultAPI const& result, hive::AgeRange const& ageRange) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            FHiveAuthV4AgeRange AgeRange(ageRange);
            Delegate.Execute(Result, AgeRange);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}


void IHiveAuthV4Impl::CheckAgeGate(bool UseCloseButton, const FHiveAuthV4OnCheckAgeGateDelegate& Delegate)
{
    hive::AuthV4::checkAgeGate(UseCloseButton, [Delegate](hive::ResultAPI const& result, std::string dateOfBirth) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            FString DateOfBirth = FString(UTF8_TO_TCHAR(dateOfBirth.c_str()));
            Delegate.Execute(Result, DateOfBirth);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveAuthV4Impl::SetAutoLoginEnabled(bool enabled, const FHiveAuthV4OnSetAutoLoginEnabledDelegate& Delegate) {
    hive::AuthV4::setAutoLoginEnabled(enabled, [Delegate](hive::ResultAPI const& result, bool isEnabled) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            Delegate.Execute(Result, isEnabled);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}