/* Copyright © 2024 Com2uS Platform Corp. All Rights Reserved. */
#include "Impl/HiveChatImpl.h"
#include "HiveUELogger.h"
#include "CPP/HIVE_Chat.h"

IHiveChatImpl::~IHiveChatImpl()
{
    {
        FScopeLock Lock(&ListenerLock);
        
        // 모든 리스너 해제
        for (auto& Pair : ChannelListeners)
        {
            if (Pair.Value)
            {
                hive::Chat::removeChannelListener(TCHAR_TO_ANSI(*Pair.Key));
                delete Pair.Value;
            }
        }
        ChannelListeners.Empty();
        
        for (auto& Pair : DirectMessageListeners)
        {
            if (Pair.Value)
            {
                hive::Chat::removeDirectMessageListener(TCHAR_TO_ANSI(*Pair.Key));
                delete Pair.Value;
            }
        }
        DirectMessageListeners.Empty();
        
        for (auto& Pair : ConnectionListeners)
        {
            if (Pair.Value)
            {
                hive::Chat::removeConnectionListener(TCHAR_TO_ANSI(*Pair.Key));
                delete Pair.Value;
            }
        }
        ConnectionListeners.Empty();
        
        for (auto& Pair : UserListeners)
        {
            if (Pair.Value)
            {
                hive::Chat::removeUserListener(TCHAR_TO_ANSI(*Pair.Key));
                delete Pair.Value;
            }
        }
        UserListeners.Empty();
        
        for (auto& Pair : CustomDataListeners)
        {
            if (Pair.Value)
            {
                hive::Chat::removeCustomDataListener(TCHAR_TO_ANSI(*Pair.Key));
                delete Pair.Value;
            }
        }
        CustomDataListeners.Empty();
    }
}

void IHiveChatImpl::CreateChannel(FHiveCreateChannelParams const & params, const FHiveChatOnCreateChannelDelegate& Delegate)
{
    hive::CreateChannelParams createChannelParams = hive::CreateChannelParams();
    FTCHARToUTF8 ChannelIdConverter(*params.channelId);
    FTCHARToUTF8 PasswordConverter(*params.password);
    FTCHARToUTF8 ChannelNameConverter(*params.channelName);
    createChannelParams.channelId = ChannelIdConverter.Get();
    createChannelParams.password = PasswordConverter.Get();
    createChannelParams.channelName = ChannelNameConverter.Get();
    createChannelParams.maxMemberCount = params.maxMemberCount;
    createChannelParams.chatHistoryAllowed = params.chatHistoryAllowed;
    switch (params.type)
    {
        case EHiveChannelType::UNKNOWN:
            createChannelParams.type = hive::ChannelType::UNKNOWN;
            break;
        case EHiveChannelType::PRIVATE:
            createChannelParams.type = hive::ChannelType::PRIVATE;
            break;
        case EHiveChannelType::PUBLIC:
            createChannelParams.type = hive::ChannelType::PUBLIC;
            break;
        case EHiveChannelType::GROUP:
            createChannelParams.type = hive::ChannelType::GROUP;
            break;
        default:
            createChannelParams.type = hive::ChannelType::UNKNOWN;
            break;
    }
    hive::Chat::createChannel(createChannelParams, [Delegate](hive::ResultAPI const & result) {
        if (!Delegate.IsBound())
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
            return;
        }
        
        FHiveResultAPI Result(result);
        Delegate.Execute(Result);
    });
}

void IHiveChatImpl::GetChannels(TOptional<FHiveGetChannelsParams> params, const FHiveChatOnGetChannelsDelegate& Delegate)
{
    std::shared_ptr<hive::GetChannelsParams> getChannelsParams;
    if (params.IsSet())
    {
        FHiveGetChannelsParams nonOptionalParams = params.GetValue();
        getChannelsParams = std::make_shared<hive::GetChannelsParams>();
        FTCHARToUTF8 ChannelIdConverter(*nonOptionalParams.channelId);
        FTCHARToUTF8 ChannelNameConverter(*nonOptionalParams.channelName);
        FTCHARToUTF8 PageOrderConverter(*nonOptionalParams.pageOrder);
        getChannelsParams->channelId = ChannelIdConverter.Get();
        getChannelsParams->channelName = ChannelNameConverter.Get();
        getChannelsParams->pageOrder = PageOrderConverter.Get();
        getChannelsParams->pageSize = nonOptionalParams.pageSize;
        getChannelsParams->pageNumber = nonOptionalParams.pageNumber;
        switch (nonOptionalParams.type)
        {
            case EHiveChannelType::UNKNOWN:
                getChannelsParams->type = hive::ChannelType::UNKNOWN;
                break;
            case EHiveChannelType::PRIVATE:
                getChannelsParams->type = hive::ChannelType::PRIVATE;
                break;  
            case EHiveChannelType::PUBLIC:
                getChannelsParams->type = hive::ChannelType::PUBLIC;
                break;
            case EHiveChannelType::GROUP:
                getChannelsParams->type = hive::ChannelType::GROUP;
                break;
            default:
                getChannelsParams->type = hive::ChannelType::UNKNOWN;
                break;
        }
        switch (nonOptionalParams.sort)
        {
            case EHiveSortType::CHANNEL_ID:
                getChannelsParams->sort = hive::SortType::ChannelId;
                break;
            case EHiveSortType::CHANNEL_NAME:
                getChannelsParams->sort = hive::SortType::ChannelName;
                break;
            case EHiveSortType::REG_TIME:
                getChannelsParams->sort = hive::SortType::RegTime;
                break;
            case EHiveSortType::UNKNOWN:
            default:
                getChannelsParams->sort = hive::SortType::Unknown;
                break;
        }
    }
    
    hive::Chat::getChannels(getChannelsParams.get(), [Delegate](hive::ResultAPI const & result, std::vector<hive::Channel> const & channels, hive::ChannelPage const & channelPage) {
        if (!Delegate.IsBound())
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
            return;
        }
        
        FHiveResultAPI Result(result);
        TArray<FHiveChannel> TarrayChannels;
        TarrayChannels.Reserve(channels.size());
        
        for (const auto& channel : channels)
        {
            TarrayChannels.Emplace(channel);
        }

        FHiveChannelPage ChannelPage(channelPage);
        Delegate.Execute(Result, TarrayChannels, ChannelPage);
    });
}

void IHiveChatImpl::GetChannelInfo(const FString& channelId, const FHiveChatOnGetChannelInfoDelegate& Delegate)
{
    FTCHARToUTF8 Converter(*channelId);
    const char* channelIdCString = Converter.Get();
    
    hive::Chat::getChannelInfo(channelIdCString, [Delegate](hive::ResultAPI const & result, hive::Channel const & channel, std::vector<hive::Member> const & members) {
        if (!Delegate.IsBound())
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
            return;
        }
        
        FHiveResultAPI Result(result);
        FHiveChannel Channel(channel);

        TArray<FHiveMember> TarrayMembers;
        TarrayMembers.Reserve(members.size());
        
        for (const auto& member : members)
        {
            TarrayMembers.Emplace(member);
        }

        Delegate.Execute(Result, Channel, TarrayMembers);
    });
}

void IHiveChatImpl::GetChannelMembers(const FString& channelId, const FHiveChatOnGetChannelMembersDelegate& Delegate)
{
    FTCHARToUTF8 Converter(*channelId);
    const char* channelIdCString = Converter.Get();
    
    hive::Chat::getChannelMembers(channelIdCString, [Delegate](hive::ResultAPI const & result, std::vector<hive::Member> const & members) {
        if (!Delegate.IsBound())
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
            return;
        }
        
        FHiveResultAPI Result(result);
        TArray<FHiveMember> TarrayMembers;
        TarrayMembers.Reserve(members.size());
        
        for (const auto& member : members)
        {
            TarrayMembers.Emplace(member);
        }

        Delegate.Execute(Result, TarrayMembers);
    });
}

void IHiveChatImpl::DeleteChannel(const FString& channelId, const FHiveChatOnDeleteChannelDelegate& Delegate)
{
    FTCHARToUTF8 Converter(*channelId);
    const char* channelIdCString = Converter.Get();
    
    hive::Chat::deleteChannel(channelIdCString, [Delegate](hive::ResultAPI const & result) {
        if (!Delegate.IsBound())
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
            return;
        }
        
        FHiveResultAPI Result(result);
        Delegate.Execute(Result);
    });
}

void IHiveChatImpl::EnterChannel(const FHiveEnterChannelParams& params, const FHiveChatOnEnterChannelDelegate& Delegate)
{
    hive::EnterChannelParams enterChannelParams = hive::EnterChannelParams();
    enterChannelParams.channelId = TCHAR_TO_ANSI(*params.channelId);
    enterChannelParams.password = TCHAR_TO_ANSI(*params.password);

    hive::Chat::enterChannel(enterChannelParams, [Delegate](hive::ResultAPI const & result) {
        if (!Delegate.IsBound())
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
            return;
        }
        
        FHiveResultAPI Result(result);
        Delegate.Execute(Result);
    });
}

void IHiveChatImpl::ExitChannel(const FString& channelId, const FHiveChatOnExitChannelDelegate& Delegate)
{
    FTCHARToUTF8 Converter(*channelId);
    const char* channelIdCString = Converter.Get();
    
    hive::Chat::exitChannel(channelIdCString, [Delegate](hive::ResultAPI const & result) {
        if (!Delegate.IsBound())
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
            return;
        }
        
        FHiveResultAPI Result(result);
        Delegate.Execute(Result);
    });
}

void IHiveChatImpl::GetChannelsByUser(const FHiveChatOnGetChannelsByUserDelegate& Delegate)
{
    hive::Chat::getChannelsByUser([Delegate](hive::ResultAPI const & result, std::vector<hive::Channel> const & channels) {
        if (!Delegate.IsBound())
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
            return;
        }
        
        FHiveResultAPI Result(result);
        TArray<FHiveChannel> TarrayChannels;
        TarrayChannels.Reserve(channels.size());
        
        for (const auto& channel : channels)
        {
            TarrayChannels.Emplace(channel);
        }

        Delegate.Execute(Result, TarrayChannels);
    });
}

void IHiveChatImpl::GetBlockMembers(const FHiveChatOnGetBlockMembersDelegate& Delegate)
{
    hive::Chat::getBlockMembers([Delegate](hive::ResultAPI const & result, std::vector<hive::BlockMember> const & blockMembers) {
        if (!Delegate.IsBound())
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
            return;
        }
        
        FHiveResultAPI Result(result);
        TArray<FHiveBlockMember> TarrayBlockMembers;
        TarrayBlockMembers.Reserve(blockMembers.size());
        
        for (const auto& blockMember : blockMembers)
        {
            TarrayBlockMembers.Emplace(blockMember);
        }

        Delegate.Execute(Result, TarrayBlockMembers);
    });
}

void IHiveChatImpl::BlockMember(int64 blockPlayerId, const FHiveChatOnBlockMemberDelegate& Delegate)
{
    hive::Chat::blockMember(blockPlayerId, [Delegate](hive::ResultAPI const & result) {
        if (!Delegate.IsBound())
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
            return;
        }
        
        FHiveResultAPI Result(result);
        Delegate.Execute(Result);
    });
}

void IHiveChatImpl::UnblockMember(int64 blockPlayerId, const FHiveChatOnUnblockMemberDelegate& Delegate)
{
    hive::Chat::unblockMember(blockPlayerId, [Delegate](hive::ResultAPI const & result) {
        if (!Delegate.IsBound())
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
            return;
        }
        
        FHiveResultAPI Result(result);
        Delegate.Execute(Result);
    });
}

void IHiveChatImpl::Connect(const FString& extraData, const FHiveChatOnConnectDelegate& Delegate)
{
    Connect(Delegate);
}

void IHiveChatImpl::Connect(const FHiveChatOnConnectDelegate& Delegate)
{
    hive::Chat::connect([Delegate](hive::ResultAPI const & result) {
        if (!Delegate.IsBound())
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
            return;
        }
        
        FHiveResultAPI Result(result);
        Delegate.Execute(Result);
    });
}

void IHiveChatImpl::Reconnect(const FHiveChatOnReconnectDelegate& Delegate)
{
    hive::Chat::reconnect([Delegate](hive::ResultAPI const & result, std::vector<std::string> const & channelIds, std::vector<std::string> const & failChannelIds) {
        if (!Delegate.IsBound())
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
            return;
        }
        
        FHiveResultAPI Result(result);
        TArray<FString> UnrealChannelIds;
        UnrealChannelIds.Reserve(channelIds.size());
        
        for (const auto& channelId : channelIds)
        {
            UnrealChannelIds.Emplace(UTF8_TO_TCHAR(channelId.c_str()));
        }
        
        TArray<FString> UnrealFailChannelIds;
        UnrealFailChannelIds.Reserve(failChannelIds.size());
        
        for (const auto& channelId : failChannelIds)
        {
            UnrealFailChannelIds.Emplace(UTF8_TO_TCHAR(channelId.c_str()));
        }
        
        Delegate.Execute(Result, UnrealChannelIds, UnrealFailChannelIds);
    });
}

void IHiveChatImpl::Disconnect(const FHiveChatOnDisconnectDelegate& Delegate)
{
    hive::Chat::disconnect([Delegate](hive::ResultAPI const & result) {
        if (!Delegate.IsBound())
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
            return;
        }
        
        FHiveResultAPI Result(result);
        Delegate.Execute(Result);
    });
}

void IHiveChatImpl::SendMessageWithChannelSendMessageParams(FHiveChannelSendMessageParams const & params, const FHiveChatOnSendMessageWithChannelSendMessageParamsDelegate& Delegate)
{
    hive::ChannelSendMessageParams channelSendMessageParams = hive::ChannelSendMessageParams();
    FTCHARToUTF8 ChannelIdConverter(*params.channelId);
    FTCHARToUTF8 MessageConverter(*params.message);
    FTCHARToUTF8 ExtraDataConverter(*params.extraData);
    FTCHARToUTF8 ReplyMessageIdConverter(*params.replyMessageId);
    channelSendMessageParams.channelId = ChannelIdConverter.Get();
    channelSendMessageParams.message = MessageConverter.Get();
    channelSendMessageParams.extraData = ExtraDataConverter.Get();
    channelSendMessageParams.replyMessageId = ReplyMessageIdConverter.Get();
    
    channelSendMessageParams.mentionedPlayerIds.reserve(params.mentionedPlayerIds.Num());
    for (const auto& playerId : params.mentionedPlayerIds)
    {
        channelSendMessageParams.mentionedPlayerIds.push_back(playerId);
    }

    hive::Chat::sendMessageWithChannelSendMessageParams(channelSendMessageParams, [Delegate](hive::ResultAPI const & result, hive::ChannelSendMessageParams const& params) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            Delegate.Execute(Result);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveChatImpl::SendMessageWithDirectSendMessageParams(FHiveDirectSendMessageParams const & params, const FHiveChatOnSendMessageWithDirectSendMessageParamsDelegate& Delegate)
{
    hive::DirectSendMessageParams directSendMessageParams = hive::DirectSendMessageParams();
    directSendMessageParams.toPlayerId = params.toPlayerId;
    FTCHARToUTF8 MessageConverter(*params.message);
    directSendMessageParams.message = MessageConverter.Get();
    FTCHARToUTF8 ExtraDataConverter(*params.extraData);
    directSendMessageParams.extraData = ExtraDataConverter.Get();
    
    hive::Chat::sendMessageWithDirectSendMessageParams(directSendMessageParams, [Delegate](hive::ResultAPI const & result, hive::DirectSendMessageParams const& params) {
        if (Delegate.IsBound())
        {
            FHiveResultAPI Result(result);
            Delegate.Execute(Result);
        }
        else
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
        }
    });
}

void IHiveChatImpl::AddChannelListener(const FString& uniqueKey, FHiveChannelListener* listener)
{
    if (!listener) return;
    
    {
        FScopeLock Lock(&ListenerLock);
        
        // 기존 리스너가 있다면 제거
        if (auto* Listener = ChannelListeners.FindRef(uniqueKey))
        {
            hive::Chat::removeChannelListener(TCHAR_TO_ANSI(*uniqueKey));
            delete Listener;
            ChannelListeners.Remove(uniqueKey);
        }
        
        // 새로운 래퍼 생성 및 등록
        auto* wrapper = new HiveChannelListenerWrapper(listener);
        wrapper->SetUniqueKey(uniqueKey);
        ChannelListeners.Add(uniqueKey, wrapper);
        hive::Chat::addChannelListener(TCHAR_TO_ANSI(*uniqueKey), wrapper);
    } 
}

void IHiveChatImpl::AddDirectMessageListener(const FString& uniqueKey, FHiveDirectMessageListener* listener)
{
    if (!listener) return;
    
    {
        FScopeLock Lock(&ListenerLock);
        
        // 기존 리스너가 있다면 제거
        if (auto* Listener = DirectMessageListeners.FindRef(uniqueKey))
        {
            hive::Chat::removeDirectMessageListener(TCHAR_TO_ANSI(*uniqueKey));
            delete Listener;
            DirectMessageListeners.Remove(uniqueKey);
        }
        
        // 새로운 래퍼 생성 및 등록
        auto* wrapper = new HiveDirectMessageListenerWrapper(listener);
        wrapper->SetUniqueKey(uniqueKey);
        DirectMessageListeners.Add(uniqueKey, wrapper);
        hive::Chat::addDirectMessageListener(TCHAR_TO_ANSI(*uniqueKey), wrapper);
    }
}

void IHiveChatImpl::AddConnectionListener(const FString& uniqueKey, FHiveConnectionListener* listener)
{
    if (!listener) return;
    
    {
        FScopeLock Lock(&ListenerLock);
        
        // 기존 리스너가 있다면 제거
        if (auto* Listener = ConnectionListeners.FindRef(uniqueKey))
        {
            hive::Chat::removeConnectionListener(TCHAR_TO_ANSI(*uniqueKey));
            delete Listener;
            ConnectionListeners.Remove(uniqueKey);
        }
        
        // 새로운 래퍼 생성 및 등록
        auto* wrapper = new HiveConnectionListenerWrapper(listener);
        wrapper->SetUniqueKey(uniqueKey);
        ConnectionListeners.Add(uniqueKey, wrapper);
        hive::Chat::addConnectionListener(TCHAR_TO_ANSI(*uniqueKey), wrapper);
    }
}

void IHiveChatImpl::RemoveChannelListener(const FString& uniqueKey)
{
    {
        FScopeLock Lock(&ListenerLock);
        
        if (auto* Listener = ChannelListeners.FindRef(uniqueKey))
        {
            hive::Chat::removeChannelListener(TCHAR_TO_ANSI(*uniqueKey));
            delete Listener;
            ChannelListeners.Remove(uniqueKey);
        }
    }
}

void IHiveChatImpl::RemoveDirectMessageListener(const FString& uniqueKey)
{
    {
        FScopeLock Lock(&ListenerLock);
        
        if (auto* Listener = DirectMessageListeners.FindRef(uniqueKey))
        {
            hive::Chat::removeDirectMessageListener(TCHAR_TO_ANSI(*uniqueKey));
            delete Listener;
            DirectMessageListeners.Remove(uniqueKey);
        }
    }
}

void IHiveChatImpl::RemoveConnectionListener(const FString& uniqueKey)
{
    {
        FScopeLock Lock(&ListenerLock);
        
        if (auto* Listener = ConnectionListeners.FindRef(uniqueKey))
        {
            hive::Chat::removeConnectionListener(TCHAR_TO_ANSI(*uniqueKey));
            delete Listener;
            ConnectionListeners.Remove(uniqueKey);
        }
    }
}

bool IHiveChatImpl::IsConnected()
{
    return hive::Chat::isConnected();
}


void IHiveChatImpl::SendMessageWithChannelSendMessageParams(FHiveChannelSendMessageParams const & params)
{
    hive::ChannelSendMessageParams channelSendMessageParams = hive::ChannelSendMessageParams();
    FTCHARToUTF8 ChannelIdConverter(*params.channelId);
    FTCHARToUTF8 MessageConverter(*params.message);
    FTCHARToUTF8 ExtraDataConverter(*params.extraData);
    FTCHARToUTF8 ReplyMessageIdConverter(*params.replyMessageId);
    channelSendMessageParams.channelId = ChannelIdConverter.Get();
    channelSendMessageParams.message = MessageConverter.Get();
    channelSendMessageParams.extraData = ExtraDataConverter.Get();
    channelSendMessageParams.replyMessageId = ReplyMessageIdConverter.Get();
    
    channelSendMessageParams.mentionedPlayerIds.reserve(params.mentionedPlayerIds.Num());
    for (const auto& playerId : params.mentionedPlayerIds)
    {
        channelSendMessageParams.mentionedPlayerIds.push_back(playerId);
    }

    hive::Chat::sendMessageWithChannelSendMessageParams(channelSendMessageParams);
}

void IHiveChatImpl::SendMessageWithDirectSendMessageParams(FHiveDirectSendMessageParams const & params)
{
    hive::DirectSendMessageParams directSendMessageParams = hive::DirectSendMessageParams();
    directSendMessageParams.toPlayerId = params.toPlayerId;
    FTCHARToUTF8 MessageConverter(*params.message);
    directSendMessageParams.message = MessageConverter.Get();
    FTCHARToUTF8 ExtraDataConverter(*params.extraData);
    directSendMessageParams.extraData = ExtraDataConverter.Get();
    
    hive::Chat::sendMessageWithDirectSendMessageParams(directSendMessageParams);
}


void IHiveChatImpl::Translate(FHiveTranslateParams const & params, const FHiveChatOnTranslateDelegate& Delegate)
{
    hive::TranslateParams translateParams = hive::TranslateParams();
    FTCHARToUTF8 MessageConverter(*params.message);
    translateParams.message = MessageConverter.Get();
    FTCHARToUTF8 SourceLanguageConverter(*params.sourceLanguage);
    translateParams.sourceLanguage = SourceLanguageConverter.Get();
    
    translateParams.targetLanguage.reserve(params.targetLanguage.Num());
    for (const auto& lang : params.targetLanguage)
    {
        FTCHARToUTF8 LangConverter(*lang);
        translateParams.targetLanguage.push_back(LangConverter.Get());
    }
    
    hive::Chat::translate(translateParams, [Delegate](hive::ResultAPI const & result, hive::TranslationData const & translationData) {
        if (!Delegate.IsBound())
        {
            HIVE_LOG_WARNING(TEXT("Delegate is not bounded."));
            return;
        }
        
        FHiveResultAPI Result(result);
        FHiveTranslationData TranslationData(translationData);
        Delegate.Execute(Result, TranslationData);
    });
}

void IHiveChatImpl::AddUserListener(const FString& uniqueKey, FHiveUserListener* listener)
{
    if (!listener) return;
    
    {
        FScopeLock Lock(&ListenerLock);
        
        // 기존 리스너가 있다면 제거
        if (auto* Listener = UserListeners.FindRef(uniqueKey))
        {
            hive::Chat::removeUserListener(TCHAR_TO_ANSI(*uniqueKey));
            delete Listener;
            UserListeners.Remove(uniqueKey);
        }
        
        // 새로운 래퍼 생성 및 등록
        auto* wrapper = new HiveUserListenerWrapper(listener);
        wrapper->SetUniqueKey(uniqueKey);
        UserListeners.Add(uniqueKey, wrapper);
        hive::Chat::addUserListener(TCHAR_TO_ANSI(*uniqueKey), wrapper);
    }
}

void IHiveChatImpl::AddCustomDataListener(const FString& uniqueKey, FHiveCustomDataListener* listener)
{
    if (!listener) return;
    
    {
        FScopeLock Lock(&ListenerLock);
        
        // 기존 리스너가 있다면 제거
        if (auto* Listener = CustomDataListeners.FindRef(uniqueKey))
        {
            hive::Chat::removeCustomDataListener(TCHAR_TO_ANSI(*uniqueKey));
            delete Listener;
            CustomDataListeners.Remove(uniqueKey);
        }
        
        // 새로운 래퍼 생성 및 등록
        auto* wrapper = new HiveCustomDataListenerWrapper(listener);
        wrapper->SetUniqueKey(uniqueKey);
        CustomDataListeners.Add(uniqueKey, wrapper);
        hive::Chat::addCustomDataListener(TCHAR_TO_ANSI(*uniqueKey), wrapper);
    }
}

void IHiveChatImpl::RemoveUserListener(const FString& uniqueKey)
{
    {
        FScopeLock Lock(&ListenerLock);
        
        if (auto* Listener = UserListeners.FindRef(uniqueKey))
        {
            hive::Chat::removeUserListener(TCHAR_TO_ANSI(*uniqueKey));
            delete Listener;
            UserListeners.Remove(uniqueKey);
        }
    }
}

void IHiveChatImpl::RemoveCustomDataListener(const FString& uniqueKey)
{
    {
        FScopeLock Lock(&ListenerLock);
        
        if (auto* Listener = CustomDataListeners.FindRef(uniqueKey))
        {
            hive::Chat::removeCustomDataListener(TCHAR_TO_ANSI(*uniqueKey));
            delete Listener;
            CustomDataListeners.Remove(uniqueKey);
        }
    }
}
