/* Copyright © 2024 Com2uS Platform Corp. All Rights Reserved. */
#include "Types/HiveAnalyticsTypes.h"

FHiveAnalyticsAdRevenue::FHiveAnalyticsAdRevenue(const hive::AnalyticsAdRevenue& AdRevenue)
: Revenue(AdRevenue.revenue)
, AdPlatform(UTF8_TO_TCHAR(AdRevenue.adPlatform.c_str()))
, AdUnitId(UTF8_TO_TCHAR(AdRevenue.adUnitId.c_str()))
, AdType(UTF8_TO_TCHAR(AdRevenue.adType.c_str()))
, AdPlacement(UTF8_TO_TCHAR(AdRevenue.adPlacement.c_str()))
, Currency(UTF8_TO_TCHAR(AdRevenue.currency.c_str()))
{ }

FHiveConsentStatus::FHiveConsentStatus(const hive::ConsentStatus& ConsentStatus)
: Id(UTF8_TO_TCHAR(ConsentStatus.id.c_str()))
, Granted(UTF8_TO_TCHAR(ConsentStatus.granted.c_str()))
, ConsentDate(static_cast<int64>(ConsentStatus.consentDate))
, ExpiredDate(static_cast<int64>(ConsentStatus.expiredDate))
{ }

FHiveConsentForm::FHiveConsentForm(const hive::ConsentForm& ConsentForm)
: Id(UTF8_TO_TCHAR(ConsentForm.id.c_str()))
, RetentionPeriod(ConsentForm.retentionPeriod)
, Title(UTF8_TO_TCHAR(ConsentForm.title.c_str()))
, Content(UTF8_TO_TCHAR(ConsentForm.content.c_str()))
{
    if (!ConsentForm.subForms.empty()) {
        for (const auto& form : ConsentForm.subForms) {
            FHiveConsentForm ConvertConsentForm(form);
            SubForms.Emplace(ConvertConsentForm);
        }
    }
}

FHiveConsentMode::FHiveConsentMode(const hive::ConsentMode& ConsentMode)
: Title(UTF8_TO_TCHAR(ConsentMode.title.c_str()))
, CompanyName(UTF8_TO_TCHAR(ConsentMode.companyName.c_str()))
, PrivacyPolicy(UTF8_TO_TCHAR(ConsentMode.privacyPolicy.c_str()))
{
    if (ConsentMode.adUserData != nullptr) 
    {
        this->AdUserData = TOptional<FHiveConsentForm>(*ConsentMode.adUserData);
    }
    else
    {
        this->AdUserData = TOptional<FHiveConsentForm>();
    }

    if (ConsentMode.adPersonalization != nullptr) 
    {
        this->AdPersonalization = TOptional<FHiveConsentForm>(*ConsentMode.adPersonalization);
    }
    else
    {
        this->AdPersonalization = TOptional<FHiveConsentForm>();
    }

    if (ConsentMode.analyticsStorage != nullptr) 
    {
        this->AnalyticsStorage = TOptional<FHiveConsentForm>(*ConsentMode.analyticsStorage);
    }
    else
    {
        this->AnalyticsStorage = TOptional<FHiveConsentForm>();
    }
}

FHiveAirbridgeSpecific::FHiveAirbridgeSpecific()
{
    SemanticAttributes = MakeShareable(new FJsonObject());
}

FString FHiveAirbridgeSpecific::GetProviderName() const
{
    return TEXT("airbridge");
}

TSharedPtr<FJsonObject> FHiveAirbridgeSpecific::ToJson() const
{
    TSharedPtr<FJsonObject> JsonObject = MakeShareable(new FJsonObject());
    JsonObject->SetObjectField(TEXT("semanticAttributes"), SemanticAttributes);

    return JsonObject;
}

FString FHiveAirbridgeSpecific::ToString() const
{
    TSharedPtr<FJsonObject> JsonObject = ToJson();

    FString OutputString;
    TSharedRef<TJsonWriter<>> Writer = TJsonWriterFactory<>::Create(&OutputString);
    FJsonSerializer::Serialize(JsonObject.ToSharedRef(), Writer);

    return OutputString;
}

void FHiveAirbridgeSpecific::SetSemanticAction(const FString& Value)
{
    SemanticAttributes->SetStringField(TEXT("action"), Value);
}

void FHiveAirbridgeSpecific::SetSemanticLabel(const FString& Value)
{
    SemanticAttributes->SetStringField(TEXT("label"), Value);
}


void FHiveAirbridgeSpecific::SetSemanticValue(double Value)
{
    SemanticAttributes->SetNumberField(TEXT("value"), Value);
}

void FHiveAirbridgeSpecific::AddSemanticAttribute(const FString& Key, TSharedPtr<FJsonValue> Value)
{
    SemanticAttributes->SetField(Key, Value);
}

FHiveAnalyticsAttributes::FHiveAnalyticsAttributes()
{
    CustomAttributes = MakeShareable(new FJsonObject());
    ModuleSpecificAttributes = MakeShareable(new FJsonObject());
}

void FHiveAnalyticsAttributes::AddCustomAttribute(const FString& Key, TSharedPtr<FJsonValue> Value)
{
    CustomAttributes->SetField(Key, Value);
}

void FHiveAnalyticsAttributes::AddModuleSpecificAttribute(const IHiveAnalyticsSpecific& Specific)
{
    ModuleSpecificAttributes->SetObjectField(Specific.GetProviderName(), Specific.ToJson());
}

TSharedPtr<FJsonObject> FHiveAnalyticsAttributes::ToJson() const
{
    TSharedPtr<FJsonObject> JsonObject = MakeShareable(new FJsonObject());
    JsonObject->SetObjectField(TEXT("customAttributes"), CustomAttributes);
    JsonObject->SetObjectField(TEXT("moduleSpecificAttributes"), ModuleSpecificAttributes);

    return JsonObject;
}

FString FHiveAnalyticsAttributes::ToString() const
{
    TSharedPtr<FJsonObject> JsonObject = ToJson();

    FString OutputString;
    TSharedRef<TJsonWriter<>> Writer = TJsonWriterFactory<>::Create(&OutputString);
    FJsonSerializer::Serialize(JsonObject.ToSharedRef(), Writer);

    return OutputString;
}