/* Copyright © 2024 Com2uS Platform Corp. All Rights Reserved. */
#include "Types/HiveAuthV4Types.h"

FHiveProviderInfo::FHiveProviderInfo(const hive::ProviderInfo& ProviderInfo)
: ProviderType(static_cast<EHiveProviderType>(ProviderInfo.providerType))
, ProviderName(UTF8_TO_TCHAR(ProviderInfo.providerName.c_str()))
, ProviderUserId(UTF8_TO_TCHAR(ProviderInfo.providerUserId.c_str()))
, ProviderEmail(UTF8_TO_TCHAR(ProviderInfo.providerEmail.c_str()))
{ }

FHiveProfileInfo::FHiveProfileInfo(const hive::ProfileInfo& ProfileInfo)
: PlayerId(ProfileInfo.playerId)
, PlayerName(UTF8_TO_TCHAR(ProfileInfo.playerName.c_str()))
, PlayerImageUrl(UTF8_TO_TCHAR(ProfileInfo.playerImageUrl.c_str()))
{ }

FHivePlayerInfo::FHivePlayerInfo(const hive::PlayerInfo& PlayerInfo)
: FHiveProfileInfo(PlayerInfo)
, PlayerToken(UTF8_TO_TCHAR(PlayerInfo.playerToken.c_str()))
, AccessToken(UTF8_TO_TCHAR(PlayerInfo.accessToken.c_str()))
, Did(UTF8_TO_TCHAR(PlayerInfo.did.c_str()))
{
    // Set ProviderInfoData
    for(const auto& pair : PlayerInfo.providerInfoData)
    {
        EHiveProviderType Key = static_cast<EHiveProviderType>(pair.first);
        FHiveProviderInfo Value = FHiveProviderInfo(pair.second);

        this->ProviderInfoData.Emplace(Key, Value);
    }

    // Set CustomProviderInfoData
    for(const auto& pair : PlayerInfo.customProviderInfoData)
    {
        FString Key = FString(UTF8_TO_TCHAR((pair.first).c_str()));
        
        FHiveProviderInfo Value = FHiveProviderInfo(pair.second);

        this->CustomProviderInfoData.Emplace(Key, Value);
    }
}

FHiveParentalConsentInfo::FHiveParentalConsentInfo(const hive::ParentalConsentInfo& parentalConsentInfo)
: dateOfBirth(UTF8_TO_TCHAR(parentalConsentInfo.dateOfBirth.c_str()))
, guardianEmail(UTF8_TO_TCHAR(parentalConsentInfo.guardianEmail.c_str()))
{}

FHiveAuthV4AgeRange::FHiveAuthV4AgeRange(const hive::AgeRange& AgeRange)
: UserStatus(static_cast<EHiveUserStatus>(AgeRange.userStatus))
, AgeLower(AgeRange.ageLower)
, AgeUpper(AgeRange.ageUpper)
, MostRecentApprovalDate(UTF8_TO_TCHAR(AgeRange.mostRecentApprovalDate.c_str()))
, AgeRangeId(UTF8_TO_TCHAR(AgeRange.ageRangeId.c_str()))
{}

FHiveAuthV4MaintenanceExtraButton::FHiveAuthV4MaintenanceExtraButton(const hive::AuthV4MaintenanceExtraButton& Button)
: Button(UTF8_TO_TCHAR(Button.button.c_str()))
, Action(static_cast<EHiveAuthV4MaintenanceActionType>(Button.action))
, Url(UTF8_TO_TCHAR(Button.url.c_str()))
{ }

FHiveAuthV4MaintenanceInfo::FHiveAuthV4MaintenanceInfo(const hive::AuthV4MaintenanceInfo& Info)
: Title(UTF8_TO_TCHAR(Info.title.c_str()))
, Message(UTF8_TO_TCHAR(Info.message.c_str()))
, Button(UTF8_TO_TCHAR(Info.button.c_str()))
, Action(static_cast<EHiveAuthV4MaintenanceActionType>(Info.action))
, Url(UTF8_TO_TCHAR(Info.url.c_str()))
, StartDate(UTF8_TO_TCHAR(Info.startDate.c_str()))
, EndDate(UTF8_TO_TCHAR(Info.endDate.c_str()))
, RemainingTime(Info.remainingTime)
, CustomerButton(UTF8_TO_TCHAR(Info.customerButton.c_str()))
, CustomerLink(UTF8_TO_TCHAR(Info.customerLink.c_str()))
{
    for(const auto& item: Info.exButtons)
    {
        FHiveAuthV4MaintenanceExtraButton AuthV4MaintenanceExtraButton(item);
        this->ExButtons.Emplace(AuthV4MaintenanceExtraButton);
    }
}

FHiveConflictSingleViewData::FHiveConflictSingleViewData(int64 ConflictPlayerId)
{
    FString Key = TEXT("player_id");
    FString Value = FString::Printf(TEXT("%lld"), ConflictPlayerId);
    this->ConflictPlayerData.Emplace(Key, Value);
}

void FHiveConflictSingleViewData::SetData(const FString& Key, const FString& _ConflictPlayerData)
{
    this->ConflictPlayerData.Emplace(Key, _ConflictPlayerData);
}

FHiveConflictViewData::FHiveConflictViewData(int64 CurrentPlayerId, int64 ConflictPlayerId)
{
    FString Key = TEXT("player_id");
    FString CurrentPlayerValue = FString::Printf(TEXT("%lld"), CurrentPlayerId);
    FString ConflictPlayerValue = FString::Printf(TEXT("%lld"), ConflictPlayerId);
    
    this->CurrentPlayerData.Emplace(Key, CurrentPlayerValue);
    this->ConflictPlayerData.Emplace(Key, ConflictPlayerValue);
}

void FHiveConflictViewData::SetData(const FString& Key, const FString& _CurrentPlayerData, const FString& _ConflictPlayerData)
{
    this->CurrentPlayerData.Emplace(Key, _CurrentPlayerData);
    this->ConflictPlayerData.Emplace(Key, _ConflictPlayerData);
}

FString GetNameFromEHiveProviderType(const EHiveProviderType& ProviderType)
{
    switch(ProviderType) {
        case EHiveProviderType::GUEST: return FString(TEXT("GUEST"));
        case EHiveProviderType::HIVE: return FString(TEXT("HIVE"));
        case EHiveProviderType::FACEBOOK: return FString(TEXT("FACEBOOK"));
        case EHiveProviderType::GOOGLE: return FString(TEXT("GOOGLE"));
        case EHiveProviderType::QQ: return FString(TEXT("QQ"));
        case EHiveProviderType::VK: return FString(TEXT("VK"));
        case EHiveProviderType::WECHAT: return FString(TEXT("WECHAT"));
        case EHiveProviderType::APPLE: return FString(TEXT("APPLE"));
        case EHiveProviderType::SIGNIN_APPLE: return FString(TEXT("SIGNIN_APPLE"));
        case EHiveProviderType::LINE: return FString(TEXT("LINE"));
        case EHiveProviderType::WEVERSE: return FString(TEXT("WEVERSE"));
        case EHiveProviderType::GOOGLE_PLAY_GAMES: return FString(TEXT("GOOGLE_PLAY_GAMES"));
        case EHiveProviderType::HUAWEI: return FString(TEXT("HUAWEI"));
        case EHiveProviderType::STEAM: return FString(TEXT("STEAM"));
        case EHiveProviderType::X: return FString(TEXT("X"));
        case EHiveProviderType::TELEGRAM: return FString(TEXT("TELEGRAM"));
        case EHiveProviderType::XIAOMI: return FString(TEXT("XIAOMI"));
        case EHiveProviderType::CUSTOM: return FString(TEXT("CUSTOM"));
        case EHiveProviderType::AUTO: return FString(TEXT("AUTO"));
    }
    return FString(TEXT("UNKNOWN"));
}

EHiveProviderType GetEHiveProviderTypeFromName(const FString& Name)
{
    static TMap<FString, EHiveProviderType> EHiveProviderTypeMap = {
        {TEXT("GUEST"), EHiveProviderType::GUEST},
        {TEXT("HIVE"), EHiveProviderType::HIVE},
        {TEXT("FACEBOOK"), EHiveProviderType::FACEBOOK},
        {TEXT("GOOGLE"), EHiveProviderType::GOOGLE},
        {TEXT("QQ"), EHiveProviderType::QQ},
        {TEXT("VK"), EHiveProviderType::VK},
        {TEXT("WECHAT"), EHiveProviderType::WECHAT},
        {TEXT("APPLE"), EHiveProviderType::APPLE},
        {TEXT("SIGNIN_APPLE"), EHiveProviderType::SIGNIN_APPLE},
        {TEXT("LINE"), EHiveProviderType::LINE},
        {TEXT("WEVERSE"), EHiveProviderType::WEVERSE},
        {TEXT("GOOGLE_PLAY_GAMES"), EHiveProviderType::GOOGLE_PLAY_GAMES},
        {TEXT("HUAWEI"), EHiveProviderType::HUAWEI},
        {TEXT("STEAM"), EHiveProviderType::STEAM},
        {TEXT("X"), EHiveProviderType::X},
        {TEXT("TELEGRAM"), EHiveProviderType::TELEGRAM},
        {TEXT("XIAOMI"), EHiveProviderType::XIAOMI},
        {TEXT("CUSTOM"), EHiveProviderType::CUSTOM},
        {TEXT("AUTO"), EHiveProviderType::AUTO}
    };

    if (const EHiveProviderType* ProviderTypePtr = EHiveProviderTypeMap.Find(Name))
    {
        return *ProviderTypePtr;
    }
    return EHiveProviderType::UNKNOWN;
}

FString GetNameFromEHiveAuthV4MaintenanceActionType(const EHiveAuthV4MaintenanceActionType& ActionType)
{
    switch(ActionType) {
        case EHiveAuthV4MaintenanceActionType::OPEN_URL: return FString(TEXT("OPEN_URL"));
        case EHiveAuthV4MaintenanceActionType::EXIT: return FString(TEXT("EXIT"));
        case EHiveAuthV4MaintenanceActionType::DONE: return FString(TEXT("DONE"));
    }
    return FString(TEXT("NONE"));
}

EHiveAuthV4MaintenanceActionType GetEHiveAuthV4MaintenanceActionTypeFromName(const FString& Name)
{
    static TMap<FString, EHiveAuthV4MaintenanceActionType> ActionTypeMap = {
        {TEXT("OPEN_URL"), EHiveAuthV4MaintenanceActionType::OPEN_URL},
        {TEXT("EXIT"), EHiveAuthV4MaintenanceActionType::EXIT},
        {TEXT("DONE"), EHiveAuthV4MaintenanceActionType::DONE}
    };

    if (const EHiveAuthV4MaintenanceActionType* ActionTypePtr = ActionTypeMap.Find(Name))
    {
        return *ActionTypePtr;
    }
    return EHiveAuthV4MaintenanceActionType::NONE;
}

FHiveIdentity::FHiveIdentity(const hive::Identity& Identity)
: PhoneNumber(UTF8_TO_TCHAR(Identity.phoneNumber.c_str()))
, PhoneCode(UTF8_TO_TCHAR(Identity.phoneCode.c_str()))
, DateOfBirth(UTF8_TO_TCHAR(Identity.dateOfBirth.c_str()))
, HashedDi(UTF8_TO_TCHAR(Identity.hashedDi.c_str()))
{ }

FString GetNameFromFHiveIdentity(const FHiveIdentity& Identity)
{
    return FString::Printf(
        TEXT("FHiveIdentity { PhoneNumber: %s, PhoneCode: %s, DateOfBirth: %s, HashedDi: %s }"), *Identity.PhoneNumber, *Identity.PhoneCode, *Identity.DateOfBirth, *Identity.HashedDi
    );
}