/**
 * @file    HIVE_Logger.h
 *
 * @author  ryuvsken
 * @date		2016-2019
 * @copyright	Copyright © GAMEVIL COM2US PLATFORM Inc. All Right Reserved.
 * @defgroup Logger
 * @{
 * @brief API 호출에 대한 결과를 담는 클래스<br/><br/>
 */

#ifndef __HIVE_LOGGER_H__
#define  __HIVE_LOGGER_H__


#include "picojson.h"
#include "HIVE_ResultAPI.h"


NS_HIVE_BEGIN

/**
 * @brief 로거 타입 형태
 *
 * @ingroup Logger
 * @ingroup hive
 */
enum class LogType {
    Verbose = 0
    ,Debug = 1
    ,Info = 2
    ,Warning = 3
    ,Error = 4
    ,None = 5
};

class LogFilter;


/**
 * @brief 로그 정보를 기록하는 클래스<br/>
 * (네이티브 영역을 호출하기 때문에 부하를 고려해서 로그를 기록해야 한다.)<br/><br/>
 *
 * @ingroup Logger
 * @author ryuvsken
 */
class HIVESDK_DLLEXPORT Logger {

public:

    /**
     * 로그 메시지 발생 시 통지 핸들러 정의
     */
    typedef std::function<void(std::string const & tag, std::string const & level, std::string const & logMessage)> onLogger;
    
    
    /**
     * 로그 메시지 발생 시 통지 받을 핸들러 설정
     */
    static void setLoggerListener(onLogger listener);
    
    
    /**
     * \~korean 로그 정보 기록<br/>
     * 로그 출력에 대한 Base API
     * HIVE에서 리모트 로깅 활성화 시 원격지로 로그를 보내는 것이 가능하다.
     * 단, 네트워크 통신이 들어가는 만큼 과도하게 많은양의 로그나 반복적인 로그는 찍지 않도록 주의할것.
     * \~english Record log information<br/>
     * (Logs should be recorded taking into account the load, because you are calling the native code.)
     * \~
     */
    static void log(const char * format, ...);
    
    /**
     * log 함수를 사용했을때 로그를 원격지 혹은 로컬에 출력하는지의 여부를 반환한다.
     *
     * @return 로그 활성화 여부
     */
    static bool isActivateLogging();

    /**
     * Hive Core/Service 로그의 노출 타입을 조정한다.
     */
    static void setLogFilter(LogFilter filter);
};

/**
 * @brief Log의 노출 정도를 조정하는 로그 필터<br/>
 * core와 service 로그의 필터를 각각 조정 가능하다.
 *
 * @ingroup Logger
 */
class HIVESDK_DLLEXPORT LogFilter{
public:
    LogFilter();

    LogType coreLog;                 ///< \~korean 코어 로그 필터 타입    \~english  coreLog filter type.
    LogType serviceLog;           ///< \~korean 서비스 로그 필터 타입    \~english  serviceLog filter type.
};

NS_HIVE_END		// namespace hive


#endif		// __HIVE_LOGGER_H__


/** @} */




