/**
 * @file    HIVE_ProviderApple.h
 *
*  @date        2016-2019
*  @copyright   Copyright © GAMEVIL COM2US PLATFORM Inc. All Right Reserved.
*  @author      Arkeo Lucid
*  @since       4.3.0 
* 
*  @defgroup    Provider
*  @defgroup    ProviderApple
*  @ingroup     Provider
*  @addtogroup ProviderApple
*  @{
* \~korean Apple GameCenter의 업적 및 리더보드 기능을 제공한다.<br />
* \~english It provides Apple GameCenter achievements and leaderboard <br />
*/

#ifndef __HIVE_PROVIDERAPPLE_H__
#define  __HIVE_PROVIDERAPPLE_H__
  
  
#include "HIVE_ResultAPI.h"

NS_HIVE_BEGIN

class ProviderAppleAchievement;
/*
* \~korean Apple GameCenter의 업적 및 리더보드 기능을 제공한다.<br />
* \~english It provides Apple GameCenter achievements and leaderboard <br />
* @author Arkeo Lucid
* @since        4.3.0 
* @ingroup ProviderApple
*/
class HIVESDK_DLLEXPORT ProviderApple {
public:

    /**
    *  \~korean 
    * Apple GameCenter 리더보드 전송 result handler.
    * 
    * @param result API 호출 결과.
    * 
    *  \~english
    * Apple GameCenter Leaderboard result handler.
    * 
    * @param result Result of API call.
    * 
    *  \~
    * @see ResultAPI
    *
    * @ingroup ProviderApple
    */
    typedef std::function<void(ResultAPI const & result)> onReportLeaderboard;

    /**
    *  \~korean
    * Apple GameCenter 리더보드 UI 노출 result handler.
    *
    * @param result API 호출 결과.
    *
    *  \~english
    * Apple GameCenter ShowLeaderboard result handler.
    *
    * @param result API 호출 결과.
    *
    *  \~
    * @see ResultAPI
    * 
    * @ingroup ProviderApple
    */
    typedef std::function<void(ResultAPI const & result)> onShowLeaderboard;
    
    /**
    *  \~korean
    * Apple GameCenter 업적 로드 result handler.
    * 
    * @param result API 호출 결과.
    * @param achievements 업적 목록.
    * 
    *  \~english
    * Apple GameCenter LoadAchievements result handler.
    * 
    * @param result Result of API call.
    * @param achievements List of Achievements.
    * 
    * @see ResultAPI, ProviderAppleAchievement
    * 
    * @ingroup ProviderApple
    */
    typedef std::function<void(ResultAPI const & result,std::vector<ProviderAppleAchievement> const & achievements)> onLoadAchievements;
    
    /**
    *  \~korean
    * Apple GameCenter 업적을 전송 result handler.
    * 
    * @param result API 호출 결과.
    * 
    *  \~english
    * Apple GameCenter ReportAchievement result handler.
    * 
    * @param result Result of API call.
    *  
    *  \~
    * 
    * @see ResultAPI
    * 
    * @ingroup ProviderApple
    */
    typedef std::function<void(ResultAPI const & result)> onReportAchievement;

    /**
    *  \~korean
    * Apple GameCenter 업적 UI 노출 result handler.
    * 
    * @param result API 호출 결과.
    * 
    *  \~english
    * Apple GameCenter ShowAchievement result handler.
    * 
    * @param result Result of API call.
    *
    *  \~
    * 
    * @see ResultAPI
    * 
    * @ingroup ProviderApple
    */
    typedef std::function<void(ResultAPI const & result)> onShowAchievement;

    /**
    *  \~korean
    * Apple GameCenter 업적 초기화 result handler.
    * 
    * @param result API 호출 결과.
    * 
    *  \~english
    * Apple GameCenter ResetAchievements result handler.
    * 
    * @param result Result of API call.
    *
    *  \~
    * 
    * @see ResultAPI
    * 
    * @ingroup ProviderApple
    */
    typedef std::function<void(ResultAPI const & result)> onResetAchievements;


    /**
    *  \~korean
    * Apple GameCenter Leaderboard에 기록을 전송한다.
    * 
    * @warning score format과 leaderboard identifier는 iTunes Connect에서 설정한다.
    * 
    * @param score Player가 얻은 score.
    * @param leaderboardIdentifier Leaderboard Identifier.
    * @param listener API 호출 result handler.
    * 
    *  \~english
    * It send record to Apple GameCenter Leaderboard.
    * 
    * @warning The score format and leaderboard identifier are set in iTunes Connect.
    * 
    * @param score Player's score.
    * @param leaderboardIdentifier Leaderboard Identifier.
    * @param listener  Result listener of API call.
    *
    *  \~
    * @ingroup ProviderApple
    */
    static void reportScore(std::string score,std::string leaderboardIdentifier,onReportLeaderboard listener);

    /**
    *  \~korean
    * Apple GameCenter Leaderboard UI를 노출한다.
    * 
    * @param listener API 호출 result listener.
    * 
    *  \~english
    * It show Apple GameCenter Leaderboard UI.
    * 
    * @param listener Result listener of API call.
    *
    *  \~
    * @ingroup ProviderApple
    */
    static void showLeaderboard(onShowLeaderboard listener);

    /**
    *  \~korean
    * Apple GameCenter 업적을 로드한다.
    * 
    * @param listener API 호출 result listener.
    * 
    *  \~english
    * It load Apple GameCenter achievement.
    * 
    * @param listener Result listener of API call.
    *
    *  \~
    * @ingroup ProviderApple
    */
    static void loadAchievements(onLoadAchievements listener);

    /**
    *  \~korean
    * Apple GameCenter 업적을 전송한다.
    * 
    * @warning achievement identifier는 iTunes Connect에서 설정한다.
    * 
    * @param percent 업적 성취 percent value(ex. 75.0)
    * @param showsCompletionBanner 업적 성공시 상단 배너 노출 여부. default is NO.
    * @param achievementIdentifier Achievement identifier
    * @param listener API 호출 result listener.
    * 
    *  \~english
    * It report Apple GameCenter achievement.
    * 
    * @warning achievement identifier is set in iTunes Connect.
    * 
    * @param percent Achievement progress percent value(ex. 75.0)
    * @param showsCompletionBanner Whether the top banner is exposed when the achievement is successful. default is NO.
    * @param achievementIdentifier Achievement identifier
    * @param listener Result listener of API call.
    * 
    * \~
    * @ingroup ProviderApple
    */
    static void reportAchievement(std::string percent,bool showsCompletionBanner,std::string achievementIdentifier,onReportAchievement listener);

    /**
    *  \~korean
    * Apple GameCenter 업적 UI를 노출한다.
    * 
    * @param listener API 호출 result listener.
    * 
    *  \~english
    * It shows Apple GameCenter Achievement UI.
    * 
    * @param listener Result listener of API call.
    * 
    *  \~
    * @ingroup ProviderApple
    */
    static void showAchievements(onShowAchievement listener);

    /**
    *  \~korean
    * Apple GameCenter 업적 정보를 초기화한다.
    * 
    * @param listener API 호출 result listener.
    * 
    *  \~english
    * It resets Apple GameCenter Achievement information.
    * 
    * @param listener Result listener of API call.
    *
    *  \~
    * @ingroup ProviderApple
    */
    static void resetAchievements(onResetAchievements listener);

    static void executeEngine(picojson::value jsonParam);
};

/**
* \~korean Apple GameCenter 업적 정보
* \~english Apple GameCenter achievements
* @ingroup ProviderApple
*/
class HIVESDK_DLLEXPORT ProviderAppleAchievement {
public:
    ProviderAppleAchievement();
    ~ProviderAppleAchievement();

    std::string identifier; ///< \~korean 업적 identifier.  \~english  Achievement identifier.
    std::string percent;    ///< \~korean 업적 진행 percent value.  \~english  Achievement progress percent value.
    bool completed;         ///< \~korean 업적 완료 여부.   \~english  Whether the achievement is complete.
};

NS_HIVE_END

#endif  //__HIVE_PROVIDERAPPLE_H__
/** @} */