/**
 * @file    HIVE_ProviderGoogle.h
 *
 *  @date		2016-2019
 *  @copyright	Copyright © GAMEVIL COM2US PLATFORM Inc. All Right Reserved.
 *  @author		hife
 *  @since		4.3.0
 *  \brief		\copydoc 	ProviderGoogle
 */


/**
 *  @defgroup	Provider
 *  @defgroup	ProviderGoogle
 *  @ingroup 	Provider
 *  @addtogroup ProviderGoogle
 *  @{
 *  \~korean
 * Google Play Games 의 기능 중 업적과 리더보드를 사용할 수 있다.<br/>
 * 연결되어있지 않을 경우 API 를 호출 할 때 자동으로 연결을 시도 한다.<br/>
 * AuthV4 의 Connect 와는 무관하게 독립적으로 사용할 수 있다.<br/>
 * 자동 연결이 되어도 현재 playerid 와 Connect 되는것은 아니다.
 *
 *  \~english
 * You can use achievements and leaderboards within Google Play Games.<br/>
 * If it is not connected to Google Play Games, it will try to connect automatically when calling API.<br/>
 * It can be used independently regardless of AuthV4 Connect.<br/>
 * Even if automatic connection is made, it is not connected with the current playerId.
 *
 *  \~
 *
 * Created by hife on 2017. 6. 7.
 *
 * @author hife
 * @since		4.3.0
 */

#ifndef __HIVE_PROVIDERGOOGLE_H__
#define  __HIVE_PROVIDERGOOGLE_H__


#include "HIVE_ResultAPI.h"

NS_HIVE_BEGIN


/**
 *  \~korean
 * Google Play Games 의 기능 중 업적과 리더보드를 사용할 수 있다.<br/>
 * 연결되어있지 않을 경우 API 를 호출 할 때 자동으로 연결을 시도 한다.<br/>
 * AuthV4 의 Connect 와는 무관하게 독립적으로 사용할 수 있다.<br/>
 * 자동 연결이 되어도 현재 playerid 와 Connect 되는것은 아니다.
 *
 *  \~english
 * You can use achievements and leaderboards within Google Play Games.<br/>
 * If it is not connected to Google Play Games, it will try to connect automatically when calling API.<br/>
 * It can be used independently regardless of AuthV4 Connect.<br/>
 * Even if automatic connection is made, it is not connected with the current playerId.
 *
 *  \~
 *
 * Created by hife on 2017. 6. 7.
 *
 * @author hife
 * @since		4.3.0
 * @ingroup ProviderGoogle
 */
class HIVESDK_DLLEXPORT ProviderGoogle {

public:

    /**
     * @brief Google Play Games 리더보드 UI 노출 result handler
     * @param result API 호출 결과.
     * @see ResultAPI
     * @ingroup ProviderGoogle
     */
    typedef std::function<void(ResultAPI const & result)> onLeaderboardsResult;
    
    /**
     * @brief Google Play Games 업적 UI 노출 result handler
     * @param result API 호출 결과.
     * @see ResultAPI
     * @ingroup ProviderGoogle
     */
    typedef std::function<void(ResultAPI const & result)> onAchievementsResult;

    /**
     * @brief Google Player Id 및 Auth Code 반환 result handler
     * @param result API 호출 결과.
     * @param googlePlayerId Google PlayerId
     * @param authCode Auth Code
     * @ingroup ProviderGoogle
     */
    typedef std::function<void(ResultAPI const & result, std::string const & googlePlayerId, std::string const & authCode)> onGooglePlayerIdResult;

    /**
     *  \~korean
     * 숨겨진 업적을 공개한다.<br/>
     * 업적이 0% 로 공개만 될 뿐 달성 되지는 않는다.<br/>
     *
     * @param achievementId 공개할 achievementId
     *
     *  \~english
     * It show hidden achievements.<br/>
     * Achievements are only revealed at 0%, not achieved.<br/>
     *
     * @param achievementId achievementId to reveal
     *
     *  \~
     * @ingroup ProviderGoogle
     */
    static void achievementsReveal(const std::string achievementId);

    /**
     *  \~korean
     * 숨겨진 업적을 공개한다.<br/>
     * 업적이 0% 로 공개만 될 뿐 달성 되지는 않는다.<br/>
     *
     * @param achievementId 공개할 achievementId
     * @param listener GoogleAchievementsListener
     *
     *  \~english
     * It show hidden achievements.<br/>
     * Achievements are only revealed at 0%, not achieved.<br/>
     *
     * @param achievementId achievementId to reveal
     * @param listener GoogleAchievementsListener
     *
     *  \~
     * @ingroup ProviderGoogle
     */
    static void achievementsReveal(const std::string achievementId, onAchievementsResult listener);
    
    /**
     *  \~korean
     * 업적을 달성한다.<br/>
     * 숨겨져 있거나 공개된 여부와 상관없이 업적이 100% 로 달성 된다.<br/>
     *
     * @param achievementId 공개할 achievementId
     *
     *  \~english
     * It achieve achievements.<br/>
     * Whether hidden or open, achievement is achieved at 100%.<br/>
     *
     * @param achievementId achievementId to achieve
     *
     *  \~
     * @ingroup ProviderGoogle
     */
    static void achievementsUnlock(const std::string achievementId);
    
    /**
     *  \~korean
     * 업적을 달성한다.<br/>
     * 숨겨져 있거나 공개된 여부와 상관없이 업적이 100% 로 달성 된다.<br/>
     *
     * @param achievementId 공개할 achievementId
     * @param listener GoogleAchievementsListener
     *
     *  \~english
     * It achieve achievements.<br/>
     * Whether hidden or open, achievement is achieved at 100%.<br/>
     *
     * @param achievementId achievementId to achieve
     * @param listener GoogleAchievementsListener
     *
     *  \~
     * @ingroup ProviderGoogle
     */
    static void achievementsUnlock(const std::string achievementId, onAchievementsResult listener);

    /**
     *  \~korean
     * 업적 수치를 증가 시킨다.<br/>
     * value 만큼 설정이 아닌 매 value 만큼 합산 된다.<br/>
     * 총 합산이 Max 가 될 경우 자동으로 업적이 달성되며, 계속 호출하여도 무방하다.<br/>
     *
     *  \~english
     * It increases achievement figures.<br/>
     * Achievement figures is added as much as value set by the API call, not by setting.<br/>
     * If the total sum is Max, the achievement is automatically accomplished.<br/>
     *
     *  \~
     * @param incrementalAchievementId incrementalAchievementId
     * @param value value
     * @ingroup ProviderGoogle
     */
    static void achievementsIncrement(const std::string incrementalAchievementId, const int argValue);
    
    /**
     *  \~korean
     * 업적 수치를 증가 시킨다.<br/>
     * value 만큼 설정이 아닌 매 value 만큼 합산 된다.<br/>
     * 총 합산이 Max 가 될 경우 자동으로 업적이 달성되며, 계속 호출하여도 무방하다.<br/>
     *
     *  \~english
     * It increases achievement figures.<br/>
     * Achievement figures is added as much as value set by the API call, not by setting.<br/>
     * If the total sum is Max, the achievement is automatically accomplished.<br/>
     *
     *  \~
     * @param incrementalAchievementId incrementalAchievementId
     * @param value value
     * @param listener GoogleAchievementsListener
     * @ingroup ProviderGoogle
     */
    static void achievementsIncrement(const std::string incrementalAchievementId, const int argValue, onAchievementsResult listener);

    /**
     *  \~korean
     * 리더보드 점수를 갱신한다.<br/>
     * leaderboardId 에 해당하는 기록에 score 수치로 갱신된다.<br/>
     *
     *  \~english
     * It update the leaderboard score.<br/>
     * The score corresponding to the leaderboardId is updated with the score value.<br/>
     *
     *  \~
     * @param leaderboardId leaderboardId
     * @param score score
     * @ingroup ProviderGoogle
     */
    static void leaderboardsSubmitScore(const std::string leaderboardId, const long long score);
    
    /**
     *  \~korean
     * 리더보드 점수를 갱신한다.<br/>
     * leaderboardId 에 해당하는 기록에 score 수치로 갱신된다.<br/>
     *
     *  \~english
     * It update the leaderboard score.<br/>
     * The score corresponding to the leaderboardId is updated with the score value.<br/>
     *
     *  \~
     * @param leaderboardId leaderboardId
     * @param score score
     * @param listener GoogleLeaderboardsListener
     * @ingroup ProviderGoogle
     */
    static void leaderboardsSubmitScore(const std::string leaderboardId, const long long score, onLeaderboardsResult listener);

    /**
     *  \~korean
     * Google 리더보드 UI 를 띄운다.<br/>
     *
     *  \~english
     * It shows Google Leaderboard UI.
     *  \~
     * @param listener GoogleLeaderboardsListener
     * @ingroup ProviderGoogle
     */
    static void showLeaderboard(onLeaderboardsResult listener);
    
    /**
     *  \~korean
     * Google 업적 UI 를 띄운다.<br/>
     *
     *  \~english
     * It shows the Google achievement UI.<br/>
     *  \~
     * @param listener GoogleAchievementsListener
     * @ingroup ProviderGoogle
     */
    static void showAchievements(onAchievementsResult listener);

    /**
     *  \~korean
     * Google PlayerId 및 authCode를 가져온다.<br/>
     * 
     *  \~english
     * It get Google PlayerId and authCode
     *  \~
     * @param listener onGooglePlayerIdResult
     * @ingroup ProviderGoogle
     */
    static void getGooglePlayerId(onGooglePlayerIdResult listener);

    static void executeEngine(picojson::value jsonParam);
    
};

NS_HIVE_END

#endif  //__HIVE_PROVIDERGOOGLE_H__
