/**
 * @file    HIVE_SocialFacebook.h
 *
 * @author  ryuvsken, nanomech
 * @date		2016-2019
 * @copyright	Copyright © GAMEVIL COM2US PLATFORM Inc. All Right Reserved.
 *  @since		4.0.0
 *  \brief		\copydoc SocialFacebook
 */
 
 
/**
 * @defgroup SocialFacebook
 * @ingroup Social
 * @addgroup SocialFacebook
 * @{
 */

#ifndef __HIVE_SOCIALFACEBOOK_H__
#define  __HIVE_SOCIALFACEBOOK_H__


#include "HIVE_ResultAPI.h"
#include "HIVE_SocialHive.h"


NS_HIVE_BEGIN


class ResultAPI;

class ProfileFacebook;
class FacebookMessage;



/**
 * @ingroup SocialFacebook
 * @author		ryuvsken, nanomech
 * @since		4.0.0
 * \~korean HIVE 유저는 자신의 프로필 정보를 조회 하거나 수정할 수 있고 친구를 맺어서 친구와 함께 게임을 즐길 수 있는 소셜 게임 서비스를 제공한다.<br/>
 * HIVE 는 Facebook, 주소록 등의 외부 유저 정보를 이용하여 HIVE 친구를 맺고 목록의 동기화를 수행할 수 있다.<br/>
 * SocialFacebook 클래스는 Facebook 프로필, Facebook 친구 목록, Facebook 글 게시 기능을 제공한다.<br/><br/>
 *
 * \~english HIVE users can view and modify their profile information and provide a social game service where they can make friends and play games with their friends together.<br/>
 * HIVE can make HIVE friends and synchronize the list by using external user information such as Facebook and contacts on your device.<br/>
 * The SocialFacebook class provides Facebook profile, Facebook friend list, and posting.<br/><br/>
 */
class HIVESDK_DLLEXPORT SocialFacebook
{
public:


    /**
     * \~korean HIVE 프로필 / 친구 정보 결과 통지
     *
     * @param result		API 호출 결과
     * @param profileList	Facebook 유저의 프로필 목록.<br/><br/>(자신의 프로필일 경우 단건, 친구 목록일 경우 여러건이 반환된다.)
     * \~english Returns HIVE Profile / Friend Information
     *
     * @param result		API call result
     * @param profileList	Facebook user's profile list.<br/><br/>(If you ask for your profile, one result is returned, and if you  ask for friends list, a list of friends is returned.)
     * \~
     * @ingroup SocialFacebook
     */
	typedef std::function<void(ResultAPI const & result, std::vector<ProfileFacebook> const & profileList)> onProfileFacebook;


    /**
     * \~korean Facebook 메시지 전송 결과 통지
     *
     * @param result		API 호출 결과
     * \~english Returns Facebook message transmission result
     *
     * @param result		API call result
     * \~
     * @ingroup SocialFacebook
     */
	typedef std::function<void(ResultAPI const & result)> onSendMessageFacebook;


    /**
     * \~korean Facebook 유저에게 친구 초대 요청 결과 통지
     *
     * @param result			API 호출 결과
     * @param invitedUserList	초대된 Facebook 유저의 ID 목록
     * \~english Returns Facebook user invite to friend
     *
     * @param result			API call result
     * @param invitedUserList	List of IDs of invited Facebook users
     * \~
     * @ingroup SocialFacebook
     */
	typedef std::function<void(ResultAPI const & result, std::vector<std::string> const & invitedUserList)> onShowInvitationDialogFacebook;


    /**
     * \~korean Facebook 게시글 등록 요청 결과 통지
     *
     * @param result			API 호출 결과
     * \~english Returns Facebook posting result
     *
     * @param result			API call result
     * \~
     * @ingroup SocialFacebook
     */
	typedef std::function<void(ResultAPI const & result)> onPostFacebook;




    /**
     * \~korean Facebook 인증 사용자의 프로필 정보 조회
     *
     * @param listener	API 호출 결과 통지
     * \~english Request profile information of Facebook certified users
     *
     * @param listener	API call result listener
     * \~
     * @ingroup SocialFacebook
     */
	static void getMyProfile(onProfileFacebook listener);


    /**
     * \~korean Facebook 친구 목록 조회
     *
     * @param listener	API 호출 결과 통지
     * \~english Request Facebook friends list
     *
     * @param listener	API call result listener
     * \~
     * @ingroup SocialFacebook
     */
	static void getFriends(onProfileFacebook listener);


    /**
     * \~korean Facebook 친구에게 메시지 전송
     *
     * @param messageContents		Facebook 메시지를 전송할 정보
     * @param listener 		API 호출 결과 통지
     * \~english Send message to Facebook friend
     *
     * @param messageContents		Facebook message to be sent
     * @param listener 		API call result listener
     * \~
     * @ingroup SocialFacebook
     */
	static void sendMessageFacebook(FacebookMessage messageContents, onSendMessageFacebook listener);


    /**
     * \~korean Facebook 유저에게 친구 초대 요청 대화상자 호출<br/>
     * (일반적으로 소셜의 친구를 늘리기 위해서 게임 친구 초대에 대한 보상을 수행하도록 운영하지만 Facebook 에서는 이런 운영 방식을 정책적으로 막고 있으니 주의해야 한다.)
     *
     * @param inviteContents		Facebook 초대 메시지를 전송할 정보
     * @param listener		API 호출 결과 통지
     * \~english Show Facebook friend invite dialog to Facebook user <br/>
     * (Note: Generally, to increase the number of social friends, you can give a reward for inviting friends. but Facebook prohibited this.)
     *
     * @param inviteContents		Facebook invite message
     * @param listener		API call result listener
     * \~
     * @ingroup SocialFacebook
     */
	static void showInvitationDialog(FacebookMessage inviteContents, onShowInvitationDialogFacebook listener);


    /**
     * \~korean 게임의 소식을 흥미 있게 전하기 위해서 메시지와 이미지를 Facebook 에 포스팅하여 글쓴이의 타임라인에 글을 노출 시키는 기능을 제공한다
     *
     * @param contentURL		Facebook 게시글 정보 URL
     * @param listener		API 호출 결과 통지
     * \~english Post messages and images on Facebook timeline.
     *
     * @param contentURL		Facebook posting information URL.
     * @param listener		API call result
     * \~
     * @ingroup SocialFacebook
     */
    static void postFacebookWithContentURL(std::string contentURL, onPostFacebook listener);

    /**
    \~korean 사용자의 Facebook 세션 정보가 client에 있는 여부를 반환한다.

    @return Boolean. YES이면 사용자의 Facebook 세션 정보가 client에 있음. 아니면 NO.

    \~english 사용자의 Facebook 세션 정보가 client에 있는 여부를 반환한다.

    @return Boolean. YES이면 사용자의 Facebook 세션 정보가 client에 있음. 아니면 NO.
    * \~
    * @ingroup SocialFacebook
    */
    static bool isLogin();
    
    
    // Native 영역에서 호출된 요청을 처리하기 위한 플러그인 내부 코드
	static void executeEngine(picojson::value jsonParam);

};




/**
	 * \~korean Facebook 유저의 프로필 정보
	 *
	 * \~english Facebook user profile information
	 * \~
	 * @ingroup SocialFacebook
	 */
class HIVESDK_DLLEXPORT ProfileFacebook
{
public:
    std::string uid;			///< \~korean 페이스북 사용자 AppScoped uid. \~english Facebook AppScoped uid.
    		
    std::string email;		///< \~korean 페이스북 사용자 email (권한을 허가한 경우) \~english Facebook user email (If permission granted)

    std::string userName;		///< \~korean 페이스북 사용자의 이름. \~english Facebook user name.

    std::string profileImageUrl;	///< \~korean 페이스북 사용자의 프로파일 이미지 url. \~english Facebook user profile image url.
    
    std::string emailSecond;
    
    
    ProfileFacebook();
    
    ProfileFacebook(picojson::value jsonParam);
    
    picojson::object toJson() const;
    
    std::string toString() const;
};


/**
	 *  \~korean Facebook 메시지를 전송 할 대상 정보
	 *
	 * @see <a href="https://developers.facebook.com/docs/games/services/gamerequests"> Facebook-AppRequest </a>
	 * @see <a href="https://developers.facebook.com/docs/reference/dialogs/requests/"> Facebook-Request </a>
	 *
	 * @see <a href="https://developers.facebook.com/docs/app-invites/overview">Facebook-AppInvite </a>
	 *
	 *  \~english Facebook message destinations
	 *
	 * @see <a href="https://developers.facebook.com/docs/games/services/gamerequests"> Facebook-AppRequest </a>
	 * @see <a href="https://developers.facebook.com/docs/reference/dialogs/requests/"> Facebook-Request </a>
	 *
	 * @see <a href="https://developers.facebook.com/docs/app-invites/overview">Facebook-AppInvite </a>
	 * \~
	 * @ingroup SocialFacebook
	 */
class HIVESDK_DLLEXPORT FacebookMessage
{
public:
    
    std::vector<std::string> recipients;	///< 수신받을 사람들의 Facebook ID 목록.<br/><br/>(초대 용도로 쓰는 경우 무시됨)///< \~korean 수신받을 사람들의 Facebook ID 목록.<br/><br/>(초대 용도로 쓰는 경우 무시됨) \~ List of Facebook IDs of people who received message.<br/><br/>(Ignored if used for invitation purposes)
    
    std::string	dialogTitle;		///< \~korean 메시지 전송 대화 상자의 제목. Max 50. \~english Title. Max 50.
    		
    std::string	message;			///< \~korean 메시지 내용 \~english Message

    std::string	data;				///< \~korean 전달할 숨김데이터. Max 255. \~english Hidden data to send. Max 255.
    
    FacebookMessage();
    
    FacebookMessage(std::vector<std::string> recipients, std::string dialogTitle, std::string message, std::string data);
    
    FacebookMessage(picojson::value jsonParam);
    
    picojson::object toJson() const;
    
    std::string toString() const;
};


NS_HIVE_END		// namespace hive


#endif		// __HIVE_SOCIALFACEBOOK_H__


/** @} */



