/**
 * @file    HIVE_SocialHive.h
 *
 * @author  ryuvsken
 * @date		2016-2019
 * @copyright	Copyright © GAMEVIL COM2US PLATFORM Inc. All Right Reserved.
 *  \brief		\copydoc SocialHive
 */
 
 

/**
 * @defgroup Social
 *
 * \~korean HIVE 유저는 자신의 프로필 정보를 조회 하거나 수정할 수 있고 친구를 맺어서 친구와 함께 게임을 즐길 수 있는 소셜 게임 서비스를 제공한다.<br/>
 * HIVE 는 Facebook, 주소록 등의 외부 유저 정보를 이용하여 HIVE 친구를 맺고 목록의 동기화를 수행할 수 있다.<br/>
 * SocialHive 클래스는 HIVE 유저의 프로필, 친구 목록, 소셜 웹뷰 기능을 제공한다.<br/><br/>
 * \~english HIVE users can view and modify their profile information and provide a social game service where they can make friends and play games with their friends together.<br/>
 * HIVE can make HIVE friends and synchronize the list by using external user information such as Facebook and contacts on your device.<br/>
 * SocialHive class provides HIVE user profile, friend list and social web <br/><br/>
 * \~
 * @defgroup SocialHive
 * @ingroup Social
 * @addgroup SocialHive
 * @{
 */
 
#ifndef __HIVE_SOCIALHIVE_H__
#define  __HIVE_SOCIALHIVE_H__


#include "HIVE_ResultAPI.h"


NS_HIVE_BEGIN


class ResultAPI;


enum class FriendType;
enum class HiveDialogType;
enum class HiveRelationRoute;

class ProfileHive;
class MessageContent;
class SocialBadge;


/**
 * \~korean HIVE 유저는 자신의 프로필 정보를 조회 하거나 수정할 수 있고 친구를 맺어서 친구와 함께 게임을 즐길 수 있는 소셜 게임 서비스를 제공한다.<br/>
 * HIVE 는 Facebook, 주소록 등의 외부 유저 정보를 이용하여 HIVE 친구를 맺고 목록의 동기화를 수행할 수 있다.<br/>
 * SocialHive 클래스는 HIVE 유저의 프로필, 친구 목록, 소셜 웹뷰 기능을 제공한다.
 *
 * \~english HIVE users can view and modify their profile information and provide a social game service where they can make friends and play games with their friends together.<br/>
 * HIVE can make HIVE friends and synchronize the list by using external user information such as Facebook and contacts on your device.<br/>
 * SocialHive class provides HIVE user profile, friend list and social web
 * \~
 * @ingroup SocialHive
 */
class HIVESDK_DLLEXPORT SocialHive
{
public:


    /**
     * \~korean HIVE 유저의 프로필 / 친구 목록 결과 통지
     *
     * @param result		API 호출 결과
     * @param profileList	HIVE 유저의 프로필 목록.<br/><br/>(자신의 프로필일 경우 단건, 친구 목록일 경우 여러건이 반환된다.)
     * \~english Returns HIVE Profile / Friend Information
     *
     * @param result		API call result
     * @param profileList	HIVE user's profile list.<br/><br/>(If you ask for your profile, one result is returned, and if you  ask for friends list, a list of friends is returned.)
     * \~
     * @ingroup SocialHive
     */
	typedef std::function<void(ResultAPI const & result, std::vector<ProfileHive> const & profileList)> onProfileHive;


    /**
     * \~korean HIVE 친구 메시지 전송 결과 통지
     *
     * @param result		API 호출 결과
     * \~english Returns HIVE message transmission result
     *
     * @param result		API call result
     * \~
     * @ingroup SocialHive
     */
	typedef std::function<void(ResultAPI const & result)> onSendMessageHive;


    /**
     * \~korean HIVE 웹뷰 대화상자 결과 통지
     *
     * @param result		API 호출 결과
     * \~english Returns showing HIVE Social dialog result
     *
     * @param result		API call result
     * \~
     * @ingroup SocialHive
     */
	typedef std::function<void(ResultAPI const & result)> onShowHiveDialog;
    
    
    /**
     * \~korean HIVE Social Badge 정보
     *
     * @param result		API 호출 결과
     * @param badge			HIVE-SocialBadge정보
     * \~english HIVE Social Badge information
     *
     * @param result		API call result
     * @param badge			HIVE-Social Badge information
     * \~
     * @ingroup SocialHive
     */
    typedef std::function<void(ResultAPI result, SocialBadge socialBadge)> onGetBadgeInfo;




    /**
    * \~korean HIVE 유저 자신의 프로필 정보 조회
    *
    * @param listener	API 결과 통지
    * \~english Request profile information of HIVE certified users
    *
    * @param listener	API call result listener
    * \~
    * @ingroup SocialHive
    */
	static void getMyProfile(onProfileHive listener);


    /**
    * \~korean HIVE 유저 자신의 프로필 정보 설정
    *
    * @param displayName	유저의 상태 메시지
    * @param listener		API 결과 통지
    * \~english Set profile information of HIVE certified users
    *
    * @param displayName	Message of user's status
    * @param listener		API call result listener
    * \~
    * @ingroup SocialHive
    */
	static void setMyProfile(std::string displayName, onProfileHive listener);


    /**
    * \~korean  HIVE 유저의 친구 정보 조회<br/>
    * 친구의 형태는 게임을 같이하고 있는 친구, 게임을 같이하고 있지 않은 친구, 게임을 초대한 친구, 모든 친구가 있고 FriendType 라는 enum 으로 정의되어 있다
    *
    * @param friendType	친구 목록 조회 형태
    * @param listener		API 결과 통지
    * \~english Request HIVE Profile / Friend Information
    * In the form of friends, there are friends who play games together, friends who do not play games, friends who invite games, all friends, and these are defined by an enum called FriendType
    *
    * @param friendType Types of Friend list
    * @param listener		API call result listener
    * \~
    * @ingroup SocialHive
    */
	static void getFriends(FriendType friendType, onProfileHive listener);


    /**
    * \~korean HIVE 사용자 정보를 조회한다.
    *
    * @param vidList 조회하고자 하는 사용자의 VID 목록
    * @param listener API 결과 통지
    * \~english HIVE 사용자 정보를 조회한다.
    *
    * @param vidList 조회하고자 하는 사용자의 VID 목록
    * @param listener API 결과 통지
    * \~
    * @ingroup SocialHive
    */
    static void getProfiles(std::vector<std::string> vidList, onProfileHive listener);
    
    
    /**
    * \~korean HIVE 친구에게 메시지 전송 요청<br/>
    * HIVE 친구에게 메시지를 발송하면 상대방이 메시지를 수신한 것을 알 수 있도록 푸시 알림이 발송된다.<br/>
    * 푸시 알림이 발송되는 매체는 발신자가 이용 중인 게임과 동일한 게임 또는 최근에 이용한 게임이 된다.<br/>
    * 수신자가 푸시 알림을 받고 메시지를 터치하면 해당 게임이 실행 되므로, 게임 내에 HIVE 메시지로 연결되는 UI를 포함해야 한다
    *
    * @param messageContent	HIVE 메시지를 전송 할 대상의 정보
    * @param listener			API 결과 통지
    * \~english Send message to HIVE friend <br/>
    * If you send a message to a HIVE friend, a push notification will be sent to the recipient who has received the message.<br/>
    * The medium to which the push notification is sent is the same game as the game that the sender is using, or a game that has been used recently.<br/>
    * When the recipient receives the push notification and touches the message, the game is executed, so the UI that leads to the HIVE message should be included in the game.
    *
    * @param messageContent	Information to send HIVE message
    * @param listener			API call result listener
    * \~
    * @ingroup SocialHive
    */
	static void sendMessage(MessageContent messageContent, onSendMessageHive listener);


    /**
     * \~korean HIVE 유저에게 초대 메시지 전송 요청
     *
     * @param messageContent	HIVE 초대 메시지를 전송 할 대상의 정보
     * @param listener			API 결과 통지
     * \~english Send a invite message to HIVE user
     *
     * @param messageContent	Information to send HIVE invite message
     * @param listener			API call result listener
     * \~
     * @ingroup SocialHive
     */
	static void sendInvitationMessage(MessageContent messageContent, onSendMessageHive listener);


    /**
     * \~korean HIVE 웹뷰 대화상자 호출<br/>
     * (- HIVE의 기본 첫 화면 (피드){@inheritDoc HiveDialogType#HOME}<br/>
     * - 자신의 프로필 페이지{@inheritDoc HiveDialogType#USER}<br/>
     * - 게임 페이지{@inheritDoc HiveDialogType#GAME}<br/>
     * - 1:1 문의 하기 {@inheritDoc HiveDialogType#INQUIRY})
     * - HIVE 쪽지 페이지. {@inheritDoc HiveDialogType#MESSAGE}
     * - 챗봇 1:1 문의 하기{@inheritDoc HiveDialogType#CHATBOT})
     *
     * @param hiveDialogType
     *            {@link HiveDialogType} HIVE 웹뷰 대화상자 형태
     * @param vid
     *            친구의 프로필 페이지로 바로 갈 경우 친구의 vid 를 설정한다
     * @param listener
     *            API 호출 결과 통지
     * \~english Show HIVE WebView dialog<br/>
     * (- HIVE Social's default first screen (feed){@inheritDoc HiveDialogType#HOME}<br/>
     * - Profile page{@inheritDoc HiveDialogType#USER}<br/>
     * - Games{@inheritDoc HiveDialogType#GAME}<br/>
     * - 1:1 Contact us {@inheritDoc HiveDialogType#INQUIRY})
     * - HIVE Note. {@inheritDoc HiveDialogType#MESSAGE}
     * - Chatbot 1:1 inquiry{@inheritDoc HiveDialogType#CHATBOT})
     *
     * @param hiveDialogType
     *            {@link HiveDialogType} Type of HIVE WebView Dialog
     * @param vid
     *            If you go directly to your friend's profile page, set your friend's vid
     * @param listener
     *            API call result listener
     * \~
     * @ingroup SocialHive
     */
    static void showHiveDialog(HiveDialogType hiveDialogType, std::string vid, onShowHiveDialog listener);

    /**
     * \~korean HIVE 웹뷰 대화상자 호출<br/>
     * (- HIVE의 기본 첫 화면 (피드){@inheritDoc HiveDialogType#HOME}<br/>
     * - 자신의 프로필 페이지{@inheritDoc HiveDialogType#USER}<br/>
     * - 게임 페이지{@inheritDoc HiveDialogType#GAME}<br/>
     * - 1:1 문의 하기 {@inheritDoc HiveDialogType#INQUIRY})
     * - HIVE 쪽지 페이지. {@inheritDoc HiveDialogType#MESSAGE}
     * - 챗봇 1:1 문의 하기 {@inheritDoc HiveDialogType#CHATBOT}
     *
     * @param hiveDialogType
     *            {@link HiveDialogType} HIVE 웹뷰 대화상자 형태
     * @param vid
     *            친구의 프로필 페이지로 바로 갈 경우 친구의 vid 를 설정한다
     * @param additionalInfo
     *            챗봇 페이지를 바로가기 위해  전달받기로한 약속된 JSON 형식의 String 데이터
     * @param handler
     *            API 호출 결과 통지
     * \~english Show HIVE WebView dialog<br/>
     * (- HIVE Social's default first screen (feed){@inheritDoc HiveDialogType#HOME}<br/>
     * - Profile page{@inheritDoc HiveDialogType#USER}<br/>
     * - Games{@inheritDoc HiveDialogType#GAME}<br/>
     * - 1:1 Contact us {@inheritDoc HiveDialogType#INQUIRY})
     * - HIVE Note. {@inheritDoc HiveDialogType#MESSAGE}
     *
     * @param hiveDialogType
     *            {@link HiveDialogType} Type of HIVE WebView Dialog
     * @param vid
     *            If you go directly to your friend's profile page, set your friend's vid
     * @param additionalInfo
     *            Promised String data (JSON format) when you want to open chatbot page directly
     * @param handler
     *            API call result handler
     * \~
     * @ingroup SocialHive
     */
    static void showHiveDialog(HiveDialogType hiveDialogType, std::string vid, std::string additionalInfo, onShowHiveDialog listener);


    /**
     * \~korean HIVE Social Badge 정보를 조회.
     *
     * @param listener
     *            {@link SocialBadgeListener} HIVE SocialBadge 정보 조회 결과 통지
     *
     * \~english Request HIVE Social Badge info.
     *
     * @param listener
     *            {@link SocialBadgeListener} HIVE SocialBadge info result listener
     * \~
     * @ingroup SocialHive
     */
	static void getBadgeInfo(onGetBadgeInfo listener);

    
	// Native 영역에서 호출된 요청을 처리하기 위한 플러그인 내부 코드
	static void executeEngine(picojson::value jsonParam);

};




/**
 * \~korean 친구 목록 조회 형태
 *
 * \~english Types of Friend list
 * \~
 * @ingroup SocialHive
 */
enum class FriendType
{
    IN_GAME			///< \~korean 게임을 같이하고 있고, HIVE 상에서 친구 관계인 경우  \~english  If you are playing a game and they are friends on HIVE.
    , OUT_GAME		///< \~korean 게임을 같이하고 있지 않지만, HIVE 상에서 친구 관계인 경우  \~english  If you are not playing games, but they are friends on HIVE.
    , INVITED		///< \~korean HIVE 소셜에서 초대한 친구  \~english  Friends invited by HIVE Social.
    , ALL_GAME		///< \~korean HIVE 상에서 친구 관계인 경우  \~english  All friends on HIVE.
    
};


/**
 * \~korean HIVE 웹뷰 대화상자의 형태
 *
 * \~english Types of HIVE Social WebView Dialog
 * \~
 * @ingroup SocialHive
 */
enum class HiveDialogType
{
    HOME			///< \~korean HIVE 소셜 웹 뷰 홈 페이지  \~english  HIVE Social home.
    		, GAME			///< \~korean HIVE 게임 목록 페이지.  \~english  HIVE Social game list.
    		, USER			///< \~korean HIVE 유저 프로필 페이지.  \~english  HIVE Social user profile.
    		, INQUIRY	///< \~korean HIVE 1:1 문의하기 페이지.  \~english  HIVE 1:1 Inquiry.
    		, MESSAGE	///< \~korean HIVE 쪽지 페이지. Hive 유저만 사용 가능.  \~english  HIVE Note. Only for HIVE user.
            , CHATBOT   ///< \~korean HIVE 챗봇 1:1 문의하기 페이지. \~english HIVE Chatbot 1:1 Inquiry.
            , MYINQUIRY   ///< \~korean 내 문의 페이지. \~english HIVE My Inquiry.
};


/**
 * \~korean HIVE 친구 관계가 맺어진 경로 형태.
 *
 * \~english HIVE The path form of friendship.
 * \~
 * @ingroup SocialHive
 */
enum class HiveRelationRoute
{
    DEFAULT, 	///< not set
    		HIVE, 			///< \~korean HIVE Social 로 맺어짐.  \~english  Through HIVE Social.
    		FACEBOOK, 	///< \~korean Facebook으로 맺어짐.  \~english  Through Facebook.
    		GAME, 			///< \~korean InGame에서 맺어짐.  \~english  Through Game.
    		CONTACT 	///< \~korean 주소록으로 맺어짐.  \~english  Through Contacts.
};




/**
 * \~korean HIVE 유저의 프로필 정보<br>
 *
 * \~english HIVE user profile information<br>
 * \~
 * @ingroup SocialHive
 */
class HIVESDK_DLLEXPORT ProfileHive
{
public:
    std::string vid;				///< \~korean HIVE 로그인을 수행하면 게임별로 발급되는 사용자의 고유 ID<br/><br/>게임에서는 vid 기준으로 유저 정보를 관리한다  \~english  The unique ID of the user issued per game when performing a HIVE login <br/> <br/> The game manages user information by vid.
    
    std::string uid;				///< \~korean HIVE Social 에서 사용하는 게임과 별개인 사용자 고유의 ID (Big Integer형태)  \~english  Unique ID (Big Integer type) used in HIVE Social
    
    std::string identifier;		///< \~korean HIVE 로그인 ID (max 12)  \~english  HIVE Login ID (max 12)
    
    std::string userName;			///< \~korean 사용자가 입력한 HIVE 닉네임 또는 facebook name (max 128)  \~english  HIVE nickname or facebook name entered by the user (max 128)
    
    std::string facebookId;		///< \~korean 페이스북 계정이 연결된 경우 페이스북 Id, 없으면 null  \~english  Facebook Id if Facebook account is connected, null if not.
    
    std::string googleplusId;		///< \~korean 구글플러스 계정이 연결된 경우 구글플러스 Id, 추후 PGS PlayerId변경 가능, 없으면 null (Android only.)  \~english  If you have a Google Sign-in account linked or null if none (Android only.)
    
    std::string profileImageUrl;	///< \~korean 프로필 이미지 URL  \~english  Profile image URL
    
    std::string country;			///< \~korean 유저 선택에 따른 국가 코드  \~english  Country code according to user selection
    
    std::string comment;			///< \~korean 유저의 상태 메시지  \~english  The user's status message (a "word" entered by the user)
    
    bool cached;			///< \~korean 데이터 캐싱 여부(테스트 필드)  \~english  Whether data is cached (test field)
    
    bool testAccount;		///< \~korean 테스트 계정 여부, true/false  \~english  Whether it is a test account, true/false
    
    
    // 이하 세개 필드는 HIVE 접속 유저의 추가 정보.
    
    std::string	email;			///< \~korean 로그인 메일 주소. 없으면 null  \~english  Login email address. Null if none
    
    std::string	birthday;		///< \~korean 생년월일 , 설정 안되었으면 null  \~english  Birthday , Null if none

    std::string	gender;			///< \~korean 성별 , M or F , 설정 안되었으면 null  \~english  Gender , Null if none
    
    
    // 이하 두개 필드는 HIVE 친구 추가정보.
    
    bool gameFriend;		///< \~korean 게임을 같이 하는 게임 친구 여부, true/false  \~english  Whether in game friend, true/false

    std::string	assnet;			///< \~korean HIVE 회원 전환 여부, C: Com2us, G : GameVil, H : HIVE, or null  \~english  HIVE Membership, C: Com2us, G : GameVil, H : HIVE, or null
    
    HiveRelationRoute	relationRoute;		///< \~korean 최초로 친구 관계가 된 경로 정보. \~english A route that you became friend for the first time.
    
    
    ProfileHive();
    
    ProfileHive(picojson::value jsonParam);
    
    picojson::object toJson() const;
    
    std::string toString() const;
};


/**
 * \~korean 메시지 전송 내용
 *
 * \~english Message information
 * \~
 * @ingroup SocialHive
 */
class HIVESDK_DLLEXPORT MessageContent
{
    
public:
    std::string vid;				///< \~korean 쪽지를 전송할 상대의 vid, uid가 null인경우, 필수 파라매터.  \~english  The vid of the person to whom you want to send the note. it is required if uid is null.
    std::string uid;				///< \~korean 쪽지를 전송할 상대의 uid, null이거나 빈문자열인 경우, vid에 해당하는 uid를 사용한다.  \~english The uid of the recipient of the note.
    std::string message;			///< \~korean 쪽지 메시지  \~english Message
    std::string imageUrl;			///< \~korean 첨부 이미지 URL  \~english Attached image URL
    std::string thumbnailUrl;		///< \~korean 첨부 이미지의 썸네일 URL  \~english Thumbnail URL of attached image
    bool usePush;			///< \~korean 쪽지 받는 사람이 접속한 디바이스에 쪽지 내용을 Push 로 보내는지 여부  \~english Whether to send the contents of the note to the device to which the recipient has connected.
    
    MessageContent();
    
    MessageContent(std::string vid, std::string uid, std::string message, std::string imageUrl, std::string thumbnailUrl, bool usePush);
    
    MessageContent(picojson::value jsonParam);
    
    picojson::object toJson() const;
    
    std::string toString() const;
};

/**
 * \~korean HIVE Social Badge 정보.
 *
 * \~english HIVE Social Badge information.
 * \~
 * @ingroup SocialHive
 *
 */
class SocialBadge
{
public:
	int messageCount;       ///< \~korean 쪽지 개수  \~english The number of message

	SocialBadge();

	SocialBadge(int messgeCount);

	SocialBadge(picojson::value jsonParam);

	picojson::object toJson();

	std::string toString();
};


NS_HIVE_END		// namespace hive


#endif		// __HIVE_SOCIALHIVE_H__


/** @} */



