/**
 * @file        HiveAnalytics.h
 *
 * @date        2025
 * @copyright   Copyright © 2025 Com2uS Platform Corp. All Rights Reserved
 * @defgroup    FHiveAnalytics
 * @{
 * \~korean
 * 앱과 사용자를 트래킹하고 분석하기 위한 기능 모음<br/>
 * (User Tracking Tool Wrapper & Collect Analytics Log API)<br/><br/>
 * \~english
 * A collection of features for tracking and analyzing apps and users<br/>
 * (User Tracking Tool Wrapper & Collect Analytics Log API)<br/><br/>
 */
#pragma once

#include "CoreMinimal.h"
#include "HiveInterface.h"
#include "Impl/HiveAnalyticsImpl.h"

/**
 * \~korean
 * 앱과 사용자를 트래킹하고 분석하기 위한 기능 모음<br/>
 * (User Tracking Tool Wrapper & Collect Analytics Log API)<br/><br/>
 * \~english
 * A collection of features for tracking and analyzing apps and users<br/>
 * (User Tracking Tool Wrapper & Collect Analytics Log API)<br/><br/>
 * \~
 * @ingroup FHiveAnalytics
 */
class HIVESDK_API FHiveAnalytics : public IHiveInterface<IHiveAnalyticsImpl>
{
public:
    /**
    * \~korean 분석용 로그 전송.
    *
    * @param LogData 로그 데이터
    *
    * \~english Send log for analysis.
    *
    * @param LogData Log data
    *
    * \~
    * @ingroup FHiveAnalytics
    */
    static void SendAnalyticsLog(const TSharedPtr<FJsonObject>& LogData);

    /**
    * \~korean 사용자 분석을 위한 사용자 정보 트래커 사용 유무 설정
    *
    * @param Name 사용자 분석을 위한 사용자 정보 트래커 형태
    * @param IsEnable 트래커 사용 유무
    *
    * \~english Set whether to use User Information Tracker for user analysis
    *
    * @param Name User information tracker type
    * @param IsEnable whether to use User Information Tracker
    * \~
    * @ingroup FHiveAnalytics
    */
    static void SetEnableTracker(const FString& Name, bool IsEnable);

    /**
    * \~korean 사용자 분석을 위한 사용자 정보 이벤트 설정
    *
    * @param EventName 사용자 정보 이벤트
    *
    * \~english Send event for user analysis
    *
    * @param EventName Event name
    * \~
    * @ingroup FHiveAnalytics
    */
    static void SendEvent(const FString& EventName);

    /**
    * \~korean 사용자 분석을 위한 사용자 정보 이벤트 설정
    *
    * @param EventName 사용자 정보 이벤트
    * @param Attributes 추가 데이터
    *
    * \~english Send event for user analysis
    *
    * @param EventName Event name
    * @param Attributes Additional Data
    * \~
    * @ingroup FHiveAnalytics
    */
    static void SendEventWithAttributes(const FString& EventName, const FHiveAnalyticsAttributes& Attributes);

    /**
    * \~korean 광고 수익 측정 이벤트 설정
    *
    * @param AdRevenue	광고 수익 측정 데이터
    *
    * \~english Send events for ad revenue measurement
    *
    * @param AdRevenue	Ad revenue data
    *
    * \~
    * @ingroup FHiveAnalytics
    */
    static void SendAdRevenueEvent(const FHiveAnalyticsAdRevenue& AdRevenue);

    /**
     * \~korean
     * @brief 사용자의 퍼널 분석을 위한 지표를 전송한다.
     *
     * 사전정의된 퍼널 목록
     * <table>
     *     <th>
     *         <td>구간명</td><td>퍼널값</td><td>HIVE SDK 자동전송 유무</td>
     *     </th>
     *     <tr>
     *         <td>권한고지</td><td>410</td><td>YES</td>
     *     </tr>
     *     <tr>
     *         <td>고지팝업</td><td>420</td><td>YES</td>
     *     </tr>
     *     <tr>
     *         <td>약관동의</td><td>430</td><td>YES</td>
     *     </tr>
     *     <tr>
     *         <td>게임서버선택</td><td>500</td><td>YES</td>
     *     </tr>
     *     <tr>
     *         <td>서버점검팝업</td><td>600</td><td>YES</td>
     *     </tr>
     *     <tr>
     *         <td>추가다운로드</td><td>700</td><td><b>NO</b></td>
     *     </tr>
     *     <tr>
     *         <td>추가다운로드 완료</td><td>800</td><td><b>NO</b></td>
     *     </tr>
     *     <tr>
     *         <td>로그인</td><td>900</td><td>YES</td>
     *     </tr>
     *     <tr>
     *         <td>전면배너</td><td>1000</td><td>YES</td>
     *     </tr>
     * </table>
     *
     * @param FunnelTrack 사전정의된 퍼널의 값
     * @param Tag 옵션으로 추가 전달할 값
     * 
     * \~
     * @ingroup FHiveAnalytics
     */
    static void SendUserEntryFunnelsLogs(const FString& FunnelTrack, const FString& Tag);

    /**
     * \~korean
     * 개인 정보를 사용하는 데 최종 사용자 동의를 받기 위한 UI를 노출합니다.
     * 이미 동의를 한 경우 노출되지 않으며 기한이 만료되었거나 추가 동의가 필요한 경우 노출 됩니다.
     * [ConsentMode]를 null 로 입력할 경우 Hive SDK 의 Analytics Provider 추가 여부에 따른 동의 목록을 자동으로 생성합니다.
     * Hive SDK 에서 판단하는 [ConsentForm.ProviderName] 은 "main", "google", "adjust", "appsflyer",
     * "singular" 입니다.
     *
     * @param CheckCmp true 일 경우 서드 파티 동의 관리 플랫폼(CMP) 또는 맞춤 솔루션으로 TCF 규격에 맞게 저장된 파일이 있다면 해당 파일을 읽어 파일에 있는 업체는 동의 설정한 것으로 간주하게 됩니다.
     * @param ConsentMode UI에 띄울 내용과 Provider를 모두 커스텀 할 수 있게 됩니다. 기본 동의인 [ConsentMode.MainForm] 은 [ConsentForm.ProviderName] 이 "main" 으로 설정됩니다.
     * @param Delegate [FHiveAnalyticsOnShowConsentModeDelegate] 로 UI에서 동의한 결과를 [ConsentStatus] 배열로 응답합니다. 취소할 경우 [FHiveResultAPI.CANCELED] [FHiveResultAPI.Code.AnalyticsConsentModeCanceled] 로 응답합니다.
     *
     * @see [ShowConsentMode]
     * 
     * \~
     * @ingroup FHiveAnalytics
     */
    static void ShowConsentModeIfRequire(bool CheckCmp, const TOptional<FHiveConsentMode>& ConsentMode, const FHiveAnalyticsOnShowConsentModeDelegate& Delegate);
    
    /**
     * \~korean
     * 개인 정보를 사용하는 데 최종 사용자 동의를 받기 위한 UI를 노출합니다.
     * 이미 동의를 한 경우 노출되지 않으며 기한이 만료되었거나 추가 동의가 필요한 경우 노출 됩니다.
     * [ConsentMode]를 null 로 입력할 경우 Hive SDK 의 Analytics Provider 추가 여부에 따른 동의 목록을 자동으로 생성합니다.
     * Hive SDK 에서 판단하는 [ConsentForm.ProviderName] 은 "main", "google", "adjust", "appsflyer",
     * "singular" 입니다.
     *
     * @param ConsentMode UI에 띄울 내용과 Provider를 모두 커스텀 할 수 있게 됩니다. 기본 동의인 [ConsentMode.MainForm] 은 [ConsentForm.ProviderName] 이 "main" 으로 설정됩니다.
     * @param Delegate [FHiveAnalyticsOnShowConsentModeDelegate] 로 UI에서 동의한 결과를 [ConsentStatus] 배열로 응답합니다. 취소할 경우 [FHiveResultAPI.CANCELED] [FHiveResultAPI.Code.AnalyticsConsentModeCanceled] 로 응답합니다.
     *
     * @see [ShowConsentModeIfRequire]
     *
     * \~
     * @ingroup FHiveAnalytics
     */
    static void ShowConsentMode(const TOptional<FHiveConsentMode>& ConsentMode, const FHiveAnalyticsOnShowConsentModeDelegate& Delegate);
};