/**
 * @file        HiveChat.h
 *
 * @date        2025
 * @copyright   Copyright © 2025 Com2uS Platform Corp. All Rights Reserved.
 * @author      herohjk
 * @since       4.25.0.0
 * @defgroup FHiveChat
 * @{
 *  \~korean
 * Hive 채팅 서비스는 실시간 채팅, 그룹 채팅, 1:1 채팅, 사용자 간 차단 기능 등 앱에서 원활한 채팅을 할 수 있도록 핵심 기능을 제공한다.<br/>
 *
 *  \~english
 * The Hive Chat service provides essential features for seamless chat within the app, including real-time chat, group chat, one-on-one chat, and user blocking functionality.<br/>
 *
 * \~
 */

#pragma once

#include "CoreMinimal.h"
#include "HiveInterface.h"
#include "Impl/HiveChatImpl.h"

/**
 *  \~korean
 * HiveChat은 실시간, 그룹, 1:1 채팅, 유저 간 차단 기능 등 게임에서 원활한 채팅을 할 수 있도록 필요한 핵심 기능을 제공하기 때문에 유저 간 협력을 증가시켜 게임 플레이 만족도를 높일 수 있습니다.<br/>
 * 별도 복잡한 개발 없이도 Hive Chat에서 제공하는 AI 채팅 필터링 기능으로 금칙어 및 광고성 채팅을 감지하고 차단해 유저의 플레이 환경을 개선할 수 있습니다.<br/>
 *
 *  \~english
 * HiveChat provides essential features for smooth chatting in games, such as real-time, group, 1:1 chat, and user blocking functions, which can increase user collaboration and improve game play satisfaction.<br/>
 * Without additional complex development, you can use the AI chat filtering feature provided by Hive Chat to detect and block prohibited words and advertising chats, improving the user's play environment.<br/>
 *  \~
 *
 * @since   4.25.0.0
 * @ingroup FHiveChat
 */
class HIVESDK_API FHiveChat : public IHiveInterface<IHiveChatImpl>
{
public:
    /**
     * \~korean
     * 채널 생성<br/>
     *
     * 채널을 생성합니다.<br/>
     * 채널명, 최대 인원수, 타입을 지정할수 있으며, PUBLIC이 아니라면 비밀번호 또한 함께 지정해야합니다. <br/>
     * 채널 생성에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Create a channel<br/>
     * Channel Name, Maximum Number of People, and Type can be specified, and if it is not PUBLIC, a password must also be specified.<br/>
     * The result of creating a channel is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void CreateChannel(FHiveCreateChannelParams const & params, const FHiveChatOnResultDelegate& Delegate);
    
    /**
     * \~korean
     * 채널 목록 조회<br/>
     * 접근 가능한 채널 목록을 조회합니다.<br/>
     * 채널 타입, 채널명, 페이지 순서, 페이지 사이즈, 페이지 번호를 지정할 수 있습니다.<br/>
     * 채널 목록 조회에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Get channel list<br/>
     * Retrieves a list of accessible channels.<br/>
     * You can specify the channel type, channel name, page order, page size, and page number.<br/>
     * The result of the channel list retrieval is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void GetChannels(TOptional<FHiveGetChannelsParams> params, const FHiveChatOnGetChannelsDelegate& Delegate);
    
    /**
     * \~korean
     * 채널 정보 조회<br/>
     * 채널 ID를 입력받아, 해당 채널의 정보를 조회합니다.<br/>
     * 조회 결과로 해당 채널의 정보와, 채널에 참여중인 멤버들의 목록이 전달됩니다.<br/>
     *
     * \~english
     * Get channel information<br/>
     * Receives the channel ID and retrieves the information of the corresponding channel.<br/>
     * The result of the query includes information about the channel and a list of members participating in the channel.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void GetChannelInfo(const FString& channelId, const FHiveChatOnGetChannelInfoDelegate& Delegate);
    
    /**
     * \~korean
     * 채널 멤버 조회<br/>
     * 채널 ID를 입력받아, 해당 채널의 멤버 목록을 조회합니다.<br/>
     * 조회 결과로 해당 채널의 멤버 목록이 전달됩니다.<br/>
     *
     * \~english
     * Get channel members<br/>
     * Receives the channel ID and retrieves the list of members in the corresponding channel.<br/>
     * The result of the query includes a list of members in the corresponding channel.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
    */
    static void GetChannelMembers(const FString& channelId, const FHiveChatOnGetChannelMembersDelegate& Delegate);
    
    /**
     * \~korean
     * 채널 삭제<br/>
     * 채널 ID를 입력받아, 해당 채널을 삭제합니다.<br/>
     * 채널 삭제에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Delete channel<br/>
     * Receives the channel ID and deletes the corresponding channel.<br/>
     * The result of deleting the channel is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void DeleteChannel(const FString& channelId, const FHiveChatOnResultDelegate& Delegate);
    
    /**
     * \~korean
     * 채널 입장<br/>
     * 채널 ID를 입력받아, 채널에 접속합니다. <br/>
     * 공개(PUBLIC)채널이 아닌 경우, 비밀번호를 함께 입력해야합니다.<br/>
     * 채널 입장에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Enter channel<br/>
     * Receives the channel ID and connects to the channel.<br/>
     * If it is not a PUBLIC channel, you must enter the password.<br/>
     * The result of entering the channel is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void EnterChannel(const FHiveEnterChannelParams& params, const FHiveChatOnResultDelegate& Delegate);
    
    /**
     * \~korean
     * 채널 퇴장<br/>
     * 채널 ID를 입력받아, 해당 채널에서 퇴장합니다.<br/>
     * 채널 퇴장에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Exit channel<br/>
     * Receives the channel ID and exits the corresponding channel.<br/>
     * The result of exiting the channel is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void ExitChannel(const FString& channelId, const FHiveChatOnResultDelegate& Delegate);
    
    /**
     * \~korean
     * 참여중인 채널 목록 조회<br/>
     * 참여중인 채널 목록을 조회합니다.<br/>
     * 채널 목록 조회에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Get a list of channels the user is participating in.<br/>
     * Retrieves a list of channels the user is participating in.<br/>
     * The result of the channel list retrieval is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void GetChannelsByUser(const FHiveChatOnGetChannelsByUserDelegate& Delegate);
    
    /**
     * \~korean
     * 차단된 사용자 목록 조회<br/>
     * 차단된 사용자 목록을 조회합니다.<br/>
     * 차단된 사용자 목록에는 차단된 사용자의 ID와 차단시간이 포함됩니다. <br/>
     * 차단된 사용자 목록 조회에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Get a list of blocked users.<br/>
     * Retrieves a list of blocked users.<br/>
     * The blocked user list includes the ID of the blocked user and the time of blocking.<br/>
     * The result of the blocked user list retrieval is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void GetBlockMembers(const FHiveChatOnGetBlockMembersDelegate& Delegate);
    
    /**
     * \~korean
     * 사용자 차단<br/>
     * 사용자 ID를 입력받아, 해당 사용자를 차단합니다.<br/>
     * 사용자 차단에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Block user<br/>
     * Receives the user ID and blocks the corresponding user.<br/>
     * The result of blocking the user is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void BlockMember(int64 blockPlayerId, const FHiveChatOnResultDelegate& Delegate);
    
    /**
     * \~korean
     * 사용자 차단 해제<br/>
     * 사용자 ID를 입력받아, 해당 사용자의 차단을 해제합니다.<br/>
     * 사용자 차단 해제에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Unblock user<br/>
     * Receives the user ID and unblocks the corresponding user.<br/>
     * The result of unblocking the user is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void UnblockMember(int64 blockPlayerId, const FHiveChatOnResultDelegate& Delegate);
    
    /**
     * \~korean
     * 채팅서버에 연결<br/>
     * 채팅서버에  연결을 시도합니다.<br/>
     * 연결에 대한 결과는 listener를 통해 전달됩니다.<br/>
     * @deprecated 이 함수는 더 이상 사용되지 않으며, 향후 버전에서 제거될 예정입니다.
     *
     * \~english
     * Connects the Chat Server<br/>
     * Attempts to connect to the Chat Server.<br/>
     * The result of the connection is delivered through the listener.<br/>
     * @deprecated This function is deprecated and will be removed in a future version.
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void Connect(const FString& extraData, const FHiveChatOnResultDelegate& Delegate);

    /**
     * \~korean
     * 채팅서버에 연결<br/>
     * 채팅서버에  연결을 시도합니다.<br/>
     * 연결에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Connects the Chat Server<br/>
     * Attempts to connect to the Chat Server.<br/>
     * The result of the connection is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void Connect(const FHiveChatOnResultDelegate& Delegate);

    /**
     * \~korean
     * 채팅서버 재연결<br/>
     * 채팅서버와의 연결을 재시도합니다.<br/>
     * 재연결에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Reconnects the Chat Server<br/>
     * Attempts to reconnect to the Chat Server.<br/>
     * The result of the reconnection is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void Reconnect(const FHiveChatOnReconnectDelegate& Delegate);
    
    /**
     * \~korean
     * 채팅서버 연결 해제<br/>
     * 채팅서버와의 연결을 해제합니다.<br/>
     * 연결 해제에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Disconnects the Chat Server<br/>
     * Disconnects from the Chat Server.<br/>
     * The result of the disconnection is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void Disconnect(const FHiveChatOnResultDelegate& Delegate);

    /**
     * \~korean
     * 채널 메시지 전송<br/>
     * 채널 ID와 메시지를 입력받아, 해당 채널에 메시지를 전송합니다.<br/>
     * 메시지 전송에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Send a channel message<br/>
     * Receives the channel ID and message and sends the message to the corresponding channel.<br/>
     * The result of sending the message is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void SendMessageWithChannelSendMessageParams(FHiveChannelSendMessageParams const & params, const FHiveChatOnResultDelegate& Delegate);
    
    /**
     * \~korean
     * 1:1 메시지 전송<br/>
     * 수신자 ID와 메시지를 입력받아, 해당 사용자에게 메시지를 전송합니다.<br/>
     * 메시지 전송에 대한 결과는 listener를 통해 전달됩니다.<br/>
     *
     * \~english
     * Send a 1:1 message<br/>
     * Receives the recipient ID and message and sends the message to the corresponding user.<br/>
     * The result of sending the message is delivered through the listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void SendMessageWithDirectSendMessageParams(FHiveDirectSendMessageParams const & params, const FHiveChatOnResultDelegate& Delegate);
    
    /**
     * \~korean
     * 채널 메시지 전송 (콜백 없음)<br/>
     * 채널 ID와 메시지를 입력받아, 해당 채널에 메시지를 전송합니다.<br/>
     * 전달된 채널 메시지를 수신하기 위해 AddChannelListener 이벤트 핸들러를 등록해야 합니다.<br/>
     *
     * \~english
     * Send a channel message (no callback)<br/>
     * Receives the channel ID and message and sends the message to the corresponding channel.<br/>
     * To receive the sent channel message, you need to register an event handler using AddChannelListener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void SendMessageWithChannelSendMessageParams(FHiveChannelSendMessageParams const & params);
    
    /**
     * \~korean
     * 1:1 메시지 전송 (콜백 없음)<br/>
     * 수신자 ID와 메시지를 입력받아, 해당 사용자에게 메시지를 전송합니다.<br/>
     * 전달된 1:1 메시지를 수신하기 위해 AddDirectMessageListener 이벤트 핸들러를 등록해야 합니다.<br/>
     *
     * \~english
     * Send a 1:1 message (no callback)<br/>
     * Receives the recipient ID and message and sends the message to the corresponding user.<br/>
     * To receive the sent 1:1 message, you need to register an event handler using AddDirectMessageListener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void SendMessageWithDirectSendMessageParams(FHiveDirectSendMessageParams const & params);
    
    /**
     * \~korean
     * 메시지 번역 요청<br/>
     * targetLanguage와 번역할 메시지를 입력받아, 해당 메시지를 번역합니다.<br/>
     * 하나의 메시지를 여러 언어로 번역할 수 있습니다.<br/>
     *
     * \~english
     * Message translation request.<br/>
     * Takes the target language and the message to be translated as input, and translates the message accordingly.<br/>
     * A single message can be translated into multiple languages.<br/>
     * \~
     *
     * @since 4.25.4.0
     * @ingroup FHiveChat
     */
    static void Translate(FHiveTranslateParams const & params, const FHiveChatOnTranslateDelegate& Delegate);
    
    /**
     * \~korean
     * 채널 리스너 등록<br/>
     * 채널 이벤트를 수신하기 위한 리스너를 등록합니다.<br/>
     * 등록된 리스너는 채널 이벤트(메시지, 공지메시지, 채널 접속 알림, 채널 퇴장 알림, 채널 삭제 알림)가 발생할 때마다 이벤트를 수신합니다.<br/>
     *
     * \~english
     * Register Channel Listener<br/>
     * Register a listener to receive channel events.<br/>
     * Registered listeners receive events (messages, notice messages, channel access notifications, channel exit notifications, channel deletion notifications) whenever they occur.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void AddChannelListener(const FString& uniqueKey, FHiveChannelListener* listener);
    
    /**
     * \~korean
     * 1:1 메시지 리스너 등록<br/>
     * 1:1 메시지를 수신하기 위한 리스너를 등록합니다.<br/>
     * 등록된 리스너는 1:1 메시지가 발생할 때마다 이벤트를 수신합니다.<br/>
     *
     * \~english
     * Register Direct Message Listener<br/>
     * Register a listener to receive 1:1 messages.<br/>
     * Registered listeners receive events whenever a 1:1 message occurs.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void AddDirectMessageListener(const FString& uniqueKey, FHiveDirectMessageListener* listener);
    
    /**
     * \~korean
     * 연결 리스너 등록<br/>
     * 채팅서버 연결 상태를 수신하기 위한 리스너를 등록합니다.<br/>
     * 등록된 리스너는 채팅서버 연결 상태가 변경될 때마다 이벤트를 수신합니다.<br/>
     *
     * \~english
     * Register Connection Listener<br/>
     * Register a listener to receive the connection status of the Chat Server.<br/>
     * Registered listeners receive events whenever the connection status of the Chat Server changes.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void AddConnectionListener(const FString& uniqueKey, FHiveConnectionListener* listener);
    
    /**
     * \~korean
     * 유저 리스너 등록<br/>
     * 유저 이벤트를 수신하기 위한 리스너를 등록합니다.<br/>
     * 등록된 리스너는 1:1 메시지, 공지메시지 이벤트가 발생할 때마다 이벤트를 수신합니다.<br/>
     *
     * \~english
     * Register User Listener<br/>
     * Register a listener to receive user events.<br/>
     * Registered listeners receive events (1:1 messages, notice messages) whenever they occur.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void AddUserListener(const FString& uniqueKey, FHiveUserListener* listener);
    
    /**
     * \~korean
     * 커스텀 데이터 리스너 등록<br/>
     * 커스텀 데이터를 수신하기 위한 리스너를 등록합니다.<br/>
     * 등록된 리스너는 커스텀 데이터가 발생할 때마다 이벤트를 수신합니다.<br/>
     *
     * \~english
     * Register Custom Data Listener<br/>
     * Register a listener to receive custom data.<br/>
     * Registered listeners receive events whenever custom data occurs.<br/>
     * \~
     *
     * @since 4.25.4.0
     * @ingroup FHiveChat
     */
    static void AddCustomDataListener(const FString& uniqueKey, FHiveCustomDataListener* listener);

    /**
     * \~korean
     * 채널 리스너 제거<br/>
     * 등록된 채널 리스너를 제거합니다.<br/>
     *
     * \~english
     * Remove Channel Listener<br/>
     * Remove the registered channel listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void RemoveChannelListener(const FString& uniqueKey);

    /**
     * \~korean
     * 1:1 메시지 리스너 제거<br/>
     * 등록된 1:1 메시지 리스너를 제거합니다.<br/>
     *
     * \~english
     * Remove Direct Message Listener<br/>
     * Remove the registered 1:1 message listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void RemoveDirectMessageListener(const FString& uniqueKey);

    /**
     * \~korean
     * 연결 리스너 제거<br/>
     * 등록된 연결 리스너를 제거합니다.<br/>
     *
     * \~english
     * Remove Connection Listener<br/>
     * Remove the registered connection listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void RemoveConnectionListener(const FString& uniqueKey);
    
    /**
     * \~korean
     * 유저 리스너 제거<br/>
     * 등록된 유저 이벤트 리스너를 제거합니다.<br/>
     *
     * \~english
     * Remove User Listener<br/>
     * Remove the registered user event listener.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static void RemoveUserListener(const FString& uniqueKey);
    
    /**
     * \~korean
     * 커스텀 데이터 리스너 제거<br/>
     * 등록된 커스텀 데이터 리스너를 제거합니다.<br/>
     *
     * \~english
     * Remove Custom Data Listener<br/>
     * Remove the registered custom data listener.<br/>
     * \~
     *
     * @since 4.25.4.0
     * @ingroup FHiveChat
     */
    static void RemoveCustomDataListener(const FString& uniqueKey);

    /**
     * \~korean
     * 소켓 연결 상태 확인<br/>
     * 현재 소켓이 연결되어 있는지 확인합니다.<br/>
     *
     * \~english
     * Check the socket connection status<br/>
     * Check if the socket is currently connected.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveChat
     */
    static bool IsConnected();
};
