/**
 * @file        HiveDataStore.h
 *
 * @date        2025
 * @copyright   Copyright © 2025 Com2uS Platform Corp. All Rights Reserved.
 * @defgroup FHiveDataStore
 * @{
 * \~korean
 * Hive 데이터 스토어는 NoSQL 클라우드 데이터베이스를 사용하여 게임 구성 요소나 설정 정보 등 <br/>
 * 클라이언트에 사용되는 데이터를 저장하고 동기화할 수 있습니다.<br/><br/>
 * \~english
 * Hive Data Store uses NoSQL Cloud Database to synchronize and store the data for use on the client, <br/>
 * such as the components of the game and the settings information.<br/><br/>
 * \~
 */
#pragma once

#include "CoreMinimal.h"
#include "HiveInterface.h"
#include "Impl/HiveDataStoreImpl.h"

/**
 *  \~korean
 * Hive 데이터 스토어는 NoSQL 클라우드 데이터베이스를 사용하여 게임 구성 요소나 설정 정보 등 <br/>
 * 클라이언트에 사용되는 데이터를 저장하고 동기화할 수 있습니다. 배포 및 유지관리할 서버가 필요없는 클라이언트 기반의 게임이나 <br/>
 * 클라이언트 간 실시간 업데이트가 필요한 경우 좋은 솔루션이 될 수 있습니다.<br/>
 *
 * Hive 데이터 스토어의 특징은 다음과 같습니다:<br/>
 * - 데이터가 각 게임의 개별 저장소에 저장됩니다.<br/>
 * - 모든 데이터는 암호화되어 안전하게 전송됩니다.<br/>
 * - SQL 데이터베이스와 달리 테이블이나 행이 없으며 컬렉션으로 구성된 문서에 데이터를 저장합니다.<br/>
 * - 데이터는 키-값 쌍으로 문서화해 저장할 수 있습니다.<br/>
 * - 다수의 작은 문서가 모인 컬렉션을 저장하는 데에 최적화되어 있습니다.<br/>
 *
 *  \~english
 * Hive Data Store uses NoSQL Cloud Database to synchronize and store the data for use on the client, <br/>
 * such as the components of the game and the settings information. Using this service is a good <br/>
 * solution for the games based on the client or if it needs real-time updates between clients.<br/>
 *
 * Hive Data Store has the following features:<br/>
 * - The data is stored in the storage separately of each game.<br/>
 * - All data is encrypted and delivered safely.<br/>
 * - Unlike SQL database, NoSQL Cloud Database has no tables and rows, and the data is stored in documents consisting of collections.<br/>
 * - Stores and documented data as key-value pairs.<br/>
 * - Optimizes for storing the collection that consists of multiple documents.<br/>
 *  \~
 *
 * @since   4.15.8
 * @ingroup FHiveDataStore
 */
class HIVESDK_API FHiveDataStore : public IHiveInterface<IHiveDataStoreImpl>
{
public:
    /**
     *  \~korean
     * 데이터 추가<br/>
     *
     * 데이터는 하나의 키-값 쌍으로 추가하거나 한꺼번에 여러 쌍을 Map 형태로 추가할 수 있습니다.<br/>
     * 요청 후 성공 또는 실패 결과를 받을 수 있습니다.<br/>
     * - 중요 : key에 해당하는 value는 단순한 문자열(String) 혹은 백슬래시(\)가 포함된 1차원의 JSON 문자열로 설정해야 합니다.<br/>
     *
     *  \~english
     * Add Data<br/>
     *
     * You can add data to the Data Store in one pair of a key-value or multiple pairs of map format at once.<br/>
     * After a request, true or false is returned.<br/>
     * - IMPORTANT : The key-value needs to be set up in a one-dimensional JSON string that includes a simple string or backslash(\).<br/>
     *  \~
     *
     * @since   4.15.8
     * @ingroup FHiveDataStore
     */
    static void Set(const FString& Key, const FString& Value, const FHiveDataStoreOnSetDelegate& Delegate);

    /**
     *  \~korean
     * 데이터 추가<br/>
     *
     * 데이터는 하나의 키-값 쌍으로 추가하거나 한꺼번에 여러 쌍을 Map 형태로 추가할 수 있습니다.<br/>
     * 요청 후 성공 또는 실패 결과를 받을 수 있습니다.<br/>
     * - 중요 : key에 해당하는 value는 단순한 문자열(String) 혹은 백슬래시(\)가 포함된 1차원의 JSON 문자열로 설정해야 합니다.<br/>
     *
     *  \~english
     * Add Data<br/>
     *
     * You can add data to the Data Store in one pair of a key-value or multiple pairs of map format at once.<br/>
     * After a request, true or false is returned.<br/>
     * - IMPORTANT : The key-value needs to be set up in a one-dimensional JSON string that includes a simple string or backslash(\).<br/>
     *  \~
     *
     * @since   4.15.8
     * @ingroup FHiveDataStore
     */
    static void Set(const DataStoreData& Data, const FHiveDataStoreOnSetDelegate& Delegate);

    /**
     *  \~korean
     * 내 데이터 하나 가져오기<br/>
     *
     *  \~english
     * Get one of my data<br/>
     *  \~
     *
     * @since   4.15.8
     * @ingroup FHiveDataStore
     */
    static void Get(const FString& Key, const FHiveDataStoreOnGetDelegate& Delegate);

    /**
     *  \~korean
     * 내 데이터 모두 가져오기<br/>
     *
     *  \~english
     * Get all my data<br/>
     *  \~
     *
     * @since   4.15.8
     * @ingroup FHiveDataStore
     */
    static void GetMyData(const FHiveDataStoreOnMyDataDelegate& Delegate);

    /**
     *  \~korean
     * 키로 전체 데이터 가져오기<br/>
     *
     *  \~english
     * Get all data using key<br/>
     *  \~
     *
     * @since   4.15.8
     * @ingroup FHiveDataStore
     */
    static void GetUsersData(const FString& Key, const FHiveDataStoreOnUsersDataDelegate& Delegate);
};