/**
 * @file        HiveIAPV4.h
 *
 * @date        2025
 * @copyright   Copyright © 2025 Com2uS Platform Corp. All Rights Reserved.
 * @defgroup FHiveIAPV4
 * @{
 * \~korean
 * Hive IAP 가 제공하는 결제 마켓은 애플 앱스토어, 구글 플레이스토어, 원스토어, 삼성 갤럭시스토어, 아마존 앱스토어,<br/> 
 * 화웨이 앱커넥트, PG, Steam, Hive 러비(Lebi) 를 제공한다.<br/><br/>
 *
 * \~english
 * Hive IAP supports Apple AppStore, Google PlayStore, ONEStore, Samsung GalaxyStore,<br/>
 * Amazon AppStore, Huawei AppConnect, PG, Steam and Hive Lebi.<br/><br/>
 *
 * \~
 */
#pragma once

#include "CoreMinimal.h"
#include "HiveInterface.h"
#include "Impl/HiveIAPV4Impl.h"

/**
 * \~korean
 * 현재 Hive IAP 가 제공하는 결제 마켓은 애플 앱스토어, 구글 플레이스토어, 원스토어, 삼성 갤럭시스토어, 아마존 앱스토어,
 * 페이스북 클라우드 게임, 화웨이 앱커넥트, PG, Steam, Hive 러비(Lebi) 를 제공한다.<br/>
 * Hive 러비(Lebi) 상점은 중국 시장에서 상점을 구성하고 상품을 구매 할 수 있는 시스템으로서 Hive 에서 자체적으로 운영하고 있다.<br/><br/>
 *
 * \~english
 * Currently, Hive IAP supports Apple AppStore, Google PlayStore, ONEStore, Samsung GalaxyStore,
 * Amazon AppStore, Facebook Cloud Games, Huawei AppConnect, PG, Steam and Hive Lebi.<br/>
 * Hive Lebi store is a system that can organize stores and purchase goods in Chinese market.
 * It is operated by Hive itself.<br/><br/>
 * 
 * \~
 * @ingroup FHiveIAPV4
 */
class HIVESDK_API FHiveIAPV4 : public IHiveInterface<IHiveIAPV4Impl>
{
public:
    /**
     *  \~korean
     * @brief IAPV4 결재 API 초기화 요청<br/>
     * Hive IAPV4 의 초기화를 수행하면 결과 통지시 사용 가능한 마켓 정보의 목록을 전달하게 된다.<br/>
     * 이 마켓 정보의 목록은 게임 클라이언트에서 사용 가능한 마켓을 의미한다.
     *
     * @param Delegate      API 결과 통지
     *
     *  \~english
     * @brief IAPV4 It initialize IAPV4 Purchase module.<br/>
     * When Hive IAPV4 is initialized, it sends a list of available market information in the result callback.<br/>
     * This list of market information means the market available on the game client.
     *
     * @param Delegate      Result of API call
     *
     *  \~
     * @ingroup FHiveIAPV4
     */
    static void MarketConnect(const FHiveIAPV4OnMarketConnectDelegate& Delegate);

    /**
     *  \~korean
     * @brief Apple Appstore 에서 상품의 구성을 위한 정보를 조회할 수 있다. <br/>
     *
     * @param Delegate          API 결과 통지
     *
     *  \~english
     * @brief Apple Appstore It returns information for composition of the product.<br/>
     *
     * @param Delegate          Result of API call
     *
     *  \~
     * @ingroup FHiveIAPV4
     */
    static void GetMarketProductInfo(const TArray<FString>& MarketPidList, const FHiveIAPV4OnProductInfoDelegate& Delegate);
    
    /**
     *  \~korean
     * @brief Hive IAPV4 에서 상품의 구성을 위한 정보를 조회할 수 있다. <br/>
     *
     * @param Delegate          API 결과 통지
     *
     *  \~english
     * @brief Hive IAPV4 It returns information for composition of the product.<br/>
     *
     * @param Delegate          Result of API call
     *
     *  \~
     * @ingroup FHiveIAPV4
     */
    static void GetProductInfo(const FHiveIAPV4OnProductInfoDelegate& Delegate);

    /**
     *  \~korean
     * @brief Hive IAPV4 에서 구독형 상품의 구성을 위한 정보를 조회할 수 있다. <br/>
     *
     * @param Delegate           API 결과 통지
     *
     *  \~english
     * @brief Hive IAPV4 It returns information for composition of the product.<br/>
     *
     * @param Delegate           Result of API call
     *
     *  \~
     * @ingroup FHiveIAPV4
     */
    static void GetSubscriptionProductInfo(const FHiveIAPV4OnProductInfoDelegate& Delegate);

    /**
     *  \~korean
     * @brief App Store, Google Store 등 에 등록된 마켓 상품(marketPid)의 구매를 요청 한다.<br/>
     * iapPayload 의 경우 구매 요청에 대한 보충 정보로 구매 후 영수증 검증에서 확인할 수 있다.<br/>
     * 프로모 코드 구매 처럼 앱 외부에서 구매 등의 경우에는 포함될 수 없게 된다.
     *
     * @param MarketPid         App Store, Google Store 등 에 등록된 하나의 상품을 지칭하는 고유 코드
     * @param IapPayload        구매 요청에 대한 보충 정보를 포함한 문자열로, 개발자가 지정합니다.
     * @param Delegate          API 결과 통지
     *
     *  \~english
     * @brief Request purchase of market product (marketPid) registered in App Store, Google Store, etc.<br/>
     * iapPayload is supplementary information for the purchase request, and it can be checked in Receipt Verification after purchase, <br/>
     * but, in case of purchasing outside of the app, such as a promo code purchase etc., it can not be included in verification result.
     *
     * @param MarketPid         Unique code registered on the App Store, Google Store or etc.
     * @param IapPayload        A string containing supplementary information about the purchase request, specified by the developer.
     * @param Delegate          Result of API call
     *
     *  \~
     * @ingroup FHiveIAPV4
     */
    static void Purchase(const FString& MarketPid, const FString& IapPayload, const FHiveIAPV4OnPurchaseDelegate& Delegate);
    
    /**
     *  \~korean
     * @brief App Store, Google Store 등 에 등록된 마켓 상품(marketPid)의 구매를 요청 한다.<br/>
     * iapPayload 의 경우 구매 요청에 대한 보충 정보로 구매 후 영수증 검증에서 확인할 수 있다.<br/>
     * 프로모 코드 구매 처럼 앱 외부에서 구매 등의 경우에는 포함될 수 없게 된다.
     *
     * @param MarketPid         App Store, Google Store 등 에 등록된 하나의 상품을 지칭하는 고유 코드
     * @param IapPayload        구매 요청에 대한 보충 정보를 포함한 문자열로, 개발자가 지정합니다.
     * @param quantity          결제 수량
     * @param Delegate          API 결과 통지
     *
     *  \~english
     * @brief Request purchase of market product (marketPid) registered in App Store, Google Store, etc.<br/>
     * iapPayload is supplementary information for the purchase request, and it can be checked in Receipt Verification after purchase, <br/>
     * but, in case of purchasing outside of the app, such as a promo code purchase etc., it can not be included in verification result.
     *
     * @param MarketPid         Unique code registered on the App Store, Google Store or etc.
     * @param IapPayload        A string containing supplementary information about the purchase request, specified by the developer.
     * @param quantity          payment quantity
     * @param Delegate          Result of API call
     *
     *  \~
     * @ingroup FHiveIAPV4
     */
    static void Purchase(const FString& MarketPid, const FString& IapPayload, int quantity, const FHiveIAPV4OnPurchaseDelegate& Delegate);
    
    /**
     *  \~korean
     * @brief App Store, Google Store 등 에 등록된 마켓 상품(marketPid)의 구매를 요청 한다.<br/>
     * iapPayload 의 경우 구매 요청에 대한 보충 정보로 구매 후 영수증 검증에서 확인할 수 있다.<br/>
     * 프로모 코드 구매 처럼 앱 외부에서 구매 등의 경우에는 포함될 수 없게 된다.
     * 구독형 상품을 업그레이드하거나 다운그레이드를 수행하는 경우 기존에 구독 중인 상품의 고유코드 oldMarketPid를 필수로 넣어야 한다.
     *
     * @param Param             구매시 필요한 parameter 의 모음
     * @param Delegate          API 결과 통지
     *
     *  \~english
     * @brief Request purchase of market product (marketPid) registered in App Store, Google Store, etc.<br/>
     * iapPayload is supplementary information for the purchase request, and it can be checked in Receipt Verification after purchase, <br/>
     * but, in case of purchasing outside of the app, such as a promo code purchase etc., it can not be included in verification result.
     * When upgrading or downgrading a subscription-type product, the unique code oldMarketPid of the existing subscription product must be entered.
     *
     * @param Param             A collection of parameters required for purchase.
     * @param Delegate          Result of API call
     *
     *  \~
     * @ingroup FHiveIAPV4
     */
    static void PurchaseSubscription(const FHiveIAPV4PurchaseParam& Param, const FHiveIAPV4OnPurchaseDelegate& Delegate);
    
    /**
     *  \~korean
     * @brief Hive IAPV4 상품에 대한 미지급된 아이템 지급을 요청한다.<br/>
     * 상품 구매 과정에서 여러가지 상황으로 인하여 아이템 지급을 실패하는 경우가 발생할 수 있다.<br/>
     * 이런 상황을 대비하기 위해서 미 지급된 아이템 지급을 요청 해야 한다.<br/>
     *
     * @param Delegate            API 결과 통지
     *
     *  \~english
     * @brief Hive IAPV4  It restore the item that has not been delivered.<br/>
     * It may happen that the item delivery fails due to various situations in the product purchase process.<br/>
     * In order to prepare for this situation, you should request to deliver the item that has not been delivered.
     *
     * @param Delegate            API 결과 통지
     *
     *  \~
     * @ingroup FHiveIAPV4
     */
    static void Restore(const FHiveIAPV4OnRestoreDelegate& Delegate);

    /**
     *  \~korean
     * @brief HIVE IAPV4 상품에 대한 미지급된 구독 상품 지급을 요청한다.<br/>
     * 상품 구매 과정에서 여러가지 상황으로 인하여 구독 상태를 갱신하지 못할 수 있다.<br/>
     * 구독 상태를 확인할 수 없는 경우, 구독 상태 복구를 요청 해야 한다.<br/>
     *
     * @param Delegate            API 결과 통지
     *
     *  \~english
     * @brief HIVE IAPV4  It restore the item that has not been delivered.<br/>
     *
     * @param Delegate            Result of API call
     *
     *  \~
     * @ingroup FHiveIAPV4
     */
    static void RestoreSubscription(const FHiveIAPV4OnRestoreDelegate& Delegate);

    /**
     *  \~korean
     * @brief HIVE IAP v4에서 지급이 완료된 상품에 대한 구매 영수증에 대한 트랜잭션 완료 처리를 요청한다.<br/>
     * 구매나 복구 기능 사용 후 아이템 지급이 완료된 시점에 이 API를 호출하여 마켓에 구매 완료 처리를 해야한다.<br/>
     *
     * @warning 만약, 지급 완료 후에도 마켓으로의 구매 완료 처리를 하지 않으면, 미지급된 상품으로 남는다.<br/>
     * 미지급된 상품은 transactionFinish API 호출 전까지 계속 복구가 되므로, 영수증 중복 여부를 확인해야 한다.<br/>
     *
     * @param MarketPid    지급이 완료되어 트랜잭션 완료 처리를 위한 상품의 고유 ID
     * @param Delegate    API 결과 통지
     *
     *  \~english
     * @brief HIVE IAP v4 It requests the transaction completion processing for the purchase receipt of the item that have been delivered.<br/>
     * At the completion of item delivery after purchasing or restoring, this API should be called to complete the purchase on the market.<br/>
     *
     * @warning If you do not notify the completion of the purchase to the market after payment has been made, it will be left as an unpaid item..<br/>
     * Since the unpaid items will continue to be recovered until the transactionFinish API call, you should check for duplicate receipts.<br/>
     *
     * @param MarketPid    The unique ID of the item for which the transaction is to be completed.
     * @param Delegate    Result of API call
     *
     *  \~
     *
     * @ingroup FHiveIAPV4
     */
    static void TransactionFinish(const FString& MarketPid, const FHiveIAPV4OnTransactionFinishDelegate& Delegate);
    
    /**
     *  \~korean
     * @brief Hive IAP v4에서 지급이 완료된 상품에 대한 구매 영수증에 대한 복수 개의 트랜잭션 완료 처리를 요청한다.<br/>
     * 구매나 복구 기능 사용 후 아이템 지급이 완료된 시점에 이 API를 호출하여 마켓에 구매 완료 처리를 해야한다.<br/>
     *
     * @warning 만약, 지급 완료 후에도 마켓으로의 구매 완료 처리를 하지 않으면, 미지급된 상품으로 남는다.<br/>
     * 미지급된 상품은 transactionFinish API 호출 전까지 계속 복구가 되므로, 영수증 중복 여부를 확인해야 한다.<br/>
     *
     * @param MarketPidList    지급이 완료되어 트랜잭션 완료 처리를 위한 상품의 고유 ID 목록
     * @param Delegate            API 결과 통지
     *
     *  \~english
     * @brief Hive IAP v4 It requests the multiple transaction completion processing for the purchase receipt of the item that have been delivered.<br/>
     * At the completion of item delivery after purchasing or restoring, this API should be called to complete the purchase on the market.<br/>
     *
     * @warning If you do not notify the completion of the purchase to the market after payment has been made, it will be left as an unpaid item..<br/>
     * Since the unpaid items will continue to be recovered until the transactionFinish API call, you should check for duplicate receipts.<br/>
     *
     * @param MarketPidList    The list of unique ID of the item for which the transaction is to be completed.
     * @param Delegate    Result of API call
     *
     *  \~
     *
     * @ingroup FHiveIAPV4
     */
    static void TransactionMultiFinish(const TArray<FString>& MarketPidList, const FHiveIAPV4OnTransactionMultiFinishDelegate& Delegate);
    
    /**
     *  \~korean
     * @brief 러비 상점이나 구글 상점을 선택하기 위한 창을 띄운다.<br/>
     * 만약 Google Store 와 HIVE 러비(Lebi) 상점을 동시에 사용 가능한 상황이라면 유저가 상점 선택을 할 수 있도록 상점 선택 화면을 노출해야 한다
     *
     * @param Delegate            API 결과 통지
     *
     *  \~english
     * @brief It shows a UI to select a Lebi store or a Google store..<br/>
     * If the Google Store and the HIVE Lebi store are available at the same time, you should expose the store selection screen to allow the user to select a store.
     *
     * @param Delegate            Result of API call
     *
     *  \~
     * @ingroup FHiveIAPV4
     */
    static void ShowMarketSelection(const FIAPV4OnMarketSelectionDelegate& Delegate);

    /**
     *  \~korean
     * @brief 러비 상점일 경우 잔액 정보 조회.
     *
     * @param Delegate            API 결과 통지
     *
     *  \~english
     * @brief It request Lebi balance information.
     *
     * @param Delegate            Result of API call
     *
     *  \~
     * @ingroup FHiveIAPV4
     */
    static void GetBalanceInfo(const FHiveIAPV4OnBalanceDelegate& Delegate);

    /**
     *  \~korean
     * @brief 러비 충전 페이지 노출
     *
     * @param Delegate            API 결과 통지
     *
     *  \~english
     * @brief It shows Lebi refill page.
     *
     * @param Delegate            Result of API call
     *
     *  \~
     * @ingroup FHiveIAPV4
     */
    static void ShowCharge(const FHiveIAPV4OnBalanceDelegate& Delegate);

    /**
     @brief Apple AppStore Promote InApp Purchase 동작으로 얻어온 MarketPID를 요청한다.<br/>
     앱의 시작, 앱의 Background -> Foreground시에 이 API를 호출하여 MarketPID를 얻어오도록 한다.<br/>

     얻어온 MarketPID로 구매를 진행하도록 한다.<br/>

     @warning iOS 11 이상에서만 지원한다.<br/>

     @param Delegate API 호출 결과 통지
     */
    static void CheckPromotePurchase(const FHiveIAPV4OnCheckPromotePurchaseDelegate& Delegate);

    /**
     * @brief 현재 선택된 마켓 타입을 반환한다. <br/>
     *
     * 얻어온 마켓 타입으로 구매를 진행하도록 한다.<br/>
     *
     *
     * @ingroup FHiveIAPV4
     */
    static TOptional<EHiveIAPV4Type> GetSelectedMarket();

    /**
     * @brief 현재 계정의 UUIDv3 값을 반환한다. <br/>
     *
     * Returns the Hive account ID converted to a UUID version 3. return null, if not logged in.
     *
     * @ingroup FHiveIAPV4
     */
    static FString GetAccountUuid();

    /**
     * Google In-App Offers
     *
     * @param categoryId 3(PROMOTIONAL)
     *
     * @param listener Int(responseCode) 가 2(NAVIGATE_TO_DEV_APP_STORE) 일 경우 유저가 제안(Offer)를 확인하고 사용하기를 원한 상황입니다.
     * 개발자는 유저가 구매할 인앱 항목을 선택할 수 있는 게임/앱 내 상점으로 (유저가 해당 화면에 없는 경우) 유저를 이동해야 합니다.
     *
     * 0 은 아무런 액션을 하지 않아도 되고, 음수는 에러상황 입니다.
     *
     * \~
     * @ingroup FHiveIAPV4
     */
    static void ShowInAppMessages(int32 CategoryId, const FHiveIAPV4OnInAppMessagesDelegate& Delegate);

    /**
     * @brief 소비정보 전송 약관에 대한 유저의 마지막 동의 상태 및 시기를 반환한다. <br/>
     *
     * Returns the lastest consent status and timing for the Consumption Information Transmission Terms.
     *
     * @ingroup FHiveIAPV4
     */
    static void GetConsumeInfoAgreement(const FHiveIAPV4OnGetConsumeInfoAgreementDelegate& Delegate);

    /**
     * @brief 소비정보 전송 약관 팝업을 노출한다. <br/>
     *
     * Displays the Consumption Information Transmission Terms Popup.
     *
     * @ingroup FHiveIAPV4
     */
    static void ShowConsumeInfoAgreement(const FHiveIAPV4OnShowConsumeInfoAgreementDelegate& Delegate);

    /* @NOT_SUPPORT
     * - PurchaseSubscriptionUpdate()
     */
};
