/**
 * @file        HiveMatchMaking.h
 *
 * @date        2025
 * @copyright   Copyright © 2025 Com2uS Platform Corp. All Rights Reserved.
 * @defgroup FHiveMatchMaking
 * @{
 * \~korean
 * Hive MatchMaking은 온라인 매치를 위해 플레이어를 연결하고 관련 데이터를 제공합니다. <br/>
 * Hive MatchMaking 기능을 통해 게임에서 구체적인 매치메이킹 로직을 구현하지 않아도 플레이어의 대전 상대를 간단하게 얻을 수 있습니다. <br/><br/>
 *
 * \~english
 * Hive MatchMaking connects players for online matches and provides relevant data. <br/>
 * Hive MatchMaking feature makes it simple to get matches for your players without having to implement specific matchmaking logic in your game. <br/><br/>
 *
 * \~
 */
#pragma once

#include "CoreMinimal.h"
#include "HiveInterface.h"
#include "Impl/HiveMatchMakingImpl.h"

/**
 *  \~korean
 * Hive MatchMaking은 온라인 매치를 위해 플레이어를 연결하고 관련 데이터를 제공합니다. <br/>
 * Hive MatchMaking 기능을 통해 게임에서 구체적인 매치메이킹 로직을 구현하지 않아도 플레이어의 대전 상대를 간단하게 얻을 수 있습니다. <br/>
 * 플레이어가 비슷한 수준을 가진 상대와 매치되도록 하여 균형잡힌 게임 경험을 제공합니다. <br/>
 *
 *  \~english
 * Hive MatchMaking connects players for online matches and provides relevant data. <br/>
 * Hive MatchMaking feature makes it simple to get matches for your players without having to implement specific matchmaking logic in your game. <br/>
 * It provides a balanced gaming experience by matching players with opponents of similar skill level. <br/>
 *
 *  \~
 * @ingroup FHiveMatchMaking
 */
class HIVESDK_API FHiveMatchMaking : public IHiveInterface<IHiveMatchMakingImpl>
{
public:

    /**
     *  \~korean
     * 매칭 요청<br/>
     *
     * 매칭 요청 시 사용할 점수(point)를 입력할 수 있습니다.<br/>
     * 매칭에 사용할 부가 정보(닉네임, 레벨, 국가 등)(extraData)를 입력할 수 있습니다.<br/>
     * 부가 정보는 매칭 결과에 포함되어 전달됩니다.<br/>
     *
     *  \~english
     * Matching Request<br/>
     *
     * You can enter the score(point) to be used when requesting matching.<br/>
     * You can also enter additional information(nickname, level, country, etc.)(extraData) to be used for matching.<br/>
     * ExtraData is delivered as part of the matching results.<br/>
     *  \~
     *
     * @since   4.24.5.0
     * @ingroup FHiveMatchMaking
     */
    static void RequestMatchMaking(int32 MatchId, int32 Point, const FString& ExtraData, const FHiveMatchMakingOnMatchMakingDataDelegate& Delegate);
    
    /**
     *  \~korean
     * 매칭 상태 확인<br/>
     *
     * 요청한 매칭 상태를 확인할 수 있습니다. <br/>
     *
     *  \~english
     * Check Matching Status<br/>
     *
     * You can check the status of your requested matching.<br>
     *  \~
     *
     * @since   4.24.5.0
     * @ingroup FHiveMatchMaking
     */
    static void GetRequestingStatus(int32 MatchId, const FHiveMatchMakingOnMatchMakingDataDelegate& Delegate);
    
    /**
     *  \~korean
     * 매칭 요청 삭제<br/>
     *
     * 요청한 매칭을 삭제합니다.<br/>
     *
     *  \~english
     * Delete Matching Request<br/>
     *
     * Deletes the requested match.<br/>
     *  \~
     *
     * @since   4.24.5.0
     * @ingroup FHiveMatchMaking
     */
    static void DeleteRequesting(int32 MatchId, const FHiveMatchMakingOnResultDelegate& Delegate);
    
    /**
     * \~korean
     * 매칭 그룹 생성<br/>
     *
     * 매칭 그룹을 생성합니다.<br/>
     *
     *  \~english
     *  Create a matching group<br/>
     *
     *  Create a matching group.<br/>
     *  \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveMatchMaking
     */
    static void CreateGroup(int32 MatchId, int32 Point, const FString& ExtraData, const FHiveMatchMakingOnMatchMakingGroupDataDelegate& Delegate);
    
    /**
     * \~korean
     * 매칭 그룹 참여<br/>
     *
     * 매칭 그룹에 참여합니다.<br/>
     *
     *  \~english
     * Join a matching group<br/>
     *
     * Join a matching group.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveMatchMaking
     */
    static void JoinGroup(int32 MatchId, const FString& GroupCode, int32 Point, const FString& ExtraData, const FHiveMatchMakingOnMatchMakingGroupDataDelegate& Delegate);
    
    /**
     * \~korean
     * 매칭 그룹 탈퇴<br/>
     *
     * 매칭 그룹에서 탈퇴합니다.<br/>
     *
     *  \~english
     * Leave a matching group<br/>
     *
     * Leave a matching group.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveMatchMaking
     */
    static void LeaveGroup(int32 MatchId, const FHiveMatchMakingOnResultDelegate& Delegate);
    
    /**
     * \~korean
     * 그룹원 추방<br/>
     *
     * 매칭 그룹에서 그룹원을 추방시킵니다.<br/>
     *
     * \~english
     * Kick a group member<br/>
     *
     * Kick a group member<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveMatchMaking
     */
    static void KickGroupUser(int32 MatchId, int64 TargetPlayerId, const FHiveMatchMakingOnMatchMakingGroupDataDelegate& Delegate);
    
    /**
     * \~korean
     * 그룹 정보 조회 (그룹 내 유저 기준)<br/>
     *
     * 유저 정보를 기준으로 그룹 정보를 조회합니다.<br/>
     *
     * \~english
     * Get group information (based on group users)<br/>
     *
     * Get group information based on user information.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveMatchMaking
     */
    static void GetGroupInfoByUser(int32 MatchId, const FHiveMatchMakingOnMatchMakingGroupDataDelegate& Delegate);
    
    /**
     * \~korean
     * 그룹 정보 조회 (그룹 코드 기준)<br/>
     *
     * 그룹 코드를 기준으로 그룹 정보를 조회합니다.<br/>
     *
     * \~english
     * Get group information (based on group code)<br/>
     *
     * Get group information based on group code.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveMatchMaking
     */
    static void GetGroupInfoByGroupCode(const FString& GroupCode, const FHiveMatchMakingOnMatchMakingGroupDataDelegate& Delegate);
    
    /**
     * \~korean
     * 멤버 정보 수정<br/>
     *
     *그룹에 속한 구성원이 자신의 정보를 변경시킵니다.<br/>
     *
     * \~english
     * Modify member information<br/>
     *
     * Group members change their information.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveMatchMaking
     */
    static void UpdateGroupUser(int32 MatchId, bool Ready, int32 Point, const FString& ExtraData, const FHiveMatchMakingOnMatchMakingGroupDataDelegate& Delegate);
    
    /**
     * \~korean
     * 그룹 매칭 요청<br/>
     *
     * 그룹 매칭을 요청합니다.<br/>
     *
     * \~english
     * Request group matching<br/>
     *
     * Request group matching.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveMatchMaking
     */
    static void RequestGroupMatching(int32 MatchId, const FHiveMatchMakingOnMatchMakingGroupDataDelegate& Delegate);
    
    /**
     * \~korean
     * 그룹 매칭 요청 취소<br/>
     *
     * 그룹 매칭 요청을 취소합니다.<br/>
     *
     * \~english
     * Cancel group matching request<br/>
     *
     * Cancel group matching request.<br/>
     * \~
     *
     * @since 4.25.0.0
     * @ingroup FHiveMatchMaking
     */
    static void DeleteGroupMatching(int32 MatchId, const FHiveMatchMakingOnMatchMakingGroupDataDelegate& Delegate);
};
