/**
 * @file        HivePlatformHelper.h
 *
 * @date        2025
 * @copyright   Copyright © 2025 Com2uS Platform Corp. All Rights Reserved.
 * @defgroup FHivePlatformHelper
 * @{
 * \~korean
 * 플랫폼 사용 편의를 위한 기능 제공 <br/><br/>
 * \~english
 * Provides features for ease of platform use <br/><br/>
 * \~
 */
#pragma once

#include "CoreMinimal.h"
#include "HiveInterface.h"
#include "Impl/HivePlatformHelperImpl.h"

/**
 * 플랫폼에서 지원하는 기능을 제공하는 클래스이다.
 * @ingroup FHivePlatformHelper
 */
class HIVESDK_API FHivePlatformHelper : public IHiveInterface<IHivePlatformHelperImpl>
{
public:
    /**
     * @brief Android에서 사용자에게 OS 권한을 재요청.
     *
     * @param Requests Android에서 요청할 권한들을 포함한 리스트.
     *
     * \~
     * @ingroup FHivePlatformHelper
     */
    static void RequestUserPermissions(const TArray<FString>& Requests, const FHivePlatformHelperOnUserPermissionsDelegate& Delegate);
    
    /**
     * @brief 업데이트 팝업 설정으로 백그라운드에서 앱 다운로드가 완료되면
     *
     * UE에서 Promotion.EngagementEventType.APPUPDATE_DOWNLOADED 로 신호를 보낸다.
     *
     * 이후 completeUpdate() 를 호출하면 completeState 값에 따라 새 버전으로 설치한다.
     *
     * 호출 하지 않으면 기본 동작으로 재시작시 설치를 진행한다.
     *
     * @param state 1 ImmediateUpdate, 2 RestartUpdate, 3 LaterUpdate. otherwise 1.
     * 
     * \~
     * @ingroup FHivePlatformHelper
     */
    static void CompleteUpdate(int32 state);

    /**
     * @brief 인 앱 브라우저를 오픈한다.
     *
     * @param param [InAppBrowserParam]
     * @param handler [onShowInAppBrowserHandler]
     *
     * @since 24.0.0
     * 
     * \~
     * @ingroup FHivePlatformHelper
     */
    static void ShowInAppBrowser(const FHiveInAppBrowserParam& Param, const FHivePlatformHelperOnShowInAppBrowserDelegate& Delegate);
    
    /**
     * \~korean
     * @brief 외부 브라우저를 오픈합니다.
     *
     * @param url 열고자 하는 웹사이트 주소
     * @param useIncognito 시크릿 모드 사용 여부
     * @return 없음
     *
     * \~english
     * @brief Opens an external browser.
     *
     * @param url URL of the website to open
     * @param useIncognito Whether to use incognito mode
     * @return None
     *
     * \~
     * @ingroup FHivePlatformHelper
     */
    static void OpenBrowser(const FHiveOpenBrowserParam& Param);
    
    /**
     * @brief 인 앱 웹뷰를 오픈한다.
     *
     * @param param [InAppWebViewParam]
     * @param handler [onShowInAppWebViewHandler]
     *
     * @since 24.5.0
     * 
     * \~
     * @ingroup FHivePlatformHelper
     */
    static void ShowInAppWebView(const FHiveInAppWebViewParam& Param, const FHivePlatformHelperOnShowInAppWebViewDelegate& Delegate);
    static void GetLaunchParameters(const FHivePlatformHelperOnGetLaunchParametersDelegate& Delegate);
};