/**
 * @file    HiveProviderApple.h
 *
*  @date        2025
*  @copyright   Copyright © 2025 Com2uS Platform Corp. All Rights Reserved.
*
*  @defgroup    FHiveProviderApple
*  @{
* \~korean
* Apple GameCenter의 업적 및 리더보드 기능을 제공한다.<br/><br/>
* \~english
* It provides Apple GameCenter achievements and leaderboard<br/><br/>
*/
#pragma once

#include "CoreMinimal.h"
#include "HiveInterface.h"
#include "Impl/HiveProviderAppleImpl.h"

/*
* \~korean
* Apple GameCenter의 업적 및 리더보드 기능을 제공한다.<br/><br/>
* \~english
* It provides Apple GameCenter achievements and leaderboard<br/><br/>
* \~
* @ingroup FHiveProviderApple
*/
class HIVESDK_API FHiveProviderApple : public IHiveInterface<IHiveProviderAppleImpl>
{
public:
    /**
    *  \~korean
    * Apple GameCenter Leaderboard에 기록을 전송한다.
    *
    * @warning score format과 leaderboard identifier는 iTunes Connect에서 설정한다.
    *
    * @param Score Player가 얻은 score.
    * @param LeaderboardIdentifier Leaderboard Identifier.
    * @param Delegate API 호출 result handler.
    *
    *  \~english
    * It send record to Apple GameCenter Leaderboard.
    *
    * @warning The score format and leaderboard identifier are set in iTunes Connect.
    *
    * @param Score Player's score.
    * @param LeaderboardIdentifier Leaderboard Identifier.
    * @param Delegate  Result listener of API call.
    *
    *  \~
    * @ingroup FHiveProviderApple
    */
    static void ReportScore(const FString& Score, const FString& LeaderboardIdentifier, const FHiveProviderAppleOnReportLeaderboard& Delegate);
    
    /**
    *  \~korean
    * Apple GameCenter Leaderboard UI를 노출한다.
    *
    * @param Delegate API 호출 result listener.
    *
    *  \~english
    * It show Apple GameCenter Leaderboard UI.
    *
    * @param Delegate Result listener of API call.
    *
    *  \~
    * @ingroup FHiveProviderApple
    */
    static void ShowLeaderboard(const FHiveProviderAppleOnShowLeaderboard& Delegate);

    /**
    *  \~korean
    * Apple GameCenter 업적을 로드한다.
    *
    * @param Delegate API 호출 result listener.
    *
    *  \~english
    * It load Apple GameCenter achievement.
    *
    * @param Delegate Result listener of API call.
    *
    *  \~
    * @ingroup FHiveProviderApple
    */
    static void LoadAchievements(const FHiveProviderAppleOnLoadAchievements& Delegate);

    /**
    *  \~korean
    * Apple GameCenter 업적을 전송한다.
    *
    * @warning achievement identifier는 iTunes Connect에서 설정한다.
    *
    * @param Percent 업적 성취 percent value(ex. 75.0)
    * @param ShowCompletionBanner 업적 성공시 상단 배너 노출 여부. default is NO.
    * @param AchievementIdentifier Achievement identifier
    * @param Delegate API 호출 result listener.
    *
    *  \~english
    * It report Apple GameCenter achievement.
    *
    * @warning achievement identifier is set in iTunes Connect.
    *
    * @param Percent Achievement progress percent value(ex. 75.0)
    * @param ShowCompletionBanner Whether the top banner is exposed when the achievement is successful. default is NO.
    * @param AchievementIdentifier Achievement identifier
    * @param Delegate Result listener of API call.
    *
    * \~
    * @ingroup FHiveProviderApple
    */
    static void ReportAchievement(const FString& Percent, bool ShowCompletionBanner, const FString& AchievementIdentifier, const FHiveProviderAppleOnReportAchievement& Delegate);
    
    /**
    *  \~korean
    * Apple GameCenter 업적 UI를 노출한다.
    *
    * @param Delegate API 호출 result listener.
    *
    *  \~english
    * It shows Apple GameCenter Achievement UI.
    *
    * @param Delegate Result listener of API call.
    *
    *  \~
    * @ingroup FHiveProviderApple
    */
    static void ShowAchievements(const FHiveProviderAppleOnShowAchievement& Delegate);

    /**
    *  \~korean
    * Apple GameCenter 업적 정보를 초기화한다.
    *
    * @param Delegate API 호출 result listener.
    *
    *  \~english
    * It resets Apple GameCenter Achievement information.
    *
    * @param Delegate Result listener of API call.
    *
    *  \~
    * @ingroup FHiveProviderApple
    */
    static void ResetAchievements(const FHiveProviderAppleOnResetAchievement& Delegate);
};