/**
 * @file        HivePush.h
 *
 *  @date        2025
 *  @copyright   Copyright © 2025 Com2uS Platform Corp. All Rights Reserved.
 *
 *  @defgroup    FHivePush
 * @{
 * \~korean  모바일 게임에서 푸시 통지 서비스는 게임 유저의 리텐션(잔존율)을 올리기 위한 중요 수단을 제공한다.<br/>
  * HIVE 푸시는 모바일 게임의 리텐션을 증대시키는 마케팅용 광고 푸시를 지원한다.<br/>
 * 또한 Google의 GCM, Apple의 APNS, Amazon의 ADM, 중국 지역의 Jpush를 지원하고, 언어별 발송 시간을 설정할 수 있어 글로벌 서비스 제공에도 대응하고 있다.<br/><br/>
 *
 * \~english In mobile games, push notification service provides an important method to increase the retention of game users.<br/>
 * HIVE Push supports push notification for marketing to increase the user retention of mobile games.<br/>
 * HIVE Push also responds to Google's GCM, Apple's APNS, Amazon's ADM, Jpush in China with language-specific time zones to support global services.<br/><br/>
 */
#pragma once

#include "CoreMinimal.h"
#include "HiveInterface.h"
#include "Impl/HivePushImpl.h"

/**
 * \~korean  모바일 게임에서 푸시 통지 서비스는 게임 유저의 리텐션(잔존율)을 올리기 위한 중요 수단을 제공한다.<br/>
 * HIVE 푸시는 모바일 게임의 리텐션을 증대시키는 마케팅용 광고 푸시를 지원한다.<br/>
 * 또한 Google의 GCM, Apple의 APNS, Amazon의 ADM, 중국 지역의 Jpush를 지원하고, 언어별 발송 시간을 설정할 수 있어 글로벌 서비스 제공에도 대응하고 있다.<br/><br/>
 *
 * \~english In mobile games, push notification service provides an important method to increase the retention of game users.<br/>
 * HIVE Push supports push notification for marketing to increase the user retention of mobile games.<br/>
 * HIVE Push also responds to Google's GCM, Apple's APNS, Amazon's ADM, Jpush in China with language-specific time zones to support global services.<br/><br/>
 *
 * \~
 * @ingroup FHivePush
 */
class HIVESDK_API FHivePush : public IHiveInterface<IHivePushImpl>
{
public:
    /**
     * \~korean 유저의 푸시 수신 상태 조회
     *
     * @param Delegate	API 결과 통지
     *
     * \~english Request the status of receiving push notification.
     *
     * @param Delegate	API call result
     * \~
     * @ingroup FHivePush
     */
    static void GetRemotePush(const FHivePushOnRemotePushDelegate& Delegate);

    /**
     * \~korean 유저의 푸시 수신 상태 설정
     *
     * @param RemotePush	리모트 푸시 정보
     * @param Delegate  	API 결과 통지
     *
     * \~english Set the status of receiving push notification.
     *
     * @param RemotePush	Remote push information.
     * @param Delegate  	API call result
     * 
     * \~
     * @ingroup FHivePush
     */
    static void SetRemotePush(const FHiveRemotePush& RemotePush, const FHivePushOnRemotePushDelegate& Delegate);

    /**
     * \~korean 로컬 푸시 등록. iOS에서는 시스템 제한으로 최대 64개 까지 등록 가능
     *
     * @param LocalPush	로컬 푸시 등록 정보
     * @param Delegate	API 결과 통지
     * 
     * \~english Register Local push notification. In iOS, up to 64 can be registered due to the system limit.
     *
     * @param LocalPush	Local push registration information.
     * @param Delegate	API call result
     * 
     * \~
     * @ingroup FHivePush
     */
    static void RegisterLocalPush(const FHiveLocalPush& LocalPush, const FHivePushOnLocalPushDelegate& Delegate);

    /**
     * \~korean 로컬 푸시 해제
     *
     * @param NoticeId		로컬 푸시 등록 ID
     * 
     * \~english Unregister Local push notification.
     *
     * @param NoticeId		Local push registration ID
     * 
     * \~
     * @ingroup FHivePush
     */
    static void UnregisterLocalPush(int32 NoticeId);

    /**
     * \~korean 로컬 푸시 해제
     *
     * @param NoticeIds        로컬 푸시 등록 ID 배열
     * 
     * \~english Unregister Local push notification.
     *
     * @param NoticeIds        Array of Local push registration ID
     * 
     * \~
     * @ingroup FHivePush
     */
    static void UnregisterLocalPushes(TArray<int32> NoticeIds);

    /**
     * \~korean Hive 로컬푸시를 포함한 모든 로컬 푸시 해제
     *
     * \~english Unregister All local pushes including Hive local pushes.
     *
     * \~
     * @ingroup FHivePush
     */
    static void UnregisterAllLocalPushes();

    /**
     * \~korean 앱 활성화 시 푸시 수신 여부 설정 값 적용
     *
     * @param Setting        유저가 앱 활성화 시 푸시를 수신하는 상태
     * @param Delegate       API 결과 통지
     * \~english Set the value whether to enable push notification or not on the activated app.
     *
     * @param Setting        The status of receiving push notification on the activated app.
     * @param Delegate       API call result
     * \~
     * @ingroup FHivePush
     */
    static void SetForegroundPush(const FHivePushSetting& Setting, const FHivePushOnPushSettingDelegate& Delegate);

    /**
     * \~korean 앱 활성화 시 푸시 수신 여부 설정 값 조회
     *
     * @param Delegate       API 결과 통지
     *
     * \~english Get the value whether to enable push notification or not on the activated app.
     *
     * @param Delegate       API call result
     * 
     * \~
     * @ingroup FHivePush
     */
    static void GetForegroundPush(const FHivePushOnPushSettingDelegate& Delegate);

    /**
     * \~korean Push Token 명시적 권한 요청
     *
     * \~english Request Push Token
     *
     * \~
     * @ingroup FHivePush
     */
    static void RequestPushPermission();
};