/* Copyright © 2025 Com2uS Platform Corp. All Rights Reserved. */
#pragma once

/* Unreal Engine Header */
#include "CoreMinimal.h"

/* Hive SDK Header */
#include "CPP/HIVE_CPP.h"
#include "HiveResultAPI.h"
#include "Types/HiveChatTypes.h"

// 리스너 래퍼의 기본 클래스
class HIVESDK_API HiveListenerBase
{
public:
    virtual ~HiveListenerBase() = default;
    
    void SetUniqueKey(const FString& InKey) { UniqueKey = InKey; }
    const FString& GetUniqueKey() const { return UniqueKey; }
    
protected:
    FString UniqueKey;
};

class HIVESDK_API HiveChannelListenerWrapper : public HiveListenerBase, public hive::ChannelListener
{
public:
    HiveChannelListenerWrapper(FHiveChannelListener* InListener) : UnrealListener(InListener) {}
    virtual ~HiveChannelListenerWrapper() = default;

    void Initialize(FHiveChannelListener* InListener) { UnrealListener = InListener; }
    
    virtual void onChannelMessage(hive::ChannelMessage const & message) override
    {
        if (UnrealListener)
        {
            FHiveChannelMessage Message(message);
            UnrealListener->onChannelMessage(Message);
        }
    }
    
    virtual void onNoticeMessage(hive::ChannelNoticeMessage const & message) override
    {
        if (UnrealListener)
        {
            FHiveChannelNoticeMessage Message(message);
            UnrealListener->onNoticeMessage(Message);
        }
    }
    
    virtual void onEnteredMember(hive::EnteredMember const & member) override
    {
        if (UnrealListener)
        {
            FHiveEnteredMember Member(member);
            UnrealListener->onEnteredMember(Member);
        }
    }
    
    virtual void onExitedMember(hive::ExitedMember const & member) override
    {
        if (UnrealListener)
        {
            FHiveExitedMember Member(member);
            UnrealListener->onExitedMember(Member);
        }
    }
    
    virtual void onDeletedChannel(hive::DeletedChannel const & channel) override
    {
        if (UnrealListener)
        {
            FHiveDeletedChannel Channel(channel);
            UnrealListener->onDeletedChannel(Channel);
        }
    }
    
    virtual void onAddReaction(hive::Reaction const & reaction) override
    {
        if (UnrealListener)
        {
            FHiveReaction Reaction(reaction);
            UnrealListener->onAddReaction(Reaction);
        }
    }
    
    virtual void onRemoveReaction(hive::Reaction const & reaction) override
    {
        if (UnrealListener)
        {
            FHiveReaction Reaction(reaction);
            UnrealListener->onRemoveReaction(Reaction);
        }
    }
    
protected:
    FHiveChannelListener* UnrealListener;
};

// 다이렉트 메시지 리스너 래퍼
class HIVESDK_API HiveDirectMessageListenerWrapper : public HiveListenerBase, public hive::DirectMessageListener
{

public:
    HiveDirectMessageListenerWrapper(FHiveDirectMessageListener* InListener) : UnrealListener(InListener) {}
    virtual ~HiveDirectMessageListenerWrapper() = default;

    void Initialize(FHiveDirectMessageListener* InListener) { UnrealListener = InListener; }
    
    virtual void onDirectMessage(hive::DirectMessage const & message) override
    {
        if (UnrealListener)
        {
            FHiveDirectMessage Message(message);
            UnrealListener->onDirectMessage(Message);
        }
    }
    
protected:
    FHiveDirectMessageListener* UnrealListener;
};

// 연결 리스너 래퍼
class HIVESDK_API HiveConnectionListenerWrapper : public HiveListenerBase, public hive::ConnectionListener
{

public:
    HiveConnectionListenerWrapper(FHiveConnectionListener* InListener) : UnrealListener(InListener) {}
    virtual ~HiveConnectionListenerWrapper() = default;

    void Initialize(FHiveConnectionListener* InListener) { UnrealListener = InListener; }
    
    virtual void onConnected() override
    {
        if (UnrealListener)
        {
            UnrealListener->onConnected();
        }
    }

    virtual void onReconnectStarted() override
    {
        if (UnrealListener)
        {
            UnrealListener->onReconnectStarted();
        }
    }

    virtual void onReconnected(hive::ResultAPI const& result, std::vector<std::string> const& channelIds, std::vector<std::string> const& failChannelIds) override
    {
        if (UnrealListener)
        {
            FHiveResultAPI Result(result);

            TArray<FString> UnrealChannelIds;
            UnrealChannelIds.Reserve(channelIds.size());
            for (const auto& channelId : channelIds)
            {
                UnrealChannelIds.Emplace(channelId.c_str());
            }

            TArray<FString> UnrealFailChannelIds;
            UnrealFailChannelIds.Reserve(failChannelIds.size());
            for (const auto& failChannelId : failChannelIds)
            {
                UnrealFailChannelIds.Emplace(failChannelId.c_str());
            }

            UnrealListener->onReconnected(Result, UnrealChannelIds, UnrealFailChannelIds);
        }
    }
    
    virtual void onDisconnected() override
    {
        if (UnrealListener)
        {
            UnrealListener->onDisconnected();
        }
    }
    
protected:
    FHiveConnectionListener* UnrealListener;
};

// 유저 리스너 래퍼
class HIVESDK_API HiveUserListenerWrapper : public HiveListenerBase, public hive::UserListener
{
public:
    HiveUserListenerWrapper(FHiveUserListener* InListener) : UnrealListener(InListener) {}
    virtual ~HiveUserListenerWrapper() = default;

    void Initialize(FHiveUserListener* InListener) { UnrealListener = InListener; }
    
    virtual void onDirectMessage(hive::DirectMessage const & message) override
    {
        if (UnrealListener)
        {
            FHiveDirectMessage Message(message);
            UnrealListener->onDirectMessage(Message);
        }
    }
    
    virtual void onNoticeMessage(hive::NoticeMessage const & message) override
    {
        if (UnrealListener)
        {
            FHiveNoticeMessage Message(message);
            UnrealListener->onNoticeMessage(Message);
        }
    }
    
protected:
    FHiveUserListener* UnrealListener;
};

// 커스텀 데이터 리스너 래퍼
class HIVESDK_API HiveCustomDataListenerWrapper : public HiveListenerBase, public hive::CustomDataListener
{
public:
    HiveCustomDataListenerWrapper(FHiveCustomDataListener* InListener) : UnrealListener(InListener) {}
    virtual ~HiveCustomDataListenerWrapper() = default;

    void Initialize(FHiveCustomDataListener* InListener) { UnrealListener = InListener; }
    
    virtual void onCustomData(std::string const & data) override
    {
        if (UnrealListener)
        {
            FString Data(UTF8_TO_TCHAR(data.c_str()));
            UnrealListener->onCustomData(Data);
        }
    }
    
protected:
    FHiveCustomDataListener* UnrealListener;
};

DECLARE_DELEGATE_OneParam(FHiveChatOnResultDelegate, const FHiveResultAPI& /* Result */);
DECLARE_DELEGATE_OneParam(FHiveChatOnCreateChannelDelegate, const FHiveResultAPI& /* Result */);
DECLARE_DELEGATE_OneParam(FHiveChatOnDeleteChannelDelegate, const FHiveResultAPI& /* Result */);
DECLARE_DELEGATE_OneParam(FHiveChatOnEnterChannelDelegate, const FHiveResultAPI& /* Result */);
DECLARE_DELEGATE_OneParam(FHiveChatOnExitChannelDelegate, const FHiveResultAPI& /* Result */);
DECLARE_DELEGATE_OneParam(FHiveChatOnBlockMemberDelegate, const FHiveResultAPI& /* Result */);
DECLARE_DELEGATE_OneParam(FHiveChatOnUnblockMemberDelegate, const FHiveResultAPI& /* Result */);
DECLARE_DELEGATE_OneParam(FHiveChatOnConnectDelegate, const FHiveResultAPI& /* Result */);
DECLARE_DELEGATE_OneParam(FHiveChatOnDisconnectDelegate, const FHiveResultAPI& /* Result */);
DECLARE_DELEGATE_OneParam(FHiveChatOnSendMessageWithChannelSendMessageParamsDelegate, const FHiveResultAPI& /* Result */);
DECLARE_DELEGATE_OneParam(FHiveChatOnSendMessageWithDirectSendMessageParamsDelegate, const FHiveResultAPI& /* Result */);
DECLARE_DELEGATE_ThreeParams(FHiveChatOnReconnectDelegate, const FHiveResultAPI& /* Result */, const TArray<FString>& channelIds /* ChannelIds */, const TArray<FString>& failedChannelIds /* FailedChannelIds */);
DECLARE_DELEGATE_ThreeParams(FHiveChatOnGetChannelsDelegate, const FHiveResultAPI& /* Result */, const TArray<FHiveChannel>& channels /* Channels */, FHiveChannelPage const & channelPage /* ChannelPage */);
DECLARE_DELEGATE_ThreeParams(FHiveChatOnGetChannelInfoDelegate, const FHiveResultAPI& /* Result */, FHiveChannel const & channel /* Channel */, const TArray<FHiveMember>& members /* Members */);
DECLARE_DELEGATE_TwoParams(FHiveChatOnGetChannelMembersDelegate, const FHiveResultAPI& /* Result */, const TArray<FHiveMember>& members /* Members */);
DECLARE_DELEGATE_TwoParams(FHiveChatOnGetChannelsByUserDelegate, const FHiveResultAPI& /* Result */, const TArray<FHiveChannel>& channels /* Channels */);
DECLARE_DELEGATE_TwoParams(FHiveChatOnGetBlockMembersDelegate, const FHiveResultAPI& /* Result */, const TArray<FHiveBlockMember>& blockMembers /* BlockMembers */);
DECLARE_DELEGATE_TwoParams(FHiveChatOnTranslateDelegate, const FHiveResultAPI& /* Result */, FHiveTranslationData const & translationData /* TranslationData */);


class HIVESDK_API IHiveChatImpl
{
public:
    virtual ~IHiveChatImpl();
public:
    virtual void CreateChannel(FHiveCreateChannelParams const & params, const FHiveChatOnCreateChannelDelegate& Delegate);
    virtual void GetChannels(TOptional<FHiveGetChannelsParams> params, const FHiveChatOnGetChannelsDelegate& Delegate);
    virtual void GetChannelInfo(const FString& channelId, const FHiveChatOnGetChannelInfoDelegate& Delegate);
    virtual void GetChannelMembers(const FString& channelId, const FHiveChatOnGetChannelMembersDelegate& Delegate);
    virtual void DeleteChannel(const FString& channelId, const FHiveChatOnDeleteChannelDelegate& Delegate);
    virtual void EnterChannel(const FHiveEnterChannelParams& params, const FHiveChatOnEnterChannelDelegate& Delegate);
    virtual void ExitChannel(const FString& channelId, const FHiveChatOnExitChannelDelegate& Delegate);
    virtual void GetChannelsByUser(const FHiveChatOnGetChannelsByUserDelegate& Delegate);
    virtual void GetBlockMembers(const FHiveChatOnGetBlockMembersDelegate& Delegate);
    virtual void BlockMember(int64 blockPlayerId, const FHiveChatOnBlockMemberDelegate& Delegate);
    virtual void UnblockMember(int64 blockPlayerId, const FHiveChatOnUnblockMemberDelegate& Delegate);
    virtual void Connect(const FString& extraData, const FHiveChatOnConnectDelegate& Delegate);
    virtual void Connect(const FHiveChatOnConnectDelegate& Delegate);
    virtual void Reconnect(const FHiveChatOnReconnectDelegate& Delegate);
    virtual void Disconnect(const FHiveChatOnDisconnectDelegate& Delegate);
    virtual void SendMessageWithChannelSendMessageParams(FHiveChannelSendMessageParams const & params, const FHiveChatOnSendMessageWithChannelSendMessageParamsDelegate& Delegate);
    virtual void SendMessageWithDirectSendMessageParams(FHiveDirectSendMessageParams const & params, const FHiveChatOnSendMessageWithDirectSendMessageParamsDelegate& Delegate);
    virtual void SendMessageWithChannelSendMessageParams(FHiveChannelSendMessageParams const & params);
    virtual void SendMessageWithDirectSendMessageParams(FHiveDirectSendMessageParams const & params);
    virtual void Translate(FHiveTranslateParams const & params, const FHiveChatOnTranslateDelegate& Delegate);

    virtual void AddChannelListener(const FString& uniqueKey, FHiveChannelListener* listener);
    virtual void AddDirectMessageListener(const FString& uniqueKey, FHiveDirectMessageListener* listener);
    virtual void AddConnectionListener(const FString& uniqueKey, FHiveConnectionListener* listener);
    virtual void AddUserListener(const FString& uniqueKey, FHiveUserListener* listener);
    virtual void AddCustomDataListener(const FString& uniqueKey, FHiveCustomDataListener* listener);
    virtual void RemoveChannelListener(const FString& uniqueKey);
    virtual void RemoveDirectMessageListener(const FString& uniqueKey);
    virtual void RemoveConnectionListener(const FString& uniqueKey);
    virtual void RemoveUserListener(const FString& uniqueKey);
    virtual void RemoveCustomDataListener(const FString& uniqueKey);
    
    virtual bool IsConnected();

private:
    FCriticalSection ListenerLock;
    TMap<FString, HiveChannelListenerWrapper*> ChannelListeners;
    TMap<FString, HiveDirectMessageListenerWrapper*> DirectMessageListeners;
    TMap<FString, HiveConnectionListenerWrapper*> ConnectionListeners;
    TMap<FString, HiveUserListenerWrapper*> UserListeners;
    TMap<FString, HiveCustomDataListenerWrapper*> CustomDataListeners;
};