/* Copyright © 2024 Com2uS Platform Corp. All Rights Reserved. */
#pragma once

/* Unreal Engine Header */
#include "CoreMinimal.h"

/* Hive SDK Header */
#include "CPP/HIVE_CPP.h"


using PlayerData = TMap<FString, FString>;
using ProviderFriendsMap = TMap<FString, int64>;

/**
 *  \~korean
 * @brief Provider 형태 정의
 * 여기서 AUTO 는 자동로그인의 용도로 쓰이며<br/>
 * isAutoSignIn() 이 true 일 경우 SignIn 시 AUTO 로 입력해 주면 된다.<br/>
 *
 *  \~english
 * @brief Provider Types
 * AUTO is for Automatic Login <br/>
 * If the result of isAutoSignIn() call is true, You need to set parameter as AUTO when you call SignIn.<br/>
 *  \~
 * @ingroup FHiveAuthV4
 *
 */
enum class EHiveProviderType
{
    UNKNOWN     = -1, // Unreal Engine 신규 추가
    GUEST       = 0,
    HIVE        = 1,
    FACEBOOK    = 2,
    GOOGLE      = 3,
    QQ          = 4,
    VK          = 6,
    WECHAT      = 7,
    APPLE       = 8,
    SIGNIN_APPLE = 9,
    LINE        = 10,
    WEVERSE     = 12,
    GOOGLE_PLAY_GAMES = 14,
    HUAWEI      = 15,
    STEAM       = 18,
    X           = 19,
    TELEGRAM    = 20,
    XIAOMI      = 21,
    CUSTOM      = 98,
    AUTO        = 99
    // 미사용 영구 결번 처리 : 5, 11, 13
};

/**
*  \~korean
* @brief 프로바이더 정보
* 연결된 프로바이더의 UserId 를 포함하고 있다. ProviderUserId 가 없다면 연결된 상태가 아니다.<br/>
* Provider 자동 로그인 (묵시적) 을 사용한다면 signIn(Provider) 호출 결과에서 ProviderUserId 도 같이 체크 하도록 한다.<br/>
* <br/>
* ProviderType : 현재 Provider 종류<br/>
* ProviderName : 현재 Provider 이름
* ProviderUserId : Provider User 고유 ID (PlayerID 가 아니다)<br/>
*
*  \~english
* @brief Identity Provider (IdP) Information
* It contains the UserId of the associated Identity Provider. If there is no ProviderUserId, it is not connected.<br/>
* If you are using Provider auto-login (implicit login), also check ProviderUserId in the result of signIn(Provider) call. <br/>
* <br/>
* ProviderType : Current Provider Type<br/>
* ProviderName : Current Provider Name
* ProviderUserId : Provider's Unique User ID (It is not PlayerID)<br/>
*
*  \~
* @ingroup FHiveAuthV4
*/
class HIVESDK_API FHiveProviderInfo
{
public:
    EHiveProviderType ProviderType;
    FString ProviderName;
    FString ProviderUserId;
    FString ProviderEmail;

    FHiveProviderInfo() {};
    FHiveProviderInfo(const hive::ProviderInfo& ProviderInfo);
    virtual ~FHiveProviderInfo() = default;
};

/**
*  \~korean
* @brief 유저의 프로필 정보
* PlayerId : 유저의 고유한 ID<br/>
* PlayerName : 외부에 보여질 유저의 닉네임, 처음 연결된 Provider 의 정보로 채워지며<br/>
* HIVE 멤버쉽으로 연동한 사용자는 변경이 가능하다.<br/>
* PlayerImageUrl : 유저의 섬네일 이미지 URL, PlayerName 과 마찬가지로 처음 연결된 Provider 의 정보로 채워지며<br/>
* HIVE 멤버쉽으로 연동한 사용자는 변경이 가능하다.<br/>
*
*  \~english
* @brief User Profile Information
* PlayerId : User's unique ID<br/>
* PlayerName : The nickname of the user to be shown outside. It is filled with information from the first connected provider, but can be changed when user register HIVE Membership. <br/>
* PlayerImageUrl : Thumbnail image URL of the user. Like PlayerName, it is filled with information from the first connected provider, but can be changed when user register HIVE Membership.<br/>
*
*  \~
* @ingroup FHiveAuthV4
*/
class HIVESDK_API FHiveProfileInfo
{
public:
    int64 PlayerId;
    FString PlayerName;
    FString PlayerImageUrl;

    FHiveProfileInfo() {};
    FHiveProfileInfo(const hive::ProfileInfo& ProfileInfo);
    virtual ~FHiveProfileInfo() = default;
};

/**
*  \~korean
* @brief 사인-인 유저의 정보
* 프로필 정보 (ProfileInfo) 와 함께 유저의 토큰과 DID 값이 포함되어 있다.<br/>
* <br/>
* PlayerToken : 사인-인 검증에 필요한 playerId 와 연결된 토큰<br/>
* Did : 단말 고유 ID (DID : Device ID). 처음 AuthV4.setup() 호출 시 생성되며 이후 앱 삭제 전까지 바뀌지 않는다.<br/>
*
*  \~english
* @brief Sign-in User Information
* it includes user's token and DID value along with ProfileInfo.<br/>
* <br/>
* PlayerToken : Token associated with playerId required for sign-in verification<br/>
* Did : Device unique ID (DID). It is created when setup () is called for the first time and does not change until after the app is deleted.<br/>
*
*  \~
* @ingroup FHiveAuthV4
*/
class HIVESDK_API FHivePlayerInfo : public FHiveProfileInfo
{
public:
    FString PlayerToken;
    FString AccessToken;
    FString Did;
    TMap<EHiveProviderType, FHiveProviderInfo> ProviderInfoData;
    TMap<FString, FHiveProviderInfo> CustomProviderInfoData;

    FHivePlayerInfo() {};
    FHivePlayerInfo(const hive::PlayerInfo& PlayerInfo);
    virtual ~FHivePlayerInfo() = default;
};

/**
*  \~korean
* @brief 보호자 동의 정보
* 유저의 생년월일과 보호자의 이메일 주소가 포함되어 있다.<br/>
* <br/>
* dateOfBirth : 유저의 생년월일<br/>
* guardianEmail : 보호자의 이메일 주소<br/>
*
*  \~english
* @brief Parental Consent Information
* It includes the user's date of birth and the guardian's email address.<br/>
* <br/>
* dateOfBirth : User's date of birth<br/>
* guardianEmail : Guardian's email address<br/>
*
*  \~
*
* @ingroup AuthV4
*
*/
class HIVESDK_API FHiveParentalConsentInfo final {
public:
	FString dateOfBirth;    ///< 유저의 생년월일
	FString guardianEmail;  ///< 보호자 이메일 주소

	FHiveParentalConsentInfo() = default;
	FHiveParentalConsentInfo(const hive::ParentalConsentInfo& parentalConsentInfo);
};

/**
 *  \~korean
 * @brief 점검 화면에서 버튼을 눌렀을때 동작될 행동
 * OPEN_URL : 외부 브라우저로 전달된 URL 을 실행<br/>
 * EXIT : 앱 종료<br/>
 * DONE : 아무 처리 하지 않고 점검 팝업 종료<br/>
 *
 *  \~english
 * @brief Actions to be taken when a button is pressed on the maintenance popup.
 * OPEN_URL : Open URL passed to external browser<br/>
 * EXIT : Exit App<br/>
 * DONE : Close the popup without any action<br/>
 *
 *  \~
 * @ingroup FHiveAuthV4
 */
enum class EHiveAuthV4MaintenanceActionType
{
    NONE        = -1,
    OPEN_URL    = 1,
    EXIT,
    DONE
};

/**
 *  \~korean
 *  @brief AgeRange API에서 사용되는 연령 상태 코드.<br/>
 *  - VERIFIED : 18세 이상<br/>
 *  - SUPERVISED : 부모 감독 계정<br/>
 *  - SUPERVISED_APPROVAL_PENDING : 보호자 승인 대기중<br/>
 *  - SUPERVISED_APPROVAL_DENIED : 보호자 승인 거부함<br/>
 *  - UNKNOWN : 검증되지 않음<br/>
 *  - REQUIRED : 해당 지역에서 검증 필요<br/>
 *
 *  \~english
 *  @brief Status codes used by the AgeRange API.<br/>
 *  - VERIFIED : User is 18+<br/>
 *  - SUPERVISED : Parent-supervised account<br/>
 *  - SUPERVISED_APPROVAL_PENDING : Approval pending<br/>
 *  - SUPERVISED_APPROVAL_DENIED : Approval denied<br/>
 *  - UNKNOWN : Not verified or supervised<br/>
 *  - REQUIRED : Verification required<br/>
 *
 *  \~
 *  @ingroup AuthV4
 */
enum class EHiveUserStatus {
    VERIFIED = 0
    ,SUPERVISED = 1
    ,SUPERVISED_APPROVAL_PENDING = 2
    ,SUPERVISED_APPROVAL_DENIED = 3
    ,UNKNOWN = 4
    ,REQUIRED = 5
};

/**
 *  \~korean
 *  @brief 연령 범위 체크 결과 정보.<br/>
 *  - userStatus : 사용자의 연령 상태 코드<br/>
 *  - ageLower : 연령 범위 하한값(미확인 시 -1)<br/>
 *  - ageUpper : 연령 범위 상한값(성인 시 -1)<br/>
 *  - mostRecentApprovalDate : 가장 최근 승인된 중요한 변경 사항의 발효일<br/>
 *  - ageRangeId : 스토어에서 부여하는 사용자 식별자<br/>
 *
 *  \~english
 *  @brief Age range check result information.<br/>
 *  - userStatus : User's age range status<br/>
 *  - ageLower : Lower bound of the age range (-1 if UNKNOWN)<br/>
 *  - ageUpper : Upper bound of the age range (-1 if adult)<br/>
 *  - mostRecentApprovalDate : Effective date of the most recent approved change<br/>
 *  - ageRangeId : Store-assigned identifier for supervised users<br/>
 *
 *  \~
 *  @ingroup AuthV4
 */
class HIVESDK_API FHiveAuthV4AgeRange {
public:
    EHiveUserStatus UserStatus = EHiveUserStatus::UNKNOWN;  // 연령 범위 상태 코드
    int AgeLower = -1;                                      // 연령 범위 하한값
    int AgeUpper = -1;                                      // 연령 범위 상한값
    FString MostRecentApprovalDate = "";                    // 최근 승인된 중요한 변경 사항의 발효일
    FString AgeRangeId = "";                                // 스토어에서 발급된 사용자 식별자

    FHiveAuthV4AgeRange() {};
    FHiveAuthV4AgeRange(const hive::AgeRange& AgeRange);
    virtual ~FHiveAuthV4AgeRange() = default;
};

/**
*  \~korean
* @brief 추가적인 점검 버튼 정보
* Button : 버튼의 Text 문구 ex) 추가 버튼<br/>
* Action : 버튼을 눌렀을때 동작될 행동<br/>
* Url : action이 OPEN_URL 일 경우에 브라우징 될 URL<br/>
*
*  \~english
* @brief Additional Maintenance Button Information
* Button : Text on the button ex) Extra Button<br/>
* Action : Actions to be taken when the button is pressed<br/>
* Url : URL to be browsed when action is OPEN_URL<br/>
*
*  \~
* @ingroup FHiveAuthV4
*/
class HIVESDK_API FHiveAuthV4MaintenanceExtraButton
{
public:
    FString Button;
    EHiveAuthV4MaintenanceActionType Action;
    FString Url;

    FHiveAuthV4MaintenanceExtraButton() {};
    FHiveAuthV4MaintenanceExtraButton(const hive::AuthV4MaintenanceExtraButton& Button);
    virtual ~FHiveAuthV4MaintenanceExtraButton() = default;
};


/**
*  \~korean
* @brief 서버 점검 및 업데이트 상태 표시 정보
* Title : 점검 제목<br/>
* Message : 점검 내용<br/>
* Button : 버튼의 Text 문구 ex) 확인<br/>
* Action : 버튼을 눌렀을때 동작될 행동<br/>
* Url : action 이 OPEN_URL 일 경우에 브라우징 될 URL<br/>
* CustomButton : 고객센터 버튼 제목<br/>
* CustomLink : 고객센터 링크<br/>
* RemainingTime : EXIT 일 경우 점검 완료까지 남은 초단위 시간. 시간은 실시간 갱신되며 0초가 되면 앱 종료<br/>
* StartDate : 점검시작일 YYYY-mm-dd HH:ii<br/>
* EndDate : 점검종료일 YYYY-mm-dd HH:ii<br/>
*
*  \~english
* @brief Server Maintenance and Update status display information
* Title : Title<br/>
* Message : Contents<br/>
* Button : Text on the button ex) OK<br/>
* Action : Actions to be taken when the button is pressed<br/>
* Url : URL to be browsed when action is OPEN_URL<br/>
* CustomButton : customer Button Title<br/>
* CustomLink : customer Link<br/>
* RemainingTime : When action is EXIT, the time in seconds remaining until the maintenance completes. The time is updated in real time, and the app will be closed at 0 seconds.<br/>
* StartDate : start time YYYY-mm-dd HH:ii<br/>
* EndDate : end time YYYY-mm-dd HH:ii<br/>
*
*  \~
* @ingroup FHiveAuthV4
*/
class HIVESDK_API FHiveAuthV4MaintenanceInfo
{
public:
    FString Title;
    FString Message;
    FString Button;
    EHiveAuthV4MaintenanceActionType Action;
    FString Url;
    FString StartDate;
    FString EndDate;
    int32 RemainingTime;
    FString CustomerButton;
    FString CustomerLink;
    TArray<FHiveAuthV4MaintenanceExtraButton> ExButtons;

    FHiveAuthV4MaintenanceInfo() {};
    FHiveAuthV4MaintenanceInfo(const hive::AuthV4MaintenanceInfo& Info);
    virtual ~FHiveAuthV4MaintenanceInfo() = default;
};

class HIVESDK_API IHiveConflictViewData
{
public:
    PlayerData CurrentPlayerData;
    PlayerData ConflictPlayerData;

    virtual ~IHiveConflictViewData() = default;
};

class HIVESDK_API FHiveConflictSingleViewData : public IHiveConflictViewData
{
public:
    FHiveConflictSingleViewData(int64 ConflictPlayerId);
    virtual ~FHiveConflictSingleViewData() = default;

    void SetData(const FString& Key, const FString& ConflictPlayerData);
};

class HIVESDK_API FHiveConflictViewData : public IHiveConflictViewData
{
public:
    FHiveConflictViewData(int64 CurrentPlayerId, int64 ConflictPlayerId);
    virtual ~FHiveConflictViewData() = default;

    void SetData(const FString& Key, const FString& CurrentPlayerData, const FString& ConflictPlayerData);
};

class HIVESDK_API FHiveIdentity
{
public: 
    FString PhoneNumber = TEXT("");
    FString PhoneCode = TEXT("");
    FString DateOfBirth = TEXT("");
    FString HashedDi = TEXT(""); 

    FHiveIdentity() {};
    FHiveIdentity(const hive::Identity& Identity);
    virtual ~FHiveIdentity() = default;
};

HIVESDK_API FString GetNameFromEHiveProviderType(const EHiveProviderType& ProviderType);
HIVESDK_API EHiveProviderType GetEHiveProviderTypeFromName(const FString& Name);
HIVESDK_API FString GetNameFromEHiveAuthV4MaintenanceActionType(const EHiveAuthV4MaintenanceActionType& ActionType);
HIVESDK_API EHiveAuthV4MaintenanceActionType GetEHiveAuthV4MaintenanceActionTypeFromName(const FString& Name);
HIVESDK_API FString GetNameFromFHiveIdentity(const FHiveIdentity& Identity);