/* Copyright © 2024 Com2uS Platform Corp. All Rights Reserved. */
#pragma once

/* Unreal Engine Header */
#include "CoreMinimal.h"
#include "Dom/JsonObject.h"
#include "Serialization/JsonReader.h"
#include "Serialization/JsonSerializer.h"

/* Hive SDK Header */
#include "CPP/HIVE_CPP.h"

/**
 * \~korean  프로모션 뷰 창의 형태
 *
 * \~english Types of Promotion view
 * \~
 * @ingroup FHivePromotion
 */
enum class EHivePromotionType
{
    BANNER_LEGACY       = -1,       ///< \~korean 전면배너  \~english  Full Banner
    BANNER,                         ///< \~korean 전면배너  \~english Full banner
    NEWS,                           ///< \~korean 새소식  \~english News
    NOTICE                          ///< \~korean 공지사항  \~english Notice
};

/**
 * \~korean  프로모션 뷰 결과 통지시 이벤트 형태
 *
 * \~english Promotion view event types
 * \~
 * @ingroup FHivePromotion
 */
enum class EHivePromotionEventType
{
    OPEN,               ///< \~korean 프로모션 뷰가 열렸을 때	\~english When the Promotion View window opens.
    CLOSE,              ///< \~korean 프로모션 뷰가 닫혔을 때	\~english When the Promotion View window is closed.
    START_PLAYBACK,     ///< \~korean 영상 재생이 시작되었을 때	\~english When Playback starts.
    FINISH_PLAYBACK,    ///< \~korean 영상 재생이 완료되었을 때	\~english When Playback is finished.
    EXIT,               ///< \~korean 종료(더 많은 게임 보기) 팝업에서 종료하기를 선택했을 때	\~english When you choose to quit from the Quit (see more games) popup.
    GOBACK
};

/**
 * \~korean  커스텀 컨텐츠의 형태
 *
 * \~english Types of Promotion CustomView
 * \~
 * @ingroup FHivePromotion
 */
enum class EHivePromotionCustomType
{
    VIEW,       ///< \~korean 커스텀 뷰	\~english Custom view
    BOARD,      ///< \~korean 커스텀 보드	\~english Custom board
    SPOT,       ///< \~korean 스팟 배너	\~english Spot banner
    DIRECT      ///< \~korean 다이렉트 캠페인	\~english Direct campaign
};

/**
 * \~korean  게임내 오퍼월(무료 충전소) 을 호출 할 수 있는 버튼 노출 가능 상태 정보
 *
 * \~english Information of Offerwall(free charging station) button
 * \~
 * @ingroup FHivePromotion
 */
enum class EHiveOfferwallState
{
    ENABLED,        ///< \~korean 게임내 무료 충전 버튼 노출 가능	\~english  Offerwall Button can be exposed.
    DISABLED,       ///< \~korean 게임내 무료 충전 버튼 노출 불가능	\~english  Offerwall Button cannot be exposed.
    UNKNOWN         ///< \~korean 버튼 노출 가능 상태를 알 수 없음	\~english  Unknown status whether available to expose buttons or not.
};

/**
 * \~korean  Promotion 컨텐츠 화면 내용<br>
 * (HIVE 프로모션 웹 뷰의 UI 를 게임 UI 의 컨셉에 맞추기 위해서 프로모션 웹 뷰를 게임에서 직접 표시하기 위한 정보)
 *
 * \~english Information of Promotion webview<br>
 * (Information to customize your promotion Webview UI)
 * \~
 * @ingroup FHivePromotion
 */
class HIVESDK_API FHivePromotionViewInfo
{
public:
    FString Url;
    FString PostString;

    FHivePromotionViewInfo() {};
    FHivePromotionViewInfo(const hive::PromotionViewInfo& ViewInfo);
    virtual ~FHivePromotionViewInfo() = default;
};

/**
 * \~korean  프로모션 뱃지 정보가 표시될 대상 정보<br/>
 * 만약 NOTICE 이면 공지사항 관련 뱃지를 표시해야한다
 *
 * \~english Information on which promotional badge is displayed<br/>
 * If NOTICE, it should indicate the badge related to the notice.
 * \~
 * @ingroup FHivePromotion
 */
enum class EHivePromotionBadgeTarget
{
    NEWS,
    NOTICE,
    CUSTOMVIEW,
    CUSTOMBOARD
};

/**
 * \~korean  유저에게 새로운 프로모션 내용이 등록 되었음을 알려주기 위한 정보
 *
 * \~english Badge information to inform users that new promotions are registered
 * \~
 * @ingroup FHivePromotion
 */
class HIVESDK_API FHivePromotionBadgeInfo
{
public:
    EHivePromotionBadgeTarget Target;
    FString ContentsKey;
    FString BadgeType;
    
    FHivePromotionBadgeInfo() {};
    FHivePromotionBadgeInfo(const hive::PromotionBadgeInfo& BadgeInfo);
    virtual ~FHivePromotionBadgeInfo() = default;
};

/**
 * \~korean  롤링 배너 데이터 요청시 선택한 캠페인 타입
 *
 * \~english Campaign type selected before requesting rolling banner data.
 * \~
 * @ingroup FHivePromotion
 */
enum class EHivePromotionCampaignType
{
    EVENT,
    NOTICE,
    ALL,
    CROSS
};

/**
 * \~korean  롤링 배너 데이터 요청시 선택한 배너 타입
 *
 * \~english Banner type selected before requesting rolling banner data.
 * \~
 * @ingroup FHivePromotion
 */
enum class EHivePromotionBannerType
{
    GREAT,
    SMALL,
    ROLLING
};

/**
 * \~korean UserEngagement 기능 타입
 *
 * \~english UserEngagement 기능 타입
 * \~
 * @ingroup FHivePromotion
 */
enum class EHiveEngagementEventType
{
    EVENT_TYPE,
    PROMOTION_VIEW,
    OFFERWALL_VIEW,
    USER_ACQUISITION,
    COUPON,
    AUTH_LOGIN_VIEW,
    SOCIAL_INQUIRY_VIEW,
    EVENT,
    IAP_UPDATED,
    IAP_PURCHASE,
    IAP_PROMOTE,
    COMPANION,
    SOCIAL_MYINQUIRY_VIEW,
    SOCIAL_PROFILE_VIEW,
    COMMUNITY_VIEW,
    APPUPDATE_DOWNLOADED
};

/**
 * \~korean HIVE SDK 가 특정한 조건에서 클라이언트에 개입 (Engagement) 하는 상태
 *
 * \~english HIVE SDK 가 특정한 조건에서 클라이언트에 개입 (Engagement) 하는 상태
 * \~
 * @ingroup FHivePromotion
 */
enum class EHiveEngagementEventState
{
    BEGIN,
    FINISH,
    EVENT_START,
    EVENT_END
};

/**
 * \~korean 특정 프로모션에 대한 배너 정보
 *
 * \~english Banner information for a specific promotion
 * \~
 * @ingroup FHivePromotion
 */
class HIVESDK_API FHivePromotionBannerInfo
{
public:
    int32 Pid;
    FString ImageUrl;
    FString LinkUrl;
    FString DisplayStartDate;
    FString DisplayEndDate;
    int64 UtcStartDate;
    int64 UtcEndDate;
    FString TypeLink;
    FString TypeBanner;
    FString TypeCampaign;
    FString InterworkData;
    
    FHivePromotionBannerInfo() {};
    FHivePromotionBannerInfo(const hive::PromotionBannerInfo& BannerInfo);
    virtual ~FHivePromotionBannerInfo() = default;
};

/**
 * \~korean  앱 초대를 위한 캠패인 정보<br>
 * 캠페인은 초대에 대한 보상을 달상하기 위한 조건을 명시한 데이터.
 *
 * \~english Campaign information for invite<br>
 * Campaigns are data that specifies the conditions for rewarding invite.
 *
 * \~
 * @ingroup FHivePromotion
 */
class HIVESDK_API FHiveAppInvitationCampaign
{
public:
    FString Title;                  ///< \~korean 캠페인 타이틀 \~english Title
    FString Description;            ///< \~korean 캠페인 설명 \~english Description
    FString ImageUrl;               ///< \~korean 캠페인 이미지 URL \~english Image URL
    int32 Order;                    ///< \~korean 캠페인 순서 \~english Order
    TSharedPtr<FJsonObject> Item;   ///< \~korean 완료 보상 정보 ("key" 는 캠페인 설정에 따라 상이함) \~english Complete reward information ("key" depends on campaign settings)
    int32 Count;                    ///< \~korean 초대 인원 중. 캠페인을 완료한 수 \~english Number of invitees who completed the campaign
    int32 Limit;                    ///< \~korean 캠페인 최대 인원 수 \~english Maximum number of user of campaigns
    FString CampaignId;             ///< \~korean 캠페인 아이디 \~english Campaign Id

    FHiveAppInvitationCampaign() {};
    FHiveAppInvitationCampaign(const hive::AppInvitationCampaign& AppInvitationCampaign);
    virtual ~FHiveAppInvitationCampaign() = default;
};

class HIVESDK_API FHiveAppInvitationCampaignStage : public FHiveAppInvitationCampaign
{
public:
    int32 GoalCount;
    int32 GoalTotal;

    FHiveAppInvitationCampaignStage() {};
    FHiveAppInvitationCampaignStage(const hive::AppInvitationCampaignStage& AppInvitationCampaignStage);
    virtual ~FHiveAppInvitationCampaignStage() = default;
};

/**
 * \~korean  앱 초대 (UserAcquisition)를 위한 정보.
 *
 * \~english Invite information for UserAcquisition.
 * 
 * \~
 * @ingroup FHivePromotion
 */
class HIVESDK_API FHiveAppInvitationData
{
public:
    TArray<uint8> QrCodes;          ///< \~korean 앱의 초대 정보가 포함된 QR Code 이미지 데이터 \english QR Code image data with app invite information
    FString InviteCommonLink;       ///< \~korean 기본 초대 링크 \english Default invite link
    FString InviteHiveMsgLink;      ///< \~korean Hive 인증 사용자용 초대 링크 \english Invite link for Hive certified users
    FString InviteFacebookLink;     ///< \~korean Facebook 인증 사용자용 초대 링크 \english Invite link for Facebook certified users
    TArray<FHiveAppInvitationCampaign> EachCampaignList;        ///< \~korean 캠페인 완료 발생시, 매번 보상을 지급하는 캠페인 목록 \english List of campaigns that will be rewarded every time when a campaign is completed.
    TArray<FHiveAppInvitationCampaignStage> StageCampaignList;  ///< \~korean 캠페인 완료 목표 초대 수를 달성했을 때 보상을 지급하는 캠페인 목록 \english List of campaigns that will be rewarded when you reach the campaign completion goal of invites.
    FString InviteMessage;

    FHiveAppInvitationData() {};
    FHiveAppInvitationData(const hive::AppInvitationData& AppInvitationData);
    virtual ~FHiveAppInvitationData() = default;
};

/**
 * \~korean  UA를 통해 자신을 앱으로 최초 초대한 유저의 정보
 *
 * \~english  First sender's userInfo who sent UA share invitation
 *
 * \~
 * @ingroup FHivePromotion
 */
class HIVESDK_API FHiveAppInvitationSenderInfo
{
public:
    FString Vid;

    FHiveAppInvitationSenderInfo() {};
    FHiveAppInvitationSenderInfo(const hive::AppInvitationSenderInfo& AppInvitationSenderInfo);
    virtual ~FHiveAppInvitationSenderInfo() = default;
};

HIVESDK_API FString GetNameFromEHivePromotionType(const EHivePromotionType& PromotionType);
HIVESDK_API EHivePromotionType GetEHivePromotionTypeFromName(const FString& Name);
HIVESDK_API FString GetNameFromEHivePromotionEventType(const EHivePromotionEventType& PromotionEventType);
HIVESDK_API EHivePromotionEventType GetEHivePromotionEventTypeFromName(const FString& Name);
HIVESDK_API FString GetNameFromEHivePromotionCustomType(const EHivePromotionCustomType& PromotionCustomType);
HIVESDK_API EHivePromotionCustomType GetEHivePromotionCustomTypeFromName(const FString& Name);

HIVESDK_API FString GetNameFromEHiveOfferwallState(const EHiveOfferwallState& OfferwallState);
HIVESDK_API EHiveOfferwallState GetEHiveOfferwallStateFromName(const FString& Name);
HIVESDK_API FString GetNameFromEHivePromotionBadgeTarget(const EHivePromotionBadgeTarget& PromotionBadgeTarget);
HIVESDK_API EHivePromotionBadgeTarget GetEHivePromotionBadgeTargetFromName(const FString& Name);

HIVESDK_API FString GetNameFromEHivePromotionCampaignType(const EHivePromotionCampaignType& PromotionCampaignType);
HIVESDK_API EHivePromotionCampaignType GetEHivePromotionCampaignTypeFromName(const FString& Name);
HIVESDK_API FString GetNameFromEHivePromotionBannerType(const EHivePromotionBannerType& PromotionBannerType);
HIVESDK_API EHivePromotionBannerType GetEHivePromotionBannerTypeFromName(const FString& Name);

HIVESDK_API FString GetNameFromEHiveEngagementEventType(const EHiveEngagementEventType& EngagementEventType);
HIVESDK_API EHiveEngagementEventType GetEHiveEngagementEventTypeFromName(const FString& Name);
HIVESDK_API FString GetNameFromEHiveEngagementEventState(const EHiveEngagementEventState& EngagementEventState);
HIVESDK_API EHiveEngagementEventState GetEHiveEngagementEventStateFromName(const FString& Name);
