// ---------- Language Utilities ----------
function getHtmlLang() {
  const lang = document.documentElement.lang?.toLowerCase() || 'en';
  const map = { 'zh-hans': 'zh', 'zh-hant': 'zh-tw', 'vi': 'vn' };
  return map[lang] || lang;
}

// Return display date format for placeholder
function getDateFormat(lang = 'en') {
  const formatMap = {
    ko: 'YYYY-MM-DD',
    ja: 'YYYY-MM-DD',
    zh: 'YYYY-MM-DD',
    'zh-tw': 'YYYY-MM-DD',
    fr: 'YYYY-MM-DD',
    ar: 'YYYY-MM-DD',
    de: 'DD-MM-YYYY',
    ru: 'DD-MM-YYYY',
    es: 'DD-MM-YYYY',
    pt: 'DD-MM-YYYY',
    id: 'DD-MM-YYYY',
    th: 'DD-MM-YYYY',
    vi: 'DD-MM-YYYY',
    it: 'DD-MM-YYYY',
    tr: 'DD-MM-YYYY',
    en: 'MM-DD-YYYY'
  };
  return formatMap[lang] || formatMap.en;
}

// Return Flatpickr date format pattern
function getFlatpickrFormat(lang = 'en') {
  const map = {
    ko: 'Y-m-d',
    ja: 'Y-m-d',
    zh: 'Y-m-d',
    'zh-tw': 'Y-m-d',
    fr: 'Y-m-d',
    ar: 'Y-m-d',
    en: 'm-d-Y',
    de: 'd-m-Y',
    ru: 'd-m-Y',
    es: 'd-m-Y',
    pt: 'd-m-Y',
    id: 'd-m-Y',
    th: 'd-m-Y',
    vi: 'd-m-Y',
    it: 'd-m-Y',
    tr: 'd-m-Y'
  };
  return map[lang] || map.en;
}

// ---------- Placeholder Initialization ----------
function initDateFormatInPickerPlaceholder(lang = 'en') {
  const birthDate = document.getElementById('birthDate');
  if (birthDate) birthDate.placeholder = getDateFormat(lang);
}

// ---------- DatePicker Initialization ----------
function initializeDatePicker(lang = 'en') {
    const flatpickrInstances = [];
  const flatpickrFormat = getFlatpickrFormat(lang);
  
  // 🔧 Normalize lang string for datepicker locale(replace "-" with "_")
  lang = lang.replace(/-/g, '_');

  document.querySelectorAll('.pop_wrap.coppa').forEach(popup => {
    const datePickerInput = popup.querySelector('.datepicker');
    const dateBirth = popup.querySelector('.date_wrap');
    const btnConfirm = popup.querySelector('.btn_start');

    if (!datePickerInput || !dateBirth || !btnConfirm) return;

    const fp = flatpickr(datePickerInput, {
            // Use default locale if the specified locale is undefined
            locale: flatpickr.l10ns[lang] || flatpickr.l10ns.default,
      dateFormat: flatpickrFormat,
      maxDate: 'today',
      allowInput: true,
      onReady(_, __, instance) {
        const cal = instance.calendarContainer;
        cal.classList.add('custom-centered');
        cal.style.opacity = '0'; // hide calendar initially
      },
      onOpen() {
        dateBirth.classList.add('enter');

        // Create dimmed background if it doesn’t exist
        if (!document.querySelector('.flatpickr-dimmed')) {
          const dim = document.createElement('div');
          dim.className = 'flatpickr-dimmed';
          document.body.appendChild(dim);
          dim.addEventListener('click', () => fp.close());
        }

        // Make the calendar visible
        fp.calendarContainer.style.opacity = '1';
      },
      onClose(_, dateStr) {
        // Remove dimmed background
        document.querySelector('.flatpickr-dimmed')?.remove();
        toggleButtonState(dateStr, btnConfirm, dateBirth);
      },
      onChange(_, dateStr) {
        toggleButtonState(dateStr, btnConfirm);
      }
    });

        flatpickrInstances.push(fp);

    // ---------- Event Handlers ----------
        datePickerInput.addEventListener('input', () => toggleButtonState(datePickerInput.value, btnConfirm));

    datePickerInput.addEventListener('click', () => dateBirth.classList.add('enter'));
    datePickerInput.addEventListener('focus', () => dateBirth.classList.add('enter'));

    datePickerInput.addEventListener('blur', () =>
      setTimeout(() => {
        if (!datePickerInput.value.trim()) {
          dateBirth.classList.remove('enter');
          // Update button state (removes 'btn_on' and disables)
          toggleButtonState(datePickerInput.value, btnConfirm, dateBirth);
        }
      }, 100)
    );

    datePickerInput.addEventListener('keydown', e => {
      if (e.key === 'Enter') {
        e.preventDefault();
        dateBirth.classList.add('enter');
      }
	  
      if (e.key === 'Escape' || e.key === 'Esc') {
        e.preventDefault();
        fp.close();
      }
    });
  });
    return flatpickrInstances;
}

// ---------- Button State Control ----------
function toggleButtonState(value, btn, dateBirth) {
  const hasValue = !!(value && value.trim());
  btn.classList.toggle('btn_on', hasValue);
    // If dateBirth element exists and there's no value, remove the 'enter' class
  if (dateBirth && !hasValue) dateBirth.classList.remove('enter');
  btn.disabled = !hasValue;
}

// ---------- Script Execution ----------
(function () {
  const currentLang = getHtmlLang();
  initDateFormatInPickerPlaceholder(currentLang);

    let flatpickrInstances;
    const afterLoad = () => {
        // Initialize DatePicker and store instances
        flatpickrInstances = initializeDatePicker(currentLang);
        const button = document.querySelector('.btn_start');
        if (button) {
            button.onclick = function () {
                const fp = flatpickrInstances?.[0];

                if (fp && fp.selectedDates && fp.selectedDates.length > 0) {
                    const selectedDateObject = fp.selectedDates[0];
                    const formattedDate = fp.formatDate(selectedDateObject, 'Y-m-d');
                    window.location.href = `hivesdk://ui?button=confirm&date=${encodeURIComponent(formattedDate)}`;
                } else {
                    console.error("No date selected or flatpickr instance not found.");
                }
            };
        }
    };
  if (currentLang === 'en') {
    afterLoad();
  } else {
    const localeScript = document.createElement('script');
    localeScript.src = `https://cdn.jsdelivr.net/npm/flatpickr/dist/l10n/${currentLang}.js`;
    localeScript.onload = afterLoad;
    localeScript.onerror = () => {
      console.warn(`flatpickr locale for "${currentLang}" not found. Falling back to English.`);
      currentLang = 'en';
      afterLoad();
    };
    document.head.appendChild(localeScript);
  }
})();