// 초기 변수 선언
let joystickState = { x: 0, y: 0 }; // 조이스틱 상태 (X, Y 축)
let joystickTimers = { x: null, y: null }; // 지속 입력 타이머
let buttonState = {}; // 버튼 상태
let buttonTimers = {}; // 버튼별 타이머
let wheelTimers = { x: null, y: null };
let lastDirection = null; // 마지막 방향
const minInterval = 200; // 최소 이동 간격 (ms)
let hasFocus = true; // 포커스 상태
const scrollableContainers = ['.pop_contents']; // 전역 컨테이너 목록
let spatialInitialized = false;
let parseState = false;

let gameControllerStatus = {
    primaryBtn: "0",
    secondaryBtn: "1",
    secondaryBtnText: "B",
    connectedGameController: false,
};


// 포커스 복원 함수
function restoreFocus() {
    if (!gameControllerStatus.connectedGameController) {
        return;
    }

    if (!spatialInitialized) {
        initGamePadNavigation(); // 초기화 호출
    }
    else{
        SpatialNavigation.restoreLastFocus();
    }

    if(!spatialInitialized) return;
    
    setKeymapInfoVisibility();
}

// 포커스 숨기기 함수 (초기화는 하지 않음)
function hideFocus() {
    const currentElement = SpatialNavigation.focusableElements[SpatialNavigation.currentIndex];
    if (currentElement) {
        currentElement.blur(); // 포커스를 해제하지만 위치는 유지
    }
}

// 포커스 상태 변경 함수
function setControllerConnectedState(state) {

    if (state) {
        if(!spatialInitialized){
            initGamePadNavigation();
        }
        else{
            SpatialNavigation.restoreLastFocus();
        }
    } else {
        hideFocus();
    }

    if(!spatialInitialized) return;

    setKeymapInfoVisibility();
}

function setKeymapInfoVisibility() {
    var keymapInfo = document.querySelector('.keymap_info');
    
    if (!keymapInfo) {
      return; // 요소가 없으면 함수 종료
    }

    const closeButtons = document.querySelectorAll('.btn_close');
    const cancelButtons = document.querySelectorAll('.btn_cancel');

    const isRealCancelButton = (button) => {
        return button &&
               isVisible(button) &&
               (!button.hasAttribute('data-custom-value') ||
                button.getAttribute('data-custom-value') !== 'not_cancel_button');
    };

    const hasVisibleCloseButton = Array.from(closeButtons).some(button => isVisible(button));
    const hasVisibleCancelButton = Array.from(cancelButtons).some(button => isRealCancelButton(button));

    if (!hasVisibleCloseButton && !hasVisibleCancelButton) {
        return; // 닫기 또는 취소 버튼이 보이지 않으면 함수 종료
    }
    
    keymapInfo.textContent = gameControllerStatus.secondaryBtnText;
    keymapInfo.style.display = gameControllerStatus.connectedGameController ? 'flex' : 'none';
  }

// 포커스 처리 함수
function handleFocusGained() {
    if(!parseState) parseAndUpdateStatus();

    if(gameControllerStatus.connectedGameController == true){
        hasFocus = true;
        restoreFocus();
    }
}

function handleFocusLost() {
    hasFocus = false;

    hideFocus();
    stopAllTimers();
}

// 모든 타이머 정지
function stopAllTimers() {
    // 버튼 타이머 정지
    for (const direction in buttonTimers) {
        if (buttonTimers[direction]) {
            clearInterval(buttonTimers[direction]);
            buttonTimers[direction] = null;
        }
    }

    // 조이스틱 타이머 정지
    for (const axis in joystickTimers) {
        if (joystickTimers[axis]) {
            clearInterval(joystickTimers[axis]);
            joystickTimers[axis] = null;
        }
    }

    // 휠 타이머 정지
    for (const axis in wheelTimers) {
        if (wheelTimers[axis]) {
            clearInterval(wheelTimers[axis]);
            wheelTimers[axis] = null;
        }
    }

    // 상태 초기화
    buttonState = {};
    joystickState = { x: 0, y: 0 };
}

// 방향키 이동 처리
function handleDirectionalKey(direction) {
    SpatialNavigation.move(direction);
}

// 버튼 이벤트 처리
function handleJoystickButton(button, state) {
    const buttonNum = parseInt(button, 10);

    const buttonConfirm = parseInt(gameControllerStatus.primaryBtn, 10);
    const buttonCancel = parseInt(gameControllerStatus.secondaryBtn, 10);

    if (state === 'DOWN') {
        if (buttonNum === buttonConfirm) {
            triggerConfirmKey();
        } else if (buttonNum === buttonCancel) {
            triggerCancelKey();
        }
    }
}

// A 버튼 동작 처리
function triggerConfirmKey() {
    const focusedElement = document.activeElement;
    if (!focusedElement) return;

    // <label> 태그가 실제 동작을 담당하는 경우
    if (focusedElement.tagName === 'INPUT' && focusedElement.type === 'checkbox') {
        const label = document.querySelector(`label[for="${focusedElement.id}"]`);
        if (label) {
            label.click(); // LABEL 클릭
            return; // 기존 동작으로 넘어가지 않음
        }
    }
            
    // focusedElement가 <a> 태그인 경우 처리
    if (focusedElement.tagName === 'A') {
        const href = focusedElement.getAttribute('href');
        if (!href || href === '#' || href === '' || href === '#none') {

            const img = focusedElement.querySelector('img');
            if(!img) {
                focusedElement.click();
                return;
            }

            const iframe = img.parentElement.querySelector('iframe');
            if(!iframe) {
                img.click();
                return;
            }

            try {
                const iframeDoc = iframe.contentDocument || iframe.contentWindow.document;
                const bigBanner = iframeDoc.querySelector("#HIVEbanner");
            
                if(bigBanner) {
                    bigBanner.click();
                } else {
                    img.click();
                }
            } catch (e) {
                img.click();
            }
            
        } else {
            // href가 유효하면 <a> 클릭
            focusedElement.click();
        }

        return;
    }

    // 다른 요소의 경우 기본 동작 수행
    focusedElement.click(); 
}

// B 버튼 동작 처리
function triggerCancelKey() {
    
    const closeButtons = document.querySelectorAll('.btn_close');
    for (let i = 0; i < closeButtons.length; i++) {
        const button = closeButtons[i];
        if (isVisible(button)) {
            processButtons(button, '닫기 버튼');
            return;
        }
    }

    const cancelButtons = document.querySelectorAll('.btn_cancel');
    for (let i = 0; i < cancelButtons.length; i++) {
        const button = cancelButtons[i];
        if (isVisible(button)) {
            processButtons(button, '취소 버튼');
            return;
        }
    }
}

// 버튼 처리 함수
function processButtons(button, label) {

    if (button) {
        try {
            // 서버점검 팝업에서 상세보기가 btn_cancel을 사용하기 때문에 커스텀데이터로 해당값을 구분한다.
            if (!button.hasAttribute('data-custom-value') || button.getAttribute('data-custom-value') !== 'not_cancel_button') {
                button.click();
            }
        } catch (error) {
            console.error(`${label} 클릭 중 오류 발생:`, error);
        }
    }
}

// 요소가 보이는지 확인
function isVisible(element) {
    if (!element) return false;

    if (element.classList.contains('hide')) {
        return false;
    }

    const style = window.getComputedStyle(element);
    if (style.display === 'none' || style.visibility === 'hidden') {
        return false;
    }
    
    let parent = element.parentElement;
    while (parent) {
        if (parent.classList.contains('hide')) {
            return false;
        }

        const parentStyle = window.getComputedStyle(parent);
        if (parentStyle.display === 'none' || parentStyle.visibility === 'hidden') {
            //console.log('hidden 상태의 부모 발견:', parent);
            return false;
        }
        parent = parent.parentElement;
    }
    return true;
}

// 십자키 이동 처리
function handleMoveEvent(direction, state) {
    function startMoveTimer(direction) {
        if (buttonTimers[direction]) {
            return;
        }
        buttonTimers[direction] = setInterval(() => {
            handleDirectionalKey(direction);
        }, minInterval);
    }

    function stopMoveTimer(direction) {
        if (buttonTimers[direction]) {
            clearInterval(buttonTimers[direction]);
            buttonTimers[direction] = null;
        }
    }

    function stopAllTimersExcept(currentDirection) {
        for (const dir in buttonTimers) {
            if (dir !== currentDirection && buttonTimers[dir]) {
                stopMoveTimer(dir);
                buttonState[dir] = false; // 다른 방향 키 상태 초기화
            }
        }
    }

    if (state === "DOWN") {
        stopAllTimersExcept(direction);

        if (!buttonState[direction]) {
            buttonState[direction] = true;
            lastDirection = direction;
            handleDirectionalKey(direction);
            startMoveTimer(direction);
        }
    } else if (state === "UP") {
        if (lastDirection) {
            buttonState[lastDirection] = false;
            stopMoveTimer(lastDirection);
            lastDirection = null;
        }
    }
}

// 조이스틱 방향 처리
function handleJoystickMove(axis, value) {
    const normalizedValue = value / 32767;
    const threshold = 0.5;

    function startTimer(direction, axisKey) {
        if (joystickTimers[axisKey]) {
            //console.log(`타이머 이미 실행 중: ${direction}`);
            return;
        }
        joystickTimers[axisKey] = setInterval(() => {
            handleDirectionalKey(direction);
        }, minInterval);
    }

    function stopTimer(axisKey) {
        if (joystickTimers[axisKey]) {
            clearInterval(joystickTimers[axisKey]);
            joystickTimers[axisKey] = null;
        }
    }

    if (axis === 0) {
        if (Math.abs(normalizedValue) > threshold) {
            if (normalizedValue > 0) {
                if (joystickState.x !== 1) {
                    handleDirectionalKey("RIGHT");
                    joystickState.x = 1;
                }
                startTimer("RIGHT", "x");
            } else if (normalizedValue < 0) {
                if (joystickState.x !== -1) {
                    handleDirectionalKey("LEFT");
                    joystickState.x = -1;
                }
                startTimer("LEFT", "x");
            }
        } else {
            if (joystickState.x !== 0) {
                joystickState.x = 0;
                stopTimer("x");
            }
        }
    }

    if (axis === 1) {
        if (Math.abs(normalizedValue) > threshold) {
            if (normalizedValue > 0) {
                if (joystickState.y !== 1) {
                    handleDirectionalKey("DOWN");
                    joystickState.y = 1;
                }
                startTimer("DOWN", "y");
            } else if (normalizedValue < 0) {
                if (joystickState.y !== -1) {
                    handleDirectionalKey("UP");
                    joystickState.y = -1;
                }
                startTimer("UP", "y");
            }
        } else {
            if (joystickState.y !== 0) {
                joystickState.y = 0;
                stopTimer("y");
            }
        }
    }
}

// 조이스틱 휠 이벤트 처리
function handleJoystickWheel(axis, value) {
    const normalizedValue = value / 32767;
    const threshold = 0.5;

    // 스크롤 가능한 컨테이너 찾기
    function findScrollableContainer() {
        for (const selector of scrollableContainers) {
            const container = document.querySelector(selector);
            if (container) {
                const scrollableHeight = container.scrollHeight - container.clientHeight;
                if (scrollableHeight > 0) {
                    //console.log(`스크롤 가능한 컨테이너 발견: ${selector}`);
                    return container; // 첫 번째 스크롤 가능한 컨테이너 반환
                }
            }
        }
        return null; // 스크롤 가능한 컨테이너가 없으면 null 반환
    }

    const scrollableContainer = findScrollableContainer();
    if (!scrollableContainer) return; // 스크롤 가능한 컨테이너가 없으면 종료

    function startTimer(direction, axisKey) {
        if (wheelTimers[axisKey]) {
            //console.log(`타이머 이미 실행 중: ${direction}`);
            return;
        }
        
        wheelTimers[axisKey] = setInterval(() => {
            if (direction === 'UP') {
                scrollableContainer.scrollBy({ top: -100, left: 0, behavior: 'smooth' });
            } else if (direction === 'DOWN') {
                scrollableContainer.scrollBy({ top: 100, left: 0, behavior: 'smooth' });
            } else if (direction === 'LEFT') {
                scrollableContainer.scrollBy({ top: 0, left: -100, behavior: 'smooth' });
            } else if (direction === 'RIGHT') {
                scrollableContainer.scrollBy({ top: 0, left: 100, behavior: 'smooth' });
            }
        }, minInterval);
    }

    function stopWheelTimer(axisKey) {
        if (wheelTimers[axisKey]) {
            clearInterval(wheelTimers[axisKey]);
            wheelTimers[axisKey] = null; // 해당 방향의 타이머를 null로 설정
        }
    }

    if (axis === 3) { // Y축 (상하 스크롤)
        if(Math.abs(normalizedValue) > threshold){
            if (value > 0) {
                startTimer("DOWN", "y");
            } else if (value < 0) {
                startTimer("UP", "y");
            }
        } else {
            stopWheelTimer("y");
        }
    } else if (axis === 2) { // X축 (좌우 스크롤)
        if(Math.abs(normalizedValue) > threshold){
            if (value > 0) {
                startTimer("RIGHT", "x");
            } else if (value < 0) {
                startTimer("LEFT", "x");
            }
        } else {
            stopWheelTimer("x");
        }
    }
}

// JSON 이벤트 처리
function handleControllerInput(inputData) {

    try {
        const { eventType, direction, state, button, axis, value, connected, primaryBtn, secondaryBtn, secondaryBtnText} = inputData;

        // CONNECT 이벤트는 포커스 상태와 상관없이 처리
        if (eventType === 'CONNECT') {
            updateGameControllerStatus(primaryBtn, secondaryBtn, secondaryBtnText, connected);
            setControllerConnectedState(connected);
            return;
        }

        if (!hasFocus) {
            return;
        }

        switch (eventType) {
            case "JOYSTICK_AXIS":
                handleJoystickMove(parseInt(axis, 10), parseInt(value, 10));
                break;

            case 'JOYSTICK_BUTTON':
                handleJoystickButton(button, state);
                break;

            case "WHEEL":
                handleJoystickWheel(parseInt(axis, 10), parseInt(value, 10));
                break;

            case 'MOVE':
                handleMoveEvent(direction, state);
                break;

            default:
                console.error(`알 수 없는 이벤트 타입: ${inputData.eventType}`);
        }
    } catch (error) {
        console.error('JSON 처리 에러:', error);
    }
}

// 게임패드 초기화
function initGamePadNavigation() {
    if(spatialInitialized == true || gameControllerStatus.connectedGameController == false){
        return;
    }

    SpatialNavigation.init({
        selector: '.focusable, button, input, a', // 사용할 클래스와 태그 추가
        excludeSelector: '.hidden, .hide' // 제외할 요소
      });
    
    SpatialNavigation.makeFocusable(); // 포커스 가능한 요소 활성화
    SpatialNavigation.setFocus(0); // 첫 번째 요소에 포커스 설정

    spatialInitialized = true;

}


window.addEventListener('keydown', (event) => {
    switch (event.key) {
        case 'Escape': // ESC 키 처리
        triggerCancelKey();
        break;
    }
  });


function parseAndUpdateStatus() {
    
    if(typeof window.gameControllerStatus == "undefined") return;

    if(!window.gameControllerStatus) return;

    let parsedStatus = window.gameControllerStatus;

    // 파싱된 값을 각 변수에 나누어 저장
    let primaryBtn = parsedStatus.primaryBtn;
    let secondaryBtn = parsedStatus.secondaryBtn;
    let secondaryBtnText = parsedStatus.secondaryBtnText;
    let isConnected = parsedStatus.isConnected;

    // 분리된 변수들로 gameControllerStatus를 업데이트
    updateGameControllerStatus(primaryBtn, secondaryBtn, secondaryBtnText, isConnected);

    parseStatue = true;
}

function updateGameControllerStatus(primaryBtn, secondaryBtn, secondaryBtnText, connectedGameController) {
    gameControllerStatus = {
        primaryBtn: primaryBtn,
        secondaryBtn: secondaryBtn,
        secondaryBtnText: secondaryBtnText,
        connectedGameController: connectedGameController
    };  
}