const SpatialNavigation = {
  focusableElements: [],
  currentIndex: -1,
  lastFocusedIndex: -1,

  // 초기화
  init: function(options = {}) {
    const { selector = 'a, button, input', excludeSelector = '', container = document } = options;

    this.focusableElements = Array.from(container.querySelectorAll(selector)).filter(elem => {
      const style = window.getComputedStyle(elem);

      return (
        (!excludeSelector || !elem.matches(excludeSelector)) &&
        style.display !== 'none' &&
        style.visibility !== 'hidden' &&
        style.opacity !== '0' &&
        !this.hasHiddenParent(elem)
      );
    });

    if (this.focusableElements.length === 0) {
      console.warn('포커스 가능한 요소가 없습니다.');
    }
    else{
      this.currentIndex = 0;
      this.setFocus(this.currentIndex);
    }

    this.observeDomChanges(options);
  },

  // 포커스 가능한 요소 활성화
  makeFocusable: function() {
    this.focusableElements.forEach(element => {
      if (!element.hasAttribute('tabindex')) {
        element.setAttribute('tabindex', '0'); // 포커스 가능하도록 설정
      }
    });
    console.log('포커스 가능한 요소 활성화 완료:', this.focusableElements);
  },

  // 숨겨진 부모가 있는지 확인
  hasHiddenParent: function(element) {
    let parent = element.parentElement;
    while (parent) {
      const style = window.getComputedStyle(parent);
      if (style.display === 'none' || style.visibility === 'hidden') {
        console.log('숨겨진 부모 요소 발견:', parent);
        return true;
      }
      parent = parent.parentElement;
    }
    return false;
  },

  // 포커스 설정
  setFocus: function(index) {
    if (index < 0 || index >= this.focusableElements.length) {
      console.warn('유효하지 않은 포커스 인덱스:', index);
      return;
    }

    this.focusableElements.forEach(el => el.classList.remove('focused'));
    const element = this.focusableElements[index];
    element.classList.add('focused');
    element.focus();

    this.currentIndex = index;
    this.lastFocusedIndex = index;
  },

  // 포커스 이동
  move: function(direction) {
    const currentElement = this.focusableElements[this.currentIndex];
    if (!currentElement) {
      console.warn('현재 포커스된 요소가 없습니다.');
      return;
    }
  
    const currentRect = currentElement.getBoundingClientRect();
    let targetIndex = -1;
    let minDistance = Infinity;
  
    //간혹 UI상에서 좌표가 엇갈리는 경우가 있음 ex) 왼쪽 UI의 right가 100, 오른쪽 UI의 left가 99
    const offset = 1;

    this.focusableElements.forEach((elem, index) => {
      if (index === this.currentIndex) return; // 자기 자신은 제외
  
      const rect = elem.getBoundingClientRect();
      let isValid = false;
      let distance = Infinity;
  
    // 방향별 유효성 판별 및 거리 계산
    switch (direction) {
      case 'UP':
        if (rect.bottom <= currentRect.top) {
          isValid = true;
          distance = Math.abs(currentRect.top - rect.bottom) + Math.abs(currentRect.left - rect.left);
        }
        break;
      case 'DOWN':
        if (rect.top >= currentRect.bottom) {
          isValid = true;
          distance = Math.abs(rect.top - currentRect.bottom) + Math.abs(currentRect.left - rect.left);
        }
        break;
      case 'LEFT':
        if (rect.right <= currentRect.left + offset) {
          isValid = true;
          distance = Math.abs(currentRect.left - rect.right) + Math.abs(currentRect.top - rect.top);
        }
        break;
      case 'RIGHT':
        if (rect.left >= currentRect.right - offset) {
          isValid = true;
          distance = Math.abs(rect.left - currentRect.right) + Math.abs(currentRect.top - rect.top);
        }
        break;
    }
  
      // 가장 가까운 유효한 요소 선택
      if (isValid && distance < minDistance) {
        minDistance = distance;
        targetIndex = index;
      }
    });
  
    if (targetIndex !== -1) {
      this.setFocus(targetIndex);
  } else {
      console.warn(`"${direction}" 방향으로 이동할 대상이 없습니다. 마지막 포커스 복원.`);
      this.restoreLastFocus(); // 이동할 대상이 없으면 마지막 포커스 복원
  }
  },

  // 마지막 포커스 복원
  restoreLastFocus: function() {
    if (this.lastFocusedIndex !== -1 && this.lastFocusedIndex < this.focusableElements.length) {
      this.setFocus(this.lastFocusedIndex);
    } else {
      this.setFocus(0);
    }
  },

  observeDomChanges: function (options = {}) {
    const observer = new MutationObserver((mutations) => {
      let shouldUpdate = false;

      mutations.forEach((mutation) => {
        if (mutation.target.matches && mutation.target.matches('span.count')) {
          return; // 해당 변경 사항을 무시
        }
        
        if (mutation.type === 'attributes' && mutation.attributeName === 'style') {
          const target = mutation.target;
          const style = window.getComputedStyle(target);

          if (style.display !== 'none' && style.visibility !== 'hidden') {
            console.log('hidden 상태에서 visible로 변경된 요소:', target);
            shouldUpdate = true;
          }
        }
        else if(mutation.type === 'childList' && mutation.addedNodes.length > 0){
          console.log('자식 노드 추가/삭제 감지:', mutation);
          shouldUpdate = true;
        }
      });

      if (shouldUpdate) {
        this.updateFocusableElements(options); // 업데이트 호출
        this.makeFocusable();

        this.restoreLastFocus();
      }
    });

    observer.observe(document.body, {
      attributes: true,
      childList: true,
      subtree: true,
    });

    console.log('MutationObserver로 DOM 변경 감시 시작');
  },

  // 포커스 가능한 요소 업데이트 (기존 요소 포함)
  updateFocusableElements: function (options = {}) {
    const { selector = 'a, button, input', excludeSelector = '', container = document } = options;

    // 기존 요소가 여전히 DOM에 존재하는지 확인
    this.focusableElements = this.focusableElements.filter((elem) => container.contains(elem));

    const newFocusableElements = Array.from(container.querySelectorAll(selector)).filter((elem) => {
      const style = window.getComputedStyle(elem);
      return (
        (!excludeSelector || !elem.matches(excludeSelector)) &&
        style.display !== 'none' &&
        style.visibility !== 'hidden' &&
        style.opacity !== '0' &&
        !this.hasHiddenParent(elem)
      );
    });

    // 기존 요소와 새 요소를 병합하여 focusableElements 업데이트
    this.focusableElements = [...new Set([...this.focusableElements, ...newFocusableElements])];

    // DOM 순서에 맞게 정렬 (compareDocumentPosition 사용)
    this.focusableElements.sort((a, b) => {
      if (a === b) return 0; // 동일한 요소는 순서 유지
      if (a.compareDocumentPosition(b) & Node.DOCUMENT_POSITION_PRECEDING) {
        return 1; // a가 b 뒤에 있다면 1 반환
      }
      return -1; // a가 b 앞에 있다면 -1 반환
    });

    this.makeFocusable(); // 업데이트된 요소에 tabindex 추가
    console.log('포커스 가능한 요소 업데이트 완료:', this.focusableElements);
  }

};


// window.addEventListener('keydown', (event) => {
//   switch (event.key) {
//     case 'ArrowUp':
//       console.log('ArrowUp 키 눌림');
//       SpatialNavigation.move('UP');
//       break;
//     case 'ArrowDown':
//       console.log('ArrowDown 키 눌림');
//       SpatialNavigation.move('DOWN');
//       break;
//     case 'ArrowLeft':
//       console.log('ArrowLeft 키 눌림');
//       SpatialNavigation.move('LEFT');
//       break;
//     case 'ArrowRight':
//       console.log('ArrowRight 키 눌림');
//       SpatialNavigation.move('RIGHT');
//       break;
//   }
// });

// 마우스 클릭 이벤트 처리
window.addEventListener('mousedown', (event) => {
  let clickedElement = event.target;

  // `img` 태그인 경우 상위의 `a` 태그로 포커스 전환
  if (clickedElement.tagName === 'IMG') {
    const parentLink = clickedElement.closest('a'); // 상위 a 태그 탐색
    if (parentLink) {
      console.log('이미지 클릭 - 상위 a 태그로 포커스 전환:', parentLink);
      clickedElement = parentLink; // 포커스 대상 요소를 상위 a 태그로 변경
    }
  }

  // 클릭된 요소가 포커스 대상인지 확인
  if (SpatialNavigation.focusableElements.includes(clickedElement)) {
    const targetIndex = SpatialNavigation.focusableElements.indexOf(clickedElement);
    console.log('마우스 클릭으로 포커스 이동:', clickedElement);

    // 포커스 이동
    SpatialNavigation.setFocus(targetIndex);
  }
});
