/* Copyright © 2025 Com2uS Platform Corp. All Rights Reserved. */

#include "ChatParamView.h"

void UChatParamView::ShowParameters(EChatAPI API)
{
    CurrentAPI = API;
    OriginalAPI = API;  // 원본 API 타입 저장
    DataManager = UChatDataManager::GetInstance();
    
    
    // 블루프린트에서 UI가 구성되어 있어야 함
    
    // 모든 파라미터 UI 숨기기
    HideAllParams();
    
    // API에 따라 적절한 UI 표시
    switch (API)
    {
        case EChatAPI::CHAT_SEND_MESSAGE_TO_CHANNEL:
            ShowSendMessageToChannelParams();
            break;
            
        case EChatAPI::CHAT_SEND_MESSAGE_TO_PLAYER:
            ShowSendMessageToPlayerParams();
            break;
            
        case EChatAPI::CHAT_CREATE_CHANNEL:
            ShowCreateChannelParams();
            break;
            
        case EChatAPI::CHAT_GET_CHANNELS:
            ShowGetChannelsParams();
            break;
            
        case EChatAPI::CHAT_CHANNEL_QUERY:
            ShowChannelQueryParams();
            break;
            
        case EChatAPI::CHAT_CHANNEL_SET_TRANSLATION_ENABLED:
            ShowSetTranslationEnabledParams();
            break;
            
        case EChatAPI::CHAT_CHANNEL_ADD_REACTION:
        case EChatAPI::CHAT_CHANNEL_REMOVE_REACTION:
            // 두 API가 같은 UI를 사용하므로 OriginalAPI로 구분
            ShowReactionParams();
            break;
            
        case EChatAPI::CHAT_TRANSLATE:
            ShowTranslateParams();
            break;
            
        default:
            break;
    }
}

void UChatParamView::OnOKButtonClicked()
{
    
    // CurrentAPI를 OriginalAPI로 복원 (Reaction의 경우 중요)
    CurrentAPI = OriginalAPI;
    
    if (OnConfirmDelegate.IsBound())
    {
        OnConfirmDelegate.Execute();
    }
    // Parent에서 제거
    RemoveFromParent();
}

void UChatParamView::OnCancelButtonClicked()
{
    if (OnCancelDelegate.IsBound())
    {
        OnCancelDelegate.Execute();
    }
    // Parent에서 제거
    RemoveFromParent();
}

void UChatParamView::SaveSendMessageToChannelParams(const FString& ChannelId, const FString& Message, 
                                                    const FString& ExtraData, const FString& ReplyMessageId, 
                                                    const FString& MentionedPlayerIds)
{
    
    if (DataManager)
    {
        DataManager->SetSendMessageChannelId(ChannelId);
        DataManager->SetSendMessageMessage(Message);
        DataManager->SetSendMessageExtraData(ExtraData);
        DataManager->SetSendMessageReplyMessageId(ReplyMessageId);
        DataManager->SetSendMessageMentionedPlayerIds(MentionedPlayerIds);
    }
}

void UChatParamView::SaveSendMessageToPlayerParams(const FString& PlayerIdStr, const FString& Message, 
                                                   const FString& ExtraData)
{
    
    if (DataManager)
    {
        int64 PlayerId = FCString::Atoi64(*PlayerIdStr);
        DataManager->SetSendMessagePlayerId(PlayerId);
        DataManager->SetSendMessageMessage(Message);
        DataManager->SetSendMessageExtraData(ExtraData);
    }
}

void UChatParamView::SaveCreateChannelParams(const FString& ChannelId, const FString& Password, 
                                             const FString& ChannelName, const FString& MaxMemberCountStr, 
                                             const FString& TypeStr, ECheckBoxState ChatHistoryAllowedState)
{
    if (DataManager)
    {
        DataManager->SetCreateChannelChannelId(ChannelId);
        DataManager->SetCreateChannelPassword(Password);
        DataManager->SetCreateChannelChannelName(ChannelName);
        
        // String to int 변환
        int32 MaxMemberCount = FCString::Atoi(*MaxMemberCountStr);
        DataManager->SetCreateChannelMaxMemberCount(MaxMemberCount);
        
        DataManager->SetCreateChannelChannelType(TypeStr);
        
        // ECheckBoxState를 bool로 변환
        bool ChatHistoryAllowed = (ChatHistoryAllowedState == ECheckBoxState::Checked);
        DataManager->SetCreateChannelChatHistoryAllowed(ChatHistoryAllowed);
    }
}

void UChatParamView::SaveGetChannelsParams(const FString& ChannelId, const FString& ChannelName, 
                                           const FString& PageOrder, const FString& PageSizeStr, const FString& PageNumberStr, 
                                           const FString& TypeStr, const FString& SortTypeStr)
{
    if (DataManager)
    {
        DataManager->SetGetChannelsChannelId(ChannelId);
        DataManager->SetGetChannelsChannelName(ChannelName);
        DataManager->SetGetChannelsPageOrder(PageOrder);
        
        // String to int 변환
        int32 PageSize = FCString::Atoi(*PageSizeStr);
        int32 PageNumber = FCString::Atoi(*PageNumberStr);
        DataManager->SetGetChannelsPageSize(PageSize);
        DataManager->SetGetChannelsPageNumber(PageNumber);
        
        DataManager->SetGetChannelsChannelType(TypeStr);
        DataManager->SetGetChannelsSortType(SortTypeStr);
    }
}

void UChatParamView::SaveChannelQueryParams(const FString& PrevSizeStr, const FString& NextSizeStr, const FString& MessageId, 
                                            const FString& Order)
{
    if (DataManager)
    {
        // String to int 변환
        int32 PrevSize = FCString::Atoi(*PrevSizeStr);
        int32 NextSize = FCString::Atoi(*NextSizeStr);
        
        DataManager->SetChannelQueryPrevSize(PrevSize);
        DataManager->SetChannelQueryNextSize(NextSize);
        DataManager->SetChannelQueryMessageId(MessageId);
        DataManager->SetChannelQueryOrder(Order);
    }
}

void UChatParamView::SaveSetTranslationEnabledParams(bool Enabled)
{
    if (DataManager)
    {
        DataManager->SetSetTranslationEnabled(Enabled);
    }
}

void UChatParamView::SaveReactionParams(const FString& MessageId, const FString& ReactionTypeStr)
{
    
    if (DataManager)
    {
        DataManager->SetReactionMessageId(MessageId);
        
        // 공백 제거하고 저장
        FString CleanedReactionType = ReactionTypeStr.TrimStartAndEnd();
        DataManager->SetReactionType(CleanedReactionType);
    }
}

void UChatParamView::SaveTranslateParams(const FString& Message, const FString& TargetLanguages)
{
    if (DataManager)
    {
        DataManager->SetTranslateMessage(Message);
        DataManager->SetTranslateTargetLanguages(TargetLanguages);
    }
}

// Load 함수 구현
void UChatParamView::LoadSendMessageToChannelParams(FString& ChannelId, FString& Message, FString& ExtraData, 
                                                    FString& ReplyMessageId, FString& MentionedPlayerIds)
{
    if (DataManager)
    {
        ChannelId = DataManager->GetSendMessageChannelId();
        Message = DataManager->GetSendMessageMessage();
        ExtraData = DataManager->GetSendMessageExtraData();
        ReplyMessageId = DataManager->GetSendMessageReplyMessageId();
        MentionedPlayerIds = DataManager->GetSendMessageMentionedPlayerIds();
    }
}

void UChatParamView::LoadSendMessageToPlayerParams(FString& PlayerIdStr, FString& Message, FString& ExtraData)
{
    if (DataManager)
    {
        PlayerIdStr = FString::Printf(TEXT("%lld"), DataManager->GetSendMessagePlayerId());
        Message = DataManager->GetSendMessageMessage();
        ExtraData = DataManager->GetSendMessageExtraData();
    }
}

void UChatParamView::LoadCreateChannelParams(FString& ChannelId, FString& Password, FString& ChannelName,
                                             FString& MaxMemberCountStr, FString& TypeStr, bool& ChatHistoryAllowed)
{
    if (DataManager)
    {
        ChannelId = DataManager->GetCreateChannelChannelId();
        Password = DataManager->GetCreateChannelPassword();
        ChannelName = DataManager->GetCreateChannelChannelName();
        MaxMemberCountStr = FString::Printf(TEXT("%d"), DataManager->GetCreateChannelMaxMemberCount());
        
        EHiveChannelType ChannelType = DataManager->GetCreateChannelChannelType();
        switch (ChannelType)
        {
            case EHiveChannelType::PUBLIC:
                TypeStr = TEXT("PUBLIC");
                break;
            case EHiveChannelType::PRIVATE:
                TypeStr = TEXT("PRIVATE");
                break;
            case EHiveChannelType::GROUP:
                TypeStr = TEXT("GROUP");
                break;
            default:
                TypeStr = TEXT("PUBLIC");
                break;
        }
        
        ChatHistoryAllowed = DataManager->GetCreateChannelChatHistoryAllowed();
    }
}

void UChatParamView::LoadGetChannelsParams(FString& ChannelId, FString& ChannelName, FString& PageOrder,
                                           FString& PageSizeStr, FString& PageNumberStr, FString& TypeStr, FString& SortTypeStr)
{
    if (DataManager)
    {
        ChannelId = DataManager->GetGetChannelsChannelId();
        ChannelName = DataManager->GetGetChannelsChannelName();
        PageOrder = DataManager->GetGetChannelsPageOrder();
        PageSizeStr = FString::Printf(TEXT("%d"), DataManager->GetGetChannelsPageSize());
        PageNumberStr = FString::Printf(TEXT("%d"), DataManager->GetGetChannelsPageNumber());
        
        EHiveChannelType ChannelType = DataManager->GetGetChannelsChannelType();
        switch (ChannelType)
        {
            case EHiveChannelType::UNKNOWN:
                TypeStr = TEXT("NONE");
                break;
            case EHiveChannelType::PUBLIC:
                TypeStr = TEXT("PUBLIC");
                break;
            case EHiveChannelType::PRIVATE:
                TypeStr = TEXT("PRIVATE");
                break;
            case EHiveChannelType::GROUP:
                TypeStr = TEXT("GROUP");
                break;
            default:
                TypeStr = TEXT("UNKNOWN");
                break;
        }
        
        EHiveSortType SortType = DataManager->GetGetChannelsSortType();
        switch (SortType)
        {
            case EHiveSortType::CHANNEL_ID:
                SortTypeStr = TEXT("ChannelId");
                break;
            case EHiveSortType::CHANNEL_NAME:
                SortTypeStr = TEXT("ChannelName");
                break;
            case EHiveSortType::REG_TIME:
                SortTypeStr = TEXT("RegTime");
                break;
            default:
                SortTypeStr = TEXT("ChannelId");
                break;
        }
    }
}

void UChatParamView::LoadChannelQueryParams(FString& PrevSizeStr, FString& NextSizeStr, FString& MessageId, FString& Order)
{
    if (DataManager)
    {
        PrevSizeStr = FString::Printf(TEXT("%d"), DataManager->GetChannelQueryPrevSize());
        NextSizeStr = FString::Printf(TEXT("%d"), DataManager->GetChannelQueryNextSize());
        MessageId = DataManager->GetChannelQueryMessageId();
        Order = DataManager->GetChannelQueryOrder();
    }
}

void UChatParamView::LoadSetTranslationEnabledParams(bool& Enabled)
{
    if (DataManager)
    {
        Enabled = DataManager->GetSetTranslationEnabled();
    }
}

void UChatParamView::LoadReactionParams(FString& MessageId, FString& ReactionTypeStr)
{
    
    if (DataManager)
    {
        MessageId = DataManager->GetReactionMessageId();
        
        // 저장된 문자열 직접 가져오기
        FString SavedReactionStr = DataManager->GetReactionTypeString();
        
        // 앞뒤 공백 제거
        SavedReactionStr = SavedReactionStr.TrimStartAndEnd();
        
        EHiveReactionType ReactionType = DataManager->GetReactionType();
        
        // 저장된 문자열이 있으면 그대로 사용 (대소문자 유지)
        if (!SavedReactionStr.IsEmpty() && SavedReactionStr.Len() > 0)
        {
            // 첫 글자를 대문자로 변환
            FString FirstChar = SavedReactionStr.Left(1).ToUpper();
            FString RestOfString = SavedReactionStr.RightChop(1).ToLower();
            ReactionTypeStr = FirstChar + RestOfString;
            
            // ComboBox Options과 정확히 일치하는지 확인
            if (ReactionTypeStr != TEXT("Like") && ReactionTypeStr != TEXT("Unknown"))
            {
                ReactionTypeStr = TEXT("Like");
            }
        }
        else
        {
            // 저장된 문자열이 없으면 기본값으로 Like
            ReactionTypeStr = TEXT("Like");
        }
    }
    else
    {
        // DataManager가 없어도 기본값 설정
        ReactionTypeStr = TEXT("Like");
    }
}

void UChatParamView::LoadTranslateParams(FString& Message, FString& TargetLanguages)
{
    if (DataManager)
    {
        Message = DataManager->GetTranslateMessage();
        TargetLanguages = DataManager->GetTranslateTargetLanguages();
    }
}

FString UChatParamView::GetCurrentAPIString() const
{
    switch (CurrentAPI)
    {
        case EChatAPI::CHAT_SEND_MESSAGE_TO_CHANNEL:
            return TEXT("CHAT_SEND_MESSAGE_TO_CHANNEL");
        case EChatAPI::CHAT_SEND_MESSAGE_TO_PLAYER:
            return TEXT("CHAT_SEND_MESSAGE_TO_PLAYER");
        case EChatAPI::CHAT_CREATE_CHANNEL:
            return TEXT("CHAT_CREATE_CHANNEL");
        case EChatAPI::CHAT_GET_CHANNELS:
            return TEXT("CHAT_GET_CHANNELS");
        case EChatAPI::CHAT_CHANNEL_QUERY:
            return TEXT("CHAT_CHANNEL_QUERY");
        case EChatAPI::CHAT_CHANNEL_SET_TRANSLATION_ENABLED:
            return TEXT("CHAT_CHANNEL_SET_TRANSLATION_ENABLED");
        case EChatAPI::CHAT_CHANNEL_ADD_REACTION:
            return TEXT("CHAT_CHANNEL_ADD_REACTION");
        case EChatAPI::CHAT_CHANNEL_REMOVE_REACTION:
            return TEXT("CHAT_CHANNEL_REMOVE_REACTION");
        case EChatAPI::CHAT_TRANSLATE:
            return TEXT("CHAT_TRANSLATE");
        default:
            return TEXT("UNKNOWN");
    }
}

TArray<FString> UChatParamView::GetReactionTypeOptions() const
{
    TArray<FString> Options;
    Options.Add(TEXT("Unknown"));
    Options.Add(TEXT("Like"));
    return Options;
}

