//
//  ChatTestView.cpp
//  HIVESDKV4Tester
//
//  Created by herohjk on 2025/02/19.
//  Copyright © 2022 Epic Games, Inc. All rights reserved.
//

#include "ChatTestView.h"
#include "HIVESDKV4Tester.h"
#include "HIVESDKV4TesterGameMode.h"
#include "ChatDataManager.h"

void UChatTestView::AddLogString(FString str)
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::Chat, TEXT("%s"), *str);
}

void UChatTestView::AddListeners()
{
    class ChannelListener : public FHiveChannelListener {
    public:
        UChatTestView* Owner;
        ChannelListener(UChatTestView* InOwner) : Owner(InOwner) {}

        virtual void onNoticeMessage(const FHiveChannelNoticeMessage& message) override {
            Owner->AddLogString(TEXT("onNoticeMessage()"));
            Owner->AddLogString(message.ToString());
        }

        virtual void onChannelMessage(const FHiveChannelMessage& message) override {
            Owner->AddLogString(TEXT("onChannelMessage()"));
            Owner->AddLogString(message.ToString());
        }

        virtual void onDeletedChannel(const FHiveDeletedChannel& channel) override {
            Owner->AddLogString(TEXT("onDeletedChannel()"));
            Owner->AddLogString(channel.ToString());
        }

        virtual void onEnteredMember(const FHiveEnteredMember& member) override {
            Owner->AddLogString(TEXT("onEnteredMember()"));
            Owner->AddLogString(member.ToString());
        }

        virtual void onExitedMember(const FHiveExitedMember& member) override {
            Owner->AddLogString(TEXT("onExitedMember()"));
            Owner->AddLogString(member.ToString());
        }
        virtual void onAddReaction(const FHiveReaction& reaction) override {
            Owner->AddLogString(TEXT("onAddReaction()"));
            Owner->AddLogString(reaction.ToString());
        }
        virtual void onRemoveReaction(const FHiveReaction& reaction) override {
            Owner->AddLogString(TEXT("onRemoveReaction()"));
            Owner->AddLogString(reaction.ToString());
        }
    };

    class DirectMessageListener : public FHiveDirectMessageListener {
    public:
        UChatTestView* Owner;
        DirectMessageListener(UChatTestView* InOwner) : Owner(InOwner) {}
    public:
        virtual void onDirectMessage(const FHiveDirectMessage& message) override {
            Owner->AddLogString(TEXT("onDirectMessage()"));
            Owner->AddLogString(message.ToString());
        }
    };

    class ConnectionListener : public FHiveConnectionListener {
    public:
        UChatTestView* Owner;
        ConnectionListener(UChatTestView* InOwner) : Owner(InOwner) {}
    public:
        virtual void onConnected() override {
            Owner->AddLogString(TEXT("onConnected()"));
        }

        virtual void onReconnectStarted() override {
            Owner->AddLogString(TEXT("onReconnectStarted()"));
        }

        virtual void onReconnected(const FHiveResultAPI& result, const TArray<FString>& channelIds, const TArray<FString>& failChannelIds) override {
            Owner->AddLogString(TEXT("onReconnected()"));
            Owner->AddLogString(FString::Printf(TEXT("Result: %s"), *result.ToString()));
            Owner->AddLogString(TEXT("channelIds: ") + ConvertChannelIdsToString(channelIds));
            Owner->AddLogString(TEXT("failChannelIds: ") + ConvertChannelIdsToString(failChannelIds));
        }

        virtual void onDisconnected() override {
            Owner->AddLogString(TEXT("onDisconnected()"));
        }
    private:
        FString ConvertChannelIdsToString(const TArray<FString>& channelIds)
        {
            FString result = TEXT("[");

            for (int32 i = 0; i < channelIds.Num(); ++i)
            {
                result += channelIds[i];

                if (i != channelIds.Num() - 1)
                {
                    result += TEXT(",");
                }
            }

            result += TEXT("]");
            return result;
        }
    };

    class UserListener : public FHiveUserListener {
    public:
        UChatTestView* Owner;
        UserListener(UChatTestView* InOwner) : Owner(InOwner) {}
    public:
        virtual void onDirectMessage(const FHiveDirectMessage& message) override {
            Owner->AddLogString(TEXT("UserListener::onDirectMessage()"));
            Owner->AddLogString(message.ToString());
        }
        
        virtual void onNoticeMessage(const FHiveNoticeMessage& message) override {
            Owner->AddLogString(TEXT("UserListener::onNoticeMessage()"));
            Owner->AddLogString(message.ToString());
        }
    };

    class CustomDataListener : public FHiveCustomDataListener {
    public:
        UChatTestView* Owner;
        CustomDataListener(UChatTestView* InOwner) : Owner(InOwner) {}
    public:
        virtual void onCustomData(const FString& data) override {
            Owner->AddLogString(TEXT("CustomDataListener::onCustomData()"));
            Owner->AddLogString(data);
        }
    };

    // 리스너 생성 및 저장
    ChannelListenerInstance = new ChannelListener(this);
    ConnectionListenerInstance = new ConnectionListener(this);
    UserListenerInstance = new UserListener(this);
    CustomDataListenerInstance = new CustomDataListener(this);
    
    // 리스너 등록
    FHiveChat::AddChannelListener("ChatChannelListener", ChannelListenerInstance);
    FHiveChat::AddConnectionListener("ChatConnectionListener", ConnectionListenerInstance);
    FHiveChat::AddUserListener("ChatUserListener", UserListenerInstance);
    FHiveChat::AddCustomDataListener("ChatCustomDataListener", CustomDataListenerInstance);
}

void UChatTestView::Connect()
{
    AddLogString(TEXT("=== FHiveChat::Connect() ==="));
    if (!bSetListener)
    {
        AddLogString(TEXT("=== Add Chat Listeners(Channel, User, Connection, CustomData) ==="));
        AddListeners();
        bSetListener = true;
    }

    FHiveChat::Connect(FHiveChatOnResultDelegate::CreateLambda([this](const FHiveResultAPI& Result) {
        if (Result.IsSuccess())
        {
            AddLogString(TEXT("Connect Success"));
        }
        else
        {
            AddLogString(FString::Printf(TEXT("Connect Failed: %s"), *Result.ToString()));
        }
    }));
}

void UChatTestView::Reconnect()
{
    AddLogString(TEXT("=== FHiveChat::Reconnect() ==="));
    if (!bSetListener)
    {
        AddLogString(TEXT("=== Add Chat Listeners(Channel, User, Connection, CustomData) ==="));
        AddListeners();
        bSetListener = true;
    }
    
    FHiveChat::Reconnect(FHiveChatOnReconnectDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TArray<FString>& channelIds, const TArray<FString>& failChannelIds) {
        if (Result.IsSuccess())
        {
            AddLogString(TEXT("Reconnect Success"));
            FString ChannelIds = TEXT("");
            for (const auto& channelId : channelIds)
            {
                ChannelIds += channelId + TEXT(",");
            }
            AddLogString(TEXT("channelIds: ") + ChannelIds);
            
            FString FailChannelIds = TEXT("");
            for (const auto& channelId : failChannelIds)
            {
                FailChannelIds += channelId + TEXT(",");
            }
            AddLogString(TEXT("failChannelIds: ") + FailChannelIds);
        }
        else
        {
            AddLogString(FString::Printf(TEXT("Reconnect Failed: %s"), *Result.ToString()));
        }
    }));
}

void UChatTestView::Disconnect()
{
    AddLogString(TEXT("=== FHiveChat::Disconnect() ==="));
    FHiveChat::Disconnect(FHiveChatOnResultDelegate::CreateLambda([this](const FHiveResultAPI& Result) {
        if (Result.IsSuccess())
        {
            AddLogString(TEXT("Disconnect Success"));
        }
        else
        {
            AddLogString(FString::Printf(TEXT("Disconnect Failed: %s"), *Result.ToString()));
        }
    }));
}

void  UChatTestView::IsConnected()
{
    AddLogString(TEXT("=== FHiveChat::IsConnected() ==="));
    bool isConnected = FHiveChat::IsConnected();
    AddLogString(FString::Printf(TEXT("IsConnected: %s"), *FString(isConnected ? TEXT("true") : TEXT("false"))));
}

void UChatTestView::GetChannelInfo(FString channelId)
{
    AddLogString(FString::Printf(TEXT("=== FHiveChat::GetChannelInfo(%s) ==="), *channelId));
    FHiveChat::GetChannelInfo(channelId, FHiveChatOnGetChannelInfoDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FHiveChannel& channel, const TArray<FHiveMember>& members) {
        if (Result.IsSuccess())
        {
            AddLogString(channel.ToString());
            AddLogString(TEXT("Members:"));
            for (const FHiveMember& Member : members)
            {
                AddLogString(Member.ToString());
            }
        }
        else
        {
            AddLogString(FString::Printf(TEXT("GetChannelInfo Failed: %s"), *Result.ToString()));
        }
    }));
}

void UChatTestView::GetChannelMembers(FString channelId)
{
    AddLogString(FString::Printf(TEXT("=== FHiveChat::GetChannelMembers(%s) ==="), *channelId));
    FHiveChat::GetChannelMembers(channelId, FHiveChatOnGetChannelMembersDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TArray<FHiveMember>& members) {
        if (Result.IsSuccess())
        {
            AddLogString(TEXT("Members:"));
            for (const FHiveMember& Member : members)
            {
                AddLogString(Member.ToString());
            }
        }
        else
        {
            AddLogString(FString::Printf(TEXT("GetChannelMembers Failed: %s"), *Result.ToString()));
        }
    }));
}

void UChatTestView::DeleteChannel(FString channelId)
{
    AddLogString(FString::Printf(TEXT("=== FHiveChat::DeleteChannel(%s) ==="), *channelId));
    FHiveChat::DeleteChannel(channelId, FHiveChatOnResultDelegate::CreateLambda([this](const FHiveResultAPI& Result) {
        if (Result.IsSuccess())
        {
            AddLogString(FString::Printf(TEXT("DeleteChannel Success")));
        }
        else
        {
            AddLogString(FString::Printf(TEXT("DeleteChannel Failed: %s"), *Result.ToString()));
        }
    }));
}

void UChatTestView::EnterChannel(FString channelId, FString password)
{
    AddLogString(FString::Printf(TEXT("=== FHiveChat::EnterChannel(%s, %s) ==="), *channelId, *password));

    FHiveEnterChannelParams params = FHiveEnterChannelParams();
    params.channelId = channelId;
    params.password = password;

    FHiveChat::EnterChannel(params, FHiveChatOnResultDelegate::CreateLambda([this](const FHiveResultAPI& Result) {
        if (Result.IsSuccess())
        {
            AddLogString(FString::Printf(TEXT("EnterChannel Success")));
        }
        else
        {
            AddLogString(FString::Printf(TEXT("EnterChannel Failed: %s"), *Result.ToString()));
        }
    }));
}

void UChatTestView::ExitChannel(FString channelId)
{
    AddLogString(FString::Printf(TEXT("=== FHiveChat::ExitChannel(%s) ==="), *channelId));
    FHiveChat::ExitChannel(channelId, FHiveChatOnResultDelegate::CreateLambda([this](const FHiveResultAPI& Result) {
        if (Result.IsSuccess())
        {
            AddLogString(FString::Printf(TEXT("ExitChannel Success")));
        }
        else
        {
            AddLogString(FString::Printf(TEXT("ExitChannel Failed: %s"), *Result.ToString()));
        }
    }));
}

void UChatTestView::GetChannelsByUser()
{
    AddLogString(FString::Printf(TEXT("=== FHiveChat::GetChannelsByUser() ===")));
    FHiveChat::GetChannelsByUser(FHiveChatOnGetChannelsByUserDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TArray<FHiveChannel>& channels) {
        if (Result.IsSuccess())
        {
            AddLogString(TEXT("Channels:"));
            for (const FHiveChannel& Channel : channels)
            {
                AddLogString(Channel.ToString());
            }
        }
        else
        {
            AddLogString(FString::Printf(TEXT("GetChannelsByUser Failed: %s"), *Result.ToString()));
        }
    }));
}

void UChatTestView::GetBlockMembers()
{
    AddLogString(FString::Printf(TEXT("=== FHiveChat::GetBlockMembers() ===")));
    FHiveChat::GetBlockMembers(FHiveChatOnGetBlockMembersDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TArray<FHiveBlockMember>& blockMembers) {
        if (Result.IsSuccess())
        {
            AddLogString(TEXT("Block Members:"));
            for (const FHiveBlockMember& Member : blockMembers)
            {
                AddLogString(Member.ToString());
            }
        }
        else
        {
            AddLogString(FString::Printf(TEXT("GetBlockMembers Failed: %s"), *Result.ToString()));
        }
    }));

}

void UChatTestView::BlockMember(FString blockPlayerIdStr)
{
    AddLogString(FString::Printf(TEXT("=== FHiveChat::BlockMember(%s) ==="), *blockPlayerIdStr));
    TCHAR* end;
    int64 blockPlayerId = FCString::Strtoi64(*blockPlayerIdStr, &end, 10);
    FHiveChat::BlockMember(blockPlayerId, FHiveChatOnResultDelegate::CreateLambda([this](const FHiveResultAPI& Result) {
        if (Result.IsSuccess())
        {
            AddLogString(FString::Printf(TEXT("BlockMember Success")));
        }
        else
        {
            AddLogString(FString::Printf(TEXT("BlockMember Failed: %s"), *Result.ToString()));
        }
    }));
}

void UChatTestView::UnblockMember(FString blockPlayerIdStr)
{
    AddLogString(FString::Printf(TEXT("=== FHiveChat::UnblockMember(%s) ==="), *blockPlayerIdStr));
    TCHAR* end;
    int64 blockPlayerId = FCString::Strtoi64(*blockPlayerIdStr, &end, 10);
    FHiveChat::UnblockMember(blockPlayerId, FHiveChatOnResultDelegate::CreateLambda([this](const FHiveResultAPI& Result) {
        if (Result.IsSuccess())
        {
            AddLogString(FString::Printf(TEXT("UnblockMember Success")));
        }
        else
        {
            AddLogString(FString::Printf(TEXT("UnblockMember Failed: %s"), *Result.ToString()));
        }
    }));
}

void UChatTestView::SendMessageToChannelWithParams()
{
    UChatDataManager* DataManager = UChatDataManager::GetInstance();
    FHiveChannelSendMessageParams Params;
    Params.channelId = DataManager->GetSendMessageChannelId();
    Params.message = DataManager->GetSendMessageMessage();
    Params.extraData = DataManager->GetSendMessageExtraData();
    Params.replyMessageId = DataManager->GetSendMessageReplyMessageId();
    
    // MentionedPlayerIds 파싱
    FString MentionedIds = DataManager->GetSendMessageMentionedPlayerIds();
    
    AddLogString(FString::Printf(TEXT("=== FHiveChat::SendMessageWithChannelSendMessageParams(channelId: %s, message: %s, extraData: %s, replyMessageId: %s, mentionedPlayerIds: %s) ==="), 
        *Params.channelId, *Params.message, *Params.extraData, *Params.replyMessageId, *MentionedIds));
    if (!MentionedIds.IsEmpty())
    {
        TArray<FString> IdStrings;
        MentionedIds.ParseIntoArray(IdStrings, TEXT(","));
        for (const FString& IdStr : IdStrings)
        {
            FString TrimmedId = IdStr.TrimStartAndEnd();
            if (!TrimmedId.IsEmpty())
            {
                Params.mentionedPlayerIds.Add(FCString::Atoi64(*TrimmedId));
            }
        }
    }
    
    FHiveChat::SendMessageWithChannelSendMessageParams(Params, 
        FHiveChatOnResultDelegate::CreateLambda([this](const FHiveResultAPI& Result) {
        if (Result.IsSuccess())
        {
            AddLogString(TEXT("SendMessageWithParams Success"));
        }
        else
        {
            AddLogString(FString::Printf(TEXT("SendMessageWithParams Failed: %s"), *Result.ToString()));
        }
    }));
}

void UChatTestView::SendMessageToPlayerWithParams()
{
    UChatDataManager* DataManager = UChatDataManager::GetInstance();
    FHiveDirectSendMessageParams Params;
    Params.toPlayerId = DataManager->GetSendMessagePlayerId();
    Params.message = DataManager->GetSendMessageMessage();
    Params.extraData = DataManager->GetSendMessageExtraData();
    
    AddLogString(FString::Printf(TEXT("=== FHiveChat::SendMessageWithDirectSendMessageParams(toPlayerId: %lld, message: %s, extraData: %s) ==="), 
        Params.toPlayerId, *Params.message, *Params.extraData));
    
    FHiveChat::SendMessageWithDirectSendMessageParams(Params, 
        FHiveChatOnResultDelegate::CreateLambda([this](const FHiveResultAPI& Result) {
        if (Result.IsSuccess())
        {
            AddLogString(TEXT("SendMessageToPlayerWithParams Success"));
        }
        else
        {
            AddLogString(FString::Printf(TEXT("SendMessageToPlayerWithParams Failed: %s"), *Result.ToString()));
        }
    }));
}

void UChatTestView::CreateChannelWithParams()
{
    UChatDataManager* DataManager = UChatDataManager::GetInstance();
    FHiveCreateChannelParams Params;
    Params.channelId = DataManager->GetCreateChannelChannelId();
    Params.password = DataManager->GetCreateChannelPassword();
    Params.channelName = DataManager->GetCreateChannelChannelName();
    Params.maxMemberCount = DataManager->GetCreateChannelMaxMemberCount();
    Params.type = DataManager->GetCreateChannelChannelType();
    Params.chatHistoryAllowed = DataManager->GetCreateChannelChatHistoryAllowed();
    
    AddLogString(FString::Printf(TEXT("=== FHiveChat::CreateChannel(channelId: %s, password: %s, channelName: %s, maxMemberCount: %d, type: %s, chatHistoryAllowed: %s) ==="), 
        *Params.channelId, *Params.password, *Params.channelName, Params.maxMemberCount, *GetChannelTypeString(Params.type), Params.chatHistoryAllowed ? TEXT("true") : TEXT("false")));
    
    FHiveChat::CreateChannel(Params, FHiveChatOnResultDelegate::CreateLambda([this](const FHiveResultAPI& Result) {
        if (Result.IsSuccess())
        {
            AddLogString(TEXT("CreateChannelWithParams Success"));
        }
        else
        {
            AddLogString(FString::Printf(TEXT("CreateChannelWithParams Failed: %s"), *Result.ToString()));
        }
    }));
}

void UChatTestView::GetChannelsWithParams()
{
    UChatDataManager* DataManager = UChatDataManager::GetInstance();
    FHiveGetChannelsParams Params;
    Params.type = DataManager->GetGetChannelsChannelType();
    Params.channelId = DataManager->GetGetChannelsChannelId();
    Params.channelName = DataManager->GetGetChannelsChannelName();
    Params.sort = DataManager->GetGetChannelsSortType();
    Params.pageOrder = DataManager->GetGetChannelsPageOrder() == TEXT("NONE") ? TEXT("DESC") : DataManager->GetGetChannelsPageOrder();
    Params.pageSize = DataManager->GetGetChannelsPageSize() < 1 ? 10 : DataManager->GetGetChannelsPageSize();
    Params.pageNumber = DataManager->GetGetChannelsPageNumber() < 1 ? 1 : DataManager->GetGetChannelsPageNumber();
    
    AddLogString(FString::Printf(TEXT("=== FHiveChat::GetChannels(type: %s, channelId: %s, channelName: %s, sort: %s, pageOrder: %s, pageSize: %d, pageNumber: %d) ==="), 
        *GetChannelTypeString(Params.type), *Params.channelId, *Params.channelName, *GetSortTypeString(Params.sort), *Params.pageOrder, Params.pageSize, Params.pageNumber));
    
    FHiveChat::GetChannels(Params, FHiveChatOnGetChannelsDelegate::CreateLambda([this](const FHiveResultAPI& Result, 
        const TArray<FHiveChannel>& Channels, const FHiveChannelPage& PageInfo) {
        if (Result.IsSuccess())
        {
            AddLogString(TEXT("GetChannelsWithParams Success"));
            AddLogString(FString::Printf(TEXT("PageInfo: Size=%d, Current=%d, Total=%d, TotalElements=%lld"), 
                PageInfo.size, PageInfo.currentPage, PageInfo.totalPages, PageInfo.totalElements));
            
            for (const FHiveChannel& Channel : Channels)
            {
                AddLogString(Channel.ToString());
            }
        }
        else
        {
            AddLogString(FString::Printf(TEXT("GetChannelsWithParams Failed: %s"), *Result.ToString()));
        }
    }));
}

void UChatTestView::QueryChannelMessages(FString channelId)
{
    AddLogString(FString::Printf(TEXT("=== Channel::Query(%s) ==="), *channelId));
    
    UChatDataManager* DataManager = UChatDataManager::GetInstance();
    FString MessageId = DataManager->GetChannelQueryMessageId();
    FString Order = DataManager->GetChannelQueryOrder();
    int32 PrevSize = DataManager->GetChannelQueryPrevSize();
    int32 NextSize = DataManager->GetChannelQueryNextSize();
    
    AddLogString(FString::Printf(TEXT("Query Params - messageId: %s, order: %s, prevSize: %d, nextSize: %d"), 
        *MessageId, *Order, PrevSize, NextSize));
    
    // 먼저 채널 정보를 가져옴
    FHiveChat::GetChannelInfo(*channelId, FHiveChatOnGetChannelInfoDelegate::CreateLambda([this, channelId](
        const FHiveResultAPI& Result, const FHiveChannel& Channel, const TArray<FHiveMember>& Members) {
        if (Result.IsSuccess())
        {
            UChatDataManager* DataManager = UChatDataManager::GetInstance();
            FHiveChannelMessageListQueryParams QueryParams;
            
            int32 PrevSize = DataManager->GetChannelQueryPrevSize();
            if (PrevSize > 0) QueryParams.prevSize = PrevSize;
            
            int32 NextSize = DataManager->GetChannelQueryNextSize();
            if (NextSize > 0) QueryParams.nextSize = NextSize;
            
            FString MessageId = DataManager->GetChannelQueryMessageId();
            if (!MessageId.IsEmpty()) QueryParams.messageId = MessageId;
            
            FString Order = DataManager->GetChannelQueryOrder();
            if (!Order.IsEmpty() && Order != TEXT("NONE")) QueryParams.order = Order;
            
            // 채널 객체의 Query 메서드 호출
            Channel.Query(QueryParams, 
                FHiveChannelQueryDelegate([this](const FHiveResultAPI& Result, 
                    const FHiveChannelMessageListQueryResponse& Response) {
                if (Result.IsSuccess())
                {
                    AddLogString(TEXT("Query Success"));
                    AddLogString(FString::Printf(TEXT("HasNext: %s, NextMessageId: %s"), 
                        Response.hasNext ? TEXT("true") : TEXT("false"), *Response.nextMessageId));
                    AddLogString(FString::Printf(TEXT("HasPrev: %s, PrevMessageId: %s"), 
                        Response.hasPrev ? TEXT("true") : TEXT("false"), *Response.prevMessageId));
                    
                    for (const FHiveChannelMessage& Message : Response.content)
                    {
                        AddLogString(Message.ToString());
                    }
                }
                else
                {
                    AddLogString(FString::Printf(TEXT("Query Failed: %s"), *Result.ToString()));
                }
            }));
        }
        else
        {
            AddLogString(FString::Printf(TEXT("GetChannelInfo Failed: %s"), *Result.ToString()));
        }
    }));
}

void UChatTestView::SetChannelTranslationEnabled(FString channelId, bool enabled)
{
    AddLogString(FString::Printf(TEXT("=== Channel::SetTranslationEnabled(%s, %s) ==="), 
        *channelId, enabled ? TEXT("true") : TEXT("false")));
    
    // 먼저 채널 정보를 가져옴
    FHiveChat::GetChannelInfo(*channelId, FHiveChatOnGetChannelInfoDelegate::CreateLambda([this, channelId, enabled](
        const FHiveResultAPI& Result, const FHiveChannel& Channel, const TArray<FHiveMember>& Members) {
        if (Result.IsSuccess())
        {
            Channel.SetTranslationEnabled(enabled, FHiveChannelTranslationSettingDelegate([this](const FHiveResultAPI& Result) {
                if (Result.IsSuccess())
                {
                    AddLogString(TEXT("SetTranslationEnabled Success"));
                }
                else
                {
                    AddLogString(FString::Printf(TEXT("SetTranslationEnabled Failed: %s"), *Result.ToString()));
                }
            }));
        }
        else
        {
            AddLogString(FString::Printf(TEXT("GetChannelInfo Failed: %s"), *Result.ToString()));
        }
    }));
}

void UChatTestView::TranslateMessage()
{
    UChatDataManager* DataManager = UChatDataManager::GetInstance();
    FString ActualMessage = DataManager->GetTranslateMessage();
    FString ActualLanguages = DataManager->GetTranslateTargetLanguages();
    
    AddLogString(FString::Printf(TEXT("=== FHiveChat::Translate(message: %s, targetLanguages: %s) ==="), 
        *ActualMessage, *ActualLanguages));
    
    FHiveTranslateParams Params;
    Params.message = ActualMessage;
    
    // 타겟 언어 파싱
    TArray<FString> LanguageArray;
    ActualLanguages.ParseIntoArray(LanguageArray, TEXT(","));
    for (FString& Lang : LanguageArray)
    {
        Params.targetLanguage.Add(Lang.TrimStartAndEnd());
    }
    
    FHiveChat::Translate(Params, FHiveChatOnTranslateDelegate::CreateLambda([this](const FHiveResultAPI& Result, 
        const FHiveTranslationData& TranslationData) {
        if (Result.IsSuccess())
        {
            AddLogString(TEXT("Translate Success"));
            for (const auto& Translation : TranslationData.translations)
            {
                AddLogString(FString::Printf(TEXT("%s: %s"), *Translation.Key, *Translation.Value));
            }
        }
        else
        {
            AddLogString(FString::Printf(TEXT("Translate Failed: %s"), *Result.ToString()));
        }
    }));
}

void UChatTestView::QueryChannelWithParams(FString channelId)
{
    // QueryChannelMessages와 동일한 기능 수행
    QueryChannelMessages(channelId);
}

void UChatTestView::AddReactionWithParams(FString channelId)
{
    UChatDataManager* DataManager = UChatDataManager::GetInstance();
    FString MessageId = DataManager->GetReactionMessageId();
    
    // EHiveReactionType을 FString으로 변환 (HIVE SDK 제공 함수 사용)
    EHiveReactionType ReactionType = DataManager->GetReactionType();
    FString ReactionTypeStr = EHiveReactionTypeToString(ReactionType);

    AddLogString(FString::Printf(TEXT("=== Channel::AddReaction(%s, %s, %s) ==="), 
        *channelId, *MessageId, *ReactionTypeStr));
    
    // 먼저 채널 정보를 가져옴
    FHiveChat::GetChannelInfo(*channelId, FHiveChatOnGetChannelInfoDelegate::CreateLambda([this, channelId, MessageId, ReactionType](
        const FHiveResultAPI& Result, const FHiveChannel& Channel, const TArray<FHiveMember>& Members) {
        if (Result.IsSuccess())
        {
            Channel.AddReaction(*MessageId, ReactionType, 
                FHiveChannelReactionDelegate([this](const FHiveResultAPI& Result, EHiveReactionType Type) {
                if (Result.IsSuccess())
                {
                    AddLogString(TEXT("AddReaction Success"));
                }
                else
                {
                    AddLogString(FString::Printf(TEXT("AddReaction Failed: %s"), *Result.ToString()));
                }
            }));
        }
        else
        {
            AddLogString(FString::Printf(TEXT("GetChannelInfo Failed: %s"), *Result.ToString()));
        }
    }));
}

void UChatTestView::RemoveReactionWithParams(FString channelId)
{
    UChatDataManager* DataManager = UChatDataManager::GetInstance();
    FString MessageId = DataManager->GetReactionMessageId();
    
    // EHiveReactionType을 FString으로 변환 (HIVE SDK 제공 함수 사용)
    EHiveReactionType ReactionType = DataManager->GetReactionType();
    FString ReactionTypeStr = EHiveReactionTypeToString(ReactionType);
    
    AddLogString(FString::Printf(TEXT("=== Channel::RemoveReaction(%s, %s, %s) ==="), 
        *channelId, *MessageId, *ReactionTypeStr));
    
    // 먼저 채널 정보를 가져옴
    FHiveChat::GetChannelInfo(*channelId, FHiveChatOnGetChannelInfoDelegate::CreateLambda([this, channelId, MessageId, ReactionType](
        const FHiveResultAPI& Result, const FHiveChannel& Channel, const TArray<FHiveMember>& Members) {
        if (Result.IsSuccess())
        {
            Channel.RemoveReaction(*MessageId, ReactionType, 
                FHiveChannelReactionDelegate([this](const FHiveResultAPI& Result, EHiveReactionType Type) {
                if (Result.IsSuccess())
                {
                    AddLogString(TEXT("RemoveReaction Success"));
                }
                else
                {
                    AddLogString(FString::Printf(TEXT("RemoveReaction Failed: %s"), *Result.ToString()));
                }
            }));
        }
        else
        {
            AddLogString(FString::Printf(TEXT("GetChannelInfo Failed: %s"), *Result.ToString()));
        }
    }));
}

EHiveChannelType UChatTestView::FStringToChannelType(const FString& TypeStr)
{
    if (TypeStr == TEXT("Public") || TypeStr == TEXT("PUBLIC")) return EHiveChannelType::PUBLIC;
    if (TypeStr == TEXT("Private") || TypeStr == TEXT("PRIVATE")) return EHiveChannelType::PRIVATE;
    if (TypeStr == TEXT("Group") || TypeStr == TEXT("GROUP")) return EHiveChannelType::GROUP;
    return EHiveChannelType::PUBLIC; // 기본값
}

FString UChatTestView::GetChannelTypeString(EHiveChannelType Type)
{
    switch (Type)
    {
        case EHiveChannelType::PUBLIC:
            return TEXT("PUBLIC");
        case EHiveChannelType::PRIVATE:
            return TEXT("PRIVATE");
        case EHiveChannelType::GROUP:
            return TEXT("GROUP");
        case EHiveChannelType::UNKNOWN:
        default:
            return TEXT("UNKNOWN");
    }
}

FString UChatTestView::GetSortTypeString(EHiveSortType Type)
{
    switch (Type)
    {
        case EHiveSortType::CHANNEL_ID:
            return TEXT("CHANNEL_ID");
        case EHiveSortType::CHANNEL_NAME:
            return TEXT("CHANNEL_NAME");
        case EHiveSortType::REG_TIME:
            return TEXT("REG_TIME");
        case EHiveSortType::UNKNOWN:
        default:
            return TEXT("UNKNOWN");
    }
}
