/* Copyright © 2024 Com2uS Platform Corp. All Rights Reserved. */
#include "ConfigurationDialog.h"
#include "HIVESDKV4Tester.h"
#include <sstream>
#include "Json.h"
#include "HIVESDKV4TesterGameMode.h"
#include "HiveTestUtils.h"

UConfigurationDialog * UConfigurationDialog ::ConstructNewObject()
{
    UConfigurationDialog * configDialog = NewObject<UConfigurationDialog>();
    UE_LOG(LogTemp, Warning, TEXT("UConfigurationDialog ConstructNewObject!.........."));

    picojson::value jsonParam;
    
    //read from config.xml
    configDialog->ResetConfig();
    
    FString LanuguageListArr[] = {
        TEXT("English,en"),
        TEXT("Deutsch,de"),
        TEXT("español,es"),
        TEXT("français,fr"),
        TEXT("Bahasa ID,id"),
        TEXT("italiano,it"),
        TEXT("日本語,ja"),
        TEXT("한국어,ko"),
        TEXT("português,pt"),
        TEXT("русский,ru"),
        TEXT("ไทย,th"),
        TEXT("Türkçe,tr"),
        TEXT("Tiếng Việt,vi"),
        TEXT("简体中文,zh-Hans"),
        TEXT("繁體中文,zh-Hant"),
        TEXT("아랍어,ar"),
        TEXT("Νέα Ελληνικά (not support),gl"),
        TEXT("Not set,null")
    };
    configDialog->LanuguageList.Append(LanuguageListArr, UE_ARRAY_COUNT(LanuguageListArr));

    
    //read from Property.
    //read file
    FString readText;
    FString SavedConfigFile = FPaths::Combine(*FPaths::ProjectDir(), *FString("HIVESDKV4ConfigSaved.txt"));
    bool loadFileResult = FFileHelper::LoadFileToString(readText,*SavedConfigFile);
    if (loadFileResult) {
        UE_LOG(LogTemp, Warning, TEXT("UConfigurationDialog file Test to Read : %s\n"), *readText);
        TSharedPtr<FJsonObject> jsonConfigObj;
        TSharedRef< TJsonReader<> > reader = TJsonReaderFactory<>::Create(readText);

        if (FJsonSerializer::Deserialize(reader, jsonConfigObj) == false) {
            //fail to read json
            UE_LOG(LogTemp, Warning, TEXT("UConfigurationDialog FJsonSerializer::Deserialize fail"));

        }
        else
        {
            if (jsonConfigObj->HasField(TEXT("appId")))
            {
                configDialog->mStrAppId = jsonConfigObj->GetStringField(TEXT("appId"));
            }
            if (jsonConfigObj->HasField(TEXT("zone")))
            {
                configDialog->mStrZone = jsonConfigObj->GetStringField(TEXT("zone"));
            }
            if (jsonConfigObj->HasField(TEXT("gameLanguage")))
            {
                configDialog->mStrLanguage = jsonConfigObj->GetStringField(TEXT("gameLanguage"));
            }
            if (jsonConfigObj->HasField(TEXT("serverId")))
            {
                configDialog->mStrServerID = jsonConfigObj->GetStringField(TEXT("serverId"));
            }
            if (jsonConfigObj->HasField(TEXT("useLog")))
            {
                configDialog->mBuseLog = jsonConfigObj->GetBoolField(TEXT("useLog"));
            }
            if (jsonConfigObj->HasField(TEXT("hiveTheme")))
            {
                configDialog->mStrHiveTheme = jsonConfigObj->GetStringField(TEXT("hiveTheme"));
            }
            if (jsonConfigObj->HasField(TEXT("hiveOrientation")))
            {
                configDialog->mStrHiveOrientation = jsonConfigObj->GetStringField(TEXT("hiveOrientation"));
            }
            if (jsonConfigObj->HasField(TEXT("channel")))
            {
                configDialog->mStrChannel = jsonConfigObj->GetStringField(TEXT("channel"));
            }
            if (jsonConfigObj->HasField(TEXT("market")))
            {
                configDialog->mStrMarket = jsonConfigObj->GetStringField(TEXT("market"));
            }
            if (jsonConfigObj->HasField(TEXT("usePrivateBrowsingForAuth")))
            {
                configDialog->mUsePrivateBrowsingForAuth = jsonConfigObj->GetBoolField(TEXT("usePrivateBrowsingForAuth"));
            }
            configDialog->mUseAgeGate = FHiveConfiguration::GetAgeGateU13();
            configDialog->mUseHercules = true;  //  default True.
            configDialog->mCrossPromotionBannerShowing = true;
            configDialog->mEnableGameController = FHiveConfiguration::GetEnableGameController();
        }
        if(jsonConfigObj->HasField(TEXT("hiveCommunityUrl")))
        {
            configDialog->mStrHiveCommunityUrl = jsonConfigObj->GetStringField(TEXT("hiveCommunityUrl"));
        }
        if(jsonConfigObj->HasField(TEXT("logType")))
        {
            configDialog->mStrLogType = jsonConfigObj->GetStringField(TEXT("logType"));
        }
    }
    return configDialog;
}
FString version;
FString UConfigurationDialog::SDKGetVersion()
{
#if !WITH_EDITOR
#if PLATFORM_IOS
	dispatch_sync(dispatch_get_main_queue(), ^{
#endif
    version = Configuration::getHiveSDKVersion().c_str();
    version.Append(TEXT("\n"));
#if PLATFORM_ANDROID
    std::string refVersion = Configuration::getReferenceSDKVersion();
    std::vector<std::string> refVersions = split(refVersion, '\n');
    std::string filteredReferenceVersion = getAndroidExternalLibraryNames(refVersions);
    
    version.Append(filteredReferenceVersion.c_str());
#else
    version.Append(Configuration::getReferenceSDKVersion().c_str());
#endif
    UE_LOG(LogTemp, Warning, TEXT("UConfigurationDialog Construction!.......... %s"), *version);
#if PLATFORM_IOS
	});
#endif
#endif
    return version;
}

std::vector<std::string> UConfigurationDialog::split(const std::string& input, char delimiter) {
    std::vector<std::string> result;
    std::stringstream ss(input);
    std::string temp;
    
    while (getline(ss, temp, delimiter)) {
        result.push_back(temp);
    }
    return result;
}

std::string UConfigurationDialog::getAndroidExternalLibraryNames(const std::vector<std::string>& input) {
    std::string result;
    
    for (const auto& str : input) {
        if (
            // Analytics
            str.find("com.adjust.sdk") != std::string::npos ||
            str.find("com.appsflyer") != std::string::npos ||
            str.find("com.google.firebase:firebase-analytics") != std::string::npos ||
            str.find("com.singular.sdk") != std::string::npos ||
            str.find("com.google.android.recaptcha:recaptcha") != std::string::npos ||
            // Auth
            str.find("facebook-core") != std::string::npos ||
            str.find("play-services-games-v2") != std::string::npos ||
            str.find("credentials-play-services-auth") != std::string::npos ||
            str.find("service.name:hwid") != std::string::npos ||
            str.find("com.linecorp.linesdk") != std::string::npos ||
            str.find("com.tencent.connect.common") != std::string::npos ||
            str.find("vk:android-sdk-core") != std::string::npos ||
            str.find("wechat-sdk-android-without-mta") != std::string::npos ||
            // IAP
            str.find("amazon-appstore-sdk") != std::string::npos ||
            str.find("billingclient:billing") != std::string::npos ||
            str.find("hms:iap") != std::string::npos ||
            str.find("com.onestorecorp.sdk") != std::string::npos ||
            str.find("com.samsung.android.sdk.iap") != std::string::npos
            
            ) {
            result += str;
            result += "\n";
        }
    }
    
    return  result;
}

FString UConfigurationDialog::GetAdvertisingID()
{
    return TEXT("");
}

void UConfigurationDialog::SaveConfig()
{
    //Set Configuration
    
    // 설정 대화 상자가 닫힐 때 설정한 내용을 설정 파일에 기록한다.

    UE_LOG(LogTemp, Warning, TEXT("######### UConfigurationDialog Before Saveing Json String. : %s"), *mStrServerID);
    
    FString SavedConfigFile = FPaths::Combine(*FPaths::ProjectDir(), *FString("HIVESDKV4ConfigSaved.txt"));
    TSharedPtr<FJsonObject> arg = MakeShareable(new FJsonObject);
    arg->SetStringField(TEXT("appId"), mStrAppId);
    arg->SetStringField(TEXT("zone"), mStrZone);
    arg->SetStringField(TEXT("gameLanguage"), mStrLanguage);
    arg->SetStringField(TEXT("serverId"), mStrServerID);
    arg->SetBoolField(TEXT("useLog"), mBuseLog);
    arg->SetStringField(TEXT("hiveTheme"), mStrHiveTheme);
    arg->SetStringField(TEXT("hiveOrientation"), mStrHiveOrientation);
    arg->SetStringField(TEXT("hiveCommunityUrl"), mStrHiveCommunityUrl);
    arg->SetStringField(TEXT("logType"), mStrLogType);
    arg->SetStringField(TEXT("channel"), mStrChannel);
    arg->SetStringField(TEXT("market"), mStrMarket);
    arg->SetBoolField(TEXT("usePrivateBrowsingForAuth"), mUsePrivateBrowsingForAuth);
    //  skip hiveUseSecurity    //  arg->SetBoolField
    FString OutputString;
    TSharedRef< TJsonWriter<> > Writer = TJsonWriterFactory<>::Create(&OutputString);
    FJsonSerializer::Serialize(arg.ToSharedRef(), Writer);
    
    UE_LOG(LogTemp, Warning, TEXT("######### UConfigurationDialog Before Saveing Json String. : %s"), *OutputString);
    FFileHelper::SaveStringToFile(OutputString, *SavedConfigFile);
    
#if !WITH_EDITOR
    // SetAppId
    FHiveConfiguration::SetAppId(mStrAppId);
    
    // SetZone
    if (mStrZone.Equals(TEXT("TEST"), ESearchCase::IgnoreCase)) {
        HiveTestUtils::ConfigurationSetTestZone();
    }
    else if (mStrZone.Equals(TEXT("SANDBOX"), ESearchCase::IgnoreCase)) {
        FHiveConfiguration::SetZone(EHiveZoneType::SANDBOX);
    }
    else {
        FHiveConfiguration::SetZone(EHiveZoneType::REAL);
    }
    
    // Set GameLanguage
    FString langCode;
    mStrLanguage.Split(TEXT(","), nullptr, &langCode);
    
    if (!langCode.Equals("null", ESearchCase::IgnoreCase))
    {
        FHiveConfiguration::SetGameLanguage(langCode);
    }
    
    // Set UseLog
    FHiveConfiguration::SetUseLog(mBuseLog);
    
    // Set ServerId
    FHiveConfiguration::SetServerId(mStrServerID);
    
    // Set AgeGateU13
    FHiveConfiguration::SetAgeGateU13(mUseAgeGate);
    
    // Set UseHercules
    FHiveConfiguration::SetUseHercules(mUseHercules);

    SetHiveTheme(mStrHiveTheme);
    SetHiveOrientation(mStrHiveOrientation);
    SetHiveCommunityUrl(mStrHiveCommunityUrl);
    SetLogFilter(mStrLogType);
    SetChannel(mStrChannel);
    SetMarket(mStrMarket);

    // Set Cross Promotion Banner Showing
    FHiveConfiguration::SetCrossPromotionBannerShowing(mCrossPromotionBannerShowing);
	
	//	PASS GAMEMODE CONFIG
	AHIVESDKV4TesterGameMode::getGameModeInstance()->modeStrLanguage = langCode;
	AHIVESDKV4TesterGameMode::getGameModeInstance()->modeStrServerID = mStrServerID;

    // Set UsePrivateBrowsingForAuth
    FHiveConfiguration::SetUsePrivateBrowsingForAuth(mUsePrivateBrowsingForAuth);

    // Set EnableGameController
    FHiveConfiguration::SetEnableGameController(mEnableGameController);

    UE_LOG(LogTemp, Warning, TEXT("this lang code : %s"), *langCode);
#endif
}


void UConfigurationDialog::ResetConfig()
{
    picojson::value jsonParam;
#if PLATFORM_WINDOWS || PLATFORM_MAC
    mStrZone = TEXT("TEST");
#else
    mStrZone = TEXT("SANDBOX");
#endif
#if !WITH_EDITOR
        std::string configJsonString = Configuration::getConfiguration();
        std::string err = picojson::parse(jsonParam, configJsonString);
        if(!err.empty())
        {
            //        return;
        }
        mStrAppId = jsonParam.get("appId").to_str().c_str();
        mStrZone = jsonParam.get("zone").to_str().c_str();
        mBuseLog = jsonParam.get("useLog").get<bool>();
#endif
    // TEST zone 확인 - getConfiguration이 TEST를 반환하지 않을 수 있으므로 직접 확인  // same as Unity
    if (HiveTestUtils::ConfigurationIsTestZone()) {       
        mStrZone = TEXT("TEST");
    }

    //setup Default value
#if !PLATFORM_WINDOWS
    mStrLanguage=TEXT("한국어,ko");
#else
    //FSlateApplication::IsInitialized() ? (FWindowsTextInputMethodSystem*)(FSlateApplication::Get().
        FString defaultLanguage
#if !WITH_EDITOR
            = jsonParam.get("gameLanguage").to_str().c_str();
#else
            = TEXT("한국어,ko");
#endif
    UE_LOG(LogTemp, Warning, TEXT("######### FGenericPlatformMisc::GetDefaultLanguage. : %s"), *defaultLanguage);
    if (defaultLanguage.Len() > 0) {
        UE_LOG(LogTemp, Warning, TEXT("######### FGenericPlatformMisc::GetDefaultLanguage. : %s"), *defaultLanguage.Left(2));
        if (defaultLanguage.Contains("zh")) {;
            if (defaultLanguage.Contains("CN") || defaultLanguage.Contains("hans") || defaultLanguage.Contains("Hans")) {
                defaultLanguage = TEXT("zh-Hans");
            }
            else {
                defaultLanguage = TEXT("zh-Hant");
            }
        }
        else {
            defaultLanguage = defaultLanguage.Left(2);
        }
    }
    TArray<FString> StrLanguageArr;
    FString LanuguageListArr[] = {
        TEXT("English,en"),
        TEXT("Deutsch,de"),
        TEXT("español,es"),
        TEXT("français,fr"),
        TEXT("Bahasa ID,id"),
        TEXT("italiano,it"),
        TEXT("日本語,ja"),
        TEXT("한국어,ko"),
        TEXT("português,pt"),
        TEXT("русский,ru"),
        TEXT("ไทย,th"),
        TEXT("Türkçe,tr"),
        TEXT("Tiếng Việt,vi"),
        TEXT("简体中文,zh-Hans"),
        TEXT("繁體中文,zh-Hant"),
        TEXT("아랍어,ar"),
        TEXT("Νέα Ελληνικά (not support),gl"),
        TEXT("Not set,null")
    };
    mStrLanguage = TEXT("English,en");
    StrLanguageArr.Append(LanuguageListArr, UE_ARRAY_COUNT(LanuguageListArr));
    for (auto It = StrLanguageArr.CreateConstIterator(); It; ++It) {
        if ((*It).Contains(defaultLanguage)) {
            mStrLanguage = *It;
        }
    }
#endif
    mStrServerID = TEXT("KR");
    mStrHiveOrientation = TEXT("all");
    mStrHiveCommunityUrl = TEXT("");
    mStrLogType = TEXT("Verbose");
    mUseHercules = true;
    mCrossPromotionBannerShowing = true;
    mUsePrivateBrowsingForAuth = true;
    mEnableGameController = true;
}

FString UConfigurationDialog::ViewConfig()
{
    FString strConfig;
#if !WITH_EDITOR
        strConfig = Configuration::getConfiguration().c_str();
#endif
    return strConfig;
}

void UConfigurationDialog::SetPermissionViewOn(bool viewOn) {
	Configuration::setHivePermissionViewOn(viewOn);
}


FString UConfigurationDialog::GetPermissionViewData() {
	FString langCode;
	mStrLanguage.Split(TEXT(","), nullptr, &langCode);
	HIVELanguage targetLanguage = HIVELanguage::HIVELanguageEN;
	if(langCode.Equals("en")) {
		targetLanguage = HIVELanguage::HIVELanguageEN;
	}
	else if(langCode.Equals("de")) {
		targetLanguage = HIVELanguage::HIVELanguageDE;
	}
	else if(langCode.Equals("es")) {
		targetLanguage = HIVELanguage::HIVELanguageES;
	}
	else if(langCode.Equals("id")) {
		targetLanguage = HIVELanguage::HIVELanguageID;
	}
	else if(langCode.Equals("it")) {
		targetLanguage = HIVELanguage::HIVELanguageIT;
	}
	else if(langCode.Equals("ja")) {
		targetLanguage = HIVELanguage::HIVELanguageJA;
	}
	else if(langCode.Equals("ko")) {
		targetLanguage = HIVELanguage::HIVELanguageKO;
	}
	else if(langCode.Equals("pt")) {
		targetLanguage = HIVELanguage::HIVELanguagePT;
	}
	else if(langCode.Equals("ru")) {
		targetLanguage = HIVELanguage::HIVELanguageRU;
	}
	else if(langCode.Equals("th")) {
		targetLanguage = HIVELanguage::HIVELanguageTH;
	}
	else if(langCode.Equals("tr")) {
		targetLanguage = HIVELanguage::HIVELanguageTR;
	}
	else if(langCode.Equals("vi")) {
		targetLanguage = HIVELanguage::HIVELanguageVI;
	}
	else if(langCode.Equals("zh-Hans")) {
		targetLanguage = HIVELanguage::HIVELanguageZHS;
	}
	else if(langCode.Equals("zh-Hant")) {
		targetLanguage = HIVELanguage::HIVELanguageZHT;
	}
    else if(langCode.Equals("ar")) {
        targetLanguage = HIVELanguage::HIVELanguageAR;
    }
	
	PermissionViewData permissionData = Configuration::getPermissionViewData(targetLanguage);
	FString permissionStringData;

//	permissionData.permissions
//	permissionData.commons

	permissionStringData.Append(TEXT("\n===PERMISSIONS===\n"));
	for(auto permission = permissionData.permissions.begin() ; permission != permissionData.permissions.end(); permission++)
	{
		PermissionViewUnitData permissionObj = (PermissionViewUnitData)*permission;
		permissionStringData.Append(FUTF8ToTCHAR(permissionObj.title.c_str()).Get());
		permissionStringData.Append(TEXT("\n"));
		permissionStringData.Append(FUTF8ToTCHAR(permissionObj.contents.c_str()).Get());
		permissionStringData.Append(TEXT("\n"));
		permissionStringData.Append(FUTF8ToTCHAR(permissionObj.nativePermissionName.c_str()).Get());
		permissionStringData.Append(TEXT("\n"));
	}
	permissionStringData.Append(TEXT("\n===COMMONS===\n"));
	for(auto permission = permissionData.commons.begin() ; permission != permissionData.commons.end(); permission++)
	{
		PermissionViewUnitData permissionObj = (PermissionViewUnitData)*permission;
		permissionStringData.Append(FUTF8ToTCHAR(permissionObj.title.c_str()).Get());
		permissionStringData.Append(TEXT("\n"));
		permissionStringData.Append(FUTF8ToTCHAR(permissionObj.contents.c_str()).Get());
		permissionStringData.Append(TEXT("\n"));
		permissionStringData.Append(FUTF8ToTCHAR(permissionObj.nativePermissionName.c_str()).Get());
		permissionStringData.Append(TEXT("\n"));
	}
	permissionStringData.Append(TEXT("\n===DESCRIPTION===\n"));
	permissionStringData.Append(FUTF8ToTCHAR(permissionData.contents.c_str()).Get());
	permissionStringData.Append(TEXT("\n"));
	
	return permissionStringData;
	
}

FString UConfigurationDialog::GetHiveTimeZone() {
    return FHiveConfiguration::GetHiveTimeZone();
}

FString UConfigurationDialog::GetHiveCountry() {
    return FHiveConfiguration::GetHiveCountry();
}

FString UConfigurationDialog::GetChannel() {
    return FHiveConfiguration::GetChannel();
}

void UConfigurationDialog::SetChannel(FString channel) {
    FHiveConfiguration::SetChannel(channel);
}

void UConfigurationDialog::SetMarket(FString market) {
    FHiveConfiguration::SetMarket(market);
}

void UConfigurationDialog::SetHiveTheme(FString hiveThemeType)
{
    EHiveThemeType Type = EHiveThemeType::Light;
    if(hiveThemeType.Equals(TEXT("hiveDark")))
    {
        Type = EHiveThemeType::Dark;
    }
    
    FHiveConfiguration::SetHiveTheme(Type);
}

void UConfigurationDialog::SetHiveOrientation(FString hiveOrientation)
{
    FHiveConfiguration::SetHiveOrientation(hiveOrientation);
}

void UConfigurationDialog::SetHiveCommunityUrl(FString url)
{
    
}

void UConfigurationDialog::SetLogFilter(FString strLogType)
{
    EHiveLogType LogType = EHiveLogType::None;
    if(strLogType.Equals(TEXT("Debug")))
    {
        LogType = EHiveLogType::Debug;
    }
    else if(strLogType.Equals(TEXT("Info")))
    {
        LogType = EHiveLogType::Info;
    }
    else if(strLogType.Equals(TEXT("Warning")))
    {
        LogType = EHiveLogType::Warning;
    }
    else if(strLogType.Equals(TEXT("Error")))
    {
        LogType = EHiveLogType::Error;
    }
    else if(strLogType.Equals(TEXT("None")))
    {
        LogType = EHiveLogType::None;
    }
    
    FHiveLogFilter LogFilter = FHiveLogFilter();
    LogFilter.CoreLog = LogType;
    LogFilter.ServiceLog = LogType;
    FHiveLogger::SetLogFilter(LogFilter);
}
void UConfigurationDialog::SetCertificationKey(FString certificationKey)
{
    FHiveConfiguration::SetHiveCertificationKey(certificationKey);
}

void UConfigurationDialog::SetCrossPromotionBannerShowing(bool showing)
{
    FHiveConfiguration::SetCrossPromotionBannerShowing(showing);
}

//write file
//    IFileHandle *configFileHandle = PlatformFile.OpenWrite(*SavedConfigFile, true, true);
//
//    if(configFileHandle)
//    {
//        FString Guid = FString(
//                               TEXT("// This file is written to disk\n")
//                               TEXT("// GUID = "))
//        + FGuid::NewGuid().ToString();
//
//        configFileHandle->Write((const uint8*)TCHAR_TO_ANSI(*Guid), Guid.Len());
//        delete configFileHandle;
//    }
//    // Directory Exists?
//    if (!PlatformFile.DirectoryExists(*TestDir))
//    {
//        PlatformFile.CreateDirectory(*TestDir);
//        UE_LOG(LogTemp, Warning, TEXT("UConfigurationDialog Create Test Dir 1" ));
//        if (!PlatformFile.DirectoryExists(*TestDir))
//        {
//            UE_LOG(LogTemp, Warning, TEXT("UConfigurationDialog create fail 1") );
//            //            return false;
//            //~~~~~~~~~~~~~~
//        }
//    }


void UConfigurationDialog::HIVEConfigurationGetMetaData(FString key, FString forceReload) 
{
    bool bForceReload = forceReload.ToBool();
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::Configuration ,TEXT("=== GetMetaData(%s, %s) ==="), *key, *forceReload);
    FHiveConfiguration::GetMetaData(key, bForceReload, FHiveConfigurationOnMetaDataDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FString& Value) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::Configuration ,TEXT("%sValue = %s"), *(Result.ToString()), *Value);
    }));
}   

void UConfigurationDialog::GetAgeGateU16Agree()
{
    const bool result = FHiveConfiguration::GetAgeGateU16Agree();

    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::Configuration ,TEXT("=== Get Age Gate U16 Agree() ==="));
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::Configuration, result ? TEXT("true") : TEXT("false"));
}

void UConfigurationDialog::GetLegalGuardianConsentMode()
{
    const bool result = FHiveConfiguration::GetLegalGuardianConsentMode();

    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::Configuration ,TEXT("=== Get Legal Guardian Consent Mode() ==="));
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::Configuration, result ? TEXT("true") : TEXT("false"));
}

void UConfigurationDialog::GetCrossPromotionBannerShowing()
{
    const bool result = FHiveConfiguration::GetCrossPromotionBannerShowing();

    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::Configuration ,TEXT("=== Get Cross Promotion Banner Showing() ==="));
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::Configuration, result ? TEXT("true") : TEXT("false"));
}


void UConfigurationDialog::SetChatConnectionTimeout(FString Seconds)
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::Configuration, TEXT("=== SetChatConnectionTimeout(%s) ==="), *Seconds);
    int32 _Seconds = FCString::Atoi(*Seconds);
    FHiveConfiguration::SetChatConnectionTimeout(_Seconds);
}


void UConfigurationDialog::GetChatConnectionTimeout()
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::Configuration, TEXT("=== GetChatConnectionTimeout() ==="));
    int32 _Seconds = FHiveConfiguration::GetChatConnectionTimeout();
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::Configuration, TEXT("Result = %d"), _Seconds);
}

FString UConfigurationDialog::GetGameLanguage()
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::Configuration, TEXT("=== GetGameLanguage() ==="));
    FString GameLanguage = FHiveConfiguration::GetGameLanguage();
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::Configuration, TEXT("GameLanguage = %s"), *GameLanguage);

    return GameLanguage;
}