/* Copyright © 2024 Com2uS Platform Corp. All Rights Reserved. */
#include "IAPV4TestView.h"
#include "HIVESDKV4Tester.h"
#include "HIVESDKV4TesterGameMode.h"
#include "HiveTestUtils.h"

#if PLATFORM_IOS
#import <StoreKit/StoreKit.h>
#include <UIKit/UIKit.h>
#endif

#define IAP4RECIPTFILENAME "iapv4Recipts"

TArray<FHiveIAPV4Receipt> ReceiptArray;

void UIAPV4TestView::PrintFHiveIAPV4Product(const FHiveIAPV4Product& Product)
{
    FString Message = TEXT("[FHiveIAPV4Product]\n");
    Message += FString::Printf(TEXT("ProductType = %s\n"), *(Product.ProductType));
    Message += FString::Printf(TEXT("MarketPid = %s\n"), *(Product.MarketPid));
    Message += FString::Printf(TEXT("Currency = %s\n"), *(Product.Currency));
    Message += FString::Printf(TEXT("Price = %f\n"), Product.Price);
    Message += FString::Printf(TEXT("DisplayPrice = %s\n"), *(Product.DisplayPrice));
    Message += FString::Printf(TEXT("Title = %s\n"), *(Product.Title));
    Message += FString::Printf(TEXT("ProductDescription = %s\n"), *(Product.ProductDescription));
    Message += FString::Printf(TEXT("OriginalMarketJson = %s\n"), *(Product.OriginalMarketJson));
    Message += FString::Printf(TEXT("DisplayOriginalPrice = %s\n"), *(Product.DisplayOriginalPrice));
    Message += FString::Printf(TEXT("OriginalPrice = %f\n"), Product.OriginalPrice);
    Message += FString::Printf(TEXT("IconUrl = %s\n"), *(Product.IconUrl));
    Message += FString::Printf(TEXT("CoinsReward = %d\n"), Product.CoinsReward);
    
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4, TEXT("%s"), *(Message));
}

FString UIAPV4TestView::DecodeBase64(const FString& BypassInfo)
{
    TArray<uint8> DecodedBytes;

    if (FBase64::Decode(BypassInfo, DecodedBytes))
    {
        return FString(UTF8_TO_TCHAR(reinterpret_cast<const char*>(DecodedBytes.GetData())));
    }

    return TEXT("Decode Failed");
}

void UIAPV4TestView::PrintFHiveIAPV4Receipt(const FHiveIAPV4Receipt& Receipt)
{
    FString Message = TEXT("[FHiveIAPV4Receipt]\n");
    Message += FString::Printf(TEXT("Type = %s\n"), *(GetNameFromEHiveIAPV4Type(Receipt.Type)));
    Message += FString::Printf(TEXT("IapPayload = %s\n"), *(Receipt.IapPayload));
    Message += FString::Printf(TEXT("HiveIapReceipt = %s\n"), *(Receipt.HiveIapReceipt));
    Message += FString::Printf(TEXT("BypassInfo = %s\n"), *(Receipt.BypassInfo));
    Message += FString::Printf(TEXT("Quantity = %d\n"), Receipt.Quantity);
    Message += FString::Printf(TEXT("BypassInfo(decode) = %s\n"), *DecodeBase64(Receipt.BypassInfo));
    
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4, TEXT("%s"), *(Message));
    
    PrintFHiveIAPV4Product(Receipt.Product);
}

void UIAPV4TestView::IAPV4marketConnect()
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("=== MarketConnnect() ==="));
    
    FHiveIAPV4::MarketConnect(FHiveIAPV4OnMarketConnectDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TArray<EHiveIAPV4Type>& MarketIds) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("%s"), *(Result.ToString()));
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("[MarketIds]\n"));
        for (const auto& IAPV4Type : MarketIds)
        {
            AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("%s"), *(GetNameFromEHiveIAPV4Type(IAPV4Type)));
        }
    }));
}

void UIAPV4TestView::IAPV4GetMarketProductInfo(FString marketPidListString)
{
    TArray<FString> MarketPidList;
    marketPidListString.ParseIntoArray(MarketPidList, TEXT(","), false);
    
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("=== GetMarketProductInfo() ==="));
    FHiveIAPV4::GetMarketProductInfo(MarketPidList, FHiveIAPV4OnProductInfoDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TArray<FHiveIAPV4Product>& ProductInfoList, uint32 Balance) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("%s\nBalance = %u"), *(Result.ToString()), Balance);
        
        if (Result.IsSuccess())
        {
            TArray<FTestIAPV4Product> tempProductLists;
            for (const auto& Product: ProductInfoList)
            {
                PrintFHiveIAPV4Product(Product);
                
                FTestIAPV4Product tempIAPV4Product;
                tempIAPV4Product.v4Currency = Product.Currency;
                tempIAPV4Product.v4DisplayPrice = Product.DisplayPrice;
                tempIAPV4Product.v4MarketPid = Product.MarketPid;
                tempIAPV4Product.v4Title = Product.Title;
                tempIAPV4Product.v4Price = Product.Price;
                tempIAPV4Product.v4ProductDescription = Product.ProductDescription;
                tempIAPV4Product.v4DisplayOriginalPrice = Product.DisplayOriginalPrice;
                
                tempProductLists.Add(tempIAPV4Product);
            }
            
            mIAPV4ProductInfo = tempProductLists;
            mIAPV4Balance = Balance;
            
            if(AHIVESDKV4TesterGameMode::getGameModeInstance()->OnUpdateIAPV4ProductInfo.IsBound())
            {
                AHIVESDKV4TesterGameMode::getGameModeInstance()->OnUpdateIAPV4ProductInfo.Broadcast();
            }
        }
    }));
}

void UIAPV4TestView::IAPV4getProductInfo()
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("=== GetProductInfo() ==="));
    
    FHiveIAPV4::GetProductInfo(FHiveIAPV4OnProductInfoDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TArray<FHiveIAPV4Product>& ProductInfoList, uint32 Balance) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("%s\nBalance = %u"), *(Result.ToString()), Balance);
        
        if (Result.IsSuccess())
        {
            TArray<FTestIAPV4Product> tempProductLists;
            for (const auto& Product: ProductInfoList)
            {
                PrintFHiveIAPV4Product(Product);
                
                FTestIAPV4Product tempIAPV4Product;
                tempIAPV4Product.v4Currency = Product.Currency;
                tempIAPV4Product.v4DisplayPrice = Product.DisplayPrice;
                tempIAPV4Product.v4MarketPid = Product.MarketPid;
                tempIAPV4Product.v4Title = Product.Title;
                tempIAPV4Product.v4Price = Product.Price;
                tempIAPV4Product.v4ProductDescription = Product.ProductDescription;
                tempIAPV4Product.v4DisplayOriginalPrice = Product.DisplayOriginalPrice;
                
                tempProductLists.Add(tempIAPV4Product);
            }
            
            mIAPV4ProductInfo = tempProductLists;
            mIAPV4Balance = Balance;
            
            if(AHIVESDKV4TesterGameMode::getGameModeInstance()->OnUpdateIAPV4ProductInfo.IsBound())
            {
                AHIVESDKV4TesterGameMode::getGameModeInstance()->OnUpdateIAPV4ProductInfo.Broadcast();
            }
        }
    }));
}

void UIAPV4TestView:: IAPV4getSubscriptionProductInfo()
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("=== GetSubscriptionProductInfo() ==="));
    
    FHiveIAPV4::GetSubscriptionProductInfo(FHiveIAPV4OnProductInfoDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TArray<FHiveIAPV4Product>& ProductInfoList, uint32 Balance) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("%s\nBalance = %u"), *(Result.ToString()), Balance);
        
        if (Result.IsSuccess())
        {
            TArray<FTestIAPV4Product> tempProductLists;
            for (const auto& Product: ProductInfoList)
            {
                PrintFHiveIAPV4Product(Product);
                
                FTestIAPV4Product tempIAPV4Product;
                tempIAPV4Product.v4Currency = Product.Currency;
                tempIAPV4Product.v4DisplayPrice = Product.DisplayPrice;
                tempIAPV4Product.v4MarketPid = Product.MarketPid;
                tempIAPV4Product.v4Title = Product.Title;
                tempIAPV4Product.v4Price = Product.Price;
                tempIAPV4Product.v4ProductDescription = Product.ProductDescription;
                tempIAPV4Product.v4DisplayOriginalPrice = Product.DisplayOriginalPrice;
                
                tempProductLists.Add(tempIAPV4Product);
            }
            
            mIAPV4ProductInfo = tempProductLists;
            mIAPV4Balance = Balance;
            
            if(AHIVESDKV4TesterGameMode::getGameModeInstance()->OnUpdateIAPV4SubscriptionProductInfo.IsBound())
            {
                AHIVESDKV4TesterGameMode::getGameModeInstance()->OnUpdateIAPV4SubscriptionProductInfo.Broadcast();
            }
        }
    }));
}

void UIAPV4TestView::IAPV4purchase(FString strMarketPid, FString strIapPayload, FString quantity) {
    
    quantity.RemoveSpacesInline();
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("Quantity: %s"), *quantity);
    if (quantity.IsEmpty()) {
        purchase(strMarketPid, strIapPayload);
    } else {
        purchaseQuantity(strMarketPid, strIapPayload, quantity);
    }
}

void UIAPV4TestView::purchase(FString strMarketPid, FString strIapPayload) {
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("=== Purchase(%s, %s) ==="), *strMarketPid, *strIapPayload);
    FHiveIAPV4::Purchase(strMarketPid, strIapPayload, FHiveIAPV4OnPurchaseDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FHiveIAPV4Receipt& Receipt){
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("%s"), *(Result.ToString()));
        
        if (Result.IsSuccess())
        {
            if (Result.ErrorCode == FHiveResultAPI::EErrorCode::ITEM_PENDING)
            {
                // When in the ITEM_PENDING state, the receipt is null. do nothing.
            }
            else if (Result.ErrorCode == FHiveResultAPI::EErrorCode::IAPSUCCESS)
            {
                // Hive Store Success. Receipt not exist.
            }
            else
            {
                PrintFHiveIAPV4Receipt(Receipt);
                ReceiptArray.Push(Receipt);
                
                if(verifyReceiptListener.IsBound())
                {
                    verifyReceiptListener.Unbind();
                }
                verifyReceiptListener.BindUObject(this, &UIAPV4TestView::gameVerifyReceiptListener);
                TestAppGameNetwork::verifyReceipt(verifyReceiptListener, Receipt.BypassInfo);
            }
        }
    }));
}

void UIAPV4TestView::purchaseQuantity(FString strMarketPid, FString strIapPayload, FString quantity) {
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("=== Purchase(%s, %s, %s) ==="), *strMarketPid, *strIapPayload, *quantity);
    
    int quantityInt = FCString::Atoi(*quantity);
    
    FHiveIAPV4::Purchase(strMarketPid, strIapPayload, quantityInt, FHiveIAPV4OnPurchaseDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FHiveIAPV4Receipt& Receipt){
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("%s"), *(Result.ToString()));
        
        if (Result.IsSuccess())
        {
            if (Result.ErrorCode == FHiveResultAPI::EErrorCode::ITEM_PENDING)
            {
                // When in the ITEM_PENDING state, the receipt is null. do nothing.
            }
            else if (Result.ErrorCode == FHiveResultAPI::EErrorCode::IAPSUCCESS)
            {
                // Hive Store Success. Receipt not exist.
            }
            else
            {
                PrintFHiveIAPV4Receipt(Receipt);
                ReceiptArray.Push(Receipt);
                
                if(verifyReceiptListener.IsBound())
                {
                    verifyReceiptListener.Unbind();
                }
                verifyReceiptListener.BindUObject(this, &UIAPV4TestView::gameVerifyReceiptListener);
                TestAppGameNetwork::verifyReceipt(verifyReceiptListener, Receipt.BypassInfo);
            }
        }
    }));
}

void UIAPV4TestView:: IAPV4purchaseSubscription(FString strMarketPid, FString strOldMarketPid, FString strIapPayload)
{
    FHiveIAPV4PurchaseParam Param;
    Param.MarketPid = strMarketPid;
    Param.IapPayload = strIapPayload;
    Param.OldMarketPid = strOldMarketPid;
    
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4, TEXT("=== PurchaseSubscription() ==="));
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4, TEXT("{MarketPid: %s, IapPayload: %s, OldMarketPid: %s}"), *strMarketPid, *strIapPayload, *strOldMarketPid);
    FHiveIAPV4::PurchaseSubscription(Param, FHiveIAPV4OnPurchaseDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FHiveIAPV4Receipt& Receipt) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("%s"), *(Result.ToString()));
        
        if (Result.IsSuccess())
        {
            PrintFHiveIAPV4Receipt(Receipt);
            ReceiptArray.Push(Receipt);
            
            if(verifyReceiptListener.IsBound())
            {
                verifyReceiptListener.Unbind();
            }
            verifyReceiptListener.BindUObject(this, &UIAPV4TestView::gameVerifyReceiptListener);
            TestAppGameNetwork::verifyReceipt(verifyReceiptListener, Receipt.BypassInfo, "subscription");
        }
    }));
}

void UIAPV4TestView::IAPV4restore() {
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4, TEXT("=== Restore() ==="));
    
    FHiveIAPV4::Restore(FHiveIAPV4OnRestoreDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TArray<FHiveIAPV4Receipt>& IAPV4ReceiptList) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("%s"), *(Result.ToString()));
        
        if (Result.IsSuccess())
        {
            //bind check before receipt loop
            if(verifyReceiptListener.IsBound())
            {
                verifyReceiptListener.Unbind();
            }
            verifyReceiptListener.BindUObject(this, &UIAPV4TestView::gameVerifyReceiptListener);
            
            ReceiptArray.Empty();

            for (const auto& Receipt : IAPV4ReceiptList)
            {
                PrintFHiveIAPV4Receipt(Receipt);
                ReceiptArray.Push(Receipt);

                TestAppGameNetwork::verifyReceiptWithLogin(verifyReceiptListener, Receipt.BypassInfo);
            }
        }
    }));
}

void UIAPV4TestView::IAPV4restoreSubscription()
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4, TEXT("=== RestoreSubscription() ==="));
    
    FHiveIAPV4::RestoreSubscription(FHiveIAPV4OnRestoreDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TArray<FHiveIAPV4Receipt>& IAPV4ReceiptList) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("%s"), *(Result.ToString()));
        
        if (Result.IsSuccess())
        {
            //bind check before receipt loop
            if(verifyReceiptListener.IsBound())
            {
                verifyReceiptListener.Unbind();
            }
            verifyReceiptListener.BindUObject(this, &UIAPV4TestView::gameVerifyReceiptListener);
            
            ReceiptArray.Empty();
            
            for (const auto& Receipt : IAPV4ReceiptList)
            {
                PrintFHiveIAPV4Receipt(Receipt);
                ReceiptArray.Push(Receipt);
                TestAppGameNetwork::verifyReceipt(verifyReceiptListener, Receipt.BypassInfo, "subscription");
            }
        }
    }));
}

void UIAPV4TestView::IAPV4showMarketSelection()
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("=== ShowMarketSelection() ==="));
    
    FHiveIAPV4::ShowMarketSelection(FIAPV4OnMarketSelectionDelegate::CreateLambda([this](const FHiveResultAPI& Result, const TArray<EHiveIAPV4Type>& MarketIds) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("%s"), *(Result.ToString()));
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("[MarketIds]\n"));
        for (const auto& IAPV4Type : MarketIds)
        {
            AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("%s"), *(GetNameFromEHiveIAPV4Type(IAPV4Type)));
        }
    }));
}

void UIAPV4TestView::IAPV4getBalanceInfo() {
    
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("=== GetBalanceInfo() ==="));
    
    FHiveIAPV4::GetBalanceInfo(FHiveIAPV4OnBalanceDelegate::CreateLambda([this](const FHiveResultAPI& Result, int32 balance) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("%sBalance = %d"), *(Result.ToString()), balance);
    }));
}
                    

void UIAPV4TestView::IAPV4getAccountUuid()
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("=== GetAccountUuid() ==="));
    
    FString AccountUuid = FHiveIAPV4::GetAccountUuid();
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("AccountUuid = %s"), *AccountUuid);
}
                    

void UIAPV4TestView::IAPV4showCharge() {
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("=== ShowCharge() ==="));
    
    FHiveIAPV4::ShowCharge(FHiveIAPV4OnBalanceDelegate::CreateLambda([this](const FHiveResultAPI& Result, int32 balance) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("%sBalance = %d"), *(Result.ToString()), balance);
    }));
}

void UIAPV4TestView::IAPV4checkPromotePurchase()
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("=== CheckPromotePurchase() ==="));
    
    FHiveIAPV4::CheckPromotePurchase(FHiveIAPV4OnCheckPromotePurchaseDelegate::CreateLambda([=](const FHiveResultAPI& Result, const FString& MarketPid) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("%sMarketPid = %s"), *(Result.ToString()), *MarketPid);
    }));
}

void UIAPV4TestView::gameVerifyReceiptListener(bool result, ITestAppGameNetworkResponse response)
{
    if(result)
    {
        FHiveIAPV4Receipt& LastReceipt = ReceiptArray.Last();

        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("IAPV4 Success Verify [GameServer]\n"));
		
		FString reponseProductId = response.resultData->GetStringField(TEXT("product_id"));
        FString reciptMarketPId = LastReceipt.Product.MarketPid;

        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("productId: %s, marketPid: %s"), *reponseProductId, *reciptMarketPId);
        
        //if (reponseProductId == reciptMarketPId)
        //{
        //    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4, TEXT("=== TransactionFinish() ==="));
        //    FHiveIAPV4::TransactionFinish(reciptMarketPId, FHiveIAPV4OnTransactionFinishDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FString& MarketPid) {
        //        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("%s\nMarketPid = %s"), *(Result.ToString()), *MarketPid);
        //    }));
        //}
    }
    else
    {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("IAPV4 Fail Verify [GameServer]\n code : %s\n message : %s\n"), *response.resultCode, *response.resultMsg);
    }
}

void UIAPV4TestView::IAPV4GetConsumeInfoAgreement()
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4, TEXT("=== GetConsumeInfoAgreement() ==="));
    FHiveIAPV4::GetConsumeInfoAgreement(
        FHiveIAPV4OnGetConsumeInfoAgreementDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FHiveIAPV4ConsumeInfoUserAgreement& Info) {
            AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4, TEXT("%s\nDidAgree = %s\nFirstAnsweredDate = %lld\nLastAnsweredDate = %lld"),
                *(Result.ToString()),
                Info.DidAgree ? TEXT("TRUE") : TEXT("FALSE"),
                Info.FirstAnsweredDate,
                Info.LastAnsweredDate);
    }));
}

void UIAPV4TestView::IAPV4ShowConsumeInfoAgreement()
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4, TEXT("=== ShowConsumeInfoAgreement() ==="));
    FHiveIAPV4::ShowConsumeInfoAgreement(FHiveIAPV4OnShowConsumeInfoAgreementDelegate::CreateLambda([this](const FHiveResultAPI& Result, bool DidAgree) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4, TEXT("%s\nDidAgree = %s\n"),
            *(Result.ToString()),
            DidAgree ? TEXT("TRUE") : TEXT("FALSE"));
    }));
}
     
void UIAPV4TestView::IAPV4FinishTransactions()
{
    if (ReceiptArray.Num() > 0)
    {
        FHiveIAPV4Receipt& LastReceipt = ReceiptArray.Last();
        FString MarketPid = LastReceipt.Product.MarketPid;

        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4, TEXT("=== TransactionFinish() ==="));
        FHiveIAPV4::TransactionFinish(MarketPid, FHiveIAPV4OnTransactionFinishDelegate::CreateLambda([this](const FHiveResultAPI& Result, const FString& MarketPid) {
            AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4, TEXT("%s\nMarketPid = %s"), *(Result.ToString()), *MarketPid);
            if (Result.IsSuccess()) {
                ReceiptArray.Pop();
            }

            }));
    }
    else
    {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4, TEXT("=== TransactionFinish() failed: No receipts available in ReceiptArray ==="));
    }
}

void UIAPV4TestView::IAPV4FinishAllTransactions()
{
    if (ReceiptArray.Num() > 0)
    {
        TArray<FString> MarketPidList;
        for (const auto& Receipt : ReceiptArray)
        {
            MarketPidList.Emplace(Receipt.Product.MarketPid);
        }

        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4, TEXT("=== TransactionMultiFinish() ==="));
        FHiveIAPV4::TransactionMultiFinish(MarketPidList, FHiveIAPV4OnTransactionMultiFinishDelegate::CreateLambda([this](const TArray<FHiveResultAPI>& ResultList, const TArray<FString>& MarketPidList) {
            for (int32 i = ResultList.Num() - 1; i >= 0; --i)
            {
                AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4, TEXT("%s\nMarketPid = %s"), *(ResultList[i].ToString()), *(MarketPidList[i]));

                if (ResultList[i].IsSuccess())
                {
                    ReceiptArray.RemoveAt(i);
                }
            }
        }));

    }
    else
    {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4, TEXT("=== TransactionMultiFinish() failed: No receipts available in ReceiptArray ==="));
    }
}


void UIAPV4TestView::IAPV4getSelectedMarket()
{
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("=== GetSelectedMarket() ==="));
    TOptional<EHiveIAPV4Type> IAPV4Type = FHiveIAPV4::GetSelectedMarket();
    
    if (IAPV4Type.IsSet())
    {
        FString StrIAPV4Type = GetNameFromEHiveIAPV4Type(IAPV4Type.GetValue());
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("Selected IAPV4Type = %s"), *StrIAPV4Type);
    }
    else
    {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("Selected IAPV4Type = Empty"));
    }
}

void UIAPV4TestView::IAPV4ShowInAppMessages(FString categoryId)
{
    int iCategoryId = FCString::Atoi(*categoryId);
    
    AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("=== GetSelectedMarket() ==="));
    FHiveIAPV4::ShowInAppMessages(iCategoryId, FHiveIAPV4OnInAppMessagesDelegate::CreateLambda([this](int32 ResponseCode) {
        AHIVESDKV4TesterGameMode::getGameModeInstance()->appendLogString(ELogType::IAPV4 ,TEXT("ResponseCode = %d"), ResponseCode);
    }));
}
