/* Copyright © 2025 Com2uS Platform Corp. All Rights Reserved. */

#include "MessageSelectView.h"
#include "ChatDataManager.h"
#include "HiveChat.h"

#define MESSAGE_SIZE 50

void UMessageSelectView::InitializeMessageSelect()
{
    AddDebugLog(TEXT("[Init] InitializeMessageSelect started"));
    
    // ChatDataManager에서 ChannelId 가져와서 표시
    UChatDataManager* ChatManager = UChatDataManager::GetInstance();
    if (ChatManager)
    {
        // MessageSelectView용 ChannelId를 먼저 확인, 없으면 ChannelQuerySelectChannelId 사용
        FString ChannelId = ChatManager->GetMessageSelectViewChannelId();
        AddDebugLog(FString::Printf(TEXT("[Init] MessageSelectViewChannelId: %s"), *ChannelId));
        
        if (ChannelId.IsEmpty())
        {
            ChannelId = ChatManager->GetChannelQuerySelectChannelId();
            AddDebugLog(FString::Printf(TEXT("[Init] Using ChannelQuerySelectChannelId: %s"), *ChannelId));
        }
        
        if (!ChannelId.IsEmpty())
        {
            SetChannelIdText(ChannelId);
            AddDebugLog(FString::Printf(TEXT("[Init] Channel ID set: %s"), *ChannelId));
        }
        else
        {
            AddDebugLog(TEXT("[Init] WARNING: No Channel ID found!"));
        }
    }
    else
    {
        AddDebugLog(TEXT("[Init] ERROR: ChatDataManager is null!"));
    }
    
    // 초기 메시지 로드 - 최신 메시지부터 이전 50개 조회 (PREV)
    AddDebugLog(TEXT("[Init] Loading initial messages with PREV..."));
    LoadMessages("", true);
}

void UMessageSelectView::PopulateMessageList()
{
    AddDebugLog(FString::Printf(TEXT("[Populate] PopulateMessageList called with %d messages"), CachedMessages.Num()));
    
    // 표시할 메시지 개수 설정
    int32 MessageCount = FMath::Min(CachedMessages.Num(), MESSAGE_SIZE);
    SetMessageListCount(MessageCount);
    AddDebugLog(FString::Printf(TEXT("[Populate] SetMessageListCount called with Count=%d (CachedMessages=%d, MESSAGE_SIZE=%d)"), 
        MessageCount, CachedMessages.Num(), MESSAGE_SIZE));
    
    // 메시지 아이템 업데이트
    if (CachedMessages.Num() > 0)
    {
        AddDebugLog(TEXT("[Populate] Updating message items..."));
        for (int32 i = 0; i < CachedMessages.Num(); i++)
        {
            const FHiveChannelMessage& Message = CachedMessages[i];
            FString TimeStamp = FDateTime::FromUnixTimestamp(Message.timestampMillis / 1000).ToString(TEXT("%Y-%m-%d %H:%M:%S"));
            FString ExtraData = Message.extraData;
            FString ExtraDataDisplay = ExtraData.IsEmpty() ? TEXT("empty") : ExtraData;
            
            AddDebugLog(FString::Printf(TEXT("[Populate] UpdateMessageItem[%d]: ID=%s, From=%lld, Extra=%s, Msg=%s"), 
                i, *Message.messageId, Message.from, *ExtraDataDisplay, *Message.message));
            
            UpdateMessageItem(i, Message.messageId, Message.message, FString::Printf(TEXT("%lld"), Message.from), TimeStamp, ExtraData);
        }
        AddDebugLog(TEXT("[Populate] All message items updated"));
    }
    else
    {
        AddDebugLog(TEXT("[Populate] No messages to display"));
    }
    
    // 페이지네이션 버튼 업데이트
    bool bCanPrev = CanLoadPreviousPage();
    bool bCanNext = CanLoadNextPage();
    AddDebugLog(FString::Printf(TEXT("[Populate] Pagination - CanPrev: %s, CanNext: %s"), 
        bCanPrev ? TEXT("true") : TEXT("false"), bCanNext ? TEXT("true") : TEXT("false")));
    UpdatePaginationButtons(bCanPrev, bCanNext);
    
    // 로딩 인디케이터 숨기기
    HideLoadingIndicator();
    AddDebugLog(TEXT("[Populate] PopulateMessageList completed"));
}

void UMessageSelectView::OnSelectButtonClicked()
{
    AddDebugLog(FString::Printf(TEXT("[Click] Select button clicked. SelectedMessageId: %s"), 
        *SelectedMessageId));
    
    if (!SelectedMessageId.IsEmpty())
    {
        OnMessageSelectedDelegate.Broadcast(SelectedMessageId);
        AddDebugLog(TEXT("[Click] Broadcasting OnMessageSelected delegate"));
    }
    else
    {
        AddDebugLog(TEXT("[Click] WARNING: No message selected!"));
    }
    RemoveFromParent();
}

void UMessageSelectView::OnCancelButtonClicked()
{
    AddDebugLog(TEXT("[Click] Cancel/Back button clicked"));
    OnCancelDelegate.Broadcast();
    RemoveFromParent();
}

void UMessageSelectView::OnMessageItemClicked(const FString& MessageId)
{
    SelectedMessageId = MessageId;
    AddDebugLog(FString::Printf(TEXT("[Click] Message item clicked. MessageId: %s"), *MessageId));
}

FString UMessageSelectView::GetMessageIdByIndex(int32 Index) const
{
    if (Index >= 0 && Index < CachedMessages.Num())
    {
        return CachedMessages[Index].messageId;
    }
    return FString();
}

void UMessageSelectView::LoadPreviousPage()
{
    AddDebugLog(FString::Printf(TEXT("[Page] LoadPreviousPage called. PrevMessageId: %s"), *PrevMessageId));
    
    if (CanLoadPreviousPage())
    {
        LoadMessages(PrevMessageId, true);
    }
    else
    {
        AddDebugLog(TEXT("[Page] Cannot load previous page - no prev message id"));
    }
}

void UMessageSelectView::LoadNextPage()
{
    AddDebugLog(FString::Printf(TEXT("[Page] LoadNextPage called. NextMessageId: %s"), *NextMessageId));
    
    if (CanLoadNextPage())
    {
        LoadMessages(NextMessageId, false);
    }
    else
    {
        AddDebugLog(TEXT("[Page] Cannot load next page - no next message id"));
    }
}

void UMessageSelectView::OnMessageButtonClicked(int32 ButtonIndex)
{
    FString MessageId = GetMessageIdByIndex(ButtonIndex);
    AddDebugLog(FString::Printf(TEXT("[Click] Message button %d clicked. MessageId: %s"), 
        ButtonIndex, *MessageId));
    
    if (!MessageId.IsEmpty())
    {
        OnMessageItemClicked(MessageId);
        OnSelectButtonClicked();
    }
    else
    {
        AddDebugLog(FString::Printf(TEXT("[Click] ERROR: No message at index %d"), ButtonIndex));
    }
}

void UMessageSelectView::LoadMessages(const FString& MessageId, bool bLoadPrevious)
{
    AddDebugLog(FString::Printf(TEXT("[Load] LoadMessages called. MessageId: %s, LoadPrevious: %s"), 
        *MessageId, bLoadPrevious ? TEXT("true") : TEXT("false")));
    
    ShowLoadingIndicator();
    CachedMessages.Empty();
    
    UChatDataManager* ChatManager = UChatDataManager::GetInstance();
    if (!ChatManager)
    {
        AddDebugLog(TEXT("[Load] ERROR: ChatDataManager not found!"));
        ShowError(TEXT("ChatDataManager not found"));
        return;
    }
    
    // MessageSelectView용 ChannelId를 먼저 확인, 없으면 ChannelQuerySelectChannelId 사용
    FString ChannelId = ChatManager->GetMessageSelectViewChannelId();
    if (ChannelId.IsEmpty())
    {
        ChannelId = ChatManager->GetChannelQuerySelectChannelId();
    }
    
    if (ChannelId.IsEmpty())
    {
        AddDebugLog(TEXT("[Load] ERROR: No channel selected!"));
        ShowError(TEXT("No channel selected"));
        return;
    }
    
    AddDebugLog(FString::Printf(TEXT("[Load] Using ChannelId: %s"), *ChannelId));
    
    // ChannelMessageListQueryParams 설정
    FHiveChannelMessageListQueryParams Params;
    Params.messageId = MessageId;
    if (bLoadPrevious)
    {
        Params.prevSize = MESSAGE_SIZE;
    }
    else
    {
        Params.nextSize = MESSAGE_SIZE;
    }
    Params.order = "DESC";
    
    AddDebugLog(FString::Printf(TEXT("[Load] Query params - MessageId: %s, PrevSize: %d, NextSize: %d, Order: %s"), 
        *Params.messageId, Params.prevSize, Params.nextSize, *Params.order));
    
    // GetChannelInfo 호출
    AddDebugLog(TEXT("[Load] Calling GetChannelInfo..."));
    FHiveChat::GetChannelInfo(ChannelId, FHiveChatOnGetChannelInfoDelegate::CreateLambda(
        [this, Params](const FHiveResultAPI& Result, const FHiveChannel& Channel, const TArray<FHiveMember>& Members)
        {
            if (!Result.IsSuccess())
            {
                FString ErrorMsg = FString::Printf(TEXT("GetChannelInfo Failed. Code: %d, Message: %s"), 
                    (int32)Result.Code, *Result.Message);
                AddDebugLog(FString::Printf(TEXT("[Load] ERROR: %s"), *ErrorMsg));
                ShowError(ErrorMsg);
                HideLoadingIndicator();
                return;
            }
            
            AddDebugLog(TEXT("[Load] GetChannelInfo success"));
            FString ChannelTypeStr = EHiveChannelTypeToString(Channel.type);
            AddDebugLog(FString::Printf(TEXT("[Load] Channel Info - ID: %s, Type: %s, Name: %s"), 
                *Channel.channelId, *ChannelTypeStr, *Channel.channelName));
            AddDebugLog(FString::Printf(TEXT("[Load] Channel Members Count: %d"), Members.Num()));
            
            // Channel 객체를 복사하여 const 문제 해결
            FHiveChannel ChannelCopy = Channel;
            
            AddDebugLog(TEXT("[Load] Calling Channel.Query..."));
            
            // Channel.Query 호출
            ChannelCopy.Query(Params, FHiveChannelQueryDelegate(
                [this](const FHiveResultAPI& QueryResult, const FHiveChannelMessageListQueryResponse& Response)
                {
                    if (!QueryResult.IsSuccess())
                    {
                        FString ErrorMsg = FString::Printf(TEXT("Query Failed. Code: %d, Message: %s"), 
                            (int32)QueryResult.Code, *QueryResult.Message);
                        AddDebugLog(FString::Printf(TEXT("[Load] ERROR: %s"), *ErrorMsg));
                        ShowError(ErrorMsg);
                        HideLoadingIndicator();
                        return;
                    }
                    
                    AddDebugLog(FString::Printf(TEXT("[Load] Query success! Messages: %d, PrevId: %s, NextId: %s"), 
                        Response.content.Num(), *Response.prevMessageId, *Response.nextMessageId));
                    
                    // 받아온 메시지 목록 상세 로그
                    if (Response.content.Num() > 0)
                    {
                        AddDebugLog(TEXT("[Load] === Message List Start ==="));
                        for (int32 i = 0; i < Response.content.Num(); i++)
                        {
                            const FHiveChannelMessage& Msg = Response.content[i];
                            FString TimeStamp = FDateTime::FromUnixTimestamp(Msg.timestampMillis / 1000).ToString(TEXT("%Y-%m-%d %H:%M:%S"));
                            FString ExtraDataStr = Msg.extraData.IsEmpty() ? TEXT("empty") : Msg.extraData;
                            
                            AddDebugLog(FString::Printf(TEXT("[Load] Msg[%d]: ID=%s, From=%lld, Time=%s, Extra=%s, Text=%s"), 
                                i, *Msg.messageId, Msg.from, *TimeStamp, *ExtraDataStr, *Msg.message));
                        }
                        AddDebugLog(TEXT("[Load] === Message List End ==="));
                    }
                    else
                    {
                        AddDebugLog(TEXT("[Load] WARNING: No messages returned from query!"));
                    }
                    
                    // 응답 데이터 저장
                    PrevMessageId = Response.prevMessageId;
                    NextMessageId = Response.nextMessageId;
                    CachedMessages = Response.content;
                    
                    // UI 업데이트 (메시지가 있든 없든 호출)
                    if (IsValid(this))
                    {
                        AddDebugLog(TEXT("[Load] Updating UI..."));
                        PopulateMessageList();
                    }
                    else
                    {
                        AddDebugLog(TEXT("[Load] ERROR: Widget is no longer valid!"));
                    }
                }));
        }));
}