/* Copyright © 2025 Com2uS Platform Corp. All Rights Reserved. */

#pragma once

#include "CoreMinimal.h"
#include "Blueprint/UserWidget.h"
#include "Components/Button.h"
#include "HiveChat.h"
#include "Types/HiveChatTypes.h"
#include "MessageSelectView.generated.h"

DECLARE_DYNAMIC_MULTICAST_DELEGATE_OneParam(FOnMessageSelected, const FString&, MessageId);
DECLARE_DYNAMIC_MULTICAST_DELEGATE(FOnMessageSelectCancel);

/**
 * 메시지 선택 UI 위젯
 */
UCLASS()
class HIVESDKV4TESTER_API UMessageSelectView : public UUserWidget
{
    GENERATED_BODY()

public:
    // 초기화
    UFUNCTION(BlueprintCallable, Category = "HIVE SDK V4|Chat|MessageSelect")
    void InitializeMessageSelect();

    // 메시지 목록 표시 - FHiveChannelMessage 직접 사용 대신 내부 저장
    UFUNCTION(BlueprintCallable, Category = "HIVE SDK V4|Chat|MessageSelect")
    void PopulateMessageList();

    // 선택된 메시지 ID 가져오기
    UFUNCTION(BlueprintCallable, BlueprintPure, Category = "HIVE SDK V4|Chat|MessageSelect")
    FString GetSelectedMessageId() const { return SelectedMessageId; }

    // Select 버튼 클릭
    UFUNCTION(BlueprintCallable, Category = "HIVE SDK V4|Chat|MessageSelect")
    void OnSelectButtonClicked();

    // Cancel 버튼 클릭
    UFUNCTION(BlueprintCallable, Category = "HIVE SDK V4|Chat|MessageSelect")
    void OnCancelButtonClicked();

    // 메시지 아이템 클릭
    UFUNCTION(BlueprintCallable, Category = "HIVE SDK V4|Chat|MessageSelect")
    void OnMessageItemClicked(const FString& MessageId);

    // 인덱스로 MessageId 가져오기
    UFUNCTION(BlueprintCallable, BlueprintPure, Category = "HIVE SDK V4|Chat|MessageSelect")
    FString GetMessageIdByIndex(int32 Index) const;

    // 페이지네이션
    UFUNCTION(BlueprintCallable, Category = "HIVE SDK V4|Chat|MessageSelect")
    void LoadPreviousPage();

    UFUNCTION(BlueprintCallable, Category = "HIVE SDK V4|Chat|MessageSelect")
    void LoadNextPage();

    UFUNCTION(BlueprintCallable, BlueprintPure, Category = "HIVE SDK V4|Chat|MessageSelect")
    bool CanLoadPreviousPage() const { return !PrevMessageId.IsEmpty() && PrevMessageId.Len() > 1; }

    UFUNCTION(BlueprintCallable, BlueprintPure, Category = "HIVE SDK V4|Chat|MessageSelect")
    bool CanLoadNextPage() const { return !NextMessageId.IsEmpty() && NextMessageId.Len() > 1; }

    // 델리게이트
    UPROPERTY(BlueprintAssignable, Category = "HIVE SDK V4|Chat|MessageSelect")
    FOnMessageSelected OnMessageSelectedDelegate;
    
    UPROPERTY(BlueprintAssignable, Category = "HIVE SDK V4|Chat|MessageSelect")
    FOnMessageSelectCancel OnCancelDelegate;
    
protected:
    // 메시지 버튼 클릭 핸들러
    UFUNCTION(BlueprintCallable, Category = "HIVE SDK V4|Chat|MessageSelect")
    void OnMessageButtonClicked(int32 ButtonIndex);

    // Blueprint에서 구현할 함수들
    UFUNCTION(BlueprintImplementableEvent, Category = "HIVE SDK V4|Chat|MessageSelect")
    void ClearMessageList();

    UFUNCTION(BlueprintImplementableEvent, Category = "HIVE SDK V4|Chat|MessageSelect")
    void AddMessageItem(const FString& MessageId, const FString& Message, const FString& PlayerName, const FString& TimeStamp);

    UFUNCTION(BlueprintImplementableEvent, Category = "HIVE SDK V4|Chat|MessageSelect")
    void UpdateMessageItem(int32 Index, const FString& MessageId, const FString& Message, const FString& PlayerName, const FString& TimeStamp, const FString& ExtraData);

    UFUNCTION(BlueprintImplementableEvent, Category = "HIVE SDK V4|Chat|MessageSelect")
    void SetMessageListCount(int32 Count);

    UFUNCTION(BlueprintImplementableEvent, Category = "HIVE SDK V4|Chat|MessageSelect")
    void ShowLoadingIndicator();

    UFUNCTION(BlueprintImplementableEvent, Category = "HIVE SDK V4|Chat|MessageSelect")
    void HideLoadingIndicator();

    UFUNCTION(BlueprintImplementableEvent, Category = "HIVE SDK V4|Chat|MessageSelect")
    void UpdatePaginationButtons(bool bCanLoadPrevious, bool bCanLoadNext);

    UFUNCTION(BlueprintImplementableEvent, Category = "HIVE SDK V4|Chat|MessageSelect")
    void ShowError(const FString& ErrorMessage);

    UFUNCTION(BlueprintImplementableEvent, Category = "HIVE SDK V4|Chat|MessageSelect")
    void HighlightSelectedButton(int32 ButtonIndex);

    UFUNCTION(BlueprintImplementableEvent, Category = "HIVE SDK V4|Chat|MessageSelect")
    void SetChannelIdText(const FString& ChannelId);

    UFUNCTION(BlueprintImplementableEvent, Category = "HIVE SDK V4|Chat|MessageSelect")
    void AddDebugLog(const FString& LogMessage);

private:
    UPROPERTY()
    FString SelectedMessageId;

    UPROPERTY()
    FString PrevMessageId;

    UPROPERTY()
    FString NextMessageId;

    // UPROPERTY 제거 - 직접 메모리 관리
    TArray<FHiveChannelMessage> CachedMessages;

    void LoadMessages(const FString& MessageId, bool bLoadPrevious);
};
